#! /usr/bin/env python
#
# $Id: splitter.py,v 1.1 2006/05/31 15:25:25 aehyvari Exp $
#

"""
Module handles the splitting of the problem instance into multiple parts.

It has three threads, the splitSearch, requestJob and
enforceResult. In addition it defines the job, jobList, result and
query -types
"""

import thread
import os
import sys
import time
import random
import socket
import re
# The languages
import splitlang
import cnflang

from logger import Logger


class JobList:
    """Lockable list of jobs"""
    jobLock = thread.allocate_lock()
    jobs = []

class Solution:
    """A single solution and the time it was found"""

    def __init__(self, sol, time):
        self.solution = sol
        self.time = time

class Job:
    """Implements the job interface"""

    def __init__(self):
        self.valid = 0
        self.learned = []
        self.solutions = []
        self.assumptions = []
        self.timeout = 0
        self.parentJob = 0
        self.baseProblem = 0
        self.sentToSatJM = 0
        self.sendTime = 0
        self.assumptionLevel = 0
        self.childList = []
        self.name = [""]
        self.filename = ""
        self.lock = thread.allocate_lock()

    def getFullName(self):
        """Full name is constructed from parents name and a local name"""
        if self.parentJob == self:
            return self.name
        else:
            myName = []
            for n in self.parentJob.getFullName():
                myName.append(n)
            myName.extend(self.name)
            return myName

    def toString(self):
        return " valid is " + str(self.valid) + "\n"\
               " learned is " + str(self.learned) + "\n"\
               " solutions is " + str(self.solutions) + "\n"\
               " assumptions is " + str(self.assumptions) + "\n"\
               " parentJob is " + str(self.parentJob) + "\n"\
               " baseProblem is " + str(self.baseProblem) + "\n"\
               " sentToSatJM is " + str(self.sentToSatJM) + "\n"\
               " sendTime is " + str(self.sendTime) + "\n"\
               " assumptionLevel is " + str(self.assumptionLevel) + "\n"\
               " childList is " + str(self.childList) + "\n"\
               " name is " + str(self.getFullName()) + "\n"\
               " filename is " + self.filename

    def isValid(self):

        if self.parentJob == self:
            return self.valid
        elif self.valid == 0:
            return 0
        else:
            return self.parentJob.isValid()

    def toCnf(self):
        """Construct a cnf description of the job.

        Read in the base file, add learned clauses and assumptions.
        Assumptions must not contain new variables. This might take some
        time, depending on the size of the instance"""

        file = open(self.filename)
        lines = file.readlines()
        logger.log("Start parsing of file", 2)
        cnf = cnflang.Cnf(lines)
        logger.log("Finished parsing", 2)

        assumptions = self.getassumptions()
#        print str(assumptions)
        for as in assumptions:
            cnf.addClause(as)

        learned = self.getlearned()
        for le in learned:
            cnf.addClause(le)

        return cnf.toString()

    def getassumptions(self):
        if (self.parentJob == self):
            return []
        else:
            assumptions = self.parentJob.getassumptions()[0:]
            assumptions.extend(self.assumptions)
            return assumptions

    def getassumptionsc(self):
        as = self.getassumptions()
        if as != []:
            m = max(map(lambda x: max(map(abs, x)), as))
        else:
            m = 0
        rval = "".join(map(lambda x: " ".join(map(str, x) + ["0"]) + "\n", as))
        return m, len(as), rval


    def getlearned(self):
        if (self.parentJob == self):
            return self.learned[0:]
        else:
            _learned = self.parentJob.getlearned()[0:]
            _learned.extend(self.learned)
            return _learned

    def getlearnedc(self):
        le = self.getlearned()
        if le != []:
            m = max(map(lambda x: max(map(abs, x)), le))
        else:
            m = 0
        rval = "".join(map(lambda x: " ".join(map(str, x) + ["0"]) + "\n", le))
        return m, len(le), rval

    def getmisccnf(self):
        ma, la, asc = self.getassumptionsc()
        ml, ll, lec = self.getlearnedc()
        rval = "" .join(["p cnf " + str(max(ma, ml)) + " "\
            + str(la + ll) + "\n", asc, lec])
        logger.log("Now using " + str(max(ma, ml)) + " vars", 2)
        return rval

    def getsolutions(self):
        if (self.parentJob == self):
            return self.solutions[0:]
        else:
            _solutions = self.parentJob.getsolutions()[0:]
            _solutions.extend(self.solutions)
            return _solutions

    def getsolutionsc(self):
        so = self.getsolutions()
        if so != []:
            m = max(map(max, so))
        else:
            m = 0
        rval = "".join(map(lambda x: " ".join(map(str, x) + ["0"]) + "\n", so))
        return m, len(so), rval

class Heuristic:
    """A wrapper to run the heuristic"""

    def __init__(self, logger, port):
        self.logger = logger
        HOST = ""
        PORT = port
        self.logger.log("Connecting to the heuristic server " + HOST +
            " port " + str(PORT), 1)
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # When we crash, don't wait until the TIME_WAIT
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, True)
        connection = False
        while not connection:
            try:
                s.connect((HOST, PORT))
                connection = True
            except socket.error:
                self.logger.log(str(socket.error), 1)
                self.logger.log("Connection to Heuristic failed, trying again", 0)
                time.sleep(2)
                connection = False

        self.logger.log("Connected", 1)
        self.sock = s

    def numSplits(self):
        return len(self.assumptions)

    def clear(self):
        self.logger.log("Resetting heuristics", 3)
        self.numAssumptions = 0
        self.assumptions = []

    def setInstance(self, jobCnf, name):
        self.jobname = str(name)
        self.logger.log("Sending job " + str(name), 3)
        self.logger.log("Job is \n" + str(jobCnf), 4)
        self.sock.send("# listing\n")
        self.sock.send("# problem " + str(name) + "\n")
        self.sock.send("# begin\n")
        self.sock.send(str(jobCnf))
        self.sock.send("# end\n")
        self.sock.send("# endproblem " + str(name) + "\n")
        self.sock.send("# endlisting\n")

    def setLocalInstance(self, job, name):
        self.jobname = str(name)
        self.logger.log("Sending local job " + str(name), 3)
        self.sock.send("# listing\n")
        self.sock.send("# problem " + str(name) + "\n")
        self.sock.send("# local " + job.filename + "\n")
        m1, a, as = job.getassumptionsc()
        m2, l, le = job.getlearnedc()

        clauses = a+l
        self.sock.send("# begin\n")
        self.sock.send("p cnf " + str(max(m1, m2)) + " " + str(clauses) + "\n")
        self.sock.send(as)
        self.sock.send(le)

        self.sock.send("# end\n")
        self.sock.send("# endproblem " + str(name) + "\n")
        self.sock.send("# endlisting\n")

    def getAssumptions(self, nth):
        self.logger.log("Returning assumptions " + str(nth), 3)
        return self.assumptions[nth]

    def start(self):
        self.logger.log("Starting the heuristic gathering", 3)
        self.sock.send("# start " + self.jobname + "\n")

    def stop(self):
        """Stop heuristic search and collect results

        A line saying in the beginnign # endlisting signifies the end of
        listing. We read the socket until we hit that line (and might
        read a bit more still) and then parse the string thus read"""

        self.logger.log("Stopping the heuristic gathering", 1)
        self.sock.send("# stop " + self.jobname + "\n")
        # We do reading and checking here in the wrong order, but it
        # does not matter because we can't have unread data.
        data = ""
        done = False
        while (done == False):
            line = self.sock.recv(10240)
            data = "".join([data, line])
            # The word # endlist in the beginning of a line signifies
            # the end of a listing, so then we're through.
            if (re.search("^#[ \t]*endlisting", data, re.M)):
                done = True


        rv, P = splitlang.parse('list', data)

        try:
            self.logger.log(str(P.globalsolution.keys()), 2)
            self.logger.log(self.jobname, 2)
            solutions = P.globalsolution[self.jobname]
            exec "name = " + self.jobname
            for cnfstr in solutions:
                self.logger.log(str(cnfstr), 2)
                cnf = cnflang.Cnf(cnfstr)
                enforcer.addSolution(name, cnf.clauses)

            self.logger.log("Solution found by heuristic", 1)
        except KeyError:
            self.logger.log("No solutions found by heuristic", 2)

        try:
            self.logger.log(str(P.globalsplit.keys()), 2)
            cnflist = P.globalsplit[self.jobname]
            for i in range(0, len(cnflist)):
                self.assumptions.append([])
                cnf = cnflang.Cnf(cnflist[i])
                self.assumptions[i].extend(cnf.clauses)
        except KeyError:
            self.logger.log("No splits found by heuristic", 2)

        try:
            if P.globalunsat[self.jobname]:
                exec "name = " + self.jobname
                self.logger.log("Job was found unsat by heuristic", 2)
                enforcer.invalidJob(name)
            else:
                self.logger.log("Job was not found unsat by heuristic", 2)

        except KeyError:
            self.logger.log("Job was not found unsat by heuristic", 2)

class QueryList:
    """Queries for jobs."""
    queryLock = thread.allocate_lock()
    queries = 0

class SplitSearch:
    """Divide solution space

    The problems are stored in a queue, so that the search will be a
    sort of breadth first. The main loop is splitter. Process gives some
    time for the heuristic collector to search for good subproblems,
    time depends on the need for problems in lower level (and time ends
    if the problem is found unsat or sat). Expand asks from the
    heuristic collector the subproblems"""

    queue = []

    def __init__(self, queryList, logger, heuristic, requester, minspool):
        self.logger = logger
        self.queryList = queryList
        self.heuristic = heuristic
        self.requester = requester
        self.minspool = minspool

    def process(self, _job, local):
        """_job is the job to be sent as a Job class. local is true if
        the local version of the base job can be passed to the solver
        (they share disk, thus) """
        # Do some processing...
        self.logger.log("Processing job " + str(_job.getFullName()), 2)
        # Start the heuristic collector here
        self.heuristic.clear()
        if (local):
            self.heuristic.setLocalInstance(_job, _job.getFullName())
        else:
            self.heuristic.setInstance(_job.toCnf(), _job.getFullName())
        self.heuristic.start()

        # Collect heuristics until job becomes invalid or new jobs are
        # queried from below.
        done = 0
        while done == 0:
            time.sleep(1)

            if _job.isValid() == 0:
                done = 1

            # If the spool is low, new queries are unsatisfied and we
            # need to get more. This threshold is controlled by the
            # configuration
            joblist.jobLock.acquire()
            if len(joblist.jobs) < self.minspool:
                done = 1
            joblist.jobLock.release()

        self.heuristic.stop()

    def expand(self, _job):
        if _job.isValid() == 0:
            return
        # Do the expanding thingie
        self.logger.log("Expanding job\n" + _job.toString(), 3)
        # Job is expanded only once
        assert(_job.childList == [])
        # Read the new job, append it to the breadth-first-search queue
        # and to the spool where jobs are searched from when queries
        # arrive
        for i in range(0, self.heuristic.numSplits()):
            job = Job()
            job.valid = 1
            job.assumptions = self.heuristic.getAssumptions(i)
            job.parentJob = _job
            job.assumptionLevel = _job.assumptionLevel + 1
            job.name = [i]
            job.filename = _job.filename
            job.childList = []
            _job.lock.acquire()
            _job.childList.append(job)
            _job.lock.release()
            self.queue.append(job)
            joblist.jobLock.acquire()
            joblist.jobs.append(job)
            joblist.jobLock.release()


    def splitter(self, _job, local):
        self.logger.threadnames[thread.get_ident()] = "splitter"

        self.logger.log("Searching for a solution", 0)
        self.queue.append(_job)
        while (self.queue != []):
            newJob = self.queue.pop(0)
            if newJob.isValid() == 0:
                continue
            self.process(newJob, local)
            self.expand(newJob)
        self.logger.log("Searching done", 0)

class RequestJob:
    """Listen for job requests."""

    def __init__(self, queryList, jobtat, logger):
        self.logger = logger
        self.queryList = queryList
        self.jobtat = jobtat

    def send(self, job, local):
        self.jobtat[str(job.getFullName())] = time.time()
        data = "# listing\n"
        data += "# problem " + str(job.getFullName()) + "\n"
        if (local):
            data += "# local " + str(job.filename) + "\n"
            data += "# begin\n"
            data += job.getmisccnf()
            data += "# end\n"
        else:
            data += "# begin\n"
            data += job.toCnf()
            data += "# end\n"
        data += "# endproblem " + str(job.getFullName()) + "\n"
        data += "# endlisting\n"
        self.conn.send(data)

    def listen(self, port, local):
        """Read the socket, parse requests"""

        HOST = ""
        PORT = port

        self.logger.threadnames[thread.get_ident()] = "reqlisten"

        self.logger.log("Setting up the request server on port " +\
            str(PORT), -1)
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # When we crash, don't wait until the TIME_WAIT
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, True)

        connected = False
        while not connected:
            try:
                s.bind((HOST, PORT))
                connected = True
            except socket.error:
                logger.log("Couldn't set up server on port " +\
                    str(PORT) + ". Waiting.", 0)
                time.sleep(2)

        s.listen(1)
        self.conn, self.addr = s.accept()

        self.logger.log("Request server connected by " + str(self.addr), -1)

        data = ""
        done = False
        while (True):
            self.logger.log("request: Reading the socket", 2)
            while not done:
                mo = re.search("^#[ \t]*endlisting", data, re.M)
                if (mo):
                    done = True
                else:
                    data += self.conn.recv(1024)

            rv, P = splitlang.parse('list', data)

            data = mo.string[mo.end():]

            requests = P.globalvars["reqjob"]

            while requests > 0:
                hasjobs = False
                # This is the only thread consuming the jobs, so we may
                # assume that if there were jobs previously, there will
                # be in the next step as well
                while not hasjobs:
                    joblist.jobLock.acquire()
                    if len(joblist.jobs) == 0:
                        joblist.jobLock.release()
                        time.sleep(1)
                    else:
                        hasjobs = True
                job = joblist.jobs.pop(0)
                joblist.jobLock.release()
                if job.isValid():
                    self.logger.log("Sending job " + \
                        str(job.getFullName()) + " to jm", 2)
                    self.send(job, local)
                    requests -= 1

            done = False

class EnforceResults:
    """Listen for results, update state of the jobs."""

    def __init__(self, jobtat, inst, logger):
        self.logger = logger
        self.solutionLock = thread.allocate_lock()
        self.solutions = []
        self.errorjob = Job()
        self.errorjob.parentjob = self.errorjob
        self.errorjob.baseProblem = "Error"
        self.errorjob.name = ["Error"]
        self.errorjob.filename = "invalid_file"
        self.inst = inst
        self.jobtat = jobtat

    def bogusResult(self, job):
        """Construct fictious results and enforce them"""
        if job.childList == []:
            return

        i = random.randint(0, 10)

        if i == 0:
            job.valid = 0
            self.logger.log("results: Job " + str(job.getFullName()) + \
                " was found invalid", 1)
        else:
            i = random.randint(0, len(job.childList) - 1)
            self.bogusResult(job.childList[i])

    def invalidJob(self, name):
        job = self.searchJob(name)
        job.lock.acquire()
        job.valid = 0
        job.lock.release()

    def searchJobProper(self, root, name):
        if (name == []):
            return root
        else:
            return self.searchJobProper(root.childList[name[0]], name[1:])

    def searchJob(self, name):
        basejob = name[0]
        name = name[1:]
        try:
            job = jobs[basejob]
        except KeyError:
            logger.log("No base job with name " + str(basejob)\
                + " found!", 0)
            return self.errorjob

        # If at some point a job is not found, we get an IndexError
        # This could be serious, as the results might belong to some
        # other job instance if user has not changed the name of job
        try:
            foundjob = self.searchJobProper(jobs[basejob], name)
        except IndexError:
            logger.log(str(name) + ": No such child for job!" +\
                " Ghosth results from previous runs?", 0)
            foundjob = self.errorjob

        return foundjob

    def addLearned(self, name, learned):
        job = self.searchJob(name)
        job.lock.acquire()
        job.learned.extend(learned)
        job.lock.release()

    def addSolution(self, name, solutions):
        job = self.searchJob(name)
        job.lock.acquire()
        job.solutions.extend([solutions])
        job.lock.release()

        # addSolution can be called from the socket and from UI
        currtime = time.time()
        self.solutionLock.acquire()
        solution = Solution(solutions, currtime)
        self.solutions.append(solution)
        self.solutionLock.release()

    def setTimeOut(self, name):
        job = self.searchJob(name)
        job.lock.acquire()
        job.timeout = 1
        job.lock.release()

    def listen(self, PORT):
        """Get results from the socket, update jobs"""

        self.logger.threadnames[thread.get_ident()] = "resultlisten"

        HOST = ""
        self.logger.log("Setting up the results server on port " +\
            str(PORT), -1)
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # When we crash, don't wait until the TIME_WAIT
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, True)

        connected = False
        while not connected:
            try:
                s.bind((HOST, PORT))
                connected = True
            except socket.error:
                logger.log("Couldn't set up server on port " + \
                    str(PORT) + " Waiting.", 0)
                time.sleep(2)

        s.listen(10)
        while (True):
            conn, addr = s.accept()
            self.logger.log("Result server connected by " + str(addr), -1)
            thread.start_new_thread(self.handleConnection, (conn, addr))

    def handleConnection(self, conn, addr):
        self.logger.threadnames[thread.get_ident()] = "resultlisten " +\
            str(addr)

        data = ""
        done = False
        while (True):

            self.logger.log("results: Reading the socket", 2)
            while not done:
                mo = re.search("^#[ \t]*endlisting", data, re.M)
                if (mo):
                    done = True
                else:
                    ibuf = conn.recv(1024)
                    if ibuf == "":
                        self.logger.log("Disconnected from " + \
                            str(addr), -1)
                        thread.exit()
                    else:
                        data += ibuf

            rv, P = splitlang.parse('list', data)

            # For every cnflist for every job, add the learned clauses
            # to that job
            for jname in P.globallearned.keys():
                self.logger.log("Learned clauses for job " + str(jname), 2)
                cnflist = P.globallearned[jname]
                exec "name = " + jname
                try:
                    self.inst.setruntime(str(name), time.time() -\
                        float(self.jobtat[str(name)]))
                except KeyError:
                    pass
                for i in range(0, len(cnflist)):
                    cnf = cnflang.Cnf(cnflist[i])
                    self.addLearned(name, cnf.clauses)

            for jname in P.globalsolution.keys():
                self.logger.log("Solutions for job " + str(jname), 2)
                cnflist = P.globalsolution[jname]
                exec "name = " + jname
                try:
                    self.inst.setruntime(str(name), time.time() -\
                        float(self.jobtat[str(name)]))
                except KeyError:
                    pass
                for i in range(0, len(cnflist)):
                    cnf = cnflang.Cnf(cnflist[i])
                    self.addSolution(name, cnf.clauses)

            for jname in P.globaltimeout.keys():
                self.logger.log("Timeout for job " + str(jname), 2)
                exec "name = " + jname
                try:
                    self.inst.setruntime(str(name), time.time() -\
                        float(self.jobtat[str(name)]))
                except KeyError:
                    pass
                self.setTimeOut(name)

            for jname in P.globalunsat.keys():
                self.logger.log("Unsat result for job " + str(jname), 2)
                exec "name = " + jname
                try:
                    self.inst.setruntime(str(name), time.time() -\
                        float(self.jobtat[str(name)]))
                except KeyError:
                    pass
                self.invalidJob(name)

            data = mo.string[mo.end():]
            self.inst.tofile()
            done = False

random.seed()

# Check that the user gave us sufficient information for jobs
if len(sys.argv) != 3:
    print "Usage: %s filename jobname" % sys.argv[0]
    print "jobname should be unique, so that old jobs do not mix with new."
    sys.exit(1)

from config import splitterloglevel
from config import splitterlog
from config import heuristicport
from config import satqueueport
from config import resultport
from config import splitterminspool
from config import instwork
from config import splittertimeout
import instrumentation
import os.path

# Init the logger
logger = Logger(splitterloglevel, splitterlog)
logger.threadnames = {}
logger.threadnames[thread.get_ident()] = "main"

# Init the instrumentation
inst = instrumentation.Instrumenter("splitter.ins")

# jobtat contains the times when certain job has been sent to the satjm
jobtat = {}

# Log messages not to be printed
#nologger = Logger()
#nologger.loglevel = -1
# set up the query and joblist -structures

joblist = JobList()
queryList = QueryList()

# Init the searcher and others
heuristic = Heuristic(logger, heuristicport)

requester = RequestJob(queryList, jobtat, logger)
searcher = SplitSearch(queryList, logger, heuristic, \
                        requester, splitterminspool)
enforcer = EnforceResults(jobtat, inst, logger)

# jobs dictionary holds the jobs handled by this splitter
jobs = {}


# Start the listening threads

thread.start_new_thread(RequestJob.listen,
                        (requester, satqueueport, True))

thread.start_new_thread(EnforceResults.listen,
                        (enforcer, resultport))


# Some debugging

def printJobs(job):
    valid = "+"
    numvalid = 1
    numsolutions = 0
    numtimeout = 0
    all = 0
    if job.isValid() == 0:
        valid = "-"
        numvalid = 0
    if job.solutions != []:
        valid = "*"
        numsolutions = len(job.solutions)
    if job.timeout == 1:
        valid = "/"
        numtimeout = 1
    space = ""
    for l in range(0, job.assumptionLevel):
        space += " "
    string = space + str(job.getFullName()) + valid + "\n"
    print string
    cstring = ""
    for i in job.childList:
        v, s, t, a, tstr = printJobs(i)
        cstring = "".join([cstring, tstr])
        numvalid += v
        numsolutions += s
        numtimeout +=t
        all += a

    return numvalid, numsolutions, numtimeout, all+1, \
        "".join([string, cstring])

# The start time initialized. Can we get it's scope to extend to the
# while loop this way?
#starttime = 0

# Initialize job
initJob = Job()
initJob.name = [sys.argv[2]]
initJob.parentJob = initJob
initJob.filename = sys.argv[1]
initJob.valid = 1
jobs[initJob.name[0]] = initJob
joblist.jobs.append(initJob)

# Start running the job
job = jobs[jobs.keys()[0]]
starttime = time.time()
logger.log("Solving started", 0)
thread.start_new_thread(SplitSearch.splitter, \
            (searcher, job, True))

class SolutionException(Exception):
    def __init__(self, sol):
        self.sol = sol

# Search the tree. The process must stop if a solution is found or if
# every subtree of the tree is unsatisfiable. A depth-first search,
# using the internal valid-flag of the job will do the job.
#
# Note: We assume here that the childList never changes after it's
# length has changed from 0. Same thing for solutions

def endcheck(job):
    job.lock.acquire()
    haschildren = (len(job.childList) > 0)
    hassolutions = (len(job.solutions) > 0)
    isvalid = job.valid
    job.lock.release()

    if hassolutions:
        # Has a solution, search is done.
        raise SolutionException(job.solutions[0][0])

    elif not isvalid:
        # Has a no-solution result, no need to check this subtree
        return False

    elif haschildren:
        # Has no no-solution but has children. Search continuation depends
        # on childrens validity status
        hasvalidchild = False
        for child in job.childList:
            hasvalidchild = endcheck(child) or hasvalidchild
        # Optimize, if childern are all invalid, parent is as well
        job.lock.acquire()
        job.valid = hasvalidchild
        job.lock.release()
        return job.valid
    else:
        # Has no no-solution and no children, we must wait for results
        # before we can say anything.
        return True

def solution(s):
    string="RESULT: SAT " + " ".join(map(str, s)) + "\n"
    print string
    resultfile.write(string)

def unsat():
    string="RESULT: UNSAT" + "\n"
    print string
    resultfile.write(string)

def timeout():
    string="RESULT: TIMEOUT" + "\n"
    print string
    resultfile.write(string)

def cleanup():
    for jobkey in jobs:
        basejob = jobs[jobkey]
        string = "TREE " + str(jobkey) + "\n"
        print string
        resultfile.write(string)
        tv, s, t, a, treestring = printJobs(basejob)
        resultfile.write(treestring)
        string = "TIME " + str(time.time() - starttime) + "\n"
        print string
        resultfile.write(string)

    resultfile.close()
    sys.exit(0)

def supervisor():
    while (True):
        for jobkey in jobs:
            basejob = jobs[jobkey]
            try:
                running = endcheck(basejob)
                if not running:
                    unsat()
                    cleanup()
            except SolutionException, s:
                solution(s.sol)
                cleanup()
        if time.time() - starttime > splittertimeout:
            timeout()
            cleanup()
        time.sleep(1)

#cmds = 0
#while(True):
#    s = raw_input('sp [' + str(cmds) + '] ')
#    cmds += 1
#
#    # Initialize job
#    if (re.search("jj[ \t]+.*", s, re.M)):
#        mo = re.search("jj[ \t]+(.*)", s, re.M)
#        filename = mo.group(1)
#        # Construct the initial job
#        initJob = Job()
#        initJob.name = [filename]
#        initJob.parentJob = initJob
#        initJob.filename = filename
#        initJob.valid = 1
#        # Add this job to the jobs dictionary
#        jobs[initJob.name[0]] = initJob
#        # Add this job to the query spool
#        joblist.jobs.append(initJob)
#
#    # Start running the job
#    elif (s == "rj"):
#        if (jobs == {}):
#            print "Set job first!"
#            continue
#        job = jobs[jobs.keys()[0]]
#        starttime = time.time()
#        print "Solving started at " + str(starttime)
#        thread.start_new_thread(SplitSearch.splitter, \
#            (searcher, job, False))
#
#    # Start running the job, local
#    elif (s == "rl"):
#        if (jobs == {}):
#            print "Set job first!"
#            continue
#        job = jobs[jobs.keys()[0]]
#        starttime = time.time()
#        print "Solving started at " + str(starttime)
#        thread.start_new_thread(SplitSearch.splitter, \
#            (searcher, job, True))
#
#    # Print job
#    elif (s == "pj"):
#        if (jobs == {}):
#            print "Set job first!"
#            continue
#        print "************* List of all jobs around ***************"
#        v, s, t, a = printJobs(initJob)
#        print "******************** That's all *********************"
#        print "valid", v
#        print "solutions", s
#        print "timeout", t
#        print "total", a
#
#    # Print pending jobs
#    elif (s == "pp"):
#        print "Pending requests:", queryList.queries
#
#    # Invalidate job (job is unsatisfiable)
#    elif (re.search("^ij[ \t]+", s)):
#        mo = re.search("^ij[ \t]+(.*)$", s)
#        exec "name = " + mo.group(1)
#        print "Invalidating job " + str(name)
#        enforcer.invalidJob(name)
#
#    # Add a learned clause list to job
#    elif (re.search("^tj[ \t]+", s)):
#        mo = re.search("^tj[ \t]+(\[.*\])[ \t]+(\[.*\])$", s)
#        exec "name = " + mo.group(1)
#        exec "learned = " + mo.group(2)
#        enforcer.addLearned(name, learned)
#        print "job " + str(name) + " learned clauses now:"
#        job = enforcer.searchJob(name)
#        print str(job.getlearned())
#
#    # Add solution list to job
#    elif (re.search("^sj[ \t]+", s)):
#        mo = re.search("^sj[ \t]+(\[.*\])[ \t]+(\[.*\])$", s)
#        exec "name = " + mo.group(1)
#        exec "solutions = " + mo.group(2)
#        enforcer.addSolution(name, solutions)
#        print "job " + str(name) + " solutions now:"
#        job = enforcer.searchJob(name)
#        print str(job.getsolutions())
#
#    # Mark job as timeout
#    elif (re.search("^oj[ \t]+", s)):
#        mo = re.search("^oj[ \t]+(\[.*\])$", s)
#        exec "name = " + mo.group(1)
#        enforcer.setTimeOut(name)
#        print "Job " + str(name) + " now marked as timeout"
#
#    # Print all solutions found
#    elif (s == "ps"):
#        for solution in enforcer.solutions:
#            print "find time ", solution.time - starttime
#            print solution.solution
#            print
#
#    # Print current running time
#    elif (s == "pt"):
#        print "running", time.time() - starttime
#
#    # Quit
#    elif (s == "qq"):
#        break
#
#    # Print help
#    elif (s == "h"):
#        print "h\t- print this help"
#        print "rj\t- run the solving process"
#        print "rl\t- run the solving process with local instances"
#        print "jj\t- set the job to be solved"
#        print "pj\t- print jobs"
#        print "pp\t- print num of pending queries"
#        print "ij job\t- invalidate a job"
#        print "\t  example: ij ['foo', 0, 1]"
#        print "tj\t- teach a clause for a job"
#        print "\t  example: tj ['foo', 0, 1] [[1, 2], [-1, -4]]"
#        print "sj\t- add a solution to a job"
#        print "\t  example: sj ['foo', 0, 1] [[1, 2, 3, 4]]"
#        print "oj job\t- Mark job as timeout"
#        print "ps\t- print all solutions found so far"
#        print "pt\t- print current running time"
#        print "qq\t- exit solver"
#
#    # Empty line is ignored
#    elif (s == ""):
#        continue
#
#    # Unknown command
#    else:
#        print "Unknown command " + s
#        print "Try h"
#
#
from config import instwork
import os.path

resultfile = open(os.path.join(instwork, "splitresults"), 'w')

supervisor()

logger.log("Not reached", 0)

