#!/usr/bin/env python
#
# $Id: satjm.py,v 1.1 2006/05/31 15:24:05 aehyvari Exp $
#

import splitlang
import time
import thread
import socket
import sets
import re
import os
import cnflang
import random
from jobutils import Job
from jobutils import tofilesysname

class GridSender:
    """Sends jobs to the grid."""

    def __init__(self, jobqueue, supervisor, uds_path, jobtat, logger, \
            gridwork, inst):
        self.queue = jobqueue
        self.logger = logger
        self.uds_path = uds_path
        self.gridwork = gridwork
        self.supervisor = supervisor
        self.jobtat = jobtat
        self.inst = inst
        from config import uds
        from config import gridjmhost
        from config import gridjmport
        self.uds = uds
        self.gridjmhost = gridjmhost
        self.gridjmport = gridjmport

    def start(self):
        """Connect to uds file, get strings from it and send the problem
        instances"""
        threadnames[thread.get_ident()] = "GridSender"

        if not self.uds:
            self.logger.log("Connecting to inet sock %s:%s" %
                (self.gridjmhost, self.gridjmport), 0)
            uds_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM, 0)
        else:
            self.logger.log("Connecting to uds sock %s" % self.uds_path,
                0)
            uds_socket = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM, 0)

        connected = False
        while not connected:
            try:
                if not self.uds:
                    uds_socket.connect((self.gridjmhost,
                        self.gridjmport))
                else:
                    uds_socket.connect(self.uds_path)
                connected = True
            except socket.error:
                if not self.uds:
                    self.logger.log("Connection to %s:%s failed. Trying again..." % (self.gridjmhost, self.gridjmport), 0)
                else:
                    self.logger.log("Connection to " + self.uds_path +
                        " failed. Trying again...", 0)
                time.sleep(2)

        self.uds_socket = uds_socket

        self.logger.log("Connected to the GridJM socket", 0)
        while (True):
            buffer = ""

            mo = None
            # Start searching for the job request from the socket
            self.logger.log("Waiting for job requests", 2)
            # XXX Fixme this is not a good way... does it matter?
            while (not mo):
                newdata = uds_socket.recv(1024)
                self.logger.log("Something arrives", 2)
                buffer = "".join([buffer, newdata])
                self.logger.log("Contents of buffer is \n" + buffer, 2)
                mo = re.search("^#[ ]*getjob[ ]+([1-9][0-9]*)", buffer, re.M)

            self.logger.log("Found a job request", 2)

            # At least one job request was found, and we ignore the
            # others
            jobreqs = int(mo.group(1))

            while jobreqs > 0:
                self.process()
                jobreqs -= 1

    def process(self):
        from config import solvers
        from config import mingridtimeout
        from config import randomgridtimeout
        # For each job, construct an xrsl file. Create the job to be
        # sent to grid and then send the xrsl file to the socket.
        gridtimeout = mingridtimeout*60 +\
            random.randint(0, randomgridtimeout*60) # in seconds
        solverpath = solvers[random.randint(0, len(solvers) - 1)]
        solvername = os.path.basename(solverpath)
        solverargs = str(gridtimeout)

        # First get hlock!
        self.supervisor.hlock.acquire()
        job = self.queue.getfirst()
        self.supervisor.has_been_sent = True
        self.supervisor.hlock.release()

        logger.log("Start writing files", 2)
        jobfile = open(self.gridwork + "/" + job.filesysname(), 'w')
        jobfile.write(job.tostring())
        jobfile.close()
        jobpath = self.gridwork + "/" + job.filesysname()
        # Construct the runnable script
        scriptname = job.fsname + ".sh"
        runscript = "#!/bin/sh\n" + \
            "# This is an automatically generated shell script from python\n" + \
            "# program " + __name__ + " " + \
            time.strftime("%Y-%m-%d at %H:%M.%S") + "\n" + \
            'echo "jobname = ' + job.name + '"\n' + \
            'chmod +x ./' + solvername + '\n' + \
            "./" + solvername + " " + job.filesysname() + " " + solverargs + \
            "\ncat /proc/cpuinfo > cpuinfo\n" +\
            "hostname -f >> cpuinfo || true\n"

        scriptfile = open(self.gridwork + "/" + scriptname, 'w')
        scriptfile.write(runscript)
        scriptfile.close()
        scriptpath = self.gridwork + "/" + scriptname

        # Calculate the size of the problem to be sent (in megabytes)
        jobstats = os.stat(jobpath)
        scriptstats = os.stat(scriptpath)

        disk = (jobstats.st_size + scriptstats.st_size)/1024/1024 + 1

        # Construct the xrsl
        from config import nogoods
        from config import restfiles
        from config import restexecutables
        xrslscript = '& (executable=' + scriptname + ')\n' +\
            '(jobname=' + job.filesysname() + ')\n' +\
            '(inputfiles=("' + solvername + '" "' + solverpath + '") ("' +\
            scriptname + '" "' + scriptpath + '") ("' + \
            job.filesysname() + '" "' + jobpath + '")' + restfiles + ')\n' +\
            '(stdout=satjob.out)\n' +\
            '(stderr=satjob.err)\n' +\
            '(gmlog=gridlog)\n' +\
            '(outputfiles=("cpuinfo" " "))\n' +\
            '(executables=' + solvername + ' ' + scriptname + ' ' + \
                restexecutables + ')\n' +\
            '(cputime=' + str(gridtimeout / 60 + 1) + ')\n' +\
            '(memory=500)\n' +\
            '(disk=' + str(disk) + ')\n' +\
            nogoods

        logger.log("File written", 2)

        self.jobtat[job.name] = time.time()
        self.inst.simstart(job.name)

        self.inst.tofile(threadnames[thread.get_ident()])

        logger.log("Sending", 2)
        self.uds_socket.send(xrslscript + "\xff")
        logger.log("Sent", 2)

class GridListener:
    """Listen for results from the grid."""

    def __init__(self, resultsender, resultdir, queuedir, inst, jobtat, logger):
        self.results = resultsender
        self.logger = logger
        self.resultdir = resultdir
        self.queuedir = queuedir
        self.inst = inst
        self.jobtat = jobtat

    def start(self):
        """Read results gotten from the grid

        Read results coming in to the given directory. The standard
        output is assumed to be in a subdirectory of this dir in file
        called satjob.out. The file is assumed to have a line
        indicating the name of the job in case. If the file exists, the
        result of the computation is read from the file. A corresponding
        result instance is constructed and sent to the resultsender.
        After this the job directory is removed with its contents.
        Finally the sent job is removed from outgoing job directory"""

        threadnames[thread.get_ident()] = "GridListener"

        while True:
            dirs = filter(lambda x: os.path.isdir(self.resultdir + \
                                                    "/" + x), \
                            os.listdir(self.resultdir))

            outputs = filter(lambda x: os.path.isfile(x[0]) and
                                        os.path.isfile(x[1]), \
                map(lambda x: [self.resultdir + "/" + x + "/satjob.out", \
                                self.resultdir + "/" + x + \
                                "/gridlog/diag"], \
                    dirs))

            logger.log("Found potential result files: "+ \
                            str(outputs), 2)

            for out in outputs:
                resulttype = ""
                file = open(out[0])
                lines = file.readlines()
                file.close()
                for line in lines:
                    # First get the name of the job from the file
                    mo = re.search("^jobname = (.*)$", line, re.M)
                    if mo:
                        jobname = mo.group(1).strip()
                        logger.log("Results from grid for job " + \
                            jobname, 2)
                        try:
                            self.inst.setruntime(jobname, float(time.time() -\
                                self.jobtat[jobname]))
                            self.inst.simstop(jobname)
                            self.inst.tofile()
                        except KeyError:
                            pass
                        continue


                    if (re.search("RESULT:[ \t]+ABORT[ \t]*:[ \t]+TIME OUT", line,
                            re.M)):
                        os.rename(out[0], out[0] + '-pr')
                        self.results.settimeout(str(jobname))
                        logger.log("Grid: timeout", 2)
                        resulttype = "timeout"
                        break

                    mo = re.search("^(-?[1-9][0-9]*[- 0-9]*)Random Seed Used",
                            line, re.M)
                    if (mo):
                        os.rename(out[0], out[0] + '-pr')
                        num_vars = max(map(abs, map(int, mo.group(1).strip().split(" "))))
                        output = ["p cnf " + str(num_vars) + " 1\n",
                                        mo.group(1) + " 0\n"]

                        self.results.setsolution([output], str(jobname))
                        logger.log("Grid: solution", 2)
                        resulttype = "solution"
                        break

                    if (re.search("RESULT:[ \t]+UNSAT", line, re.M)):
                        os.rename(out[0], out[0] + '-pr')
                        self.results.setunsat(str(jobname))
                        logger.log("Grid: unsat", 2)
                        resulttype = "unsat"
                        break

                # If none of above trigger, we get here.
                if resulttype == "":
                    logger.log("Strange, grid stdout was " + \
                        "incomprehensible for file " + out[0], 0)

                else:
                    fsname = tofilesysname(jobname) + ".cnf"
                    try:
                        os.unlink(os.path.join(self.queuedir, fsname))
                    except OSError, e:
                        logger.log("Ghost results " + str(e), 0)

            time.sleep(10)


class Queue:
    """The queue of the jobs to be sent to the grid"""

    def __init__(self, port, host, logger, queuesize):
        self.port = port
        self.host = host
        self.logger = logger
        self.queuesize = queuesize
        self.queue = []
        self.sent = False # A signal for the local solver
        self.queuelock = thread.allocate_lock()

    def start(self):
        """Create a connection to the request port, request jobs

        The queue must be full, so more jobs are requested if it is
        not"""
        # First add my name to the threads
        threadnames[thread.get_ident()] = "Queue"

        self.reqs = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        connected = False
        while not connected:
            try:
                self.reqs.connect((self.host, self.port))
                connected = True
            except socket.error:
                self.logger.log("Connection to " + self.host + " port " +
                    str(self.port) + " failed. Trying again...", 0)
                time.sleep(2)

        self.logger.log("Connected to the request port", 0)
        try:
            while True:
                self.queuelock.acquire()
                difference = self.queuesize - len(self.queue)
                self.queuelock.release()
                if (difference > 0):
                    self.request(difference)
                    self.retrieve(difference)
                time.sleep(2)
        except socket.error, descr:
            self.logger.log("Job requester: " + str(descr), 0)
            self.reqs.close()

    def request(self, numjobs):
        """Request numjobs jobs from the splitter"""
        self.logger.log("Requesting " +
            str(numjobs) + " jobs", 2)
        self.reqs.send("# listing\n")
        self.reqs.send("# reqjob " + \
            str(numjobs) + "\n")
        self.reqs.send("# endlisting\n")

    def retrieve(self, numjobs):
        """Retrieve numjobs jobs from the splitter

        Read the result until numjobs jobs have been transmitted. Create
        a job description from each of the jobs"""
        data = ""
        while numjobs > 0:
            done = False
            while not done:
                mo = re.search("^#[ \t]*endlisting", data, re.M)
                if (mo):
                    done = True
                else:
                    data = "".join([data, self.reqs.recv(10240)])
            rv, P = splitlang.parse('list', data[0:mo.end()])
            data = mo.string[mo.end():]
            keys = sets.Set(P.globalproblem.keys() + \
                P.localproblem.keys())
            numjobs -= len(keys)
            for jobkey in keys:
                # Notation expects keys not to be false
                job = Job(jobkey,
                    P.globalproblem.has_key(jobkey) and\
                        P.globalproblem[jobkey] or [],\
                    P.localproblem.has_key(jobkey) and\
                        P.localproblem[jobkey] or [])
                self.queuelock.acquire()
                self.queue.append(job)
                self.queuelock.release()

    def getfirst(self):
        """Get the first job on the queue.

        Blocking if the queue is empty"""

        done = False
        while not done:
            self.logger.log("Trying to get first from queue", 2)
            self.queuelock.acquire()
            if len(self.queue) > 0:
                self.logger.log("Got first from queue", 2)
                job = self.queue.pop(0)
                done = True
                self.queuelock.release()
            else:
                self.logger.log("Queue was empty, sleeping", 2)
                self.queuelock.release()
                time.sleep(2)
        return job

    def getfirstref(self):
        """Get reference to the first job on the queue. Do not pop.

        Blocking if the queue is empty"""

        done = False
        while not done:
            self.logger.log("Trying to get ref from queue", 2)
            self.queuelock.acquire()
            if len(self.queue) > 0:
                self.logger.log("Got ref from queue", 2)
                job = self.queue[0]
                done = True
                self.queuelock.release()
            else:
                self.logger.log("ref: Queue was empty, sleeping", 2)
                self.queuelock.release()
                time.sleep(2)
        return job

    def getbyname(self, name):
        """Get the job identified by the name.

        If job is not found, return None"""

        namedjob = None

        self.queuelock.acquire()
        # By first setting i and then popping, we have too many i's, so
        # break
        for i in range(0, len(self.queue)):
            if self.queue[i].name == name:
                namedjob = self.queue.pop(i)
                break
        self.queuelock.release()

        return namedjob

#class Job:
#    """The description of a job
#
#    Job consists of the clauses given explicitly and a possible file
#    containing the base job"""
#
#    def __init__(self, name, globals, locals):
#        self.globals = globals
#        self.locals = locals
#        self.name = name
#
#    def tosplitlang(self):
#        retstr = "# problem " + self.name + "\n"
#        for gstr in self.globals:
#            retstr = "".join([retstr, "# begin\n"])
#            retstr = "".join([retstr, "".join(gstr)])
#            retstr = "".join([retstr, "# end\n"])
#
#        for lstr in self.locals:
#            retstr = "".join([retstr, "# local " + lstr + "\n"])
#
#        retstr = "".join([retstr, "# endproblem " + self.name + "\n"])
#        return retstr

class Logger:

    def __init__(self, loglevel, logfilename):
        self.loglevel = loglevel
        self.logfilename = logfilename
        logfile = open(logfilename, 'w')
        logfile.write("Starting " + __name__ + " logger")
        logfile.close()

    def log(self, string, level):
        if level <= self.loglevel:
            diftime = "[" + str(time.time()) + "]"
            logfile = open(self.logfilename, 'a')
            threadname = threadnames[thread.get_ident()]
            logfile.write(diftime + " " + threadname + ": " + string + "\n")
            logfile.close()

class LocalSupervisor:
    """Read the first job from the queue, send it for local solving, get
    results.

    Results are that the first job from the queue has been sent to the
    grid, and the checking time is over or local solving finds some
    solution (sat/unsat). In both cases the solving must be stopped and
    a new job must be fed to the local solver."""

    def __init__(self, solver, queue, result, logger):
        """Solver is the local solver interface, queue is where the jobs
        come from and result is the interface for sending results"""

        self.solver = solver
        self.queue = queue
        self.result = result
        self.logger = logger
        self.has_been_sent = False
        self.hlock = thread.allocate_lock()

    def start(self):
        # First set my name to threadnames
        threadnames[thread.get_ident()] = "LocalSupervisor"
        while True:
            job = self.queue.getfirstref()
            self.solver.solve(job)
            ok = True
            while ok:

                # Here we hold two locks. Make sure that the grid sender
                # does not attempt to grab the first job before it can
                # acquire hlock (is this enough?). Hmm... While we
                # sleep, grid sender manages to send two jobs. In this
                # case has_been_sent is true and we set it false. One
                # of the jobs is then found to have a solution on else
                # branch here, and we pop third job, thinking that this
                # was the job we found a solution for. We need to
                # remove the jobs by name!
                self.hlock.acquire()
                if self.has_been_sent == True:
                    self.has_been_sent = False
                    self.solver.stop()
                    ok = False
                else:
                    self.solver.solutionlock.acquire()
                    if len(self.solver.solutions) != 0:
                        self.queue.getbyname(job.name)
                        self.result.setsolution(self.solver.solutions, job.name)
                        ok = False
                    elif self.solver.unsat == True:
                        self.queue.getbyname(job.name)
                        self.result.setunsat(job.name)
                        ok = False
                    self.solver.solutionlock.release()
                self.hlock.release()

                time.sleep(2)

class SocketError(Exception):
    def __init__(self, description):
        self.description = description
    def __str__(self):
        return self.description

class LocalSolver:
    """Solve jobs locally.

    Send problems to solver socket, stop solver if required, read
    solutions provided by the solver from the socket."""

    def __init__(self, port, logger):
        self.port = port
        self.host = ""
        self.logger = logger
        self.solutionlock = thread.allocate_lock()
        self.solutions = []
        self.unsat = False
        self.conn = None

    def listen(self):
        # First add my name to the threadnames
        threadnames[thread.get_ident()] = "Localsolver"
        self.logger.log("Setting up the local solver on port " +
            str(self.port), 0)

        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # When we crash, don't wait until the TIME_WAIT
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, True)

        connected = False
        while not connected:
            try:
                s.bind((self.host, self.port))
                connected = True
            except socket.error:
                logger.log("Couldn't set up server on port " +\
                    str(self.port) + ". Trying again", 0)
                time.sleep(2)

        s.listen(1)
        self.conn, self.addr = s.accept()

        self.logger.log("Local solver port connected by " +
            str(self.addr), 0)

        self.logger.log("Starting listen", 2)

        data = ""
        try:
            while True:
                done = False
                # Parse one block of cnflang
                while (done == False):
                    line = self.conn.recv(10240)
                    if line == "":
                        logger.log("Connection to local solver lost", 0)
                        raise SocketError, "Connection lost"
                    data = "".join([data, line])
                    logger.log("Data is now " + data + "\n", 2)
                    mo = re.search("^#[ \t]*endlisting", data, re.M)
                    if (mo != None):
                        done = True
                # The to-be-parsed part is in block, rest is in data
                block = data[:mo.end()]
                data = data[mo.end():]

                rv, P = splitlang.parse('list', block)
                jobnames = P.globalsolution.keys()
                # There must be at most one job
                assert len(jobnames) <= 1
                for jobname in jobnames:
                    self.solutionlock.acquire()
                    self.solutions = P.globalsolution[jobname]
                    self.solutionlock.release()
                # There must be at most one unsat job as well
                jobnames = P.globalunsat.keys()
                assert len(jobnames) <= 1
                if len(jobnames) == 1:
                    self.solutionlock.acquire()
                    self.unsat = True
                    self.solutionlock.release()
        except SocketError, desc:
            self.logger.log("Localsolver:" + str(desc), 0)
            self.conn.close()
            thread.exit()

    def solve(self, job):
        """Send problem to local solver, clear previous results"""
        while self.conn == None:
            logger.log("No connection to local solver, waiting...", 0)
            time.sleep(2)

        self.unsat = False
        self.solutions = []
        self.conn.send("# listing\n")
        self.conn.send(job.tosplitlang())
        self.conn.send("# endlisting\n")

    def stop(self):
        while self.conn == None:
            logger.log("No connection to local solver, waiting...", 0)
            time.sleep(2)

        self.conn.send("# stop\n")

class ResultSender:
    """Methods for setting solutions and a thread running to send them.

    The results are sent to splitter as a batch. Sending is triggered by
    the length of the queue. If size is greater than 0, queue is locked
    and results are sent to the splitter."""

    def __init__(self, port, host, logger, refresh):
        self.port = port
        self.host = host
        self.logger = logger
        self.results = []
        self.resultlock = thread.allocate_lock()
        self.refresh = refresh

    def setsolution(self, solutions, name):
        self.logger.log("Solution " + str(solutions) + " for job " +
            name, 1)
        self.resultlock.acquire()
        for solution in solutions:
            solstr = "# solution " + name + "\n# begin\n" \
                + "".join(solution) + "# end\n# endsolution " + name + "\n"
        self.results.append(solstr)
        self.resultlock.release()

    def setunsat(self, name):
        self.logger.log("Job " + name + " is unsat", 2)
        self.resultlock.acquire()
        unsatstr = "# unsat " + name + "\n"
        self.results.append(unsatstr)
        self.resultlock.release()

    def settimeout(self, name):
        self.logger.log("Job " + name + " is timeout", 2)
        self.resultlock.acquire()
        timeoutstr = "# timeout " + name + "\n"
        self.results.append(timeoutstr)
        self.resultlock.release()

    def send(self):
        """The solution sending thread"""
        # First add my name to threadnames
        threadnames[thread.get_ident()] = "Solutionsender"

        self.sends = socket.socket(socket.AF_INET, socket.SOCK_STREAM)

        connected = False
        while not connected:
            try:
                self.sends.connect((self.host, self.port))
                connected = True
            except socket.error:
                self.logger.log("Connection to " + self.host + " port "
                    + str(self.port) + " failed. Trying again...", 0)
                time.sleep(2)

        self.logger.log("Connected to the results port", 0)
        localresults = []
        try:
            while True:
                time.sleep(self.refresh)
                self.resultlock.acquire()
                if len(self.results) > 0:
                    for r in self.results:
                        localresults.append(r)
                    self.results = []
                self.resultlock.release()

                # Send the results to the splitter without holding the lock
                if len(localresults) > 0:
                    sendstring = "# listing\n"
                    sendstring = "".join([sendstring] + localresults)
                    sendstring = "".join([sendstring, '# endlisting'])
                    logger.log("Sending\n" + sendstring, 3)
                    self.sends.send(sendstring)
                    localresults = []
        except socket.error, desc:
            self.logger.log("Resultsending: " + str(desc), 0)
            self.sends.close()
        thread.exit()

starttime = time.time()

from config import queueloglevel
from config import queuesize
from config import satqueueport
from config import localsolverport
from config import resultport
from config import xrsluds
from config import queuework
from config import dlwork
from config import resultrefresh
from config import queuelogfile

logger = Logger(queueloglevel, queuelogfile)

class S:
    pass

ss = S()

import instrumentation

def s(reqport, solveport, solport, jobtat, inst):
    ss.queue = Queue(reqport, "", logger, queuesize)
    thread.start_new_thread(ss.queue.start, ())
    ss.solver = LocalSolver(solveport, logger)
    thread.start_new_thread(ss.solver.listen, ())
    ss.results = ResultSender(solport, "", logger, resultrefresh)
    thread.start_new_thread(ss.results.send, ())
    ss.supervisor = LocalSupervisor(ss.solver, ss.queue, ss.results, logger)
    thread.start_new_thread(ss.supervisor.start, ())
    ss.gridlistener = GridListener(ss.results, dlwork, queuework, \
        inst, jobtat, logger)
    thread.start_new_thread(ss.gridlistener.start, ())

def binit():
    inst = instrumentation.Instrumenter("satjm.ins")
    jobtat = {}
    s(satqueueport, localsolverport, resultport, jobtat, inst)
    ss.sender = GridSender(ss.queue, ss.supervisor, \
        xrsluds, jobtat, logger, queuework, inst)

def grid():
    thread.start_new_thread(ss.sender.start, ())

threadnames = {}

threadnames[thread.get_ident()] = "Main"

if __name__ != '__main__':
    print "To start solving, type " + __name__ + \
        ".s(reqport, solveport, solport, {})"

    print "Or, if everything's fine, " + __name__ + ".binit() and " + __name__ +\
            ".grid()"
else:
    binit()
    grid()

import signal
signal.pause()
