#!/usr/bin/env python
#
# $Id: instrumentation.py,v 1.1 2006/05/31 14:33:41 aehyvari Exp $
#

import time
import os
from thread import get_ident
from config import instwork

class Eventobject:
    def __init__(self, eventname):
        self.name = eventname
        self.start = 0
        self.stop = 0
        self.jobname = ""

    def stoptime(self):
        """If event has not yet stopped, return current time"""
        if self.stop == 0:
            return time.time()
        else:
            return self.stop

class Instrumenter:
    """Records per-thread event timings

    The event timings are triples of form (event, start, stop) and are
    stored in a hash table indexed by the per process unique thread number.
    Timing should first be initialized by giving the file name to which
    timing data will be stored. Events are started by calling the function
    start(event) and closed by calling stop(event). It is an error to have
    two events running simultaneously in the same thread"""

    def __init__(self, filename):

        file = open(instwork + filename, 'w')
        file.close()
        self.filename = instwork + filename
        self.events = {}
        self.evtypes = {}
        self.threaddict = {}
        self.firstentry = 0
        self.lastentry = 0
        self.jobnames = {}
        self.runtimes = {}
        self.simevents = {}

    def start(self, event, jobname = "generic"):
        """Call to make the event start, jobname optional"""

        # check if this is the first entry
        if self.firstentry == 0:
            self.firstentry = time.time()
        else:
            self.lastentry = time.time()

        # Event types are recorded here
        try:
            self.evtypes[event] += 1
        except KeyError:
            self.evtypes[event] = 1

        # Job names recorded here
        try:
            self.jobnames[jobname] += 1
        except KeyError:
            self.jobnames[jobname] = 1

        thname = get_ident()

        # If the event is the first for this thread, we must initialize
        # the thread's array
        try:
            latestev = \
                self.events[thname][len(self.events[thname]) - 1]
        except KeyError:
            # We don't have entries for this thread, so add the initial
            # event
            initev = Eventobject('instinit')
            initev.start = time.time()
            initev.stop = time.time()
            initev.jobname = "generic"
            try:
                self.jobnames["generic"] += 1
            except KeyError:
                self.jobnames["generic"] = 1

            self.events[thname] = [initev]
            try:
                self.evtypes['instinit'] += 1
            except KeyError:
                self.evtypes['instinit'] = 1
            latestev = initev

        evobj = Eventobject(event)
        evobj.start = time.time()
        evobj.jobname = jobname

        assert(latestev.stop <= time.time() and latestev.stop != 0)

        self.events[thname].append(evobj)

    def stop(self, event):

        self.lastentry = time.time()

        thname = get_ident()

        evobj = \
            self.events[thname][len(self.events[thname]) -1]

        assert(evobj.name == event)

        evobj.stop = time.time()

    def simstart(self, evname):
        assert (not self.simevents.has_key(evname))
        self.simevents[evname] = [time.time()]

    def simstop(self, evname):
        assert (self.simevents.has_key(evname))
        self.simevents[evname].append(time.time())

    def setruntime(self, pname, time):
        self.runtimes[pname] = time

    def _th2n(self, th):
        """Convert threads to names according to a thread dictionary.

        If name is not found, return th (as string)"""

        try:
            return str(self.threaddict[th])
        except KeyError:
            return str(th)

    # Tofile is quite a hack, but I don't want to lock anything!
    def tofile(self, threaddict = {}):
        """threaddict, argument is the dictionary mapping
        thread.get_ident() -values to user given names"""

        self.threaddict = threaddict

        file = open(self.filename + ".tmp", 'w')
        timescale = "(\n(timescale\n"
        timescale = "".join([timescale, ' (', \
            str(float(self.firstentry)), " ", \
            str(float(self.lastentry)), ")\n)\n"])

        file.write(timescale)

#        file.write("(\n")
#        file.write('(timescale\n')
#        file.write(' (' + str(float(self.firstentry)) + " " + str(float(self.lastentry)) + \
#            ")\n")
#        file.write(')\n')

        evtypes = "(evtypes\n("
#        file.write('(evtypes\n (')
        try:
            for ev in self.evtypes:
                evtypes = "".join([evtypes, ev + " "])
        except RuntimeError, e:
            pass # Dictionary size change

        evtypes = "".join([evtypes, ')\n)\n'])
        file.write(evtypes)

        thnames = '(thnames\n ('
        try:
            for thkey in self.events:
                thnames = "".join([thnames, self._th2n(thkey), " "])
        except RuntimeError, e:
            pass
        thnames = "".join([thnames, ')\n)\n'])
        file.write(thnames)

        jobnames = '(jobnames\n ('
        try:
            for jname in self.jobnames:
                jobnames = "".join([jobnames, jname,  " "])
        except RuntimeError, e:
            pass
        jobnames = "".join([jobnames, '))\n'])
        file.write(jobnames)

        runtimes = '(runtimes (\n'
        try:
            for jobkey in self.runtimes:
                runtimes = "".join([runtimes, " (", jobkey, " ", \
                    str(self.runtimes[jobkey]), ")\n"])
        except RuntimeError, e:
            pass
        runtimes = "".join([runtimes, '))\n'])
        file.write(runtimes)

        events = '(events (\n'
        try:
            for th in self.events:
                events = "".join([events, " (", self._th2n(th), " (\n"])
                evlist = self.events[th]
                for ev in evlist:
                    events = "".join([events, "  (", ev.name, " (", \
                        ev.jobname, " ", str(ev.start), " ", \
                        str(ev.stoptime()), "))\n"])
                events = "".join([events, " ))\n"])
        except RuntimeError, e:
            pass
        events = "".join([events, '))\n'])
        file.write(events)

        try:
            se = '(simevents (\n'
            for s in self.simevents:
                start = self.simevents[s][0]
                if len(self.simevents[s]) == 1:
                    stop = time.time() # Up 'till now
                else:
                    stop = self.simevents[s][1]
                se = "".join([se, " (" + s + "\n"])
                se = "".join([se, "  (", str(start), " ", str(stop), \
                    "))\n"])
            se = "".join([se, '))\n'])
            file.write(se)
        except RuntimeError, e:
            pass # Dictionary has changed during execution, ignore

        file.write(')\n')
        file.close()
        try:
            os.rename(self.filename + ".tmp", self.filename)
        except OSError, e:
            # Some other thread has already moved it
            pass

if __name__ == '__main__':
    import thread
    import time
    def mark(type, job):
        inst.simstart(type)
        inst.start(type, job)
        time.sleep(2)
        inst.stop(type)
        time.sleep(3)
        inst.start(type, job)
        time.sleep(5)
        inst.stop(type)
        inst.simstop(type)

    inst = Instrumenter('foo.ins')

    mark('foo', "['bar', 0]")
    th1 = thread.start_new_thread(mark, ('bar', "['bar', 1]"))
    th2 = thread.start_new_thread(mark, ('quux', "['bar', 2]"))

    inst.setruntime("['bar', 0]", 234.35)
    inst.setruntime("['bar', 1]", 345.363)
    time.sleep(20)
    threaddict = {thread.get_ident(): 'main', th1: 'barthread', th2: 'quuxthread'}
    inst.tofile(threaddict)
    print "ok."
