// $Id: picosat_2.cpp.patch,v 1.1 2006/05/31 13:56:12 aehyvari Exp $
#include <vector>
#include <map>
#include <queue>
#include <iostream>
#include <fstream>
#include <assert.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <csignal>

using namespace std;

//======================================================================
double get_cpu_time()
{
    double res;
    struct rusage usage;

    getrusage(RUSAGE_SELF, &usage);

    res = usage.ru_utime.tv_usec + usage.ru_stime.tv_usec;
    res *= 1e-6;
    res += usage.ru_utime.tv_sec + usage.ru_stime.tv_sec; 
	
    return res;
}

void get_line(ifstream &fs, vector<char> &buf)
{
    buf.clear();
    buf.reserve(4096);
    while(!fs.eof()) {
        char ch = fs.get();
        if(ch == '\n' || ch == '\377')
	    break;
        if(ch == '\r') 
	    continue;
        buf.push_back(ch);
    }
    buf.push_back('\0');
    return;
}

int get_token (char * & lp, char * token)
{
    char * wp = token;
    while (*lp && ((*lp == ' ') || (*lp == '\t'))) {
	lp++;
    }
    while (*lp && (*lp != ' ') && (*lp != '\t') && (*lp != '\n')) {
	*(wp++) = *(lp++);
    }
    *wp = '\0';                                 // terminate string
    return wp - token; 
}


//======================================================================

typedef unsigned CLit;

#define UNKNOWN 	2

#define VID(lit)	(lit>>1 )
#define SIGN(lit)	(lit & 1)

#define CID(cl_id)      (cl_id>>1)
#define LEARNED(cl_id)  (cl_id & 1)

// A global variable. Cannot use c++ -features in signal handlers...
// True when splitting should start
bool start_split;

void hup(int sig)
{
    start_split = 1;
}

enum EDeductionStatus 
{
    CONFLICT = 100,
    NO_CONFLICT,
    DLIMIT

};

enum ESolvingStatus
{
    SATISFIABLE = 500,
    UNSATISFIABLE,
    TIME_OUT,
    PARALLEL_LIMIT
};

struct CClause 
{
    vector<CLit> literals;	//literal array, i.e. all the literals in this clause
    unsigned num_0;		//number of value 0 literals
    unsigned num_1;		//number of value 1 literals
};

struct CVariable
{
    unsigned	value;		//could be 0 (false), 1 (true) or UNKNOWN (unassigned)
    int		dlevel;		//decision level
    int 	antecedent;
    bool	marked;
    bool        flipped;
    vector<unsigned> cl_idx[2];	//the clauses that this variable appears
};

struct CImplication
{
    CLit	lit;
    int		antecedent;
};

class CSolver
{
public:
    double		time_limit;
    unsigned		num_decisions;
    unsigned		num_conflicts;
    unsigned		num_implications;
    unsigned            num_learned; // This is almost the same as num_conflicts
    int                 dlimit;         //decision limit
    int                 pinstances;
    int                 plimit; //max num of parallel instances to create
    int                 climit; //min mun of conflicts before instances are created.
    int                 max_clause_len; //>Length of learned clauses
    double              h_accept_prop; // Accept propability for heuristics
    int                 h_pool_size; // If possible, minimum amount of
                                     // variables in heuristic pool.
    string              name;   //instance
    bool                vsids;  //Use vsids-heuristics
    bool                minmaxvsids; //Use modified vsids-heuristics
    bool                randheur; // Use pure random heuristic
    vector<CClause>     learned_clauses; //

private:
    double		init_time;
    unsigned		init_num_clauses;
    
    int			dlevel;		//current decision level
    vector<CClause> 	init_clauses;	//Initial clauses
    vector<CClause>     constraint_clauses; // The split clauses
    vector<CVariable> 	variables;	//all the variables, index 0 is not used		
    vector<vector<CLit> > assign_stack;	//variables assigned at decision level n

    queue<CImplication>	implication_queue;	
    vector<int> 	conflicts;
    int                 first_backtrack; // Indicates whether this is the first backtrack
                                         // after the collecting of
                                         // conflict clauses.
    void verify_integrity(void);
private:
    unsigned literal_value(CLit lit) {	//it will return 0, 1 or something else (but may not be UNKNOWN)
	unsigned vid = VID(lit);
	unsigned sign = SIGN(lit);
	return (sign ^ variables[vid].value); 
    }

    bool time_out(void)	{
	return (get_cpu_time() - init_time > time_limit);
    }

    bool parallel_limit(void) {
        return (plimit >= 0)?(pinstances >= plimit):0;
    }

    void set_var_value (int vid, int value);
    void unset_var_value (int vid);

    void init_solve(void);
    int preprocess(void);
    bool decide_next_branch(void);
    int deduce	(void);
    int analyze_conflicts(void);
    void backtrack(int dlevel);

    int analyze_dlimit(void);

    bool dlimit_trig(int); // True if given dlevel is greater than current dlimit

    map<CLit,unsigned> score;          // score list for heuristics
    map<CLit,unsigned> recent; // recent occurences

public:
    int add_clause( vector<CLit> & lits);
    void set_var_number(int n);

    void read_cnf (char * filename);
    int solve (void);
    void print_solution(ostream & o = cout);

    int add_init_clause( vector<CLit> & lits);
    // Constraint (split) clauses are stored here as well
    void add_constraint_clause( vector<CLit> & lits);
    void dump_cnf(ostream & o = cout);
    void update_heuristics(void);
    void reset_heuristics(void);
};



void CSolver::dump_cnf(ostream & o) {
    // Total num of clauses is 
    // num of clauses + num of set variables - num of too long learned
    unsigned good_learned_clauses = 0;
    if (max_clause_len >= 0) {
        for (unsigned i = 0; i < learned_clauses.size(); i++) {
            if (learned_clauses[i].literals.size() <= max_clause_len)
                ++ good_learned_clauses;
        }
    }
    else {
        // Take 'em all.
        good_learned_clauses = learned_clauses.size();
    }

    unsigned num_clauses = good_learned_clauses + constraint_clauses.size() + dlevel;
/*
    for (unsigned i=1; i < variables.size(); ++i) {
        if (variables[i].value != UNKNOWN)
            ++ num_clauses;
    }
*/

    char *tmp = (char *)malloc(1000);
    char *pname = (char *)malloc(1000);
    char *h_vsids_name = "vsids";
    char *h_minmax_name = "minmax";
    char *h_random_name = "random";
    char *h_name;
    char *fmt = "-%d-%d-%d-%d-%f-%d-%0*d-%s.cnf";

    if (vsids)
        h_name = h_vsids_name;
    else if (minmaxvsids)
        h_name = h_minmax_name;
    else if (randheur)
        h_name = h_random_name;
    else
        h_name = "unknown";

    int n = name.size();
    strncpy(tmp, name.c_str(), 80);
    strncpy(tmp + n, fmt, 80);

    sprintf(pname, tmp, dlimit, plimit, climit,
            max_clause_len, h_accept_prop, h_pool_size,
            5, pinstances, h_name);
    ofstream file(pname, ios::out); 
    file << "p cnf " << variables.size() - 1 << " " << num_clauses << endl;
    file << "c dumped state of minisat" << endl;

    file << "c decision literals" << endl;
    // Print the first literal from each decision level, except 0.
    for (unsigned i=1; i <= dlevel ; ++i) {
        if (SIGN(assign_stack[i][0]) == 0) {
            file << " " << VID(assign_stack[i][0]) << " 0" << endl;
        }
        else if (SIGN(assign_stack[i][0]) == 1) {
            file << "-" << VID(assign_stack[i][0]) << " 0" << endl;
        }
    }


    file << "c learned clauses" << endl;
    for (unsigned i = 0; i < learned_clauses.size(); i++) {

        if (max_clause_len >= 0 &&
                learned_clauses[i].literals.size() > max_clause_len)
            continue;

        CClause cl = learned_clauses[i];
        for (unsigned j = 0; j < cl.literals.size(); j++) {
            if (SIGN(cl.literals[j]) == 1)
                file << "-";
            else
                file << " ";
            file << VID(cl.literals[j]) << " ";
        }
        file << "0" << endl;
    }

    file << "c constraint clauses" << endl;
    for (unsigned i = 0; i < constraint_clauses.size(); ++i) {
        CClause cl = constraint_clauses[i];
        for (unsigned j = 0; j < cl.literals.size(); j++) {
            if (SIGN(cl.literals[j]) == 1)
                file << "-";
            else
                file << " ";
            file << VID(cl.literals[j]) << " ";
        }
        file << "0" << endl;
    }
    file.close();
    free(tmp);
    free(pname);
}

void CSolver::verify_integrity(void)
{
    for (unsigned i=0; i< learned_clauses.size(); ++i) {
	unsigned n0 = 0, n1 = 0;
	CClause & cl = learned_clauses[i];
	for (unsigned j=0; j< cl.literals.size(); ++j) {
	    if (literal_value(cl.literals[j]) == 0)
		++ n0;
	    else if (literal_value(cl.literals[j]) == 1)
		++ n1;
	}
	assert (n0 == cl.num_0 && n1 == cl.num_1);
    }

    for (unsigned i=0; i< init_clauses.size(); ++i) {
	unsigned n0 = 0, n1 = 0;
	CClause & cl = init_clauses[i];
	for (unsigned j=0; j< cl.literals.size(); ++j) {
	    if (literal_value(cl.literals[j]) == 0)
		++ n0;
	    else if (literal_value(cl.literals[j]) == 1)
		++ n1;
	}
	assert (n0 == cl.num_0 && n1 == cl.num_1);
    }

    for (unsigned i=1; i< variables.size(); ++i) {
	if (variables[i].value != UNKNOWN) {
	    if (variables[i].antecedent == -1) {
		assert (variables[i].dlevel == 0 ||
			VID(assign_stack[variables[i].dlevel][0]) == i);
	    }
	    else {
                CClause & cl = (LEARNED(variables[i].antecedent))?
                    learned_clauses[CID(variables[i].antecedent)]:
                    init_clauses[CID(variables[i].antecedent)];

		int dl = 0;
		for (unsigned j=0; j< cl.literals.size(); ++j) {	
		    CLit lit = cl.literals[j];
		    if (VID(lit) == i)
			continue;
		    else {
			assert (literal_value(lit) == 0);
			if (dl < variables[VID(lit)].dlevel)
			    dl = variables[VID(lit)].dlevel;
		    }
		}
		assert (dl == variables[i].dlevel);
	    }
	}
    }
}

/* The dlimit_trig is true for dl if 1/2^dl <= 1 / (plimit - pinstances +
 * 1) <= 1/2^dl + 1
 * Assert that low and hi are not too big. If the first_backtrack is
 * true, we have always crossed the decision limit (unless it is 0)
 */

bool CSolver::dlimit_trig(int dl)
{
    if ( dl == 0 )
        return false;
    if ( first_backtrack )
        return true;

    int r = plimit - pinstances + 1;
    int low = 2 << (dl-1);
    int hi = 2 << dl;
    assert(dl < 29); // Use smaller parallel limit!
    return (r >= low) && (r < hi);
//    return dl >= dlimit;
}

void CSolver::set_var_number(int n)
{
    variables.resize( n + 1);
    for (unsigned i=0; i< variables.size(); ++i) {
	variables[i].value = UNKNOWN;
	variables[i].dlevel = -1;
	variables[i].marked = false;
        variables[i].flipped = false;
    }
}

int CSolver::add_init_clause (vector<CLit> & lits) {
    int cl_id = init_clauses.size();
    int clause = cl_id + cl_id;

    init_clauses.resize(cl_id + 1);

    init_clauses[cl_id].num_0 = 0;
    init_clauses[cl_id].num_1 = 0;

    for (unsigned i = 0; i < lits.size(); ++i) {
        int lit = lits[i];
        int vid = VID(lit);
        int sign = SIGN(lit);
        int l_value = literal_value(lit);

        init_clauses[cl_id].literals.push_back(lit);
        variables[vid].cl_idx[sign].push_back(clause);

        if (l_value == 0)
            ++ init_clauses[cl_id].num_0;
        else if (l_value == 1)
            ++ init_clauses[cl_id].num_1;

        map<CLit,unsigned>::iterator pos;

        pos = score.find(lit);
        if (pos != score.end())
            ++ score[lit];
        else {
            score[lit] = 1;
            recent[lit] = 0;
        }
    }
    return clause;
}

void CSolver::add_constraint_clause(vector<CLit> & lits) {
    int cl_id = constraint_clauses.size();
    constraint_clauses.resize(cl_id + 1);
    for (unsigned i = 0; i < lits.size(); ++i) {
        int lit = lits[i];
        constraint_clauses[cl_id].literals.push_back(lit);
    }
}

void CSolver::reset_heuristics() {
    map<CLit,unsigned>::iterator pos;

/*    pos = init_occurences.find(lit);
    if (pos != init_occurences.end())
        ++ init_occurences[lit];
    else {
        init_occurences[lit] = 1;
        score[lit] = 0;
    } */
}

void CSolver::update_heuristics() {
    map<CLit,unsigned>::iterator pos;
/*
    pos = score.find(lit);
    ++ score[lit];
*/
}

int CSolver::add_clause (vector<CLit> & lits) 
{
    int cl_id = learned_clauses.size();
    int clause = cl_id + cl_id + 1;

    learned_clauses.resize( cl_id + 1);

    learned_clauses[cl_id].num_0 = 0;
    learned_clauses[cl_id].num_1 = 0;

    for (unsigned i=0; i< lits.size(); ++i) {
	int lit = lits[i];
	int vid = VID(lit);
	int sign = SIGN(lit);
	int l_value = literal_value (lit);

	learned_clauses[cl_id].literals.push_back(lit);
	variables[vid].cl_idx[sign].push_back(clause);

	if (l_value == 0)
	    ++ learned_clauses[cl_id].num_0 ;
	else if (l_value == 1)
	    ++ learned_clauses[cl_id].num_1 ;

        map<CLit,unsigned>::iterator pos;

        pos = score.find(lit);
        if (pos != score.end())
            ++ recent[lit];
        else
            recent[lit] = 1;

    }
    return clause;
}

bool CSolver::decide_next_branch()
{
    assert (implication_queue.empty());
    ++ num_decisions;

    ++ dlevel;

    map<CLit, unsigned>::iterator pos;
    unsigned max = 0;

    CLit svar = 0;

    if (vsids) {
        for (pos = score.begin(); pos != score.end(); ++ pos) {
            unsigned vid = VID(pos->first);
            if (variables[vid].value == UNKNOWN) {
                if (pos->second >= max) {
                    max = pos->second;
                    svar = pos->first;
                }
            }
        }
        vector<unsigned> accepts;
        // Get variables that are within p percents of the max score
    for (unsigned i=1; i< variables.size(); ++i) {
	if (variables[i].value == UNKNOWN) {
                int vscore = (score[i+i] > score[i+i+1]) ?
                     score[i+i] : score[i+i+1];
                if ((double)(vscore) >= (double)max*h_accept_prop)
                    accepts.push_back(i);
            }
        }
        if (accepts.size() != 0) {
            // An unbound variable found
            if (h_accept_prop < 1) {
                // Many vars, check that we have enough
                int threshold = (int)(max*h_accept_prop);
                while ((accepts.size() < h_pool_size) &&
                        (threshold >= 0)) {
                    for (unsigned i = 1; i < variables.size(); ++ i) {
                        if (variables[i].value == UNKNOWN) {
                            int vscore = (score[i+i] > score[i+i+1]) ?
                                score[i+i] : score[i+i+1];
                            if (vscore >= threshold)
                                accepts.push_back(i);
                        }
                    }
                    threshold --;
                }
            }
            unsigned vid = accepts[ random() % accepts.size() ];
            svar = (score[vid+vid] > score[vid+vid+1]) ?
                    vid+vid : vid+vid+1;
        }
    }

    else if (minmaxvsids) {
        // search the greatest minimum phase from unassigned vars
        for (unsigned i = 1; i < variables.size(); ++ i) {
            if (variables[i].value == UNKNOWN) {
                unsigned min = (score[i+i] > score[i+i+1]) ?
                    score[i+i+1] : score[i+i];
                if (min >= max)
                    max = min;
            }
        }
        vector<unsigned> mins;
        assert(mins.size() == 0);
        // collect the variables that have the property
        // i.e. have score within p percents of the maximum score.
        for (unsigned i = 1; i < variables.size(); ++ i) {
            if (variables[i].value == UNKNOWN) {
                unsigned min = (score[i+i] > score[i+i+1]) ?
                    score[i+i+1] : score[i+i];
                if ((double)min >= (double)max*h_accept_prop)
                    mins.push_back(i);
            }
        }
        if (mins.size() != 0) {
            // A free variable was found.
            // Choose the highest phase-score from these vars.
            if (h_accept_prop < 1) {
                // If random is applied, check that we have at least the
                // amount of variables we have fixed for the pool size.
                // Or we run out of variables...
                int threshold = (int)(max*h_accept_prop); // floor
                while ((mins.size() < h_pool_size) &&
                        (threshold >= 0)) {
                    // get more variables...
                    for (unsigned i = 1; i < variables.size(); ++ i) {
                        if (variables[i].value == UNKNOWN) {
                            unsigned min = (score[i+i] > score[i+i+1]) ?
                                score[i+i+1] : score[i+i];
                            if (min >= threshold)
                                mins.push_back(i);
                        }
                    }
                    -- threshold;
                }

                // If random is applied, take one of the possibilities
                int var = mins[random() % mins.size()];
                svar = (score[var+var] > score[var+var+1]) ?
                    var+var : var+var+1;
            }
            else {
                // If not, take the best literal among the equally good
                // choices.
                unsigned mscore = 0;
                for (unsigned i = 0; i < mins.size(); ++ i) {
                    int vid = mins[i];
                    unsigned gscore = (score[vid+vid] > score[vid+vid+1]) ?
                        score[vid+vid] : score[vid+vid+1];
                    if (gscore >= mscore) {
                        mscore = gscore;
                        svar = (score[vid+vid] > score[vid+vid+1]) ?
                            vid+vid : vid+vid+1;
                    }
                }
            }
        }
    }

    else if (randheur) {
        vector<unsigned> accepts;
        for (unsigned i = 1; i < variables.size(); ++ i) {
            if (variables[i].value == UNKNOWN) {
                accepts.push_back(i);
            }
        }

        if (accepts.size() != 0) {
            int vid = accepts[random() % accepts.size()];
            int sign = random() % 2;
            svar = vid+vid+sign;
        }
    }

    // update scores
    if ((num_decisions & 255) == 0) {
        for (pos = score.begin(); pos != score.end(); ++ pos) {
            pos->second = recent[pos->first] + pos->second / 2;
        }
        for (pos = recent.begin(); pos != recent.end(); ++ pos) {
            pos->second = 0;
	}
    }

    if (svar != 0) {
	CImplication imp;
	imp.lit = svar;
	imp.antecedent = -1;
	implication_queue.push(imp);
	return true;
    }
    return false;
}

void CSolver::init_solve(void)
{
    init_num_clauses 	= init_clauses.size();
    init_time	 	= get_cpu_time();
    num_decisions	= 0;
    num_conflicts	= 0;
    num_implications	= 0;
    num_learned         = 0;
    pinstances          = 0;
    first_backtrack     = 1;
    assign_stack.resize(variables.size());
    dlevel = 0;
}


int CSolver::preprocess (void)
{
    for (unsigned i=0; i< init_clauses.size(); ++i) {
	assert (init_clauses[i].literals.size() > 0);
	if (init_clauses[i].literals.size() == 1) {
	    CImplication imp;
	    imp.lit = init_clauses[i].literals[0];
	    imp.antecedent = -1;
	    implication_queue.push(imp);
	}
    }
    return deduce();
}

int CSolver::analyze_dlimit(void) {
    assert(conflicts.size() == 0);

}

int CSolver::analyze_conflicts(void)
{
    if (conflicts.size() == 0) {
        if (first_backtrack) {
            // Indicate that the splitting phase will start. We want to
            // do a new selection starting from dl 1 to ascertain that
            // the first problem will not have too many selected
            // literals.
            first_backtrack = 0;
            return 1;
        }
        // We have reached the decision level limit
        assert((dlimit > 0) && dlimit_trig(dlevel));
        vector<CLit> lits;
        dump_cnf();
        for (int i = 1; i <= dlevel; i++) {
            lits.push_back(assign_stack[i][0]^1);
        }
        add_init_clause(lits);
        add_constraint_clause(lits);
        ++ pinstances;
        return 1;
    }

    assert (conflicts.size() > 0) ;

    unsigned cl_id = CID(conflicts[0]);
    CClause & conf_cl = (LEARNED(conflicts[0]))?
        learned_clauses[cl_id]:
        init_clauses[cl_id];

    conflicts.clear();
    ++ num_conflicts;

    if (dlevel == 0)	//current dlevel is 0, so unsatisfiable
	return 0;
    
    // If not, we learn
    ++ num_learned;

    vector<CLit> learned_clause;
    int num_marked_current_dl = 0;

    for (unsigned i=0; i< conf_cl.literals.size(); ++i) {
	CLit lit = conf_cl.literals[i];
	assert (literal_value (lit) == 0);
	variables[VID(lit)].marked = true;	

	if (variables[VID(lit)].dlevel != dlevel)
	    learned_clause.push_back(lit);
	else 
	    ++ num_marked_current_dl;
    }
    assert (num_marked_current_dl > 1);

    vector<CLit> & assign_last_dl = assign_stack[dlevel];

    int i;
    for (i=assign_last_dl.size() - 1; i >= 0; --i) {
	CLit lit = assign_last_dl[i];
	if (variables[VID(lit)].marked == false) //it's not involved
	    continue;

	if (num_marked_current_dl == 1) {
	    learned_clause.push_back(lit^1);
	    break;
	}

        CClause & ante_cl = (LEARNED(variables[VID(lit)].antecedent))?
            learned_clauses[CID(variables[VID(lit)].antecedent)]:
            init_clauses[CID(variables[VID(lit)].antecedent)];

	for (unsigned j=0; j< ante_cl.literals.size(); ++j) {
	    CLit l = ante_cl.literals[j];
	    if (l == lit) //the implied variable
		continue;
	    //not the lit
	    assert (literal_value (l) == 0);
	    if (variables[VID(l)].marked != true) {
		variables[VID(l)].marked = true;
		if (variables[VID(l)].dlevel != dlevel)
		    learned_clause.push_back(l);
		else 
		    ++ num_marked_current_dl;
	    }
	}
	variables[VID(lit)].marked = false;
	-- num_marked_current_dl;
    }
    assert (i >= 0);

    int assert_lit = learned_clause.back();
    int assert_level = 0;
    assert (literal_value(assert_lit) == 0);
    variables[VID(assert_lit)].marked = false;

    for (unsigned i = 0; i < learned_clause.size() - 1; ++i) {
	int lit = learned_clause[i];
	assert (literal_value(lit) == 0);

	CVariable & var = variables[VID(lit)];
	assert (var.marked == true);
	var.marked = false;
	if (var.dlevel > assert_level)
	    assert_level = var.dlevel;
    }

    unsigned clause = add_clause(learned_clause);
    CImplication imp;
    imp.lit = assert_lit;
    imp.antecedent = clause;
    implication_queue.push(imp);

    assert (assert_level < dlevel);
    return assert_level + 1;
}

void CSolver::set_var_value(int vid, int value)
{
    assert (value == 0 || value == 1);
    ++ num_implications;

    CVariable & var = variables[vid];
    var.value = value;

    vector<unsigned> & pos_clauses = var.cl_idx [1 - value];
    for (unsigned i=0; i< pos_clauses.size(); ++i) {
        CClause & cl = (LEARNED(pos_clauses[i]))?
            learned_clauses[CID(pos_clauses[i])]:
            init_clauses[CID(pos_clauses[i])];
	++ cl.num_1;
    }

    vector<unsigned> & neg_clauses = var.cl_idx[value];
    for (unsigned i=0; i< neg_clauses.size(); ++i) {
	int clause = neg_clauses[i];
        CClause & cl = (LEARNED(clause))?
            learned_clauses[CID(clause)]:
            init_clauses[CID(clause)];

	++ cl.num_0;
	if (cl.num_0 == cl.literals.size())
	    conflicts.push_back(neg_clauses[i]);
	else if (cl.num_0 == cl.literals.size() - 1 &&
		 cl.num_1 == 0) {
	    unsigned j;
	    for (j=0; j< cl.literals.size(); ++j) {
		CLit lit = cl.literals[j];
		if (variables[VID(lit)].value == UNKNOWN) {
		    CImplication imp;
		    imp.lit = lit;
		    imp.antecedent = clause;
		    implication_queue.push(imp);
		    break;
		}
	    }
	    assert (j < cl.literals.size());
	}
    }
}

void CSolver::unset_var_value(int vid)
{
    int value = variables[vid].value;
    assert (value != UNKNOWN);

    CVariable & var = variables[vid];

    vector<unsigned> & pos_clauses = var.cl_idx [1 - value];
    for (unsigned i=0; i< pos_clauses.size(); ++i) {
        CClause & cl = (LEARNED(pos_clauses[i]))?
            learned_clauses[CID(pos_clauses[i])]:
            init_clauses[CID(pos_clauses[i])];

	-- cl.num_1;
    }

    vector<unsigned> & neg_clauses = var.cl_idx[value];
    for (unsigned i=0; i< neg_clauses.size(); ++i) {
        CClause & cl = (LEARNED(neg_clauses[i]))?
            learned_clauses[CID(neg_clauses[i])]:
            init_clauses[CID(neg_clauses[i])];
	-- cl.num_0;
    }
    
    var.value = UNKNOWN;
}

void CSolver::backtrack(int back_dl)
{
    assert (back_dl <= dlevel);
    for (int i=dlevel; i >= back_dl; --i) {
	vector<CLit> & assigned_at_dlevel = assign_stack[i];
	for (unsigned j=0; j< assigned_at_dlevel.size(); ++j) {
	    int vid = VID(assigned_at_dlevel[j]);
	    unset_var_value(vid);
	    variables[vid].dlevel = -1;
	}
	assigned_at_dlevel.clear();
    }
    dlevel = back_dl - 1;
}

int CSolver::deduce(void)
{
    while (!implication_queue.empty() && conflicts.size()==0) {
	CLit lit = implication_queue.front().lit;
	int ante = implication_queue.front().antecedent;

	int vid = VID(lit);
	implication_queue.pop();
	
	CVariable & var = variables[vid];
	if ( var.value == UNKNOWN) { // an implication
	    assert (var.dlevel == -1);

	    set_var_value(vid, 1 - SIGN(lit));

	    var.dlevel = dlevel;
	    var.antecedent = ante;
	    assign_stack[dlevel].push_back(lit);
	}
    }
    //if loop exited because of a conflict, we need to clean implication queue
    while(!implication_queue.empty()) 
	implication_queue.pop();

    if (conflicts.size())
        return CONFLICT;
    // This is triggered in case the number of conflicts has reached the
    // initial conflict collecting maximum and dlevel is over dlimit.
    // That means two cases: After the initial collecting, marking the
    // start of splitting phase and when the splitting is in process
    // and dlevel is greater than or equal to dlimit, marking a split.
    // collected conflicts is greater than climit and start_split
    // is on.
    // Since we want to have dlimit as a function of parallel instances,
    // we need one more condition, a function
    // bool dlimit_trig(int dlevel)
    else if ((dlimit >= 0) && (num_conflicts >= climit) &&
                start_split && dlimit_trig(dlevel))
        return DLIMIT;
    else
        return NO_CONFLICT;
}

int CSolver::solve(void)
{
    int status;

    init_solve();
/*
    if(preprocess()==CONFLICT) 
	return UNSATISFIABLE;
*/
    while(!time_out() && (!parallel_limit())) {
        if (preprocess() == CONFLICT) {
            return UNSATISFIABLE;
        }
//	verify_integrity();
	if (decide_next_branch()) {
	    while (deduce()!=NO_CONFLICT) { 
		int blevel = analyze_conflicts();
		if (blevel == 0)
		    return UNSATISFIABLE;
		else {
		    backtrack(blevel);
                    if (parallel_limit())
                        dump_cnf();
                }
	    }
	}
	else 
	    return SATISFIABLE;
    }
    if (parallel_limit()) {
        return PARALLEL_LIMIT;
    }
    return TIME_OUT;
}

void CSolver::read_cnf (char * filename)
{
    ifstream in_file(filename);
    if (!in_file) {
	cerr << "Can't open input CNF file " << filename << endl;
	exit(1);
    }
    unsigned num_cls = 0, num_vars = 0;
    
    vector<char> buffer;
    vector<CLit> literals;
    bool header_encountered = false;
    char token[64];	
    while (!in_file.eof()) {
	get_line(in_file, buffer);
	char * ptr = &(*buffer.begin());
	if (get_token(ptr, token)) {
	    if (strcmp(token, "c")==0)
		continue;
	    else if (strcmp(token, "p")==0) {
		assert (literals.empty());
		assert (header_encountered == false);
		get_token(ptr, token);
		if (strcmp(token, "cnf") != 0) {
		    cerr << "Format Error, p cnf NumVar NumCls " << endl;
		    exit(1);
		}
		get_token(ptr, token);
		num_vars = atoi(token);
		set_var_number(num_vars);

		get_token(ptr, token);
		num_cls = atoi(token);

		header_encountered = true;
		continue;
	    }
	    else {
		int lit = atoi(token);
		assert (lit <= (int) num_vars || lit >= - (int) num_vars);
		if (lit != 0) {
		    if (lit > 0) 
			literals.push_back(lit + lit);
		    else 
			literals.push_back(1- lit - lit);
		}
		else {
		    add_init_clause(literals);
		    literals.clear();
		}
	    }
	}
	while (get_token(ptr, token)) {
	    int lit = atoi(token);
	    assert (lit <= (int) num_vars || lit >= - (int) num_vars);
	    if (lit != 0) {
		if (lit > 0) 
		    literals.push_back(lit + lit);
		else 
		    literals.push_back(1- lit - lit);
	    }
	    else {
		add_init_clause( literals );
		literals.clear();
	    }
	}
    }
    if (!literals.empty()) {
	cerr << "Trailing literals in input without termination " << endl;
	exit(1);
    }
    if ( init_clauses.size() != num_cls ) 
	cerr << "WARNING : Clause count inconsistant with the header " << endl;
    cout << "Successfully read " << init_clauses.size() << " Clauses " << endl;
}

void CSolver::print_solution(ostream & o)
{
    o << "A Satisfiable Assignment is: " << endl;
    for (unsigned i=1; i< variables.size(); ++i) {
	CVariable & var = variables[i];
	if (var.value == 0) 
	    o << "-" << i;
	else if (var.value == 1)
	    o << "+" << i;
	else 
	    o << "(" << i << ")";
	o << "  ";
	if (i%10 == 0) 
	    o << endl;
    }
    o << endl;
}

int main(int argc, char * * argv)
{
    signal(SIGHUP, hup); // Register the SIGHUP handler
    if (argc < 3) {
	cerr << "MiniSAT: A Simple SAT Solver " << endl;
	cerr << "Copyright 2003, Microsoft Corp. " << endl << endl;;
	cerr << "Usage: "<< argv[0] <<
                " cnf { vsids | minmax | random } " <<
                "[tlimit [dlimit plimit climit mlen spool [accept_prop]]]" << endl;
	return 2;
    }

    // Gets set when a SIGHUP is received,
    // or if we are running random heuristic
    start_split = 0;


    double init_time = get_cpu_time();
    CSolver solver;


    if (strncmp(argv[2], "vsids", 5) == 0) {
        solver.vsids = true;
        solver.minmaxvsids = false;
        solver.randheur = false;
    }
    else if (strncmp(argv[2], "minmax", 6) == 0) {
        solver.minmaxvsids = true;
        solver.vsids = false;
        solver.randheur = false;
    }
    else if (strncmp(argv[2], "random", 6) == 0) {
        solver.randheur = true;
        solver.vsids = false;
        solver.minmaxvsids = false;
        start_split = 1; // In random heuristic, we don't need to wait for HUP
    }
    else {
        cerr << "second argument is either random, vsids or minmax" << endl;
        return 2;
    }

    if (argc > 3) {
	solver.time_limit = atoi(argv[3]);
        if (argc >= 9) {
            solver.dlimit = atoi(argv[4]);
            solver.plimit = atoi(argv[5]);
            solver.climit = atoi(argv[6]);
            solver.max_clause_len = atoi(argv[7]);
            solver.h_pool_size = atoi(argv[8]);
            if (argc == 10)
                solver.h_accept_prop = atof(argv[9]);
    else 
                solver.h_accept_prop = 1;
        }
        else {
            solver.dlimit = -1;
            solver.plimit = -1;
            solver.climit = 0;
            solver.max_clause_len = 0;
            solver.h_accept_prop = 1;
            solver.h_pool_size = 0;
        }
    }
    else {
	solver.time_limit = 120;	//default to 2 minutes
        solver.dlimit = -1;
        solver.plimit = -1;
        solver.h_accept_prop = 1;
        solver.h_pool_size = 0;
    }


    solver.name = argv[1];

    string::size_type idx = solver.name.find('.');
    if (idx != string::npos) {
        solver.name = solver.name.substr(0,idx);
    }

    cout << "name is " << solver.name << endl;

    solver.read_cnf(argv[1]);
    cout <<"Solving " << argv[1] << " ......" << endl;


    int status = solver.solve();

    switch(status) {
    case SATISFIABLE: 
	cout << "Instance SATISFIABLE" << endl;
	solver.print_solution();
	break;
    case UNSATISFIABLE:
	cout << "Instance UNSATISFIABLE" << endl;
	break;
    case TIME_OUT:
	cout << "Time Out" << endl;
	break;
    case PARALLEL_LIMIT:
        cout << "Parallel limit reached" << endl;
        break;
    }

    unsigned learned = 0;

    cout << "Num Decisions   \t" << solver.num_decisions << endl;
    cout << "Num Conflicts   \t" << solver.num_conflicts << endl;
    cout << "Num Implications\t" << solver.num_implications << endl;
    cout << "Num Parallels   \t" << solver.pinstances << endl;
    vector<CClause> & learned_clauses = solver.learned_clauses;
    if (solver.max_clause_len >= 0) {
        for (unsigned i = 0; i < learned_clauses.size(); i++)
            if (learned_clauses[i].literals.size() <= solver.max_clause_len)
                ++ learned;
}
    else
        learned = learned_clauses.size();

    cout << "Num Learned     \t" << learned << endl;
    cout << "Total Runtime   \t" << get_cpu_time() - init_time << endl;

    return 0;
}
