#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <string>
#include <vector>

#include "Preferences.h"
#include "CertInfo.h"
#include "DateTime.h"
#include "MdsQuery.h"
#include "Xrsl.h"
#include "Giis.h"
#include "JobSubmission.h"

#include "ngui.h"


extern "C"
int ngsubxx (const std::vector <std::string> & xrslfiles,
	     const std::vector <std::string> & xrslstrings,
	     const std::vector <std::string> & clusterselect_,
	     const std::vector <std::string> & clusterreject_,
	     const std::vector <std::string> & giisurls,
	     const std::string & joblist,
	     const bool dryrun,
	     const bool dumpxrsl,
	     int timeout,
	     int debug,
	     const bool anonymous) {

  std::vector <std::string> clusterselect = ResolveAliases (clusterselect_);
  std::vector <std::string> clusterreject = ResolveAliases (clusterreject_);

  if (xrslfiles.empty() && xrslstrings.empty()) {
    std::cerr << "ngsub: no xrsl input specified" << std::endl;
    std::cerr << "Use \"ngsub -help\" for help" << std::endl;
    return 1;
  }

  std::vector <Xrsl> xrsllist;
  std::vector <Giis> giislist;
  std::vector <Cluster> clusterlist;

  int error = 0;
  ActivateGlobus();

  // setting defaults

  if (timeout == UNDEFINED) timeout = iGetEnv ("NGTIMEOUT");
  if (timeout == UNDEFINED) timeout = iGetDef ("NGTIMEOUT");
  if (timeout == UNDEFINED) timeout = DEFAULT_TIMEOUT;

  if (debug == UNDEFINED) debug = iGetEnv ("NGDEBUG");
  if (debug == UNDEFINED) debug = iGetDef ("NGDEBUG");
  if (debug == UNDEFINED) debug = 0;

  // get user info

  CertInfo user;
  if (!user) {
    error = 1;
    goto errorexit;
  }

  if (debug) {
    std::cout << "User subject name: " << user.GetSN() << std::endl;
    std::cout << "Remaining proxy lifetime: "
	      << Period (user.TimeLeft()) << std::endl;
  }

  // parse xrsl

  for (std::vector <std::string>::const_iterator vsi = xrslfiles.begin();
       vsi != xrslfiles.end(); vsi++) {
    Xrsl xrsl (*vsi, 0);
    if (!xrsl) {
      std::cerr << "The XRSL file \"" << *vsi << "\" could not be parsed" << std::endl;
      error = 1;
      goto errorexit;
    }
    bool done = false;
    for (int i = 0; !done; i++) {
      Xrsl axrsl = xrsl[i];
      if (!axrsl)
	done = true;
      else
	xrsllist.push_back (axrsl);
    }
  }

  for (std::vector <std::string>::const_iterator vsi = xrslstrings.begin();
       vsi != xrslstrings.end(); vsi++) {
    Xrsl xrsl (*vsi);
    if (!xrsl) {
      std::cerr << "The XRSL string \"" << *vsi << "\" could not be parsed" << std::endl;
      error = 1;
      goto errorexit;
    }
    bool done = false;
    for (int i = 0; !done; i++) {
      Xrsl axrsl = xrsl[i];
      if (!axrsl)
	done = true;
      else
	xrsllist.push_back (axrsl);
    }
  }

  for (std::vector <Xrsl>::iterator vxi = xrsllist.begin();
       vxi != xrsllist.end(); vxi++) {

    if (debug > 1) {
      std::cout << "Parsed input XRSL:" << std::endl;
      vxi->Print();
    }

    vxi->Eval();

    if (vxi->Check()) {
      error = 1;
      goto errorexit;
    }
  }

  // find available clusters

  for (std::vector <std::string>::const_iterator vsi = clusterselect.begin();
       vsi != clusterselect.end(); vsi++) {
    bool found = false;
    for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	 !found && cli != clusterlist.end(); cli++)
      if (*vsi == cli->GetName()) found = true;
    if (!found) clusterlist.push_back (*vsi);
  }

  if (clusterlist.empty()) {
    if (GetGiises (giisurls, giislist)) {
      error = 1;
      goto errorexit;
    }

    clusterlist = FindClusters (giislist, user.GetSNx(), anonymous,
				timeout, debug);
    if (clusterlist.empty()) {
      std::cerr << "ngsub: could not retrieve cluster list from giis" << std::endl;
      error = 1;
      goto errorexit;
    }
  }

  for (std::vector <std::string>::const_iterator vsi = clusterreject.begin();
       vsi != clusterreject.end(); vsi++)
    for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	 cli != clusterlist.end(); cli++)
      if (cli->GetName() == *vsi) {
	if (debug) std::cout << "Rejecting cluster: " << *vsi << std::endl;
	clusterlist.erase (cli);
	break;
      }

  FindClusterInfo (clusterlist, Mds::JobSubmission, user.GetSNx(), anonymous,
		   timeout, debug);

  for (std::vector <Cluster>::iterator cli = clusterlist.begin();
       cli != clusterlist.end(); cli++)
    for (std::vector <Queue>::iterator qli = cli->queues.begin();
	 qli != cli->queues.end(); qli++)
      for (std::vector <Job>::iterator jli = qli->jobs.begin();
	   jli != qli->jobs.end(); jli++)
	qli->Accept (jli->GetReqCpuTime(), jli->GetCpuCount());

  for (std::vector <Xrsl>::iterator vxi = xrsllist.begin();
       vxi != xrsllist.end(); vxi++) {
    std::string oldcluster;
    if (JobSubmission (clusterlist, oldcluster, *vxi, joblist, dryrun,
		       dumpxrsl, timeout, debug)) {
      error = 1;
      goto errorexit;
    }
  }

 errorexit:
  DeactivateGlobus();
  return error;
}
