#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <string>
#include <vector>

#include "Preferences.h"
#include "CertInfo.h"
#include "DateTime.h"
#include "MdsQuery.h"
#include "Giis.h"

#include "ngui.h"


extern "C"
int ngstatxx (const std::vector <std::string> & jobs,
	      const std::vector <std::string> & joblists,
	      const std::vector <std::string> & clusterselect_,
	      const std::vector <std::string> & clusterreject_,
	      const std::vector <std::string> & status,
	      const bool all,
	      const std::vector <std::string> & giisurls,
	      const bool clusters,
	      const bool longlist,
	      int timeout,
	      int debug,
	      const bool anonymous) {

  std::vector <std::string> clusterselect = ResolveAliases (clusterselect_);
  std::vector <std::string> clusterreject = ResolveAliases (clusterreject_);

  if ((all || !jobs.empty() || !joblists.empty() || !status.empty()) &&
      clusters) {
    std::cerr << "ngstat: incompatible options" << std::endl;
    std::cerr << "Use \"ngstat -help\" for help" << std::endl;
    return 1;
  }

  int error = 0;
  ActivateGlobus();

  // setting defaults

  if (timeout == UNDEFINED) timeout = iGetEnv ("NGTIMEOUT");
  if (timeout == UNDEFINED) timeout = iGetDef ("NGTIMEOUT");
  if (timeout == UNDEFINED) timeout = DEFAULT_TIMEOUT;

  if (debug == UNDEFINED) debug = iGetEnv ("NGDEBUG");
  if (debug == UNDEFINED) debug = iGetDef ("NGDEBUG");
  if (debug == UNDEFINED) debug = 0;

  // get user info

  CertInfo user;
  if (!user) {
    error = 1;
    goto errorexit;
  }

  if (debug) {
    std::cout << "User subject name: " << user.GetSN() << std::endl;
    std::cout << "Remaining proxy lifetime: "
	      << Period (user.TimeLeft()) << std::endl;
  }

  // finding the information

  if (clusters) {

    std::vector <Giis> giislist;
    std::vector <Cluster> clusterlist;

    for (std::vector <std::string>::const_iterator vsi = clusterselect.begin();
	 vsi != clusterselect.end(); vsi++) {
      bool found = false;
      for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	   !found && cli != clusterlist.end(); cli++)
	if (*vsi == cli->GetName()) found = true;
      if (!found) clusterlist.push_back (*vsi);
    }

    if (clusterlist.empty()) {
      if (GetGiises (giisurls, giislist)) {
	error = 1;
	goto errorexit;
      }

      clusterlist = FindClusters (giislist, user.GetSNx(), anonymous,
				  timeout, debug);
      if (clusterlist.empty()) {
	std::cerr << "ngstat: could not retrieve cluster list from giis" << std::endl;
	error = 1;
	goto errorexit;
      }
    }

    for (std::vector <std::string>::const_iterator vsi = clusterreject.begin();
	 vsi != clusterreject.end(); vsi++)
      for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	   cli != clusterlist.end(); cli++)
	if (cli->GetName() == *vsi) {
	  if (debug) std::cout << "Rejecting cluster: " << *vsi << std::endl;
	  clusterlist.erase (cli);
	  break;
	}

    FindClusterInfo (clusterlist, Mds::ClusterInfo, user.GetSNx(), anonymous,
		     timeout, debug);

    for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	 cli != clusterlist.end(); cli++) {
      if (longlist)
	cli->Print();
      else
	cli->PrintShort();
      for (std::vector <Queue>::iterator qli = cli->queues.begin();
	   qli != cli->queues.end(); qli++) {
	if (longlist)
	  qli->Print();
	else
	  qli->PrintShort();
      }
      std::cout << std::endl;
    }
  }

  else {

    std::vector <std::string> jobids;
    std::vector <Cluster> clusterlist;

    jobids = GetJobIDs (jobs, joblists, clusterselect, clusterreject, status,
			all);

    if (all && jobids.empty()) {
      std::cout << "ngstat: No jobs" << std::endl;
      goto errorexit;
    }

    if (jobids.empty()) {
      std::cerr << "ngstat: no valid jobids given" << std::endl;
      std::cerr << "Use \"ngstat -help\" for help" << std::endl;
      error = 1;
      goto errorexit;
    }

    for (std::vector <std::string>::iterator vsi = jobids.begin();
	 vsi != jobids.end(); vsi++) {
      int pos0 = 0;
      if (vsi->substr (0, 9) == "gsiftp://") pos0 = 9;
      int pos = vsi->find_first_of (":/", pos0);
      if (pos == std::string::npos) {
	std::cerr << "ngstat: invalid jobid: " << *vsi << std::endl;
	error = 1;
	continue;
      }
      std::string clustername = vsi->substr (pos0, pos - pos0);
      bool found = false;
      for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	   !found && cli != clusterlist.end(); cli++)
	if (clustername == cli->GetName()) found = true;
      if (!found) clusterlist.push_back (clustername);
    }

    if (clusterlist.empty()) {
      std::cerr << "ngstat: no valid jobids given" << std::endl;
      error = 1;
      goto errorexit;
    }

    FindClusterInfo (clusterlist, Mds::JobInfo, user.GetSNx(), anonymous,
		     timeout, debug);

    for (std::vector <std::string>::iterator vsi = jobids.begin();
	 vsi != jobids.end(); vsi++) {
      bool found = false;
      for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	   !found && cli != clusterlist.end(); cli++)
	for (std::vector <Queue>::iterator qli = cli->queues.begin();
	     !found && qli != cli->queues.end(); qli++)
	  for (std::vector <Job>::iterator jli = qli->jobs.begin();
	       !found && jli != qli->jobs.end(); jli++)
	    if (*vsi == jli->GetId()) {
	      if (jli->IsStatus (status))
		if (longlist)
		  jli->Print();
		else
		  jli->PrintShort();
	      found = true;
	    }
      if (!found) {
	std::cerr << "ngstat: no job with jobid " << *vsi << " found" << std::endl;
	error = 1;
      }
    }
  }

 errorexit:
  DeactivateGlobus();
  return error;
}
