#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <map>

#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#include <globus_common.h>
#include <globus_rsl.h>

#include "Xrsl.h"
#include "MdsQuery.h"
#include "DateTime.h"
#include "Target.h"

#include "misc/checksum.h"
#include "string.h"


Xrsl::Xrsl (const std::string & xrslstring) {

  xrsl = globus_rsl_parse ((char *) xrslstring.c_str());
}


Xrsl::Xrsl (const std::string & xrslfilename, int dummy) {

  std::ifstream xrslfile (xrslfilename.c_str());
  xrslfile.seekg (0, std::ios::end);
  int length = xrslfile.tellg();
  xrslfile.seekg (0, std::ios::beg);
  char * buffer = new char [length + 1];
  xrslfile.read (buffer, length);
  buffer [length] = 0;
  xrslfile.close();
  xrsl = globus_rsl_parse (buffer);
  delete buffer;
}


Xrsl::Xrsl (const Xrsl & axrsl) {

  xrsl = globus_rsl_copy_recursive (axrsl.xrsl);
}


Xrsl::Xrsl (globus_rsl_t * axrsl) : xrsl (axrsl) {}


Xrsl::~Xrsl () {

  if (xrsl) globus_rsl_free_recursive (xrsl);
}


Xrsl & Xrsl::operator= (const Xrsl & axrsl) {

  if (this != &axrsl) {
    if (xrsl) globus_rsl_free_recursive (xrsl);
    xrsl = globus_rsl_copy_recursive (axrsl.xrsl);
  }
  return *this;
}


bool Xrsl::operator! () const { return !xrsl; }

bool Xrsl::valid () const { return xrsl; }


Xrsl Xrsl::operator[] (const int ix) const {

  if (globus_rsl_is_boolean_multi (xrsl)) {
    if (ix >= 0) {
      globus_list_t * rlist = globus_rsl_boolean_get_operand_list (xrsl);
      for (int i = 0; i < ix && !globus_list_empty (rlist); i++)
	rlist = globus_list_rest (rlist);
      if (!globus_list_empty (rlist))
	return globus_rsl_copy_recursive
	  ((globus_rsl_t *) globus_list_first (rlist));
    }
  }
  else if (ix == 0)
    return *this;
  return NULL;
}


void Xrsl::Print () const {

  if (xrsl) globus_rsl_print_recursive (xrsl);
}


const std::string Xrsl::str () const {

  char * buffer = globus_rsl_unparse (xrsl);
  const std::string xrslstr = buffer;
  globus_free (buffer);
  return xrslstr;
}


void Xrsl::AddSimpleRelation (const std::string & attr, const std::string & value) {

  globus_rsl_value_t * rvalue =
    globus_rsl_value_make_literal (strdup (value.c_str()));
  globus_list_t * rlist =
    globus_list_cons ((void *) rvalue, NULL);
  globus_rsl_value_t * rsequence =
    globus_rsl_value_make_sequence (rlist);
  globus_rsl_t * relation =
    globus_rsl_make_relation (GLOBUS_RSL_EQ, strdup (attr.c_str()), rsequence);
  globus_list_t ** head = FindHead();
  globus_list_insert (head, (void *) relation);
}


void Xrsl::AddSimpleRelationSet (const std::string & attr,
				 const std::vector <std::string> & values, bool select) {

  globus_list_t * oplist = NULL;
  for (std::vector <std::string>::const_iterator vsi = values.begin();
       vsi != values.end(); vsi++) {
    globus_rsl_value_t * rvalue =
      globus_rsl_value_make_literal (strdup (vsi->c_str()));
    globus_list_t * rlist =
      globus_list_cons ((void *) rvalue, NULL);
    globus_rsl_value_t * rsequence =
      globus_rsl_value_make_sequence (rlist);
    globus_rsl_t * relation =
      globus_rsl_make_relation ((select ? GLOBUS_RSL_EQ : GLOBUS_RSL_NEQ),
				strdup (attr.c_str()), rsequence);
    oplist = globus_list_cons ((void  *) relation, oplist);
  }
  globus_rsl_t * relation =
    globus_rsl_make_boolean ((select ? GLOBUS_RSL_OR : GLOBUS_RSL_AND),
			     oplist);
  globus_list_t ** head = FindHead();
  globus_list_insert (head, (void *) relation);
}


int Xrsl::RemoveRelation (const std::string & attr, globus_rsl_t * axrsl) {

  if (!axrsl) axrsl = xrsl;

  if (globus_rsl_is_boolean (axrsl)) {
    globus_list_t * rlist = globus_rsl_boolean_get_operand_list (axrsl);
    while (!globus_list_empty (rlist)) {
      bool deleted = false;
      globus_rsl_t * bxrsl = (globus_rsl_t *) globus_list_first (rlist);
      if (globus_rsl_is_relation (bxrsl)) {
	if (globus_rsl_is_relation_attribute_equal (bxrsl,
						    (char *) attr.c_str())) {
	  globus_list_t ** head =
	    globus_rsl_boolean_get_operand_list_ref (axrsl);
	  globus_list_t * rlistsave = rlist;
	  rlist = globus_list_rest (rlist);
	  globus_list_remove (head, rlistsave);
	  globus_rsl_free_recursive (bxrsl);
	  deleted = true;
	}
      }
      else
	if (RemoveRelation (attr, bxrsl))
	  return 1;
      if (!deleted) rlist = globus_list_rest (rlist);
    }
  }

  return 0;
}


int Xrsl::Check (globus_rsl_t * axrsl) const {

  if (!axrsl) axrsl = xrsl;

  if (globus_rsl_is_boolean (axrsl)) {
    globus_list_t * rlist = globus_rsl_boolean_get_operand_list (axrsl);
    while (!globus_list_empty (rlist)) {
      if (Check ((globus_rsl_t *) globus_list_first (rlist))) return 1;
      rlist = globus_list_rest (rlist);
    }
    return 0;
  }

  else if (globus_rsl_is_relation (axrsl)) {

    if (globus_rsl_is_relation_attribute_equal (axrsl, "executable") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "arguments") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "inputfiles") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "outputfiles") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "executables") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "jobname") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "stdin") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "stdout") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "stderr") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "gmlog") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "join") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "notify") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "cluster") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "queue") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "starttime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "cputime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "walltime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "gridtime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "lifetime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "memory") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "disk") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "runtimeenvironment") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "middleware") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "opsys") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "replicacollection") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "rerun") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "architecture") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "dryrun") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "rsl_substitution") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "environment") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "ftpthreads") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "jobtype") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "count") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "cache") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "nodeaccess") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "jobreport") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "benchmarks"))
      return 0;

    if (globus_rsl_is_relation_attribute_equal (axrsl, "sstdin") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "action") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "savestate") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "lrmstype") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "hostname") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "jobid")) {
      std::cerr << "Error: The XRSL contains the internal argument \""
	   << globus_rsl_relation_get_attribute (axrsl) << "\"" << std::endl;
      return 1;
    }

    if (globus_rsl_is_relation_attribute_equal (axrsl,
						"resourcemanagercontact") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "directory") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "maxwalltime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "maxcputime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "maxtime") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "maxmemory") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "minmemory") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "maxdisk") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "stdlog") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "grammyjob") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "project") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "hostcount") ||
	globus_rsl_is_relation_attribute_equal (axrsl,
						"parallelenvironment") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "label") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "subjobcommstype") ||
	globus_rsl_is_relation_attribute_equal (axrsl, "subjobstarttype")) {
      std::cerr << "Warning: the XRSL contains the deprecated argument \""
	   << globus_rsl_relation_get_attribute (axrsl) << "\"" << std::endl;
      std::cerr << "It will be ignored" << std::endl;
      return 0;
    }

    std::cerr << "Warning: the XRSL contains the unknown argument \""
	 << globus_rsl_relation_get_attribute (axrsl) << "\"" << std::endl;
    std::cerr << "It will be ignored" << std::endl;
    return 0;

  }
  else {
    std::cerr << "Error: Unexpected XRSL token" << std::endl;
    return 1;
  }
}


int Xrsl::FindRelation (const std::string & attr, globus_rsl_t ** relation,
			globus_rsl_t * axrsl) const {

  if (!axrsl) {
    axrsl = xrsl;
    *relation = NULL;
  }

  if (globus_rsl_is_boolean (axrsl)) {
    globus_list_t * rlist = globus_rsl_boolean_get_operand_list (axrsl);
    while (!globus_list_empty (rlist)) {
      if (FindRelation (attr, relation,
			(globus_rsl_t *) globus_list_first (rlist)))
	return 1;
      rlist = globus_list_rest (rlist);
    }
    return 0;
  }
  else if (globus_rsl_is_relation (axrsl)) {
    if (globus_rsl_is_relation_attribute_equal (axrsl,
						(char *) attr.c_str())) {
      if (*relation) {
	std::cerr << "Error: Multiple definitions of XRSL attribute \"" << attr
	     << "\"" << std::endl;
	return 1;
      }
      *relation = axrsl;
    }
    return 0;
  }
  else {
    std::cerr << "Error: Unexpected XRSL token" << std::endl;
    return 1;
  }
}


Env::Sign IntToSign (const int i) {

  switch (i) {
  case GLOBUS_RSL_EQ:
    return Env::eq;
    break;
  case GLOBUS_RSL_NEQ:
    return Env::ne;
    break;
  case GLOBUS_RSL_GT:
    return Env::gt;
    break;
  case GLOBUS_RSL_GTEQ:
    return Env::ge;
    break;
  case GLOBUS_RSL_LT:
    return Env::lt;
    break;
  case GLOBUS_RSL_LTEQ:
    return Env::le;
    break;
  }
}


int Xrsl::Collect (Target & target, const std::string & attr,
		   globus_rsl_t * axrsl) {

  if (!axrsl) axrsl = xrsl;

  if (globus_rsl_is_boolean (axrsl)) {

    std::vector <EnvironmentTest> envtestlist;

    globus_list_t * rlist = globus_rsl_boolean_get_operand_list (axrsl);
    while (!globus_list_empty (rlist)) {
      bool deleted = false;
      globus_rsl_t * bxrsl = (globus_rsl_t *) globus_list_first (rlist);
      if (globus_rsl_is_relation (bxrsl)) {
	if (globus_rsl_is_relation_attribute_equal (bxrsl,
						    (char *) attr.c_str())) {
	  globus_rsl_value_t * relvalue =
	    globus_rsl_relation_get_single_value (bxrsl);
	  std::string relstr = globus_rsl_value_literal_get_string (relvalue);
	  bool found = false;
	  for (std::vector <EnvironmentTest>::iterator veti = envtestlist.begin();
	       !found && veti != envtestlist.end(); veti++)
	    found = veti->AddCondition
	      (relstr, IntToSign (globus_rsl_relation_get_operator (bxrsl)));
	  if (!found) {
	    EnvironmentTest envtest = EnvironmentTest
	      (relstr, IntToSign (globus_rsl_relation_get_operator (bxrsl)));
	    envtestlist.push_back (envtest);
	  }
	  globus_list_t ** head =
	    globus_rsl_boolean_get_operand_list_ref (axrsl);
	  globus_list_t * rlistsave = rlist;
	  rlist = globus_list_rest (rlist);
	  globus_list_remove (head, rlistsave);
	  globus_rsl_free_recursive (bxrsl);
	  deleted = true;
	}
      }
      else
	if (Collect (target, attr, bxrsl))
	  return 1;
      if (!deleted) rlist = globus_list_rest (rlist);
    }

    if (!envtestlist.empty()) {
      globus_list_t * xlist = NULL;
      globus_list_t ** ylist = &xlist;
      for (std::vector <EnvironmentTest>::iterator veti = envtestlist.begin();
	   veti != envtestlist.end(); veti++) {
	std::string value = target.FindEnvironment (attr, *veti)->GetOriginal();
	globus_rsl_value_t * rvalue =
	  globus_rsl_value_make_literal (strdup (value.c_str()));
	*ylist = globus_list_cons ((void *) rvalue, NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      globus_rsl_value_t * rsequence =
	globus_rsl_value_make_sequence (xlist);
      globus_rsl_t * relation =
	globus_rsl_make_relation (GLOBUS_RSL_EQ, strdup (attr.c_str()),
				  rsequence);
      globus_list_t ** head = globus_rsl_boolean_get_operand_list_ref (axrsl);
      globus_list_insert (head, (void *) relation);
    }
  }
  return 0;
}


globus_list_t ** Xrsl::FindHead () {

  if (!globus_rsl_is_boolean_and (xrsl)) {
    globus_list_t * children = globus_list_cons ((void *) xrsl, NULL);
    xrsl = globus_rsl_make_boolean (GLOBUS_RSL_AND, children);
  }
  return globus_rsl_boolean_get_operand_list_ref (xrsl);
}


int Xrsl::FixJoin () {

  globus_rsl_t * stdoutrelation = NULL;
  globus_rsl_t * stderrrelation = NULL;
  globus_rsl_t * joinrelation = NULL;

  if (FindRelation ("stdout", &stdoutrelation)) return 1;
  if (FindRelation ("stderr", &stderrrelation)) return 1;
  if (FindRelation ("join", &joinrelation)) return 1;

  bool join = false;

  if (joinrelation) {
    globus_rsl_value_t * joinvalue =
      globus_rsl_relation_get_single_value (joinrelation);
    if (!joinvalue) {
      std::cerr << "Error: XRSL attribute \"join\" not single value" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (joinvalue)) {
      std::cerr << "Error: XRSL attribute \"join\" not string literal" << std::endl;
      return 1;
    }
    if (!strcasecmp (globus_rsl_value_literal_get_string (joinvalue), "yes") ||
	!strcasecmp (globus_rsl_value_literal_get_string (joinvalue), "true"))
      join = true;
  }

  if (join && !stdoutrelation) {
    std::cerr << "Error: XRSL attribute \"join\" is true, but XRSL attribute \"stdout\" is not set" << std::endl;
    return 1;
  }

  if (join && stderrrelation) {
    std::cerr << "Error: XRSL attribute \"join\" is true, but XRSL attribute \"stderr\" is set" << std::endl;
    return 1;
  }

  if (join) {
    globus_rsl_value_t * stdoutvalue =
      globus_rsl_relation_get_single_value (stdoutrelation);
    if (!stdoutvalue) {
      std::cerr << "Error: XRSL attribute \"stdout\" not single value" << std::endl;
      return 1;
    }
    globus_list_t * stderrlist = globus_list_cons
      ((void *) globus_rsl_value_copy_recursive (stdoutvalue), NULL);
    globus_rsl_value_t * stderrsequence =
      globus_rsl_value_make_sequence (stderrlist);
    stderrrelation =
      globus_rsl_make_relation (GLOBUS_RSL_EQ, strdup ("stderr"),
				stderrsequence);
    globus_list_t ** head = FindHead();
    globus_list_insert (head, (void *) stderrrelation);
  }

  return 0;
}


int Xrsl::FixInOut (const std::string & stdattr, const std::string & inoutattr) {

  globus_rsl_t * stdrelation = NULL;
  globus_rsl_t * inoutrelation = NULL;

  if (FindRelation (stdattr, &stdrelation)) return 1;
  if (FindRelation (inoutattr, &inoutrelation)) return 1;

  if (stdrelation) {
    globus_rsl_value_t * stdvalue =
      globus_rsl_relation_get_single_value (stdrelation);
    if (!stdvalue) {
      std::cerr << "Error: XRSL attribute \"" << stdattr << "\" not single value"
	   << std::endl;
      return 1;
    }
    if (globus_rsl_value_is_literal (stdvalue)) {
      std::string stdfile = globus_rsl_value_literal_get_string (stdvalue);
      if (stdfile[0] != '/' && stdfile[0] != '$') {
	if (inoutrelation) {
	  globus_rsl_value_t * inoutsequence =
	    globus_rsl_relation_get_value_sequence (inoutrelation);
	  globus_list_t * inoutlist =
	    globus_rsl_value_sequence_get_value_list (inoutsequence);
	  bool found = false;
	  while (!globus_list_empty (inoutlist) && !found) {
	    globus_rsl_value_t * filesequence =
	      (globus_rsl_value_t *) globus_list_first (inoutlist);
	    if (!globus_rsl_value_is_sequence (filesequence)) {
	      std::cerr << "Error: XRSL syntax error in attribute \"" << inoutattr
		   << "\"" << std::endl;
	      return 1;
	    }
	    globus_list_t * filelist =
	      globus_rsl_value_sequence_get_value_list (filesequence);
	    globus_rsl_value_t * filevalue =
	      (globus_rsl_value_t *) globus_list_first (filelist);
	    if (globus_rsl_value_is_literal (filevalue))
	      if (globus_rsl_value_literal_get_string (filevalue) == stdfile)
		found = true;
	    inoutlist = globus_list_rest (inoutlist);
	  }
	  if (!found) {
	    globus_rsl_value_t * rvalue =
	      globus_rsl_value_make_literal (strdup (""));
	    globus_list_t * rlist = globus_list_cons
	      ((void *) globus_rsl_value_copy_recursive (stdvalue), NULL);
	    globus_list_insert (globus_list_rest_ref (rlist), (void *) rvalue);
	    globus_rsl_value_t * rsequence =
	      globus_rsl_value_make_sequence (rlist);
	    globus_list_t ** inoutlist =
	      globus_rsl_value_sequence_get_list_ref (inoutsequence);
	    globus_list_insert (inoutlist, (void *) rsequence);
	  }
	}
	else {
	  globus_rsl_value_t * rvalue =
	    globus_rsl_value_make_literal (strdup (""));
	  globus_list_t * rlist = globus_list_cons
	    ((void *) globus_rsl_value_copy_recursive (stdvalue), NULL);
	  globus_list_insert (globus_list_rest_ref (rlist), (void *) rvalue);
	  globus_rsl_value_t * rsequence =
	    globus_rsl_value_make_sequence (rlist);
	  globus_list_t * inoutlist =
	    globus_list_cons ((void *) rsequence, NULL);
	  globus_rsl_value_t * inoutsequence =
	    globus_rsl_value_make_sequence (inoutlist);
	  inoutrelation =
	    globus_rsl_make_relation (GLOBUS_RSL_EQ,
				      strdup (inoutattr.c_str()),
				      inoutsequence);
	  globus_list_t ** head = FindHead();
	  globus_list_insert (head, (void *) inoutrelation);
	}
      }
    }
  }

  return 0;
}


int Xrsl::FixExec () {

  globus_rsl_t * execrelation = NULL;
  globus_rsl_t * argsrelation = NULL;

  if (FindRelation ("executable", &execrelation)) return 1;
  if (FindRelation ("arguments", &argsrelation)) return 1;

  if (!execrelation) {
    std::cerr << "Error: XRSL attribute \"executable\" undefined" << std::endl;
    return 1;
  }

  globus_rsl_value_t * execvalue =
    globus_rsl_relation_get_single_value (execrelation);

  if (!execvalue) {
    std::cerr << "Error: XRSL attribute \"executable\" not single value" << std::endl;
    return 1;
  }

  if (argsrelation) {
    globus_rsl_value_t * argssequence =
      globus_rsl_relation_get_value_sequence (argsrelation);
    globus_list_t ** argslist =
      globus_rsl_value_sequence_get_list_ref (argssequence);
    globus_list_insert
      (argslist, (void *) globus_rsl_value_copy_recursive (execvalue));
  }
  else {
    globus_list_t * argslist = globus_list_cons
      ((void *) globus_rsl_value_copy_recursive (execvalue), NULL);
    globus_rsl_value_t * argssequence =
      globus_rsl_value_make_sequence (argslist);
    argsrelation =
      globus_rsl_make_relation (GLOBUS_RSL_EQ, strdup ("arguments"),
				argssequence);
    globus_list_t ** head = FindHead();
    globus_list_insert (head, (void *) argsrelation);
  }

  globus_rsl_value_t * execsequence =
    globus_rsl_relation_get_value_sequence (execrelation);
  globus_list_t * execlist =
    globus_rsl_value_sequence_get_value_list (execsequence);
  globus_rsl_value_t * newvalue =
    globus_rsl_value_make_literal (strdup ("/bin/echo"));
  globus_rsl_value_free_recursive
    ((globus_rsl_value_t *) globus_list_first (execlist));
  globus_list_replace_first (execlist, newvalue);

  return 0;
}


int Xrsl::FixExecs () {

  globus_rsl_t * execrelation = NULL;
  globus_rsl_t * execsrelation = NULL;

  if (FindRelation ("executable", &execrelation)) return 1;
  if (FindRelation ("executables", &execsrelation)) return 1;

  if (!execrelation) {
    std::cerr << "Error: XRSL attribute \"executable\" undefined" << std::endl;
    return 1;
  }

  globus_rsl_value_t * execvalue =
    globus_rsl_relation_get_single_value (execrelation);

  if (!execvalue) {
    std::cerr << "Error: XRSL attribute \"executable\" not single value" << std::endl;
    return 1;
  }

  if (globus_rsl_value_is_literal (execvalue)) {
    std::string executable = globus_rsl_value_literal_get_string (execvalue);
    if (executable[0] != '/' && executable[0] != '$') {
      if (execsrelation) {
	globus_rsl_value_t * execssequence =
	  globus_rsl_relation_get_value_sequence (execsrelation);
	globus_list_t ** execslist =
	  globus_rsl_value_sequence_get_list_ref (execssequence);
	globus_list_insert
	  (execslist, (void *) globus_rsl_value_copy_recursive (execvalue));
      }
      else {
	globus_list_t * execslist = globus_list_cons
	  ((void *) globus_rsl_value_copy_recursive (execvalue), NULL);
	globus_rsl_value_t * execssequence =
	  globus_rsl_value_make_sequence (execslist);
	execsrelation =
	  globus_rsl_make_relation (GLOBUS_RSL_EQ, strdup ("executables"),
				    execssequence);
	globus_list_t ** head = FindHead();
	globus_list_insert (head, (void *) execsrelation);
      }
    }
  }

  return 0;
}


int Xrsl::FixStdin () {

  globus_rsl_t * stdinrelation = NULL;

  if (FindRelation ("stdin", &stdinrelation)) return 1;

  if (stdinrelation) {
    globus_free (globus_rsl_relation_get_attribute (stdinrelation));
    stdinrelation->req.relation.attribute_name = strdup ("sstdin");
  }

  return 0;
}


int Xrsl::FixTime (const std::string & attr) {

  globus_rsl_t * timerelation;
  if (FindRelation (attr, &timerelation)) return 1;

  if (timerelation) {
    globus_rsl_value_t * timevalue =
      globus_rsl_relation_get_single_value (timerelation);

    if (!timevalue) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" not single value"
	   << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (timevalue)) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" not string literal"
	   << std::endl;
      return 1;
    }

    std::string time_s = globus_rsl_value_literal_get_string (timevalue);

    if (!IsGlobusTime (time_s)) {
      if (!IsUserTime (time_s)) {
	std::cerr << "Error: XRSL attribute \"" << attr << "\" has invalid format"
	     << std::endl;
	return 1;
      }
      time_s = GlobusTime (time_s);

      globus_rsl_value_t * timesequence =
	globus_rsl_relation_get_value_sequence (timerelation);
      globus_list_t * timelist =
	globus_rsl_value_sequence_get_value_list (timesequence);
      globus_rsl_value_t * newvalue =
	globus_rsl_value_make_literal (strdup (time_s.c_str()));
      globus_rsl_value_free_recursive
	((globus_rsl_value_t *) globus_list_first (timelist));
      globus_list_replace_first (timelist, newvalue);
    }
  }

  return 0;
}


int Xrsl::FixPeriod (const std::string & attr) {

  globus_rsl_t * periodrelation;
  if (FindRelation (attr, &periodrelation)) return 1;

  if (periodrelation) {
    globus_rsl_value_t * periodvalue =
      globus_rsl_relation_get_single_value (periodrelation);

    if (!periodvalue) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" not single value"
	   << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (periodvalue)) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" not string literal"
	   << std::endl;
      return 1;
    }

    int seconds = Seconds (globus_rsl_value_literal_get_string (periodvalue));

    if (seconds == -1) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" has invalid format"
	   << std::endl;
      return 1;
    }

#ifdef HAVE_SSTREAM
    std::stringstream ss;
#else
    strstream ss;
#endif

    ss << seconds;

    globus_rsl_value_t * periodsequence =
      globus_rsl_relation_get_value_sequence (periodrelation);
    globus_list_t * periodlist =
      globus_rsl_value_sequence_get_value_list (periodsequence);
#ifdef HAVE_SSTREAM
    globus_rsl_value_t * newvalue =
      globus_rsl_value_make_literal (strdup (ss.str().c_str()));
#else
    ss << ends;
    globus_rsl_value_t * newvalue =
      globus_rsl_value_make_literal (strdup (ss.str()));
    ss.freeze (false);
#endif
    globus_rsl_value_free_recursive
      ((globus_rsl_value_t *) globus_list_first (periodlist));
    globus_list_replace_first (periodlist, newvalue);
  }

  return 0;
}


int Xrsl::UpdateAttribute (const std::string & attr, const std::string & value) {

  globus_rsl_t * xrslrelation;
  if (FindRelation (attr, &xrslrelation)) return 1;

  if (xrslrelation) {
    globus_rsl_value_t * xrslvalue =
      globus_rsl_relation_get_single_value (xrslrelation);

    if (!xrslvalue) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" not single value"
       << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (xrslvalue)) {
      std::cerr << "Error: XRSL attribute \"" << attr << "\" not string literal"
       << std::endl;
      return 1;
    }

    globus_rsl_value_t * xrslsequence =
      globus_rsl_relation_get_value_sequence (xrslrelation);
    globus_list_t * xrsllist =
      globus_rsl_value_sequence_get_value_list (xrslsequence);
    globus_rsl_value_t * newvalue =
      globus_rsl_value_make_literal (strdup (value.c_str()));
    globus_rsl_value_free_recursive
      ((globus_rsl_value_t *) globus_list_first (xrsllist));
    globus_list_replace_first (xrsllist, newvalue);
  }

  return 0;
}


int Xrsl::PrepareUpload (std::vector <std::string> & filelist) {

  globus_rsl_t * inputfilesrelation = NULL;

  if (FindRelation ("inputfiles", &inputfilesrelation)) return 1;

  if (inputfilesrelation) {

    globus_rsl_value_t * sequence =
      globus_rsl_relation_get_value_sequence (inputfilesrelation);
    if (!globus_rsl_value_is_sequence (sequence)) {
      std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\"" << std::endl;
      return 1;
    }

    globus_list_t * list1 =
      globus_rsl_value_sequence_get_value_list (sequence);

    while (!globus_list_empty (list1)) {

      globus_rsl_value_t * value1 =
	(globus_rsl_value_t *) globus_list_first (list1);
      if (!globus_rsl_value_is_sequence (value1)) {
	std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\"" << std::endl;
	return 1;
      }

      globus_list_t * list2 =
	globus_rsl_value_sequence_get_value_list (value1);

      int paircnt = 0;
      std::string rurl;
      std::string lurl;

      while (!globus_list_empty (list2)) {

	globus_rsl_value_t * value2 =
	  (globus_rsl_value_t *) globus_list_first (list2);

	switch (paircnt) {

	case 0:
	  if (globus_rsl_value_is_literal (value2))
	    rurl = globus_rsl_value_literal_get_string (value2);
	  break;

	case 1:
	  if (globus_rsl_value_is_literal (value2)) {
	    lurl = globus_rsl_value_literal_get_string (value2);
	    if (lurl.empty()) lurl = rurl;
	    if (lurl.substr (0, 7) == "file://" ||
		lurl.find ("://") == std::string::npos) {

	      if (rurl.empty()) {
		std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\""
		     << std::endl;
		return 1;
	      }

	      if (lurl.substr (0, 7) == "file://")
		lurl = lurl.substr (7);
	      else if (lurl[0] != '/') {
		char buffer[PATH_MAX];
		getcwd (buffer, PATH_MAX);
		lurl = (std::string) buffer + '/' + lurl;
	      }

	      std::ifstream file (lurl.c_str());

	      if (!file) {
		std::cerr << "Error: could not open input file " << lurl << std::endl;
		return 1;
	      }

	      file.seekg (0, std::ios::end);
	      size_t length = file.tellg();
	      file.seekg (0, std::ios::beg);

	      unsigned long int crc;
	      if (length < 1024 * 1024) {
		CRC32Sum cs;
		char buffer [4096];
		while (file) {
		  file.read (buffer, 4096);
		  ssize_t l = file.gcount();
		  cs.add (buffer, l);
		}
		cs.end();
		crc = cs.crc();
	      }

	      file.close();

#ifdef HAVE_SSTREAM
	      std::stringstream ss;
#else
	      strstream ss;
#endif

	      if (length < 1024 * 1024)
		ss << length << "." << crc;
	      else
		ss << length;

#ifdef HAVE_SSTREAM
	      globus_rsl_value_t * newvalue =
		globus_rsl_value_make_literal (strdup (ss.str().c_str()));
#else
	      ss << ends;
	      globus_rsl_value_t * newvalue =
		globus_rsl_value_make_literal (strdup (ss.str()));
	      ss.freeze (false);
#endif
	      globus_rsl_value_free_recursive
		((globus_rsl_value_t *) globus_list_first (list2));
	      globus_list_replace_first (list2, newvalue);

	      filelist.push_back (rurl);
	      filelist.push_back (lurl);
	    }
	  }
	  break;
	}

	paircnt++;
	list2 = globus_list_rest (list2);
      }
      if (paircnt != 2) {
	std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\"" << std::endl;
	return 1;
      }
      list1 = globus_list_rest (list1);
    }
  }

  return 0;
}


int Xrsl::GetJobName (std::string & jobname) const {

  jobname = "";

  globus_rsl_t * jobnamerelation;
  if (FindRelation ("jobname", &jobnamerelation)) return 1;

  if (jobnamerelation) {
    globus_rsl_value_t * jobnamevalue =
      globus_rsl_relation_get_single_value (jobnamerelation);
    if (!jobnamevalue) {
      std::cerr << "Error: XRSL attribute \"jobname\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (jobnamevalue)) {
      std::cerr << "Error: XRSL attribute \"jobname\" not string literal" << std::endl;
      return 1;
    }
    jobname = globus_rsl_value_literal_get_string (jobnamevalue);
  }

  return 0;
}


std::string Xrsl::GetName () const {

  std::string jobname;
  GetJobName (jobname);
  return jobname;
}


int Xrsl::GetCount (int * count) const {

  *count = 1;

  globus_rsl_t * countrelation;
  if (FindRelation ("count", &countrelation)) return 1;

  if (countrelation) {
    globus_rsl_value_t * countvalue =
      globus_rsl_relation_get_single_value (countrelation);
    if (!countvalue) {
      std::cerr << "Error: XRSL attribute \"count\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (countvalue)) {
      std::cerr << "Error: XRSL attribute \"count\" not string literal" << std::endl;
      return 1;
    }
    *count = atoi (globus_rsl_value_literal_get_string (countvalue));
  }

  return 0;
}


int Xrsl::GetCpuCount () const {

  int count;
  if (GetCount (&count)) return -1;
  return count;
}


int Xrsl::GetCpuTime (long int * cputime) const {

  *cputime = UNDEFINED;

  globus_rsl_t * cputimerelation;
  if (FindRelation ("cputime", &cputimerelation)) return 1;

  if (cputimerelation) {
    globus_rsl_value_t * cputimevalue =
      globus_rsl_relation_get_single_value (cputimerelation);
    if (!cputimevalue) {
      std::cerr << "Error: XRSL attribute \"cputime\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (cputimevalue)) {
      std::cerr << "Error: XRSL attribute \"cputime\" not string literal" << std::endl;
      return 1;
    }

    *cputime = Seconds (globus_rsl_value_literal_get_string (cputimevalue));

    if (*cputime == -1) {
      std::cerr << "Error: XRSL attribute \"cputime\" has invalid format" << std::endl;
      return 1;
    }
  }

  return 0;
}


int Xrsl::GetWallTime (long int * walltime) const {

  *walltime = UNDEFINED;

  globus_rsl_t * walltimerelation;
  if (FindRelation ("walltime", &walltimerelation)) return 1;

  if (walltimerelation) {
    globus_rsl_value_t * walltimevalue =
      globus_rsl_relation_get_single_value (walltimerelation);
    if (!walltimevalue) {
      std::cerr << "Error: XRSL attribute \"walltime\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (walltimevalue)) {
      std::cerr << "Error: XRSL attribute \"walltime\" not string literal" << std::endl;
      return 1;
    }

    *walltime = Seconds (globus_rsl_value_literal_get_string (walltimevalue));

    if (*walltime == -1) {
      std::cerr << "Error: XRSL attribute \"walltime\" has invalid format" << std::endl;
      return 1;
    }
  }

  return 0;
}


int Xrsl::GetGridTime (long int * gridtime) const {

  *gridtime = UNDEFINED;

  globus_rsl_t * gridtimerelation;
  if (FindRelation ("gridtime", &gridtimerelation)) return 1;

  if (gridtimerelation) {
    globus_rsl_value_t * gridtimevalue =
      globus_rsl_relation_get_single_value (gridtimerelation);
    if (!gridtimevalue) {
      std::cerr << "Error: XRSL attribute \"gridtime\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (gridtimevalue)) {
      std::cerr << "Error: XRSL attribute \"gridtime\" not string literal" << std::endl;
      return 1;
    }

    *gridtime = Seconds (globus_rsl_value_literal_get_string (gridtimevalue));

    if (*gridtime == -1) {
      std::cerr << "Error: XRSL attribute \"gridtime\" has invalid format" << std::endl;
      return 1;
    }
  }

  return 0;
}


int Xrsl::GetBenchmarks (std::map<std::string, std::pair<float, long int> > & bmrk) const {

  globus_rsl_t * benchmarksrelation = NULL;

  if (FindRelation ("benchmarks", &benchmarksrelation)) return 1;

  if (benchmarksrelation) {

    globus_rsl_value_t * sequence =
      globus_rsl_relation_get_value_sequence (benchmarksrelation);
    if (!globus_rsl_value_is_sequence (sequence)) {
      std::cerr << "Error: XRSL syntax error in attribute \"benchmarks\"" << std::endl;
      return 1;
    }

    globus_list_t * list1 =
      globus_rsl_value_sequence_get_value_list (sequence);

    while (!globus_list_empty (list1)) {

      globus_rsl_value_t * value1 =
	(globus_rsl_value_t *) globus_list_first (list1);
      if (!globus_rsl_value_is_sequence (value1)) {
	std::cerr << "Error: XRSL syntax error in attribute \"benchmarks\"" << std::endl;
	return 1;
      }

      globus_list_t * list2 =
	globus_rsl_value_sequence_get_value_list (value1);

      int tripletcnt = 0;
      std::string name;
      float value;
      long int time;

      while (!globus_list_empty (list2)) {

	globus_rsl_value_t * value2 =
	  (globus_rsl_value_t *) globus_list_first (list2);

	switch (tripletcnt) {

	case 0:
	  if (globus_rsl_value_is_literal (value2))
	    name = globus_rsl_value_literal_get_string (value2);
	  break;

	case 1:
	  if (globus_rsl_value_is_literal (value2))
	    value = atof (globus_rsl_value_literal_get_string (value2));
	  break;

	case 2:
	  if (globus_rsl_value_is_literal (value2))
	    time = Seconds (globus_rsl_value_literal_get_string (value2));
	  break;
	}

	tripletcnt++;
	list2 = globus_list_rest (list2);
      }
      if (tripletcnt != 3) {
	std::cerr << "Error: XRSL syntax error in attribute \"benchmarks\"" << std::endl;
	return 1;
      }

      bmrk [name] = std::make_pair (value, time);

      list1 = globus_list_rest (list1);
    }
  }

  return 0;
}


int Xrsl::GetDisk (long long int * disk) const {

  *disk = UNDEFINED;

  globus_rsl_t * diskrelation;
  if (FindRelation ("disk", &diskrelation)) return 1;

  if (diskrelation) {
    globus_rsl_value_t * diskvalue =
      globus_rsl_relation_get_single_value (diskrelation);
    if (!diskvalue) {
      std::cerr << "Error: XRSL attribute \"disk\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (diskvalue)) {
      std::cerr << "Error: XRSL attribute \"disk\" not string literal" << std::endl;
      return 1;
    }
    *disk =
      atoll (globus_rsl_value_literal_get_string (diskvalue)) * 1024 * 1024;
  }

  return 0;
}


int Xrsl::GetClientXrsl (std::string & clientxrsl) const {

  clientxrsl = "";

  globus_rsl_t * clientxrslrelation;
  if (FindRelation ("clientxrsl", &clientxrslrelation)) return 1;

  if (clientxrslrelation) {
    globus_rsl_value_t * clientxrslvalue =
      globus_rsl_relation_get_single_value (clientxrslrelation);
    if (!clientxrslvalue) {
      std::cerr << "Error: XRSL attribute \"clientxrsl\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (clientxrslvalue)) {
      std::cerr << "Error: XRSL attribute \"clientxrsl\" not string literal"
	   << std::endl;
      return 1;
    }
    clientxrsl = globus_rsl_value_literal_get_string (clientxrslvalue);
  }

  return 0;
}


int Xrsl::GetInputFiles (std::vector <std::string> & filelist) const {

  globus_rsl_t * inputfilesrelation = NULL;

  if (FindRelation ("inputfiles", &inputfilesrelation)) return 1;

  if (inputfilesrelation) {

    globus_rsl_value_t * sequence =
      globus_rsl_relation_get_value_sequence (inputfilesrelation);
    if (!globus_rsl_value_is_sequence (sequence)) {
      std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\"" << std::endl;
      return 1;
    }

    globus_list_t * list1 =
      globus_rsl_value_sequence_get_value_list (sequence);

    while (!globus_list_empty (list1)) {

      globus_rsl_value_t * value1 =
	(globus_rsl_value_t *) globus_list_first (list1);
      if (!globus_rsl_value_is_sequence (value1)) {
	std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\"" << std::endl;
	return 1;
      }

      globus_list_t * list2 =
	globus_rsl_value_sequence_get_value_list (value1);

      int paircnt = 0;
      std::string rurl;
      std::string lurl;

      while (!globus_list_empty (list2)) {

	globus_rsl_value_t * value2 =
	  (globus_rsl_value_t *) globus_list_first (list2);

	switch (paircnt) {

	case 0:
	  if (globus_rsl_value_is_literal (value2))
	    rurl = globus_rsl_value_literal_get_string (value2);
	  break;

	case 1:
	  if (globus_rsl_value_is_literal (value2)) {
	    lurl = globus_rsl_value_literal_get_string (value2);
	    if (lurl.empty()) lurl = rurl;
	    filelist.push_back (lurl);
	  }
	  break;
	}

	paircnt++;
	list2 = globus_list_rest (list2);
      }
      if (paircnt != 2) {
	std::cerr << "Error: XRSL syntax error in attribute \"inputfiles\"" << std::endl;
	return 1;
      }
      list1 = globus_list_rest (list1);
    }
  }

  return 0;
}


int Xrsl::GetRc (std::string & rc) const {

  rc = "";

  globus_rsl_t * rcrelation;
  if (FindRelation ("replicacollection", &rcrelation)) return 1;

  if (rcrelation) {
    globus_rsl_value_t * rcvalue =
      globus_rsl_relation_get_single_value (rcrelation);
    if (!rcvalue) {
      std::cerr << "Error: XRSL attribute \"replicacollection\" not single valued"
	   << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (rcvalue)) {
      std::cerr << "Error: XRSL attribute \"replicacollection\" not string literal"
	   << std::endl;
      return 1;
    }
    rc = globus_rsl_value_literal_get_string (rcvalue);
  }

  return 0;
}


int Xrsl::GetDefaultCache (bool * cache) const {

  *cache = true;

  globus_rsl_t * cacherelation;
  if (FindRelation ("cache", &cacherelation)) return 1;

  if (cacherelation) {
    globus_rsl_value_t * cachevalue =
      globus_rsl_relation_get_single_value (cacherelation);
    if (!cachevalue) {
      std::cerr << "Error: XRSL attribute \"cache\" not single valued" << std::endl;
      return 1;
    }
    if (!globus_rsl_value_is_literal (cachevalue)) {
      std::cerr << "Error: XRSL attribute \"cache\" not string literal" << std::endl;
      return 1;
    }
    std::string cachestring = globus_rsl_value_literal_get_string (cachevalue);
    if (cachestring [0] == 'n' || cachestring [0] == 'N' || // no
	cachestring [0] == 'f' || cachestring [0] == 'F' || // false
	cachestring [0] == '0')
      *cache = false;
  }

  return 0;
}


void Xrsl::Eval () {

  globus_symboltable_t table;

  globus_symboltable_init (&table,
			   globus_hashtable_string_hash,
			   globus_hashtable_string_keyeq);
  globus_rsl_eval (xrsl, &table);
  globus_symboltable_destroy (&table);
}


int Xrsl::Test (Target & target, std::string & failattr,
		globus_rsl_t * axrsl) const {

  static globus_list_t * xlist;
  static globus_list_t ** ylist;
  static bool result;

  if (!axrsl) {
    axrsl = xrsl;
    xlist = NULL;
    ylist = &xlist;
  }

  if (globus_rsl_is_boolean (axrsl)) {

    if (globus_rsl_is_boolean_and (axrsl)) {
      globus_list_t * rlist = globus_rsl_boolean_get_operand_list (axrsl);
      while (!globus_list_empty (rlist)) {
	if (Test (target, failattr,
		  (globus_rsl_t *) globus_list_first (rlist)))
	  goto errorexit;
	if (!result) break;
	rlist = globus_list_rest (rlist);
      }
    }

    else if (globus_rsl_is_boolean_or (axrsl)) {
      globus_list_t * xlistsave = xlist;
      globus_list_t ** ylistsave = ylist;
      globus_list_t * rlist = globus_rsl_boolean_get_operand_list (axrsl);
      while (!globus_list_empty (rlist)) {
	xlist = NULL;
	ylist = &xlist;
	if (Test (target, failattr,
		  (globus_rsl_t *) globus_list_first (rlist))) {
	  xlist = xlistsave;
	  ylist = ylistsave;
	  goto errorexit;
	}
	if (result) {
	  if (!xlist) ylist = ylistsave;
	  if (xlistsave) {
	    *ylistsave = xlist;
	    xlist = xlistsave;
	  }
	  break;
	}
	while (!globus_list_empty (xlist)) {
	  globus_rsl_free_recursive
	    ((globus_rsl_t *) globus_list_first (xlist));
	  globus_list_remove (&xlist, xlist);
	}
	rlist = globus_list_rest (rlist);
      }
      if (!result) {
	xlist = xlistsave;
	ylist = ylistsave;
      }
    }

    else {
      std::cerr << "Error: Unimplemented XRSL boolean operator" << std::endl;
      goto errorexit;
    }
  }

  else if (globus_rsl_is_relation (axrsl)) {
    if (globus_rsl_is_relation_attribute_equal (axrsl, "cluster")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"cluster\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"cluster\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      if (globus_rsl_relation_get_operator (axrsl) == GLOBUS_RSL_EQ) {
	result = (strcasecmp (target.cluster->GetName().c_str(), svalue) == 0);
      }
      else if (globus_rsl_relation_get_operator (axrsl) == GLOBUS_RSL_NEQ) {
	result = (strcasecmp (target.cluster->GetName().c_str(), svalue) != 0);
      }
      else {
	std::cerr << "Error: Unexpected operator for XRSL attribute \"cluster\""
	     << std::endl;
	goto errorexit;
      }
      if (!result) failattr = "cluster";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "queue")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"queue\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"queue\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      if (globus_rsl_relation_get_operator (axrsl) == GLOBUS_RSL_EQ) {
	result = (strcasecmp (target.queue->GetName().c_str(), svalue) == 0);
      }
      else if (globus_rsl_relation_get_operator (axrsl) == GLOBUS_RSL_NEQ) {
	result = (strcasecmp (target.queue->GetName().c_str(), svalue) != 0);
      }
      else {
	std::cerr << "Error: Unexpected operator for XRSL attribute \"queue\""
	     << std::endl;
	goto errorexit;
      }
      if (!result) failattr = "queue";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "count")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"count\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"count\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      int count = atoi (svalue);
      result = (target.GetTotalCpus() >= count);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "count";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "memory")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"memory\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"memory\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      int memory = atoi (svalue);
      result = (target.GetNodeMemory() != UNDEFINED &&
		target.GetNodeMemory() >= memory);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "memory";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "architecture")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"architecture\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"architecture\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      result = !strcasecmp (target.GetArchitecture().c_str(), svalue);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "architecture";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "middleware")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"middleware\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"middleware\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      std::string strvalue = svalue;
      EnvironmentTest envtest = EnvironmentTest
	(strvalue, IntToSign (globus_rsl_relation_get_operator (axrsl)));
      globus_list_t * tmplist = xlist;
      while (!globus_list_empty (tmplist)) {
	globus_rsl_t * tmpxrsl = (globus_rsl_t *) globus_list_first (tmplist);
	if (globus_rsl_is_relation_attribute_equal (tmpxrsl, "middleware")) {
	  globus_rsl_value_t * tmpvalue =
	    globus_rsl_relation_get_single_value (tmpxrsl);
	  std::string tmpstr = globus_rsl_value_literal_get_string (tmpvalue);
	  envtest.AddCondition
	    (tmpstr, IntToSign (globus_rsl_relation_get_operator (tmpxrsl)));
	}
	tmplist = globus_list_rest (tmplist);
      }
      result = target.FindEnvironment ("middleware", envtest);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "middleware";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl,
						     "runtimeenvironment")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"runtimeenvironment\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"runtimeenvironment\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      std::string strvalue = svalue;
      EnvironmentTest envtest = EnvironmentTest
	(strvalue, IntToSign (globus_rsl_relation_get_operator (axrsl)));
      globus_list_t * tmplist = xlist;
      while (!globus_list_empty (tmplist)) {
	globus_rsl_t * tmpxrsl = (globus_rsl_t *) globus_list_first (tmplist);
	if (globus_rsl_is_relation_attribute_equal (tmpxrsl,
						    "runtimeenvironment")) {
	  globus_rsl_value_t * tmpvalue =
	    globus_rsl_relation_get_single_value (tmpxrsl);
	  std::string tmpstr = globus_rsl_value_literal_get_string (tmpvalue);
	  envtest.AddCondition
	    (tmpstr, IntToSign (globus_rsl_relation_get_operator (tmpxrsl)));
	}
	tmplist = globus_list_rest (tmplist);
      }
      result = target.FindEnvironment ("runtimeenvironment", envtest);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "runtimeenvironment";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "opsys")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"opsys\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"opsys\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      std::string strvalue = svalue;
      EnvironmentTest envtest = EnvironmentTest
	(strvalue, IntToSign (globus_rsl_relation_get_operator (axrsl)));
      globus_list_t * tmplist = xlist;
      while (!globus_list_empty (tmplist)) {
	globus_rsl_t * tmpxrsl = (globus_rsl_t *) globus_list_first (tmplist);
	if (globus_rsl_is_relation_attribute_equal (tmpxrsl, "opsys")) {
	  globus_rsl_value_t * tmpvalue =
	    globus_rsl_relation_get_single_value (tmpxrsl);
	  std::string tmpstr = globus_rsl_value_literal_get_string (tmpvalue);
	  envtest.AddCondition
	    (tmpstr, IntToSign (globus_rsl_relation_get_operator (tmpxrsl)));
	}
	tmplist = globus_list_rest (tmplist);
      }
      result = target.FindEnvironment ("opsys", envtest);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "opsys";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "nodeaccess")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"nodeaccess\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"nodeaccess\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      result = target.cluster->HaveNodeAccess (svalue);
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "nodeaccess";
    }
    else if (globus_rsl_is_relation_attribute_equal (axrsl, "lifetime")) {
      globus_rsl_value_t * avalue =
	globus_rsl_relation_get_single_value (axrsl);
      if (!avalue) {
	std::cerr << "Error: XRSL attribute \"lifetime\" is not single valued"
	     << std::endl;
	goto errorexit;
      }
      char * svalue = globus_rsl_value_literal_get_string (avalue);
      if (!svalue) {
	std::cerr << "Error: XRSL attribute \"lifetime\" is not a string literal"
	     << std::endl;
	goto errorexit;
      }
      result = (target.cluster->GetSessionDirLifetime() == UNDEFINED ||
		target.cluster->GetSessionDirLifetime() >= Seconds (svalue));
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
      if (!result) failattr = "lifetime";
    }
    else {
      result = true;
      if (result) {
	*ylist = globus_list_cons ((void *) globus_rsl_copy_recursive (axrsl),
				   NULL);
	ylist = globus_list_rest_ref (*ylist);
      }
    }
  }
  else {
    std::cerr << "Error: Unexpected XRSL token" << std::endl;
    goto errorexit;
  }

  if (xrsl == axrsl) {
    if (result)
      target.xrsl = globus_rsl_make_boolean (GLOBUS_RSL_AND, xlist);
    else {
      while (!globus_list_empty (xlist)) {
	globus_rsl_free_recursive
	  ((globus_rsl_t *) globus_list_first (xlist));
	globus_list_remove (&xlist, xlist);
      }
    }
  }

  return 0;

 errorexit:
  while (!globus_list_empty (xlist)) {
    globus_rsl_free_recursive
      ((globus_rsl_t *) globus_list_first (xlist));
    globus_list_remove (&xlist, xlist);
  }
  return 1;
}


std::string Xrsl::TestTarget (Target & target) const {

  std::string failattr;
  if (Test(target, failattr)) return "-1"; // bit of a hack, but..
  return failattr;
}
