#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "datamove/datapoint.h"
#include "datamove/datahandle.h"

#include <algorithm>
#include <string>
#include <vector>
#include <map>
#include <iostream>

#include "RemoteFile.h"
#include "MdsQuery.h"
#include "Xrsl.h"
#include "Utils.h"


LocationInfo::LocationInfo (const std::string & url_) : url (url_) {}


const std::string & LocationInfo::GetUrl () const { return url; }


const std::string LocationInfo::GetHost () const {

  int pos0 = url.find ("://");
  if (pos0 == std::string::npos)
    pos0 = 0;
  else
    pos0 += 3;
  int pos1 = url.find_first_of (":/", pos0);
  if (pos1 == std::string::npos)
    return url.substr (pos0);
  else
    return url.substr (pos0, pos1 - pos0);
}


Location::Location (const std::string & url_,
		    const std::string & optstring) : url (url_) {

  options = ParseOptionString (optstring);
}


const std::string & Location::GetUrl () const { return url; }


bool Location::GetBooleanOption (const std::string & opt, bool * value) const {

  return ::GetBooleanOption (options, opt, value);
}


RemoteFileInfo::RemoteFileInfo (const std::string & url_) : url (url_),
							    size (0),
							    queried (false),
							    iscatalog (false) {}


bool RemoteFileInfo::IsCatalog () const { return iscatalog; }


RemoteFile::RemoteFile (const std::string & url,
			const std::string & optstring,
			const std::string & locstring) {

  fileinfo = new RemoteFileInfo (url);
  if (!fileinfo) return;

  options = ParseOptionString (optstring);

  if (!locstring.empty()) {
    int pos = 0;
    while (pos != std::string::npos) {
      int barpos = locstring.find ('|', pos);
      std::string entry;
      if (barpos == std::string::npos)
	entry = locstring.substr (pos);
      else
	entry = locstring.substr (pos, barpos - pos);
      int semicpos = entry.find (';');
      if (semicpos == 0)
	defaultoptions = ParseOptionString (entry.substr (1));
      else {
	std::string name;
	std::string optstring;
	if (semicpos == std::string::npos)
	  name = entry;
	else {
	  name = entry.substr (0, semicpos);
	  optstring = entry.substr (semicpos + 1);
	}
	loclist.push_back (Location (name, optstring));
      }
      pos = barpos;
      if (pos != std::string::npos) pos++;
    }
  }

  bool found = false;
  for (std::vector <RemoteFileInfo *>::iterator vrcii = filecloud.begin();
       !found && vrcii != filecloud.end(); vrcii++)
    if (*fileinfo == **vrcii) {
      delete fileinfo;
      fileinfo = *vrcii;
      found = true;
    }
  if (!found) filecloud.push_back (fileinfo);
}


const std::string & RemoteFileInfo::GetUrl () const {

  return url;
}


long long int RemoteFileInfo::GetSize () {

  if (Query()) return -1;
  return size;
}


bool RemoteFile::IsCatalog () const {

  return fileinfo->IsCatalog ();
}


void RemoteFileInfo::RegisterCachedFile (const std::string & cluster) {

  if (!IsCached (cluster)) cached.push_back (cluster);
}


bool RemoteFileInfo::IsCached (const std::string & cluster) const {

  return (find (cached.begin(), cached.end(), cluster) != cached.end());
}


bool RemoteFileInfo::operator== (const RemoteFileInfo & file) const {

  return (url == file.url);
}


int RemoteFileInfo::Query () {

  if (queried) return 0;
  queried = true;

  size = 0;

  DataPoint u (url.c_str());
  if (u) {
    iscatalog = u.meta();
    std::list<DataPoint::FileInfo> files;
    if (iscatalog) {
      u.list_files (files);
    }
    else {
      DataHandle h(&u);
      h.secure (false);
      h.list_files(files);
    }
    if (files.empty()) {
      std::cerr << "Error when querying " << url << std::endl;
      return 1;
    }
    for (std::list<DataPoint::FileInfo>::iterator i = files.begin();
	 i != files.end(); i++) {
      if(i->size_available) {
	size = i->size;
      }
      for (std::list<std::string>::iterator u = i->urls.begin();
	   u != i->urls.end(); u++) {
	LocationInfo locinfo (*u);
	if (locinfo.GetUrl().substr (0, 8) == "cache://")
	  RegisterCachedFile (locinfo.GetHost());
	loclist.push_back (locinfo);
      }
    }
    return 0;
  }

  std::cerr << "Error when querying " << url << std::endl;
  return 1;
}


long long int RemoteFile::GetSize () const {

  return fileinfo->GetSize();
}


void RemoteFile::RegisterCachedFile (const std::string & cluster) const {

  fileinfo->RegisterCachedFile (cluster);
}


FileLocation RemoteFile::Local (Cluster * c, const bool defaultcache) const {

  if (fileinfo->Query()) return IOERROR;

  if (fileinfo->IsCatalog()) {

    bool foundlocal = false;
    bool found = false;
    bool cache = defaultcache;

    for (std::vector <Location>::const_iterator vrli = loclist.begin();
	 !foundlocal && vrli != loclist.end(); vrli++) {
      for (std::vector <LocationInfo>::const_iterator vrlii =
	     fileinfo->loclist.begin(); !foundlocal &&
	     vrlii != fileinfo->loclist.end(); vrlii++) {
	if (vrli->GetUrl() == vrlii->GetUrl() ||
	    vrli->GetUrl() == vrlii->GetHost()) {
	  if (c->MatchLocalSe (vrlii->GetUrl())) foundlocal = true;
	  bool local = false;
	  if (!vrli->GetBooleanOption ("local", &local))
	    GetBooleanOption (defaultoptions, "local", &local);
	  if (foundlocal || (!found && !local)) {
	    found = true;
	    cache = defaultcache;
	    if (!vrli->GetBooleanOption ("cache", &cache))
	      GetBooleanOption (defaultoptions, "cache", &cache);
	  }
	}
      }
    }

    if (!foundlocal && (loclist.empty() || !defaultoptions.empty())) {
      for (std::vector <LocationInfo>::const_iterator vrlii =
	     fileinfo->loclist.begin(); !foundlocal &&
	     vrlii != fileinfo->loclist.end(); vrlii++) {
	if (c->MatchLocalSe (vrlii->GetUrl())) foundlocal = true;
	bool local = false;
	GetBooleanOption (defaultoptions, "local", &local);
	if (foundlocal || (!found && !local)) {
	  found = true;
	  cache = defaultcache;
	  GetBooleanOption (defaultoptions, "cache", &cache);
	}
      }
    }

    if (!found) return NOLOCATION;
    if (cache && fileinfo->IsCached (c->GetName())) return CACHED;
    if (foundlocal) return (cache ? LOCAL : LOCALNOCACHE);
    return (cache ? REMOTE : REMOTENOCACHE);
  }

  else if (fileinfo->GetUrl().substr (0, 7) == "file://")
    return REMOTENOCACHE;

  else {
    bool foundlocal = c->MatchLocalSe (fileinfo->GetUrl());
    bool local = false;
    bool cache = defaultcache;

    GetBooleanOption (options, "local", &local);
    GetBooleanOption (options, "cache", &cache);
    if (!foundlocal && local) return NOLOCATION;
    if (cache && fileinfo->IsCached (c->GetName())) return CACHED;
    if (foundlocal) return (cache ? LOCAL : LOCALNOCACHE);
    return (cache ? REMOTE : REMOTENOCACHE);
  }
}


RemoteFileQuery::RemoteFileQuery (const RemoteFileQuery & query) : filelist (query.filelist) {}


RemoteFileQuery::~RemoteFileQuery () {}


RemoteFileQuery & RemoteFileQuery::operator= (const RemoteFileQuery & query) {

  if (this != &query)
    filelist = query.filelist;
  return *this;
}


void RemoteFileQuery::AddFile (RemoteFile * const file) {

  filelist.push_back (file);
}


int RemoteFileQuery::AddSizes (Cluster * c,
			       long long int * cachesize,
			       long long int * sessdirsize,
			       long long int * remotesize,
			       long long int * localsize,
			       const bool defaultcache) const {

  *cachesize = 0;
  *sessdirsize = 0;
  *remotesize = 0;
  *localsize = 0;

  for (std::vector <RemoteFile *>::const_iterator vrfi = filelist.begin();
       vrfi != filelist.end(); vrfi++) {
    switch ((*vrfi)->Local (c, defaultcache)) {
    case REMOTE:
      *remotesize += (*vrfi)->GetSize();
      *cachesize += (*vrfi)->GetSize();
      break;
    case LOCAL:
      *localsize += (*vrfi)->GetSize();
      *cachesize += (*vrfi)->GetSize();
      break;
    case REMOTENOCACHE:
      *remotesize += (*vrfi)->GetSize();
      *sessdirsize += (*vrfi)->GetSize();
      break;
    case LOCALNOCACHE:
      *localsize += (*vrfi)->GetSize();
      *sessdirsize += (*vrfi)->GetSize();
      break;
    case CACHED:
      break;
    case NOLOCATION:
      *cachesize = *sessdirsize = *remotesize = *localsize = UNDEFINED;
      return 0;
      break;
    case IOERROR:
      *cachesize = *sessdirsize = *remotesize = *localsize = UNDEFINED;
      return 1;
      break;
    }
  }
  return 0;
}


int RemoteFileQuery::RegisterCachedFiles (Cluster * c,
					  const bool defaultcache) const {

  for (std::vector <RemoteFile *>::const_iterator vrfi = filelist.begin();
       vrfi != filelist.end(); vrfi++) {
    switch ((*vrfi)->Local (c, defaultcache)) {
    case REMOTE:
    case LOCAL:
      (*vrfi)->RegisterCachedFile (c->GetName());
      break;
    case REMOTENOCACHE:
    case LOCALNOCACHE:
    case CACHED:
      break;
    case NOLOCATION:
    case IOERROR:
      return 1;
      break;
    }
  }
  return 0;
}


std::vector <RemoteFileInfo *> RemoteFile::filecloud;
