#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <string>
#include <vector>
#include <map>
#include <algorithm>
#include <ctype.h>

#include <globus_gsi_credential.h>
#include <globus_gsi_system_config.h>

#include "MdsQuery.h"
#include "LdapQuery.h"
#include "Giis.h"
#include "DateTime.h"
#include "Environment.h"
#include "Utils.h"


// need to wrap for gcc 3.2
inline int to_lower (int x) { return tolower(x); }


struct callback_type {
  std::vector <Giis> * giislist;
  std::vector <Cluster> * clusterlist;
};


Job::Job (const std::string & id_) : id           (id_),
				     reqcputime   (UNDEFINED),
				     reqwalltime  (UNDEFINED),
				     queuerank    (UNDEFINED),
				     usedcputime  (UNDEFINED),
				     usedwalltime (UNDEFINED),
				     usedmem      (UNDEFINED),
				     cpucount     (UNDEFINED),
				     exitcode     (UNDEFINED) {}


const std::string & Job::GetId () const { return id; }

const std::string & Job::GetStatus () const { return status; }

const std::string & Job::GetName () const { return jobname; }

const std::string & Job::GetErrors() const { return errors; }

const std::string & Job::GetStdout () const { return stdout; }

const std::string & Job::GetStderr () const { return stderr; }

const std::string & Job::GetRerunable () const { return rerunable; }

long int Job::GetReqCpuTime () const { return reqcputime; }

int Job::GetCpuCount () const { return cpucount; }

void Job::SetAttr (const std::string & attr, const std::string & value) {

  if (attr == "nordugrid-job-globalid")
    id = value;
  else if (attr == "nordugrid-job-globalowner")
    owner = UnX509 (value);
  else if (attr == "nordugrid-job-execcluster")
    execcluster = value;
  else if (attr == "nordugrid-job-execqueue")
    execqueue = value;
  else if (attr == "nordugrid-job-stdout")
    stdout = value;
  else if (attr == "nordugrid-job-stderr")
    stderr = value;
  else if (attr == "nordugrid-job-stdin")
    stdin = value;
  else if (attr == "nordugrid-job-reqcputime")
    reqcputime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-job-reqcput")
    // backward compatibility
    reqcputime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-job-status") {
    status = value;
    // backward compatibility for pre-0.5.14 servers
    int pos = status.find(" at: ");
    if (pos != std::string::npos) {
      completiontime = status.substr (pos + 5, 15);
      status = status.substr (0, pos);
    }
    // backward compatibility for pre-0.5.14 servers
    if (status == "FINISHED" && !errors.empty()) {
      if (errors.find ("User requested to cancel the job") != std::string::npos)
	status = "KILLED";
      else
	status = "FAILED";
    }
    // map some obsolete states to new ones
    if (status == "PENDING: ACCEPTED")  status = "ACCEPTED";
    if (status == "PENDING:ACCEPTED")   status = "ACCEPTED";
    if (status == "PENDING: PREPARING") status = "PREPARED";
    if (status == "PENDING:PREPARING")  status = "PREPARED";
    if (status == "PENDING: INLRMS")    status = "EXECUTED";
    if (status == "PENDING:INLRMS")     status = "EXECUTED";
    if (status == "SUBMIT")             status = "SUBMITTING";
    if (status == "CANCELING")          status = "KILLING";
    if (status == "CANCELLING")         status = "KILLING";
    if (status == "CANCELED")           status = "KILLED";
    if (status == "CANCELLED")          status = "KILLED";
    if (status.substr (0, 7) == "INLRMS:") {
      if (status[7] == ' ') status.erase (7, 1);
      status = status.substr (0, 8);
    }
  }
  else if (attr == "nordugrid-job-queuerank")
    queuerank = atoi (value.c_str());
  else if (attr == "nordugrid-job-comment")
    comment = value;
  else if (attr == "nordugrid-job-lrmscomment")
    // backward compatibility
    comment = value;
  else if (attr == "nordugrid-job-submissionui")
    submissionui = value;
  else if (attr == "nordugrid-job-submissiontime")
    submissiontime = value;
  else if (attr == "nordugrid-job-usedcputime")
    usedcputime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-job-usedwalltime")
    usedwalltime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-job-sessiondirerasetime")
    erasetime = value;
  else if (attr == "nordugrid-job-usedmem")
    usedmem = atoi (value.c_str());
  else if (attr == "nordugrid-job-errors") {
    errors = value;
    // backward compatibility for pre-0.5.14 servers
    if (status == "FINISHED") {
      if (errors.find ("User requested to cancel the job") != std::string::npos)
	status = "KILLED";
      else
	status = "FAILED";
    }
    int pos = errors.find ("Job exit code is ");
    if (pos != std::string::npos) {
      pos += 17;
      int pos2 = errors.find (" != 0", pos);
      if (pos2 != std::string::npos)
	exitcode = atoi (errors.substr (pos, pos2 - pos).c_str());
    }
  }
  else if (attr == "nordugrid-job-jobname")
    jobname = value;
  else if (attr == "nordugrid-job-runtimeenvironment")
    runtimeenv.push_back (value);
  else if (attr == "nordugrid-job-cpucount")
    cpucount = atoi (value.c_str());
  else if (attr == "nordugrid-job-executionnodes")
    execnodes.push_back (value);
  else if (attr == "nordugrid-job-gmlog")
    gmlog = value;
  else if (attr == "nordugrid-job-clientsoftware")
    clientsoftware = value;
  else if (attr == "nordugrid-job-proxyexpirationtime")
    proxyexptime = value;
  else if (attr == "nordugrid-job-completiontime")
    completiontime = value;
  else if (attr == "nordugrid-job-exitcode")
    exitcode = atoi (value.c_str());
  else if (attr == "nordugrid-job-rerunable")
    rerunable = value;
  else if (attr == "nordugrid-job-reqwalltime")
    reqwalltime = atol (value.c_str()) * 60;
}


void Job::Print (std::ostream & s) const {

  s << "Job " << id << std::endl;
  if (!jobname.empty())
    s << "  Jobname: " << jobname << std::endl;
  s << "  Owner: " << owner << std::endl;
  s << "  Status: " << status << std::endl;
  if (exitcode != UNDEFINED)
    s << "  Exitcode: " << exitcode << std::endl;
  if (!errors.empty())
    s << "  Error: " << errors << std::endl;
  if (!comment.empty())
    s << "  Comment: " << comment << std::endl;
  if (!execcluster.empty())
    s << "  Cluster: " << execcluster << std::endl;
  if (!execqueue.empty())
    s << "  Queue: " << execqueue << std::endl;
  if (cpucount != UNDEFINED)
    s << "  Requested number of CPUs: " << cpucount << std::endl;
  if (!execnodes.empty()) {
    s << "  Execution nodes:" << std::endl;
    for (std::vector <std::string>::const_iterator it = execnodes.begin();
	 it != execnodes.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (queuerank != UNDEFINED)
    s << "  Rank: " << queuerank << std::endl;
  if (!stdin.empty())
    s << "  stdin: " << stdin << std::endl;
  if (!stdout.empty())
    s << "  stdout: " << stdout << std::endl;
  if (!stderr.empty())
    s << "  stderr: " << stderr << std::endl;
  if (!gmlog.empty())
    s << "  Grid manager log directory: " << gmlog << std::endl;
  if (!submissiontime.empty())
    s << "  Submitted: " << UserTime (submissiontime) << std::endl;
  if (!completiontime.empty())
    s << "  Completed: " << UserTime (completiontime) << std::endl;
  if (!submissionui.empty())
    s << "  Submitted from: " << submissionui << std::endl;
  if (!clientsoftware.empty())
    s << "  Submitting client: " << clientsoftware << std::endl;
  if (!runtimeenv.empty()) {
    s << "  Requested runtime environments:" << std::endl;
    for (std::vector <Environment>::const_iterator it = runtimeenv.begin();
	 it != runtimeenv.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (reqcputime != UNDEFINED)
    s << "  Required CPU time: " << Period (reqcputime) << std::endl;
  if (reqwalltime != UNDEFINED)
    s << "  Required wall time: " << Period (reqwalltime) << std::endl;
  if (usedcputime != UNDEFINED)
    s << "  Used CPU time: " << Period (usedcputime) << std::endl;
  if (usedwalltime != UNDEFINED)
    s << "  Used wall time: " << Period (usedwalltime) << std::endl;
  if (usedmem != UNDEFINED)
    s << "  Used memory: " << usedmem << " kB" << std::endl;
  if (!erasetime.empty())
    s << "  Results " << (status == "DELETED" ? "should have been" : "must be")
      << " retrieved before: " << UserTime (erasetime) << std::endl;
  if (!proxyexptime.empty())
    s << "  Proxy expiration time: " << UserTime (proxyexptime) << std::endl;
  if (!rerunable.empty())
    s << "  Rerun can be attempted at state: " << rerunable << std::endl;
}

void Job::PrintShort (std::ostream & s) const {

  s << "Job " << id << std::endl;
  if (!jobname.empty())
    s << "  Jobname: " << jobname << std::endl;
  s << "  Status: " << status << std::endl;
  if (!errors.empty())
    s << "  Error: " << errors << std::endl;
}


bool Job::IsStatus (const std::vector <std::string> & stat) {

  if (stat.empty())
    return true;
  for (std::vector <std::string>::const_iterator vsci = stat.begin();
       vsci != stat.end(); vsci++)
    if (strcasecmp (vsci->c_str(), status.c_str()) == 0) return true;
  return false;
}


Queue::Queue (const std::string & name_) : name              (name_),
				      running           (UNDEFINED),
				      queued            (UNDEFINED),
				      maxrunning        (UNDEFINED),
				      maxqueuable       (UNDEFINED),
				      maxuserrun        (UNDEFINED),
				      maxcputime        (UNDEFINED),
				      mincputime        (UNDEFINED),
				      defaultcputime    (UNDEFINED),
				      totalcpus         (UNDEFINED),
				      cpufreq           (UNDEFINED),
				      nodemem           (UNDEFINED),
				      gridrunning       (UNDEFINED),
				      gridqueued        (UNDEFINED),
				      userdiskspace     (UNDEFINED),
				      userqueuelength   (UNDEFINED),
				      homogeneity       (false) {}


const std::string & Queue::GetName () const { return name; }

const std::string & Queue::GetStatus () const { return status; }

int Queue::GetRunning () const { return running; }

int Queue::GetQueued () const { return queued; }

int Queue::GetMaxRunning () const { return maxrunning; }

int Queue::GetMaxQueuable () const { return maxqueuable; }

long int Queue::GetMaxCpuTime () const { return maxcputime; }

long int Queue::GetMinCpuTime () const { return mincputime; }

long int Queue::GetDefaultCpuTime () const { return defaultcputime; }

int Queue::GetTotalCpus () const { return totalcpus; }

int Queue::GetNodeMemory () const { return nodemem; }

const std::string & Queue::GetArchitecture () const { return arch; }

int Queue::GetFrequency () const { return cpufreq; }


int Queue::GetUserFreeCpus (long int seconds) const {

  if (userfreecpus.empty()) return UNDEFINED;
  if (seconds == UNDEFINED) seconds == LONG_MAX;
  std::map <long int, int>::const_iterator it = userfreecpus.lower_bound (seconds);
  if (it == userfreecpus.end()) return 0;
  return it->second;
}


long long int Queue::GetUserDiskSpace () const { return userdiskspace; }

void Queue::ClaimUserDiskSpace (long long int size) { userdiskspace -= size; }


const Environment * Queue::FindEnvironment (const std::string & attr,
					    const EnvironmentTest &
					    envtest) const {

  std::string lcattr (attr.length(), '\0');
  std::transform (attr.begin(), attr.end(), lcattr.begin(), to_lower);
  const Environment * env = NULL;
  if (lcattr == "middleware") {
    for (std::vector <Environment>::const_iterator vei = middleware.begin();
	 vei != middleware.end(); vei++)
      if (envtest.Test (*vei))
	if (!env || (env->GetVersion() < vei->GetVersion())) env = &*vei;
  }
  else if (lcattr == "runtimeenvironment") {
    for (std::vector <Environment>::const_iterator vei = runtimeenv.begin();
	 vei != runtimeenv.end(); vei++)
      if (envtest.Test (*vei))
	if (!env || (env->GetVersion() < vei->GetVersion())) env = &*vei;
  }
  else if (lcattr == "opsys") {
    for (std::vector <Environment>::const_iterator vei = opsys.begin();
	 vei != opsys.end(); vei++)
      if (envtest.Test (*vei))
	if (!env || (env->GetVersion() < vei->GetVersion())) env = &*vei;
  }
  else
    std::cerr << "Unexpected attribute in FindEnvironment - should never happen"
	      << std::endl;
  return env;
}


float Queue::GetBenchmark (const std::string & name) const {

  if (benchmarks.find (name) != benchmarks.end())
    return benchmarks.find (name)->second;
  return -1.;
}


void Queue::SetAttr (const std::string & attr, const std::string & value) {

  if (attr == "nordugrid-queue-name")
    name = value;
  else if (attr == "nordugrid-queue-status")
    status = value;
  else if (attr == "nordugrid-queue-running")
    running = atoi (value.c_str());
  else if (attr == "nordugrid-queue-queued")
    queued = atoi (value.c_str());
  else if (attr == "nordugrid-queue-maxrunning")
    maxrunning = atoi (value.c_str());
  else if (attr == "nordugrid-queue-maxqueuable")
    maxqueuable = atoi (value.c_str());
  else if (attr == "nordugrid-queue-maxuserrun")
    maxuserrun = atoi (value.c_str());
  else if (attr == "nordugrid-queue-maxcputime")
    maxcputime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-queue-mincputime")
    mincputime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-queue-defaultcputime")
    defaultcputime = atol (value.c_str()) * 60;
  else if (attr == "nordugrid-queue-schedulingpolicy")
    schedpolicy = value;
  else if (attr == "nordugrid-queue-totalcpus")
    totalcpus = atoi (value.c_str());
  else if (attr == "nordugrid-queue-nodecpu") {
    nodecpu = value;
    int pos = nodecpu.find (" MHz");
    if (pos != std::string::npos) {
      int numpos = nodecpu.rfind (' ', pos - 1) + 1;
      cpufreq = atoi (nodecpu.substr (numpos, pos - numpos).c_str());
      benchmarks ["gridtime"] = cpufreq;
    }
  }
  else if (attr == "nordugrid-queue-nodememory")
    nodemem = atoi (value.c_str());
  else if (attr == "nordugrid-queue-architecture")
    arch = value;
  else if (attr == "nordugrid-queue-opsys")
    opsys.push_back (value);
  else if (attr == "nordugrid-queue-gridrunning")
    gridrunning = atoi (value.c_str());
  else if (attr == "nordugrid-queue-gridqueued")
    gridqueued = atoi (value.c_str());
  else if (attr == "nordugrid-queue-comment")
    comment = value;
  else if (attr == "nordugrid-queue-runtimeenvironment")
    runtimeenv.push_back (value);
  else if (attr == "nordugrid-queue-middleware")
    middleware.push_back (value);
  else if (attr == "nordugrid-queue-benchmark") {
    int pos = value.find (" @ ");
    if (pos != std::string::npos) {
      std::string name = value.substr (0, pos);
      float bmark = atof (value.substr (pos + 3).c_str());
      if (bmark != 0.) benchmarks [name] = bmark;
    }
  }
  else if (attr == "nordugrid-queue-homogeneity")
    homogeneity = (value == "True" || value == "TRUE");
  else if (attr == "nordugrid-authuser-freecpus")
    userfreecpus = ParseFreeCpuStringToMap (value);
  else if (attr == "nordugrid-authuser-diskspace")
    userdiskspace = atoll (value.c_str()) * 1024 * 1024;
  else if (attr == "nordugrid-authuser-queuelength")
    userqueuelength = atoi (value.c_str());
}


void Queue::Print (std::ostream & s) const {

  s << "Queue " << name << std::endl;
  s << "  Status: " << status << std::endl;
  if (!comment.empty())
    s << "  Comment: " << comment << std::endl;
  if (!arch.empty())
    s << "  Architecture: " << arch << std::endl;
  if (!opsys.empty()) {
    s << "  Operating system:" << std::endl;
    for (std::vector <Environment>::const_iterator it = opsys.begin();
	 it != opsys.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!nodecpu.empty())
    s << "  CPU" << (homogeneity ? ": " : " (worst): ") << nodecpu << std::endl;
  if (nodemem != UNDEFINED)
    s << "  Memory on each node" << (homogeneity ? ": " : " (smallest): ")
      << nodemem << " MB" << std::endl;
  if (!middleware.empty()) {
    s << "  Installed middleware:" << std::endl;
    for (std::vector <Environment>::const_iterator it = middleware.begin();
	 it != middleware.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!runtimeenv.empty()) {
    s << "  Installed runtime environments:" << std::endl;
    for (std::vector <Environment>::const_iterator it = runtimeenv.begin();
	 it != runtimeenv.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!benchmarks.empty()) {
    s << "  Evaluated benchmarks:" << std::endl;
    for (std::map <std::string, float>::const_iterator it = benchmarks.begin();
	 it != benchmarks.end(); it++)
      s << "    " << it->first << ": " << it->second << std::endl;
  }
  if (running != UNDEFINED)
    s << "  Number of running jobs: " << running << std::endl;
  if (queued != UNDEFINED)
    s << "  Number of queued jobs: " << queued << std::endl;
  if (maxrunning != UNDEFINED)
    s << "  Max number of running jobs: " << maxrunning << std::endl;
  if (maxqueuable != UNDEFINED)
    s << "  Max number of queued jobs: " << maxqueuable << std::endl;
  if (maxuserrun != UNDEFINED)
    s << "  Max number of running jobs per local user: " << maxuserrun << std::endl;
  if (maxcputime != UNDEFINED)
    s << "  Max CPU time: " << Period (maxcputime) << std::endl;
  if (mincputime != UNDEFINED)
    s << "  Min CPU time: " << Period (mincputime) << std::endl;
  if (defaultcputime != UNDEFINED)
    s << "  Default CPU time: " << Period (defaultcputime) << std::endl;
  if (!schedpolicy.empty())
    s << "  Scheduling policy: " << schedpolicy << std::endl;
  if (totalcpus != UNDEFINED)
    s << "  Number of CPUs: " << totalcpus << std::endl;
  if (!userfreecpus.empty()) {
    s << "  User number of free CPUs:" << std::endl;
    for (std::map <long int, int>::const_iterator it = userfreecpus.begin();
	 it != userfreecpus.end(); it++)
      if (it->second)
	s << "    " << it->second
	  << (it->second == 1 ? " processor for " : " processors for ")
	  << (it->first == LONG_MAX ? "infinite time" : Period (it->first))
	  << std::endl;
      else
	s << "    None" << std::endl;
  }
  if (userdiskspace != UNDEFINED)
    s << "  User disk space: " << userdiskspace << " bytes" << std::endl;
  if (userqueuelength != UNDEFINED)
    s << "  User queue length: " << userqueuelength << std::endl;
}


void Queue::PrintShort (std::ostream & s) const {

  s << "Queue " << name << std::endl;
  s << "  Status: " << status << std::endl;
}


void Queue::Accept (long int seconds, int count) {

  if (running < maxrunning)
    running++;
  else
    queued++;

  if (seconds == UNDEFINED) seconds = defaultcputime;
  if (seconds == UNDEFINED) seconds = LONG_MAX;
  std::map <long int, int>::iterator last = userfreecpus.lower_bound (seconds);
  if (last != userfreecpus.end() && last->second >= count) {
    for (std::map <long int, int>::iterator it = userfreecpus.begin();
	 it != userfreecpus.end(); it++) {
      if (it->first <= last->first)
	userfreecpus [it->first] -= count;
      else if (it->second >= last->second) {
	userfreecpus [it->first] = last->second;
	int oldkey = last->first;
	last++;
	userfreecpus.erase (oldkey);
      }
    }
    if (last->second == 0) userfreecpus.erase (last->first);
    if (userfreecpus.empty())
      userfreecpus [(maxcputime != UNDEFINED ? maxcputime : LONG_MAX)] = 0;
  }
}


Cluster::Cluster (const std::string & name_) : name        (name_),
					  homogeneity (false),
					  cpufreq     (UNDEFINED),
					  nodemem     (UNDEFINED),
					  totalcpus   (UNDEFINED),
					  totaljobs   (UNDEFINED),
					  usedcpus    (UNDEFINED),
					  queuedjobs  (UNDEFINED),
					  sessiondirfree  (UNDEFINED),
					  sessiondirtotal (UNDEFINED),
					  cachefree   (UNDEFINED),
					  cachetotal  (UNDEFINED),
					  sessiondirlifetime (UNDEFINED) {}


const std::string & Cluster::GetName () const { return name; }


const std::string Cluster::GetLongName () const {

  if (!alias.empty())
    return alias + " (" + name + ")";
  else
    return name;
}


const std::string & Cluster::GetContact () const { return contact; }

const std::string & Cluster::GetArchitecture () const { return arch; }

int Cluster::GetFrequency () const { return cpufreq; }

int Cluster::GetNodeMemory () const { return nodemem; }

int Cluster::GetTotalCpus () const { return totalcpus; }

int Cluster::GetQueued () const { return queuedjobs; }

long long int Cluster::GetCacheFree () const { return cachefree; }

void Cluster::ClaimCache (long long int size) { cachefree -= size; }

long long int Cluster::GetSessionDirFree () const { return sessiondirfree; }

void Cluster::ClaimSessionDir (long long int size) { sessiondirfree -= size; }


const Environment * Cluster::FindEnvironment (const std::string & attr,
					      const EnvironmentTest &
					      envtest) const {

  std::string lcattr (attr.length(), '\0');
  std::transform (attr.begin(), attr.end(), lcattr.begin(), to_lower);
  const Environment * env = NULL;
  if (lcattr == "middleware") {
    for (std::vector <Environment>::const_iterator vei = middleware.begin();
	 vei != middleware.end(); vei++)
      if (envtest.Test (*vei))
	if (!env || (env->GetVersion() < vei->GetVersion())) env = &*vei;
  }
  else if (lcattr == "runtimeenvironment") {
    for (std::vector <Environment>::const_iterator vei = runtimeenv.begin();
	 vei != runtimeenv.end(); vei++)
      if (envtest.Test (*vei))
	if (!env || (env->GetVersion() < vei->GetVersion())) env = &*vei;
  }
  else if (lcattr == "opsys") {
    for (std::vector <Environment>::const_iterator vei = opsys.begin();
	 vei != opsys.end(); vei++)
      if (envtest.Test (*vei))
	if (!env || (env->GetVersion() < vei->GetVersion())) env = &*vei;
  }
  else
    std::cerr << "Unexpected attribute in FindEnvironment - should never happen"
	      << std::endl;
  return env;
}


float Cluster::GetBenchmark (const std::string & name) const {

  if (benchmarks.find (name) != benchmarks.end())
    return benchmarks.find (name)->second;
  return -1.;
}


long int Cluster::GetSessionDirLifetime () const { return sessiondirlifetime; }


EnvVersion Cluster::GetServerVersion () const {

  for (std::vector <Environment>::const_iterator vei = middleware.begin();
       vei != middleware.end(); vei++)
    if (vei->GetName() == "nordugrid") return vei->GetVersion();
  return EnvVersion ();
}


bool Cluster::MatchLocalSe (const std::string & location) const {

  std::string loc = location;
  RemoveDefaultPort (loc);
  for (std::vector <std::string>::const_iterator vsi = localse.begin();
       vsi != localse.end(); vsi++) {
    std::string lse = *vsi;
    RemoveDefaultPort (lse);
    if (lse [lse.length() - 1] == '/') lse.erase (lse.length() - 1);
    if (lse == loc) return true;
    if (lse == loc.substr (0, lse.length()) && loc [lse.length()] == '/')
      return true;
  }
  return false;
}


bool Cluster::HaveNodeAccess (const std::string & attr) const {

  for (std::vector <std::string>::const_iterator vsi = nodeaccess.begin();
       vsi != nodeaccess.end(); vsi++)
    if (*vsi == attr) return true;
  return false;
}


bool Cluster::HaveKeys () const {

  globus_result_t res;
  bool found = false;

  char * certdir;
  res = GLOBUS_GSI_SYSCONFIG_GET_CERT_DIR (&certdir);

  globus_fifo_t ca_cert_list;
  globus_fifo_init (&ca_cert_list);

  res = GLOBUS_GSI_SYSCONFIG_GET_CA_CERT_FILES (certdir, &ca_cert_list);
  free (certdir);
  if (res != GLOBUS_SUCCESS) return false;

  char * filename;
  while ((filename = (char *) globus_fifo_dequeue (&ca_cert_list)) != NULL) {

    if (!found) {
      globus_gsi_cred_handle_attrs_t handle_attrs;
      globus_gsi_cred_handle_t handle;

      res = globus_gsi_cred_handle_attrs_init (&handle_attrs);
      if (res == GLOBUS_SUCCESS) {
	res = globus_gsi_cred_handle_init (&handle, handle_attrs);
	if (res == GLOBUS_SUCCESS) {
	  res = globus_gsi_cred_read_cert (handle, filename);
	  if (res == GLOBUS_SUCCESS) {
	    char * iss;
	    res = globus_gsi_cred_get_identity_name (handle, &iss);
	    if (res == GLOBUS_SUCCESS)
	      if (strcasecmp (iss, issuer.c_str()) == 0) found = true;
	    OPENSSL_free(iss);
	  }
	  res = globus_gsi_cred_handle_destroy (handle);
	}
	res = globus_gsi_cred_handle_attrs_destroy (handle_attrs);
      }
    }
    free (filename);
  }

  globus_fifo_destroy (&ca_cert_list);

  return found;
}


void Cluster::SetAttr (const std::string & attr, const std::string & value) {

  if (attr == "nordugrid-cluster-name")
    name = value;
  else if (attr == "nordugrid-cluster-aliasname")
    alias = value;
  else if (attr == "nordugrid-cluster-contactstring")
    contact = value;
  else if (attr == "nordugrid-cluster-support")
    support.push_back (value);
  else if (attr == "nordugrid-cluster-lrms-type")
    lrmstype = value;
  else if (attr == "nordugrid-cluster-lrms-version")
    lrmsversion = value;
  else if (attr == "nordugrid-cluster-lrms-config")
    lrmsconfig = value;
  else if (attr == "nordugrid-cluster-architecture")
    arch = value;
  else if (attr == "nordugrid-cluster-opsys")
    opsys.push_back (value);
  else if (attr == "nordugrid-cluster-homogeneity")
    homogeneity = (value == "True" || value == "TRUE");
  else if (attr == "nordugrid-cluster-nodecpu") {
    nodecpu = value;
    int pos = nodecpu.find (" MHz");
    if (pos != std::string::npos) {
      int numpos = nodecpu.rfind (' ', pos - 1) + 1;
      cpufreq = atoi (nodecpu.substr (numpos, pos - numpos).c_str());
      benchmarks ["gridtime"] = cpufreq;
    }
  }
  else if (attr == "nordugrid-cluster-nodememory")
    nodemem = atoi (value.c_str());
  else if (attr == "nordugrid-cluster-totalcpus")
    totalcpus = atoi (value.c_str());
  else if (attr == "nordugrid-cluster-cpudistribution")
    cpudist = ParseStringToMap (value);
  else if (attr == "nordugrid-cluster-sessiondir-free")
    sessiondirfree = atoll (value.c_str()) * 1024 * 1024;
  else if (attr == "nordugrid-cluster-sessiondir-total")
    sessiondirtotal = atoll (value.c_str()) * 1024 * 1024;
  else if (attr == "nordugrid-cluster-cache-free")
    cachefree = atoll (value.c_str()) * 1024 * 1024;
  else if (attr == "nordugrid-cluster-cache-total")
    cachetotal = atoll (value.c_str()) * 1024 * 1024;
  else if (attr == "nordugrid-cluster-runtimeenvironment")
    runtimeenv.push_back (value);
  else if (attr == "nordugrid-cluster-localse")
    localse.push_back (value);
  else if (attr == "nordugrid-cluster-middleware")
    middleware.push_back (value);
  else if (attr == "nordugrid-cluster-totaljobs")
    totaljobs = atoi (value.c_str());
  else if (attr == "nordugrid-cluster-usedcpus")
    usedcpus = atoi (value.c_str());
  else if (attr == "nordugrid-cluster-queuedjobs")
    queuedjobs = atoi (value.c_str());
  else if (attr == "nordugrid-cluster-location")
    location = value;
  else if (attr == "nordugrid-cluster-owner")
    owner.push_back (value);
  else if (attr == "nordugrid-cluster-issuerca")
    issuer = value;
  else if (attr == "nordugrid-cluster-nodeaccess")
    nodeaccess.push_back (value);
  else if (attr == "nordugrid-cluster-comment")
    comment = value;
  else if (attr == "nordugrid-cluster-interactive-contactstring")
    intcontact = value;
  else if (attr == "nordugrid-cluster-benchmark") {
    int pos = value.find (" @ ");
    if (pos != std::string::npos) {
      std::string name = value.substr (0, pos);
      float bmark = atof (value.substr (pos + 3).c_str());
      if (bmark != 0.) benchmarks [name] = bmark;
    }
  }
  else if (attr == "nordugrid-cluster-sessiondir-lifetime")
    sessiondirlifetime = atol (value.c_str()) * 60;
  // kept for backward compatibility
  else if (attr == "nordugrid-cluster-opsysdistribution")
    opsys.push_back (value);
}


void Cluster::Print (std::ostream & s) const {

  s << "Cluster " << name << std::endl;
  if (!alias.empty())
    s << "  Alias: " << alias << std::endl;
  if (!comment.empty())
    s << "  Comment: " << comment << std::endl;
  if (!contact.empty())
    s << "  Contact: " << contact << std::endl;
  if (!intcontact.empty())
    s << "  Interactive contact: " << intcontact << std::endl;
  if (!location.empty())
    s << "  Location: " << location << std::endl;
  if (!issuer.empty()) {
    s << "  Host certificate issuer:" << std::endl;
    s << "    " << issuer << std::endl;
  }
  if (!owner.empty()) {
    s << "  Owners:" << std::endl;
    for (std::vector <std::string>::const_iterator it = owner.begin();
	 it != owner.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!arch.empty())
    s << "  Architecture: " << arch << std::endl;
  if (!opsys.empty()) {
    s << "  Operating system:" << std::endl;
    for (std::vector <Environment>::const_iterator it = opsys.begin();
	 it != opsys.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!nodecpu.empty())
    s << "  CPU" << (homogeneity ? ": " : " (worst): ") << nodecpu << std::endl;
  if (nodemem != UNDEFINED)
    s << "  Memory on each node" << (homogeneity ? ": " : " (smallest): ")
      << nodemem << " MB" << std::endl;
  if (sessiondirtotal != UNDEFINED)
    s << "  Size of scratch dir: " << sessiondirtotal << " bytes" << std::endl;
  if (sessiondirfree != UNDEFINED)
    s << "  Free space in scratch dir: " << sessiondirfree << " bytes" << std::endl;
  if (cachetotal != UNDEFINED)
    s << "  Size of cache dir: " << cachetotal << " bytes" << std::endl;
  if (cachefree != UNDEFINED)
    s << "  Free space in cache dir: " << cachefree << " bytes" << std::endl;
  if (totalcpus != UNDEFINED)
    s << "  Number of CPUs: " << totalcpus << std::endl;
  if (usedcpus != UNDEFINED)
    s << "  Number of used CPUs: " << usedcpus << std::endl;
  if (totaljobs != UNDEFINED)
    s << "  Number of jobs: " << totaljobs << std::endl;
  if (queuedjobs != UNDEFINED)
    s << "  Number of queued jobs: " << queuedjobs << std::endl;
  if (!cpudist.empty()) {
    s << "  CPU distribution:" << std::endl;
    for (std::map <int, int>::const_iterator it = cpudist.begin();
	 it != cpudist.end(); it++) {
      s << "    ";
      if      (it->first == 1) s << "single";
      else if (it->first == 2) s << "dual";
      else                     s << it->first;
      s << " CPU machines: " << it->second << std::endl;
    }
  }
  if (!benchmarks.empty()) {
    s << "  Evaluated benchmarks:" << std::endl;
    for (std::map <std::string, float>::const_iterator it = benchmarks.begin();
	 it != benchmarks.end(); it++)
      s << "    " << it->first << ": " << it->second << std::endl;
  }
  if (!support.empty()) {
    s << "  Cluster support:" << std::endl;
    for (std::vector <std::string>::const_iterator it = support.begin();
	 it != support.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!localse.empty()) {
    s << "  Local Storage Elements:" << std::endl;
    for (std::vector <std::string>::const_iterator it = localse.begin();
	 it != localse.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!middleware.empty()) {
    s << "  Installed middleware:" << std::endl;
    for (std::vector <Environment>::const_iterator it = middleware.begin();
	 it != middleware.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!runtimeenv.empty()) {
    s << "  Installed runtime environments:" << std::endl;
    for (std::vector <Environment>::const_iterator it = runtimeenv.begin();
	 it != runtimeenv.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (!nodeaccess.empty()) {
    s << "  Network access on cluster nodes:" << std::endl;
    for (std::vector <std::string>::const_iterator it = nodeaccess.begin();
	 it != nodeaccess.end(); it++)
      s << "    " << *it << std::endl;
  }
  if (sessiondirlifetime != UNDEFINED)
    s << "  Session directory lifetime: " << Period (sessiondirlifetime)
      << std::endl;
}


void Cluster::PrintShort (std::ostream & s) const {

  s << "Cluster " << name << std::endl;
  if (!alias.empty())
    s << "  Alias: " << alias << std::endl;
}


void FindInfoCallback (const std::string & attr,
		       const std::string & value, void * ref) {

  Cluster * c = (Cluster *) ref;

  static int qix;
  static int jix;

  std::string lcattr (attr.length(), '\0');
  std::transform (attr.begin(), attr.end(), lcattr.begin(), to_lower);
  if (lcattr == "dn") {
    std::string dn (value.length(), '\0');
    std::transform (value.begin(), value.end(), dn.begin(), to_lower);
    int pos;
    qix = UNDEFINED;
    pos = dn.find ("nordugrid-queue-name");
    if (pos != std::string::npos) {
      pos = dn.find ('=', pos);
      std::string qname = value.substr (pos + 1, dn.find (',', pos + 1) - pos - 1);
      while (qname [qname.length() - 1] == ' ')
	qname.erase (qname.length() - 1);
      for (int i = 0; i < c->queues.size(); i++)
	if (qname == c->queues[i].GetName()) {qix = i; break;}
      if (qix == UNDEFINED) {
	qix = c->queues.size();
	c->queues.push_back (qname);
      }
    }
    jix = UNDEFINED;
    pos = dn.find ("nordugrid-job-globalid");
    if (pos != std::string::npos) {
      pos = dn.find ('=', pos);
      std::string jobid = value.substr (pos + 1, dn.find (',', pos + 1) - pos - 1);
      while (jobid [jobid.length() - 1] == ' ')
	jobid.erase (jobid.length() - 1);
      for (int i = 0; i < c->queues[qix].jobs.size(); i++)
	if (jobid == c->queues[qix].jobs[i].GetId()) {jix = i; break;}
      if (jix == UNDEFINED) {
	jix = c->queues[qix].jobs.size();
	c->queues[qix].jobs.push_back (jobid);
      }
    }
  }
  else {
    if (qix == UNDEFINED)
      c->SetAttr (lcattr, value);
    else if (jix == UNDEFINED)
      c->queues[qix].SetAttr (lcattr, value);
    else
      c->queues[qix].jobs[jix].SetAttr (lcattr, value);
  }
}


int Cluster::Connect (const std::string & usersn, bool anonymous,
		      int timeout, int debug) {

  return ldap.Connect (name, 2135, usersn, anonymous, timeout, debug);
}


int Cluster::Query (Mds::Filter mdsfilter, const std::string & usersn,
		    int timeout, int debug) {

  std::string usersn1 (usersn);
  std::string usersn2 (usersn);

  int pos;

  pos = 0;
  while ((pos = usersn1.find_first_of ("()*\\", pos)) != std::string::npos) {
    if (usersn1 [pos] == '\\' && usersn1 [pos + 1] == 'x') {
      usersn1.erase (pos + 1, 1);
      pos += 1;
    }
    else {
      usersn1.insert (pos, 1, '\\');
      pos += 2;
    }
  }

  pos = 0;
  while ((pos = usersn2.find_first_of ("()*\\", pos)) != std::string::npos) {
    usersn2.insert (pos, 1, '\\');
    pos += 2;
  }

  std::string filter;
  switch (mdsfilter) {
  case Mds::ClusterInfo:
    filter = "(|(objectclass=nordugrid-cluster)(objectclass=nordugrid-queue)(nordugrid-authuser-sn=" + usersn1 + ")(nordugrid-authuser-sn=" + usersn2 + "))";
    break;
  case Mds::JobInfo:
    filter = "(|(nordugrid-job-globalowner=" + usersn1 + ")(nordugrid-job-globalowner=" + usersn2 + "))";
    break;
  case Mds::JobSubmission:
    filter = "(|(objectclass=nordugrid-cluster)(objectclass=nordugrid-queue)(nordugrid-authuser-sn=" + usersn1 + ")(nordugrid-authuser-sn=" + usersn2 + ")(nordugrid-job-status=*ACCEPT*)(nordugrid-job-status=*PREPAR*))";
    break;
  case Mds::JobManipulation:
    filter = "(|(objectclass=nordugrid-cluster)(nordugrid-job-globalowner=" + usersn1 + ")(nordugrid-job-globalowner=" + usersn2 + "))";
    break;
  }

  const std::vector <std::string> attributes;

  return ldap.Query ("Mds-Vo-name=local,o=grid", filter, attributes,
		     LdapQuery::subtree, timeout, debug);
}


int Cluster::Result (int timeout, int debug) {

  return ldap.Result (FindInfoCallback, this, timeout, debug);
}


int Cluster::Find (Mds::Filter mdsfilter, const std::string & usersn,
		   bool anonymous, int timeout, int debug) {

  if (Connect (usersn, anonymous, timeout, debug)) return 1;
  if (Query (mdsfilter, usersn, timeout, debug)) return 1;
  return Result (timeout, debug);
}


void FindClusterInfo (std::vector <Cluster> & clusterlist,
		      Mds::Filter mdsfilter,
		      const std::string & usersn,
		      bool anonymous,
		      int timeout,
		      int debug) {

  for (std::vector <Cluster>::iterator cli = clusterlist.begin();
       cli != clusterlist.end(); cli++)
    cli->Connect (usersn, anonymous, timeout, debug);
  for (std::vector <Cluster>::iterator cli = clusterlist.begin();
       cli != clusterlist.end(); cli++)
    cli->Query (mdsfilter, usersn, timeout, debug);
  for (std::vector <Cluster>::iterator cli = clusterlist.begin();
       cli != clusterlist.end(); cli++)
    cli->Result (timeout, debug);
}


void FindClusterInfo (Cluster & cluster,
		      Mds::Filter mdsfilter,
		      const std::string & usersn,
		      bool anonymous,
		      int timeout,
		      int debug) {

  cluster.Connect (usersn, anonymous, timeout, debug);
  cluster.Query (mdsfilter, usersn, timeout, debug);
  cluster.Result (timeout, debug);
}


void FindClustersCallback (const std::string & attr,
			   const std::string & value, void * ref) {

  struct callback_type * cb = (struct callback_type *) ref;

  static std::string host;
  static int port;
  static std::string suffix;
  static bool newgiis;
  static bool newcluster;

  std::string lcattr (attr.length(), '\0');
  std::transform (attr.begin(), attr.end(), lcattr.begin(), to_lower);
  if (lcattr == "mds-service-hn") {
    host = value;
    newcluster = false;
    newgiis = false;
  }
  else if (lcattr == "mds-service-port") {
    port = atoi (value.c_str());
  }
  else if (lcattr == "mds-service-ldap-suffix") {
    std::string lcvalue (value.length(), '\0');
    std::transform (value.begin(), value.end(), lcvalue.begin(), to_lower);
    if (lcvalue.substr (0, 17) == "mds-vo-name=local" ||
	lcvalue.substr (0, 22) == "nordugrid-cluster-name") {
      newcluster = true;
    }
    else if (lcvalue.substr (0, 11) == "mds-vo-name") {
      newgiis = true;
      suffix = value;
    }
  }
  else if (lcattr == "mds-reg-status") {
    if (value == "VALID")
      if (newcluster) {
	bool old = false;
	for (std::vector <Cluster>::iterator cli = cb->clusterlist->begin();
	     !old && cli != cb->clusterlist->end(); cli++)
	  if (cli->GetName() == host) old = true;
	if (!old) cb->clusterlist->push_back (host);
      }
      else if (newgiis) {
	Giis giis (host, port, suffix);
	bool old = false;
	for (std::vector <Giis>::iterator vgi = cb->giislist->begin();
	     !old && vgi != cb->giislist->end(); vgi++) {
	  if (*vgi == giis) old = true;
	}
	if (!old) cb->giislist->push_back (giis);
      }
  }
}


std::vector <Cluster> FindClusters (std::vector <Giis> giislist,
				    const std::string & usersn,
				    bool anonymous,
				    int timeout,
				    int debug) {

  callback_type cb;
  std::vector <Cluster> cl;
  std::vector <Giis> gl = giislist;
  cb.giislist = &gl;
  cb.clusterlist = &cl;

  int nqueried = 0;

  while (nqueried < giislist.size()) {
    int last = giislist.size();
    for (int i = nqueried; i < last; i++)
      giislist [i].Connect (usersn, anonymous, timeout, debug);
    for (int i = nqueried; i < last; i++)
      giislist [i].Query (usersn, timeout, debug);
    for (int i = nqueried; i < last; i++)
      giislist [i].Result (FindClustersCallback, &cb, timeout, debug);
    giislist = gl;
    nqueried = last;
  }

  return cl;
}
