#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <vector>
#include <string>
#include <fstream>

#include "Giis.h"
#include "LdapQuery.h"
#include "Preferences.h"


Giis::Giis (const std::string & url) {

  if (url.substr (0, 7) != "ldap://") return;
  int pos0 = url.find_first_of (":/", 7);
  if (pos0 == std::string::npos) return;
  host = url.substr (7, pos0 - 7);
  int pos = url.find ('/', pos0);
  if (pos == pos0)
    port = 2135;
  else
    port = atoi (url.substr (pos0 + 1, pos - pos0 - 1).c_str());
  while (pos != std::string::npos) {
    int nextpos = url.find ('/', pos + 1);
    int apos = (nextpos == std::string::npos) ? url.length() : nextpos;
    if (basedn.empty())
      basedn = url.substr (pos + 1, apos - pos - 1);
    else
      basedn = url.substr (pos + 1, apos - pos - 1) + ',' + basedn;
    pos = nextpos;
  }
}


Giis::Giis (const std::string & host_,
	    int port_,
	    const std::string & basedn_) : host (host_),
					   port (port_),
					   basedn (basedn_) {}


bool Giis::operator== (const Giis & giis) const {

  return ((host == giis.host) &&
	  (port == giis.port) &&
	  (basedn == giis.basedn));
}


bool Giis::operator! () const { return host.empty() || basedn.empty(); }


int Giis::Connect (const std::string & usersn, bool anonymous,
		   int timeout, int debug) {

  return ldap.Connect (host, port, usersn, anonymous, timeout, debug);
}


int Giis::Query (const std::string & usersn, int timeout, int debug) {

  const std::string filter;
  const std::string attr = "giisregistrationstatus";
  std::vector <std::string> attrs;
  attrs.push_back (attr);

  return ldap.Query (basedn, filter, attrs, LdapQuery::base, timeout, debug);
}


int Giis::Result (void callback (const std::string & attr,
				 const std::string & value,
				 void * ref),
		  void * ref, int timeout, int debug) {

  return ldap.Result (callback, ref, timeout, debug);
}


int GetGiises (const std::vector <std::string> & giisurls, std::vector <Giis> & giislist) {

  if (!giisurls.empty()) {
    for (std::vector <std::string>::const_iterator vsi = giisurls.begin();
	 vsi != giisurls.end(); vsi++) {
      Giis giis (*vsi);
      if (!giis) {
	std::cerr << "Error: Syntax error in giisurl: " << *vsi << std::endl;
	return 1;
      }
      else
	giislist.push_back (giis);
    }
    return 0;
  }

  int i = 0;
  bool found = false;
  while (!found) {
    std::string filename;
    switch (i) {
    case 0:
      filename = GetEnv ("HOME");
      filename.append ("/.nggiislist");
      break;
    case 1:
      filename = GetEnv ("ARC_LOCATION");
      if (filename.empty()) filename = GetEnv ("NORDUGRID_LOCATION");
      if (filename.empty()) filename = DEFAULT_ARC_LOCATION;
      filename.append ("/etc/giislist");
      break;
    case 2:
      filename = "/etc/giislist";
      break;
    case 3:
      std::cerr << "Error: No giislist found at standard locations" << std::endl;
      return 1;
      break;
    }
    i++;
    std::ifstream giislistfile (filename.c_str());
    if (!giislistfile) continue;
    std::string line;
    while (getline (giislistfile, line)) {
      if (line.empty()) continue;
      if (line[0] == '#') continue;
      Giis giis (line);
      if (!giis) {
	std::cerr << "Error: Syntax error in " << filename << ": " << line << std::endl;
	return 1;
      }
      else
	giislist.push_back (giis);
    }
    giislistfile.close();
    if (giislist.empty()) {
      std::cerr << "Error: Giislist file " << filename << " contains no giises"
		<< std::endl;
      return 1;
    }
    return 0;
  }
}
