#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef _BSD_SOURCE
#define _BSD_SOURCE
#endif

#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#include <iomanip>
#include <iostream>
#include <string>
#include <time.h>

#ifndef HAVE_GLOBUS_GSI_PROXY_H
#include <sys/stat.h>
#include <sslutils.h>
#include <openssl/x509.h>
#else
#include <globus_gsi_proxy.h>
#include <globus_gsi_system_config.h>
#endif

#include <ldap.h>

#include "CertInfo.h"


#ifndef HAVE_GLOBUS_GSI_PROXY_H

CertInfo::CertInfo (const char * proxy) {

  char * proxy_file = NULL;
  struct stat stx;

  good = false;

  proxy_cred_desc * pcd = proxy_cred_desc_new();
  if (!pcd) {
    std::cerr << "Error: Could not create proxy credential descriptor." << std::endl;
    goto exit;
  }

  if (proxy) {
    proxy_file = strdup (proxy);
  }
  else {
    proxy_get_filenames (pcd, 1, NULL, NULL, &proxy_file, NULL, NULL);
    if (!proxy_file) {
      std::cerr << "Error: Unable to determine proxy file name." << std::endl;
      goto exit;
    }
  }

  if (stat (proxy_file, &stx) != 0) {
    std::cerr << "Error: Proxy file " << proxy_file << " not found." << std::endl;
    goto exit;
  }

  pcd->type = CRED_TYPE_PROXY;

  if (proxy_load_user_cert (pcd, proxy_file, NULL, NULL)) {
    std::cerr << "Error: Unable to load proxy from " << proxy_file << "." << std::endl;
    goto exit;
  }

  if ((pcd->upkey = X509_get_pubkey (pcd->ucert)) == NULL) {
    std::cerr << "Error: Unable to load public key from proxy." << std::endl;
    goto exit;
  }

  good = true;

  sn = X509_NAME_oneline (X509_get_issuer_name (pcd->ucert), NULL, 0);
  expires = ASN1_UTCTIME_mktime (X509_get_notAfter (pcd->ucert));

 exit:
  if (pcd) proxy_cred_desc_free (pcd);
  if (proxy_file) free (proxy_file);
}

#else

CertInfo::CertInfo (const char * proxy) {

  globus_result_t result;
  char * proxy_filename = NULL;
  globus_gsi_cred_handle_t proxy_cred = NULL;
  X509 * proxy_cert = NULL;
  EVP_PKEY * proxy_pubkey = NULL;
  char * subject = NULL;
  time_t goodtill;

  good = false;

  if (proxy) {
    proxy_filename = strdup (proxy);
  }
  else {
    result = GLOBUS_GSI_SYSCONFIG_GET_PROXY_FILENAME (&proxy_filename,
						      GLOBUS_PROXY_FILE_INPUT);
    if (result != GLOBUS_SUCCESS) {
      std::cerr << "Error: Couldn't find a valid proxy." << std::endl;
      goto exit;
    }
  }

  result = globus_gsi_cred_handle_init (&proxy_cred, NULL);
  if (result != GLOBUS_SUCCESS) {
    std::cerr << "Error: Couldn't initialize proxy credential handle." << std::endl;
    goto exit;
  }

  result = globus_gsi_cred_read_proxy (proxy_cred, proxy_filename);
  if (result != GLOBUS_SUCCESS) {
    std::cerr << "Error: Couldn't read proxy from " << proxy_filename << '.' << std::endl;
    goto exit;
  }

  result = globus_gsi_cred_get_cert (proxy_cred, &proxy_cert);
  if (result != GLOBUS_SUCCESS) {
    std::cerr << "Error: Couldn't get the proxy certificate from the proxy credential." << std::endl;
    goto exit;
  }

  if ((proxy_pubkey = X509_get_pubkey(proxy_cert)) == NULL) {
    std::cerr << "Error: Unable to load public key from proxy." << std::endl;
    goto exit;
  }

  result = globus_gsi_cred_get_identity_name (proxy_cred, &subject);
  if (result != GLOBUS_SUCCESS) {
    std::cerr << "Error: Couldn't get a valid identity name from the proxy credential." << std::endl;
    goto exit;
  }

  result = globus_gsi_cred_get_goodtill (proxy_cred, &goodtill);
  if (result != GLOBUS_SUCCESS) {
    std::cerr << "Error: Couldn't get a valid lifetime for the proxy credential." << std::endl;
    goto exit;
  }

  sn = subject;
  expires = goodtill;
  good = true;

 exit:
  if (proxy_cred) globus_gsi_cred_handle_destroy (proxy_cred);
  if (proxy_filename) free (proxy_filename);
  if (subject) OPENSSL_free (subject);
}

#endif


std::string CertInfo::GetSN () const {

  std::string sn8 (sn);
  int pos = 0;
  while ((pos = sn8.find ("\\x", pos)) != std::string::npos) {
#ifdef HAVE_SSTREAM
    std::stringstream ss (sn8.substr (pos + 2, 2));
#else
    std::strstream ss;
    ss << sn8.substr (pos + 2, 2) << std::ends;
#endif
    int i;
    ss >> std::hex >> i;
    sn8.replace (pos, 4, 1, i);
  }
  return sn8;
}


std::string CertInfo::GetSNx () const { return sn; }

std::string CertInfo::GetDN () const { return ldap_dcedn2dn(sn.c_str()); }

time_t CertInfo::Expires () const { return expires; }


time_t CertInfo::TimeLeft () const {

  time_t now = time (NULL);
  time_t timeleft = expires - now;
  return (timeleft > 0) ? timeleft : 0;
}


bool CertInfo::operator! () const {

  return expired();
}


bool CertInfo::expired () const {

  if (!good) return true;

  if (TimeLeft() == 0) {
    std::cerr << "Error: Your proxy has expired" << std::endl;
    std::cerr << "Please rerun grid-proxy-init" << std::endl;
    return true;
  }

  if (TimeLeft() < 600) {
    std::cerr << "Error: Your proxy will soon expire" << std::endl;
    std::cerr << "Please rerun grid-proxy-init" << std::endl;
    return true;
  }

  return false;
}
