#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ngui.h"

#include <cstdlib>
#include <fstream>
#include <iostream>
#include <cstdio>
#include <string>
#include <vector>
#include <unistd.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <getopt.h>

using namespace std;

#include "Preferences.h"
#include "LdapQuery.h"
#include "MdsQuery.h"
#include "Giis.h"
#include "CertInfo.h"
#include "Environment.h"

//#include "ftpClient.h"

static int number_testcases = 83;

void print_usage( string );
void print_cases( );

void executable_args( vector<string> &components );
void arguments_args( vector<string> &components );
void executables_args( vector<string> &components );
void inputfiles_args( vector<string> &components );
void outputfiles_args( vector<string> &components );
void runtimeenvironment_args( vector<string> &components );

void VOQueryCallback( const string& attr, const string& value, void* ref );
vector<string> FindNorduGridVO( );
string convert_int_to_string( unsigned int number );

int main( int argc, char* argv[] )
{
  string progname = argv[0];
  unsigned int pos = progname.rfind('/');
  if (pos != string::npos)
    progname = progname.substr (pos + 1);

  vector<string> members = FindNorduGridVO( );
  CertInfo user;
  if( !user ) {
    return 1;
  }
  string userSubject = user.GetSN();
  
  bool match = false;
  for( unsigned int i=0; i<members.size(); i++ ) {
    if( strcmp( userSubject.c_str(), members[i].c_str() )==0 ) match = true;
  }
  if( match==false )
    cout << "Warning: You're not a member of the NorduGrid VO. You will not be able to run all tests." << endl;

  // check whether rls-server is up

  vector<string> jobs;
  vector<string> ngsubargs;

  bool dropintofile = false;
  string runtime = "5";
  bool actuallysubmitted = true;

  // partly ngsub-options -- partly ngtest-options
  char* optstring = "c:C:g:G:f:o:DPt:d:xXvhr:sl";
  struct option longoptions[18] = { {"cluster", 1, 0, 'c'}, {"clustlist", 1, 0, 'C'}, {"giisurl", 1, 0, 'g'}, {"giislist", 1, 0, 'G'}, {"file", 1, 0, 'f'}, {"joblist", 1, 0, 'o'}, {"timeout", 1, 0, 't'}, {"version", 0, 0, 'v'}, {"debug", 1, 0, 'd'}, {"anonymous", 0, 0, 'x'}, {"gsi", 0, 0, 'X'}, {"help", 0, 0, 'h'}, {"dryrun", 0, 0, 'D'}, {"dumpxrsl", 0, 0, 'P'}, {"list-cases", 0, 0, 'l'}, {"dropintofile", 0, 0, 's'}, {"runtime", 1, 0, 'r'}, {0, 0, 0, 0} };

  jobs.clear();
  ngsubargs.clear();

  // initial parsing of specified arguments

  while( true ) {

    int optc = getopt_long_only( argc, argv, optstring, longoptions, NULL );
    if( optc == -1 ) {
      for( int i = optind; i< argc; i++ )
	jobs.push_back( argv[i] );
      break;
    }
    switch( optc ) {
    case 'v': {
      cout << progname << ": version " << VERSION << endl;
      exit(0);
    }
    case 'h': {
      print_usage( progname );
      exit(0);
    }
    case '?': {
      exit(1);
    }
    case 's': {
      dropintofile = true;
      continue;
    }
    case 'r': {
      runtime = optarg;
      continue;
    }
    case 'l': {
      print_cases( );
      exit(0);
    }
    default: {
      if( (char)optc == 'P' ) { 
	actuallysubmitted = false;
      }
      string arg = "-";
      arg += (char)optc;
      ngsubargs.push_back( arg );
      if( optarg != 0 ) ngsubargs.push_back( optarg );
      continue;
    }
    }
  }

  //  for( int i = 0; i<ngsubargs.size(); i++ )
  //    cout << "Option " << i << " : " << ngsubargs[i] << endl;
  //  cout << "runtime: " << runtime << endl;

  vector<int> jobids;

  for(unsigned int i=0; i<jobs.size(); i++) {
    if( jobs[i].find_first_not_of("0123456789-:") != string::npos ) {
      cout << "Do not understand jobid: " << jobs[i] << endl;
      exit(1);
    }

    if( (jobs[i].find_first_of(":") != string::npos) && (jobs[i].find_first_of("-") != string::npos) ) {
      cout << "Cannot specify both : and - in a jobid: " << jobs[i] << endl;
      exit(1);
    }

    if( jobs[i].find_first_of(":") != string::npos ) {
      // there is a colon in the jobid
      
      unsigned int pos = 0;
      if( (pos = jobs[i].find_first_of(":")) != jobs[i].find_last_of(":") ) {
	cout << "Can only specify one : in a jobid: " << jobs[i] << endl;
	exit(1);
      }
      // there is only one colon in the jobid

      // find the two numbers to the left and right of the colon
      int start = atoi( jobs[i].substr( 0,pos ).c_str() );
      int number = atoi( jobs[i].substr( pos+1 ).c_str() );

      // build jobids
      for(int id=start; id<start+number; id++)
	if( (id>=0) && (id<=number_testcases) ) {
	  jobids.push_back( id );
	} else {
	  cout << "Ignoring jobid: " << id << endl;
	}
    } else if( jobs[i].find_first_of("-") != string::npos ) {
      // there is a minus in the jobid

      unsigned int pos = 0;
      if( (pos = jobs[i].find_first_of("-")) != jobs[i].find_last_of("-") ) {
        cout << "Can only specify one - in a jobid: " << jobs[i] << endl;
        exit(1);
      }
      // there is only one minus in the jobid

      // find the two numbers to the left and right of the minus
      int start = atoi( jobs[i].substr( 0,pos ).c_str() );
      int end   = atoi( jobs[i].substr( pos+1 ).c_str() );

      // build jobids
      for(int id=start; id<=end; id++)
	if( (id>=0) && (id<=number_testcases) ) {
	  jobids.push_back( id );
	} else {
	  cout << "Ignoring jobid: " << id << endl;
	}
    } else {
      // there are no colons nor minuses in the jobid
      int id = atoi( jobs[i].c_str() );
      if( (id>=0) && (id<=number_testcases) ) {
	jobids.push_back( id );
      } else {
	cout << "Ignoring jobid: " << id << endl;
      }
    }

  }

  if(jobids.size()==0) {
    cout << "No testjob specified. Submitting default test-job 0." << endl;
    jobids.push_back( 0 );
  }

  // sort the jobids
  cout << "Submitting test-job numbers: ";

  for( unsigned int i=0; i<jobids.size(); i++)
    cout << jobids[i] << " "; 
  cout << endl;

  // constructing test-jobs
  vector<string> executable, arguments, executables, inputfiles, runtimeenvironments, outputfiles;
  executable_args( executable );
  arguments_args( arguments );
  executables_args( executables );
  inputfiles_args( inputfiles );
  outputfiles_args( outputfiles );
  runtimeenvironment_args( runtimeenvironments );

  // now construct the job-xrsl
  const unsigned int numberofjobs = jobids.size();
  string jobxrsl[numberofjobs];

  for( unsigned int i=0; i<jobids.size(); i++) {
    
    jobxrsl[i] += "&";  // initial &

    // now build test-jobs
    if( jobids[i]==0 || jobids[i]==83 ) {
      jobxrsl[i] += executable[7];
      jobxrsl[i] = jobxrsl[i] + "(arguments=" + runtime + ")";
      jobxrsl[i] += "(inputfiles=";

      if( jobids[i]==0 ) jobxrsl[i] = jobxrsl[i] + "(\"run.sh\" \"http://www.nordugrid.org;cache=no/data/run.sh\")";
      if( jobids[i]==83 ) jobxrsl[i] = jobxrsl[i] + "(\"run.sh\" \"ftp://ftp.nordugrid.org;cache=no/applications/test/localdisk.sh\")";
      if( match==true ) {
	jobxrsl[i] = jobxrsl[i] + "(\"Makefile\" \"rls://rls.nordugrid.org/Makefile\")";
      } else {
	jobxrsl[i] = jobxrsl[i] + "(\"Makefile\" \"ftp://ftp.nordugrid.org;cache=no/applications/test/Makefile\")";
      }
      jobxrsl[i] = jobxrsl[i] + "(\"prime.cpp\" \"ftp://ftp.nordugrid.org;cache=no/applications/test/prime.cpp\"))";

      jobxrsl[i] += "(stderr=\"primenumbers\")";

      jobxrsl[i] += "(outputfiles=";
      jobxrsl[i] = jobxrsl[i] + "(\"primenumbers\" \"\"))";
    }

    if( jobids[i]==1 ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[0];
    }

    if( jobids[i]==2 ) {
      jobxrsl[i] += executable[1];
      jobxrsl[i] += arguments[0];
    }

    if( jobids[i]==3 ) {
      jobxrsl[i] += executable[2];
      jobxrsl[i] += arguments[0];
      jobxrsl[i] += runtimeenvironments[0];
    }      

    if( jobids[i]==4 ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[1];
    }      

    if( jobids[i]==5 ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[2];
    }      

    if( jobids[i]==6 ) {
      jobxrsl[i] += executable[4];
      jobxrsl[i] += arguments[6];
      jobxrsl[i] += runtimeenvironments[0];
    }

    if( jobids[i]==7 ) {
      jobxrsl[i] += executable[5];
      jobxrsl[i] += runtimeenvironments[0];
      jobxrsl[i] += runtimeenvironments[1];
    }

    if( jobids[i]==8 ) {
      jobxrsl[i] += executable[4];
      jobxrsl[i] += arguments[3];
      jobxrsl[i] += executables[0];
      jobxrsl[i] = jobxrsl[i] + "(inputfiles=(\"echo.sh\" " +inputfiles[0] + "))";
    }      

    if( (jobids[i]>8) && (jobids[i]<61) ) {
      
      if( (jobids[i]-9) % 13 != 12) {
	
	int type = ( (jobids[i]-9) % 13 ) / 3;
	int number = ( (jobids[i]-9) % 13 ) % 3;

	if( (number==0)||(number==1) ) {
	  jobxrsl[i] += executable[0];
	  jobxrsl[i] += arguments[0];
	} else {
	  jobxrsl[i] += executable[3];
	  jobxrsl[i] += arguments[5];
	}

	jobxrsl[i] += "(inputfiles=";

	if(number==2) number=8;
	
	for(int j=0; j<=number; j++) {
	  jobxrsl[i] = jobxrsl[i] + "(" + "\"downloaded" + (char)(48+j) + "\" ";
	  jobxrsl[i] = jobxrsl[i] + inputfiles[3*type+(j % 3)] + ")";
	}

	if(number==8) jobxrsl[i] += "(\"md5sum.sh\" " + inputfiles[15] + ")";
	jobxrsl[i] += ")";
      }      
      
      if( (jobids[i]-9) % 13 == 12 ) {
	jobxrsl[i] += executable[3];
	jobxrsl[i] += arguments[5];

	jobxrsl[i] += "(inputfiles=";
	for(int j=0; j<=7; j++) {
	  jobxrsl[i] = jobxrsl[i] + "(" + "\"downloaded" + (char)(48+j) + "\" ";
	  jobxrsl[i] = jobxrsl[i] + inputfiles[3*(j / 2)+(j % 2)] + ")";
	}

	jobxrsl[i] += "(\"md5sum.sh\" " + inputfiles[15] + ")";
	jobxrsl[i] += ")";

      }
      
      if( (jobids[i]-9) >= 26 )
	jobxrsl[i] += "(ftpThreads=10)";
      if( ((jobids[i]-9)/13==1) || ((jobids[i]-9)/13==3) ) {
	unsigned int j = 0;
	while( (j = jobxrsl[i].find( "yes" )) != string::npos )
	  jobxrsl[i].replace( j, 3, "no");
      }
    }

    if( (jobids[i]>=61) && (jobids[i]<=66) ) {
      jobxrsl[i] += executable[3];
      jobxrsl[i] += arguments[5];
	
      int number = (jobids[i]-61) % 3;
      if(number==2) number=8;

      jobxrsl[i] += "(inputfiles=";
      for(int j=0; j<=number; j++) {
	jobxrsl[i] = jobxrsl[i] + "(" + "\"downloaded" + (char)(48+j) + "\" ";
	jobxrsl[i] = jobxrsl[i] + inputfiles[12+(j % 3)] + ")"; 
      }
      jobxrsl[i] += "(\"md5sum.sh\" " + inputfiles[15] + ")";
      jobxrsl[i] += ")";

      if( jobids[i]>=64 )
	jobxrsl[i] += "(ftpThreads=10)";

    }

    if( (jobids[i]==67)||(jobids[i]==68) ) {
      jobxrsl[i] += executable[3];
      jobxrsl[i] += arguments[5];

      jobxrsl[i] += "(inputfiles=";
      for(int j=0; j<=9; j++) {
	jobxrsl[i] = jobxrsl[i] + "(" + " \"downloaded" + (char)(48+j) + "\" ";
	jobxrsl[i] = jobxrsl[i] + inputfiles[3*(j / 2)+(j % 2)] + ")";
      }
      jobxrsl[i] += "(\"md5sum.sh\" " + inputfiles[15] + ")";
      jobxrsl[i] += ")";

      if( jobids[i]==68 )
	jobxrsl[i] += "(ftpThreads=10)";
    }

    if( (jobids[i]>=69)&&(jobids[i]<=71) ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[0];
      
      if(jobids[i]==69) jobxrsl[i] += "(cluster=\"lscf.nbi.dk\")";
      if(jobids[i]==70) jobxrsl[i] += "(|(cluster=\"grid.tsl.uu.se\")(cluster=\"quark.hep.lu.se\"))";
      if(jobids[i]==71) jobxrsl[i] += "(cluster!=\"lscf.nbi.dk\")";
    }
    
    if( jobids[i]==72 ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[0];
      
      jobxrsl[i] += "(cluster=\"grid.nbi.dk\")";
      jobxrsl[i] += "(queue=\"short\")";
    }

    if( (jobids[i]>=73)&&(jobids[i]<=75) ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[0];
      
      if(jobids[i]==73) jobxrsl[i] = jobxrsl[i] + "(middleware=\"nordugrid-" + VERSION + "\")";
      if(jobids[i]==74) jobxrsl[i] = jobxrsl[i] + "(middleware>=\"nordugrid-" + VERSION + "\")";
      if(jobids[i]==75) jobxrsl[i] = jobxrsl[i] + "(middleware<=\"nordugrid-" + VERSION + "\")";
    }

    if( jobids[i]==76 ) {
      jobxrsl[i] += executable[6];
      jobxrsl[i] += arguments[0];

      jobxrsl[i] += "(environment=(\"ECHOBIN\" \"/bin\"))";
    }
    
    if( jobids[i]==77 ) {
      jobxrsl[i] += "(rsl_substitution=(\"ECHOBIN\" \"/bin\"))";
      jobxrsl[i] += executable[8];
      jobxrsl[i] += arguments[0];

    }
    
    if( jobids[i]==78 ) {
      jobxrsl[i] += executable[0];
      jobxrsl[i] += "(rsl_substitution=(\"ECHOBIN\" \".\"))";
      jobxrsl[i] += arguments[8];
    }

    if( jobids[i]==79 ) { 
      jobxrsl[i] += executable[0];
      jobxrsl[i] += arguments[0];
      jobxrsl[i] += "(cluster=lscf.nbi.dk)(queue=gridshort)(CPUTime=\"2 hours\")";
    }

    if( (jobids[i]>=80)&&(jobids[i]<=82) ) {
      struct timeval time;
      struct timezone zone;
      int result = gettimeofday( &time, &zone );
      if ( result != 0 ) {
	cout << "Could not gettimeofday! Exiting." << endl;
	return 1;
      }

      unsigned int seed = time.tv_usec;
      srandom( seed );

      jobxrsl[i] += executable[0];
      
      string argument_string = "The following files have been uploaded:";

      int number = (jobids[i]-80) % 3;
      if(number==2) number = 8;

      jobxrsl[i] += "(inputfiles=";
      for(int j=0; j<=number; j++) {
	jobxrsl[i] = jobxrsl[i] + "(" + "\"downloaded" + (char)(48+j) + "\" ";
	jobxrsl[i] = jobxrsl[i] + inputfiles[6+(j % 3)] + ")"; 
      }
      jobxrsl[i] += ")";

      jobxrsl[i] += "(outputfiles=";
      for(int j=0; j<=number; j++) {
	unsigned int randomnumber = (unsigned int)random();
	string randomnumber_str = convert_int_to_string( randomnumber );

	jobxrsl[i] = jobxrsl[i] + "(" + "\"downloaded" + (char)(48+j) + "\" ";
	jobxrsl[i] = jobxrsl[i] + "\"" + outputfiles[0] + "uploaded" + randomnumber_str + "\")";
	argument_string = argument_string + " " + outputfiles[0] + "uploaded" + randomnumber_str;
      }
      jobxrsl[i] += ")";
      argument_string += ". They can be removed with ngremove.";

      jobxrsl[i] = jobxrsl[i] + "(arguments=\"" + argument_string + "\")";

    }

    string jobid_str = convert_int_to_string( jobids[i] );
    
    jobxrsl[i] = jobxrsl[i] + "(jobName=\"ngtest-job-" + jobid_str + "\")";
    jobxrsl[i] = jobxrsl[i] + "(stdout=\"stdout\")";

    if( jobids[i]!=0 && jobids[i]!=83 ) {
      jobxrsl[i] = jobxrsl[i] + "(stderr=\"stderr\")";
    }

    jobxrsl[i] = jobxrsl[i] + "(gmlog=\"gmlog\")";

    if( jobids[i]!=79) jobxrsl[i] = jobxrsl[i] + "(CPUTime=" + runtime + ")";

  }

  cout << "Client middleware: nordugrid-" << VERSION;

  char* globusloc = getenv("GLOBUS_LOCATION");
  if( globusloc==NULL ) globusloc = DEFAULT_GLOBUS_LOCATION;
  
  string versionfile = (string)globusloc + "/share/doc/VERSION";
  ifstream file(versionfile.c_str());
  
  if(file) {
    string readline;
    file >> readline;
    if( readline.size()>0 )
      cout << ", globus-" << readline;
    file.close();
  }

  cout << endl;   

  //Make sure jobid is dropped into file

  //First create temporary file
  char* temp = (char*)malloc( 19 );
  strncpy( temp, "/tmp/ngtest.XXXXXX", 19 );
  temp[18]=0;
  if( mkstemp(temp) == -1 ) {
    cout << "Could not create temporary file for the jobid. Exiting." << endl;
    exit(1);
  }

  //then add -o option to ngsubargs
  int found = -1;
  if( ngsubargs.size()>=1 ) {
    for( unsigned int i=0; i<ngsubargs.size()-1; i++ ) {
      if( strncmp( ngsubargs[i].c_str(), "-o", 2 ) == 0 ) {
	ngsubargs[i+1] = temp;
	found = i;
      }
    }
  }
  if( found == -1 ) {
    ngsubargs.push_back( "-o" );
    ngsubargs.push_back( temp );
  }

  //Now construct the arguments for the ngsub-command
  int numargs = ngsubargs.size();
  char* ngsub_argv[numargs+4];

  ngsub_argv[0] = "ngsub";
  for(int i=0; i<numargs; i++) {
    ngsub_argv[i+1] = (char *)ngsubargs[i].c_str();
    // cout << ngsubargs[i] << endl;
  }

  ngsub_argv[numargs+3] = 0;
  ngsub_argv[numargs+1] = "-xrsl";

  //Drop job-xrsls into file
  if( dropintofile == true ) {
    string xrsl;
    for( unsigned int i=0; i<numberofjobs; i++ ) {
      string filename("ngtest");
      filename = filename + "-" + convert_int_to_string( jobids[i] ) + ".xrsl";
      ofstream out( filename.c_str() );
      out << jobxrsl[i] << endl;
      out.close();
    }
  
  }

  //Submit jobs
  for( unsigned int i=0; i<numberofjobs; i++ ) {
    
    ngsub_argv[numargs+2] = (char *)jobxrsl[i].c_str();
    
    // now fork before submitting
    pid_t pid;
    pid = fork();
    
    switch( pid ) {
      
    case -1:
      cout << "Fork failed. Exiting." << endl;
      exit(1);
      
    case 0:
      cout << "Submitting xrsl: " << jobxrsl[i] << endl;
      execvp("ngsub", ngsub_argv);
      
    default:
      int child_status = 0;
      wait( &child_status );

      if( WIFEXITED(child_status)!=0 && WEXITSTATUS(child_status)==0 ) {
	if( actuallysubmitted ) {
	  ifstream file( temp );
      
	  string jobid;
	  while( file >> jobid ) { };
	  int end = jobid.find(':',9);
	  string clustername = jobid.substr( 9, end-9 );

	  CertInfo user;
	  vector<Cluster> clusters;
	  clusters.push_back( Cluster( clustername ) );
	  
	  FindClusterInfo( clusters, Mds::ClusterInfo, user.GetSNx(), true, 40, 0 );
	  
	  Environment dummyenv( "globus" );
	  EnvironmentTest envtest( dummyenv, Env::eq );
	  const Environment* globusenv = clusters[0].FindEnvironment( "middleware", envtest); 

	  Environment dummyenv2( "nordugrid" );
	  EnvironmentTest envtest2( dummyenv2, Env::eq );
	  const Environment* nordugridenv = clusters[0].FindEnvironment( "middleware", envtest2); 

      cout << "Cluster (" << clustername << ") middleware: ";
      if (nordugridenv!=NULL) {
        cout << nordugridenv->GetOriginal() << ", ";
      } else {
        cout << "nordugrid-<unknown>, ";
      }

      if (globusenv!=NULL) {
        cout << globusenv->GetOriginal() << endl;
      } else {
        cout << "globus-<unknown>" << endl;
      }
	}
      } else {
	cout << "Job not submitted!" << endl;
	continue;
      }
    }
  }

  unlink( temp );
  return 0;
}

vector<string> FindNorduGridVO( )
{ 
  // query VO-server at base given
  string ldapserver( "grid-vo.nordugrid.org" );
  string base( "ou=People,dc=nordugrid,dc=org" );
  int timeout = 20;
  int debug = 0;

  vector<string> members;

  string dummy;
  vector<string> dummyvector;

  LdapQuery query;
  query.Connect( ldapserver, 389, dummy, true, timeout, debug );
  query.Query( base, dummy, dummyvector, LdapQuery::subtree, timeout, debug );
  query.Result( VOQueryCallback, &members, timeout, debug );

  return members;
}

void VOQueryCallback( const string& attr, const string& value, void* ref )
{
  vector<string>* save = (vector<string>*)ref;

  // save all subjects into saved pointer
  if( strcmp( attr.c_str(), "description" )==0 && strncmp( value.c_str(), "subject=", 8)==0 ) {
    string cutvalue = value.substr( 8 ); // skip subject=
    while( cutvalue[0]==' ' ) cutvalue = cutvalue.substr( 1 );
    save->push_back( cutvalue );
  }
}

string convert_int_to_string( unsigned int number )
{
  char b[100];
  snprintf( b, 99, "%d", number );
  
  return b;
}

void executable_args( vector<string> &components )
{
  string begin = "(executable=";
  string end   = ")";

  components.push_back( begin + "\"/bin/echo\"" + end );
  components.push_back( begin + "\"echo.sh\""   + end );
  components.push_back( begin + "\"$ECHOBIN/echo.sh\"" + end );
  components.push_back( begin + "\"md5sum.sh\"" + end );
  components.push_back( begin + "\"/bin/ls\""   + end );
  components.push_back( begin + "\"/usr/bin/printenv\"" + end );
  components.push_back( begin + "\"$ECHOBIN/echo\"" + end );
  components.push_back( begin + "\"run.sh\"" + end );
  components.push_back( begin + "$(ECHOBIN)/echo" + end );
  components.push_back( begin + "\"localdisk.sh\"" + end );
}

void arguments_args( vector<string> &components )
{
  string begin = "(arguments=";
  string end   = ")";

  components.push_back( begin + "\"hello, grid\"" + end );
  components.push_back( begin + "1 2 3" + end );
  components.push_back( begin + "1 2 3 4 5 6 7 8 9 0" + end );
  components.push_back( begin + "-l" + end );
  components.push_back( begin + "\"echo.sh\"" + end );
  components.push_back( begin + "downloaded[0-9]" + end );
  components.push_back( begin + "\"$ECHOBIN/echo.sh\"" + end );
  components.push_back( begin + "\"hello, testsuite\"" + end );
  components.push_back( begin + "$(ECHOBIN)/echo.sh" + end );
}

void executables_args( vector<string> &components )
{
  string begin = "(executables=\"";
  string end   = "\")";

  components.push_back( begin + "echo.sh" + end );
  components.push_back( begin + "cat.sh" + end );
}

void inputfiles_args( vector<string> &components )
{
  components.push_back( "\"http://www.nordugrid.org;cache=yes/data/echo.sh\"" );
  components.push_back( "\"http://www.nordugrid.org;cache=yes/data/somefile\"" );
  components.push_back( "\"http://www.nordugrid.org;cache=yes/data/somehugefile\"" );

  components.push_back( "\"gsiftp://hathi.hep.lu.se;cache=yes/public/ngtest/echo.sh\"" );
  components.push_back( "\"gsiftp://hathi.hep.lu.se;cache=yes/public/ngtest/somefile\"" );
  components.push_back( "\"gsiftp://hathi.hep.lu.se;cache=yes/public/ngtest/somehugefile\"" );

  components.push_back( "\"ftp://ftp.nordugrid.org;cache=yes/applications/test/echo.sh\"" );
  components.push_back( "\"ftp://ftp.nordugrid.org;cache=yes/applications/test/somefile\"" );
  components.push_back( "\"ftp://ftp.nordugrid.org;cache=yes/applications/test/somehugefile\"" );

  components.push_back( "\"rls://rls.nordugrid.org/echo.sh\"" );
  components.push_back( "\"rls://rls.nordugrid.org/somefile\"" );
  components.push_back( "\"rls://rls.nordugrid.org/somehugefile\"" );

  components.push_back( "\"echo.sh\"" );
  components.push_back( "\"somefile\"" );
  components.push_back( "\"somehugefile\"" );

  // misc
  components.push_back( "\"ftp://ftp.nordugrid.org;cache=yes/applications/test/md5sum.sh\"" );
  components.push_back( "\"ftp://ftp.nordugrid.org;cache=yes/applications/test/ECHOENV\"");
  components.push_back( "\"ftp://ftp.nordugrid.org;cache=yes/applications/test/MD5SUMENV\"");

}

void outputfiles_args( vector<string> &components )
{
  components.push_back( "gsiftp://hathi.hep.lu.se/public/ngtest/uploaded/" );
}

void runtimeenvironment_args( vector<string> &components )
{
  string begin = "(runTimeEnvironment=\"";
  string end   = "\")";

  components.push_back( begin + "ECHOENV" + end );
  components.push_back( begin + "MD5SUMENV" + end );
}

void print_usage( string progname )
{
  cout << "Usage: " << progname << " [jobids] [xrsl-options ...]" << endl;
  cout << endl;
  cout << "Jobids is a list of job-partitions which can take the form:" << endl;
  cout << endl;
  cout << "x    means: x" << endl;
  cout << "x:n  means: x, x+1,..., x+(n-1)" << endl;
  cout << "x-y  means: x,...,y" << endl;
  cout << endl;
  cout << "i.e. 4 5:3 14-17 gets expanded to:" << endl;
  cout << endl;
  cout << "4 5 6 7 14 15 16 17" << endl;
  cout << endl;
  cout << "ngtest accepts the following options besides the usual xrsl-options:" << endl;
  cout << endl;
  cout << "-v - prints version number" << endl;
  cout << "-h - prints a short usage-description" << endl;
  cout << "-s - drop the submitted xrsls into the files ngtest-<testnumber>.xrsl" << endl;
  cout << "-l - lists the different test-cases" << endl;
  cout << "-r <int> - the number of minutes that testcase 0 should run. Default is 5." << endl;
  cout << endl;
  cout << "The xrsl-options are passed on directly to ngsub." << endl;
}

void print_cases( )
{
  cout << "The following test-jobs are included in the program at the moment:" << endl;
  cout << endl;
  cout << "#0 : A complete job that tests a number of different things." << endl;
  cout << "#1 : basic executable xrsl (hello grid)" << endl;
  cout << "#2 : UI uploads executable                        (requires echo.sh locally)" << endl;
  cout << "#3 : executable specified via runtime-environment (requires ECHOENV on cluster)" << endl;
  cout << "#4 : tests  3 arguments" << endl;
  cout << "#5 : tests 10 arguments" << endl;
  cout << "#6 : tests runtime-environment in arguments-attr  (requires ECHOENV on cluster)" << endl;
  cout << "#7 : tests 2 runtime-environments                 (requires ECHOENV, MD5SUMENV)" << endl;
  cout << "#8 : test executables attribute                   (requires echo.sh locally)" << endl;
  cout << endl;
  cout << "#9 : tests downloading 1 inputfile via http" << endl;
  cout << "#10: tests downloading 2 inputfiles via http" << endl;
  cout << "#11: tests downloading 9 inputfiles via http" << endl;
  cout << "#12: tests downloading 1 inputfiles via gsiftp" << endl;
  cout << "#13: tests downloading 2 inputfiles via gsiftp" << endl;
  cout << "#14: tests downloading 9 inputfiles via gsiftp" << endl;
  cout << "#15: tests downloading 1 inputfile via ftp" << endl;
  cout << "#16: tests downloading 2 inputfiles via ftp" << endl;
  cout << "#17: tests downloading 9 inputfiles via ftp" << endl;
  cout << "#18: tests downloading 1 inputfiles via rls" << endl;
  cout << "#19: tests downloading 2 inputfiles via rls" << endl;
  cout << "#20: tests downloading 9 inputfiles via rls" << endl;
  cout << "#21: tests downloading 8 files, 2 of each via http, ftp, gsiftp and rls" << endl;
  cout << endl;
  cout << "#22-#34: as above with cache=no" << endl;
  cout << "#35-#47: as above with ftpThreads=10" << endl;
  cout << "#48-#60: as above with cache=no and ftpThreads=10" << endl;
  cout << endl;
  cout << "#61: UI uploads 1 file      (requires echo.sh locally)" << endl;
  cout << "#62: UI uploads 2 files     (requires echo.sh, somefile locally)" << endl;
  cout << "#63: UI uploads 9 files     (requires echo.sh, somefile, somehugefile locally)" << endl;
  cout << endl;
  cout << "#64-#66: as above with ftpThreads=10" << endl;
  cout << endl;
  cout << "#67: combination of #21 and #62 -- 10 files processed" << endl;
  cout << "#68: as #67 with ftpThreads=10" << endl;
  cout << endl;
  cout << "#69: tests cluster-attribute" << endl;
  cout << "#70: tests cluster-attribute and |" << endl;
  cout << "#71: tests cluster-attribute and !=" << endl;
  cout << "#72: tests cluster- and queue attribute" << endl;
  cout << endl;
  cout << "#73: tests middleware-attribute (=)" << endl;
  cout << "#74: tests middleware-attribute (>=)" << endl;
  cout << "#75: tests middleware-attribute (<=)" << endl;
  cout << endl;
  cout << "#76: tests environment-attribute" << endl;
  cout << "#77: tests rsl_substitution in executable-attribute" << endl;
  cout << "#78: tests rsl_substitution in arguments-attribute" << endl;
  cout << "#79: tests brokering and CpuTime with \"xx hours\"  (see ng-bug-63)" << endl;

  cout << "#80: uploading 1 file to gsiftp-server" << endl;
  cout << "#81: uploading 2 file to gsiftp-server" << endl;
  cout << "#82: uploading 9 file to gsiftp-server" << endl;
  cout << endl;
  cout << "#83: tests the LOCALDISK attribute. Runs testcase 0 on the localdisk." << endl;

}
