package LRMS_Condor;

use strict;
use warnings;

BEGIN {
    use base 'Exporter';

    # Set the version for version checking.
    our $VERSION = '2.000';

    # This export list can be generated with the following Vim command:
    # :r !awk '/\<sub (lrms|nord|condor_)/ { print "       ", $2 }' % | sort
    our @EXPORT = qw(
        condor_config
        condor_location
        lrms_get_job_executionnodes
        lrms_get_job_executionnodes_completed
        lrms_get_jobinfo_logged
        lrms_get_job_status
        lrms_get_localids
        lrms_get_localids_running
        lrms_get_queued
        lrms_get_total
        nordugrid_authuser_freecpus
        nordugrid_authuser_queuelength
        nordugrid_cluster_cpudistribution
        nordugrid_cluster_lrms_type
        nordugrid_cluster_lrms_version
        nordugrid_cluster_totalcpus
        nordugrid_cluster_usedcpus
        nordugrid_job_lrmscomment
        nordugrid_job_queuerank
        nordugrid_job_reqcput
        nordugrid_job_reqwalltime
        nordugrid_job_usedcputime
        nordugrid_job_usedmem
        nordugrid_job_usedwalltime
        nordugrid_queue_defaultcputime
        nordugrid_queue_gridqueued
        nordugrid_queue_gridrunning
        nordugrid_queue_maxcputime
        nordugrid_queue_maxqueuable
        nordugrid_queue_maxrunning
        nordugrid_queue_maxuserrun
        nordugrid_queue_mincputime
        nordugrid_queue_queued
        nordugrid_queue_running
        nordugrid_queue_status
        nordugrid_queue_totalcpus
    );
}

my $qdef = '';
my $queuename = $main::config{queue};
if ($queuename) {
    $qdef = join "", split /\[separator\]/, $main::config{requirements};
    die "Option 'requirements' is not defined for queue $queuename" unless $qdef;
}


#
# Total number of CPUs available for running jobs.
#
sub nordugrid_cluster_totalcpus {
    return scalar condor_cluster_totalcpus();
}

#
# Total number of CPUs in the queue
#
sub nordugrid_queue_totalcpus {
    return scalar condor_queue_get_nodes();
}

#
# Number of CPUs that are busy, either in a job or interactive use.
#
sub nordugrid_cluster_usedcpus {
    return condor_cluster_get_usedcpus();
}

#
# Text string containing the LRMS type.
#
sub nordugrid_cluster_lrms_type {
    return 'Condor';
}

#
# String containing LRMS version.  ('UNKNOWN' in case of errors.)
#
sub nordugrid_cluster_lrms_version {
    my ($out, $err, $ret) = condor_run('bin/condor_version');
    return 'UNKNOWN' if $ret != 0;
    $out =~ /\$CondorVersion:\s+(\S+)/;
    return $1 || 'UNKNOWN';
}

#
# Returns the number of queued jobs (idle and held) in the LRMS pool.  Counts
# both Grid jobs and jobs submitted directly to the LRMS by local users.
#
sub nordugrid_queue_queued {
    return condor_queue_get_queued('global');
}

#
# Returns the number of queued jobs (idle and held) in the LRMS pool.  Counts
# only Grid jobs.
#
sub nordugrid_queue_gridqueued {
    return condor_queue_get_queued();
}

#
# Counts all jobs in the custer, queued and running, including machines in interactive use
#
sub lrms_get_total {
    return condor_cluster_get_usedcpus() + condor_cluster_get_queued();
}

# Counts all queued jobs in the cluster
sub lrms_get_queued {
    return condor_cluster_get_queued();
}

#
# Returns 'inactive' if condor_status fails.
#
sub nordugrid_queue_status {
    return condor_error() ? 'inactive' : 'active';
}

#
# Returns total number of CPUs claimed by jobs, plus the number of CPUs in
# interactive use.
#
sub nordugrid_queue_running {
    return condor_queue_get_running();
}

#
# Returns number of running jobs on Condor that came from the Grid.  Since
# condor_q by default lists only the jobs submitted from the machine where
# condo_q is running, and only the Grid Manager is allowed to submit from
# there, we can easily tell how many jobs belong to the Grid.
#
sub nordugrid_queue_gridrunning {
    return condor_queue_get_gridrunning();
}

#
# Returns the number of CPUs in the Condor pool, which is always equal to the
# maximum number of running jobs.
#
sub nordugrid_queue_maxrunning {
    return scalar condor_queue_get_nodes();
}

#
# Returns 2 * maxrunning, which is an arbitrary number.  There is (as far as I
# know) no limit on the number of queued jobs.
#
sub nordugrid_queue_maxqueuable {
    return (2 * nordugrid_queue_maxrunning());
}

#
# Returns the maximum number of jobs that a single user can run at once.
# TODO: I don't know the details as to how Condor handles this.
#
sub nordugrid_queue_maxuserrun {
    return nordugrid_queue_maxrunning();
}

#
# There's no limit on the CPU time in Condor, so leave this blank.
#
sub nordugrid_queue_maxcputime {
    return '';
}

#
# There's no limit on the CPU time in Condor, so leave this blank.
#
sub nordugrid_queue_mincputime {
    return '';
}

#
# Always returns maxcputime, since there's no limit on the CPU time in Condor.
#
sub nordugrid_queue_defaultcputime {
    return nordugrid_queue_maxcputime();
}

#
# Currently set to the number of free nodes.
#
sub nordugrid_authuser_freecpus {
    return condor_queue_get_nodes() - condor_queue_get_running();
}

#
# Returns number of jobs queued by Grid.
#
sub nordugrid_authuser_queuelength {
    return nordugrid_queue_gridqueued();
}

#
# (Used in 'nordugrid-job-status'.)
# Takes two arguments:
# 1. The LRMS job id as represented in the GM.  (In Condor terms,
#    it's <cluster>.condor.  <proc> is not included, since only
#    one job is submitted at a time, so <proc> is always zero.)
# 2. The 'controldir' attribute from arc.conf.
#
# Returns the current status of the job by mapping Condor's JobStatus
# integer into corresponding one-letter codes used by ARC:
#
#   1 (Idle)       --> Q (job is queuing, waiting for a node, etc.)
#   2 (Running)    --> R (running on a host controlled by the LRMS)
#   2 (Suspended)  --> S (an already running job in a suspended state)
#   3 (Removed)    --> E (finishing in the LRMS)
#   4 (Completed)  --> E (finishing in the LRMS)
#   5 (Held)       --> O (other)
#
# If the job couldn't be found, E is returned since it is probably finished.
#
sub lrms_get_job_status {
    my %num2letter = qw(1 Q 2 R 3 E 4 E 5 O);
    my $s = condor_job_get_field('JobStatus', $_[0]);
    return 'E' if !defined $s;
    $s = $num2letter{$s};
    if ($s eq 'R') {
        $s = 'S' if condor_job_suspended(@_);
    }
    return $s;
}

#
# There's no easy way to define the job's queue "position" in Condor.
#
sub nordugrid_job_queuerank {
    return '';
}

#
# Takes one argument, the LRMS job id as represented in the GM.  (In Condor
# terms, it's <cluster>.condor.  <proc> is not included, since only one job is
# submitted at a time, so <proc> is always zero.)
#
# Returns number of minutes of CPU time the job has consumed, rounded to the
# nearest minute.
#
sub nordugrid_job_usedcputime {
    my $time = condor_job_get_field('RemoteUserCpu', $_[0]);
    return 0 if !defined $time;
    return sprintf "%.0f", $time / 60;
}

#
# Takes one argument, the LRMS job id as represented in the GM.  (In Condor
# terms, it's <cluster>.condor.  <proc> is not included, since only one job is
# submitted at a time, so <proc> is always zero.)
#
# Returns number of minutes the job has been allocated to a machine, rounded to
# the nearest minute.
#
sub nordugrid_job_usedwalltime {
    my $time = condor_job_get_field('RemoteWallClockTime', $_[0]);
    return 0 if !defined $time;
    return sprintf "%.0f", $time / 60;
}

#
# Takes one argument, the LRMS job id as represented in the GM.  (In Condor
# terms, it's <cluster>.condor.  <proc> is not included, since only one job is
# submitted at a time, so <proc> is always zero.)
#
# Returns virtual image size of executable in KB.
#
sub nordugrid_job_usedmem {
    my $size = condor_job_get_field('ImageSize', $_[0]);
    return defined $size ? $size : 0;
}

#
# Condor has no "requested CPU time" attribute.
#
sub nordugrid_job_reqcput {
    return '';
}

#
# Condor has no "requested walltime" attribute.
#
sub nordugrid_job_reqwalltime {
    return '';
}

#
# TODO: Returns no useful comment yet, but I'll improve this in the future.
#
sub nordugrid_job_lrmscomment {
    return '';
}

#
# Parses long output from condor_q -l
# and condor_status -l into and hash.
# OBS: Does NOT remove quotes around strings
# OBS: Filed names are lowercased! 
#
sub condor_digest_classad($) {
    my %classad;
    for (split /\n+/, shift) {
        /^(\w+)\s*=\s*(.*\S|)\s*$/ && ($classad{lc $1} = $2);
    }
    return %classad;
}

#
# Takes an optional constraint description string and returns the names of the
# nodes which satisfy this contraint. If no constraint is given, returns all
# the nodes in the pool
#
sub condor_grep_nodes {
    my $req = shift;
    my $cmd = 'bin/condor_status -format "%s\n" Machine';
    $cmd .= " -constraint '$req'" if $req;
    my ($out, $err, $ret) = condor_run($cmd);
    return () if $ret != 0;
    return split /\n/, $out;
}

{
    my @queuenodes;
    my $queuenodes_initialized;

    #
    # returns the list of nodes belonging to the current queue
    #
    sub condor_queue_get_nodes() {
        return @queuenodes if $queuenodes_initialized;
        @queuenodes = condor_grep_nodes($qdef);
        $queuenodes_initialized = 1;
        return @queuenodes;
    }
}

{
    my @allnodedata;
    my $allnodedata_initialized;

    my %alljobdata;
    my $alljobdata_initialized;

    my $error;

    sub condor_error {
        collect_node_data();
        collect_job_data();
        return $error;
    }

    #
    # Helper funtion which collects all the information about condor nodes.
    #
    sub collect_node_data() {
        return if $allnodedata_initialized;
        @allnodedata = ();
        my ($out, $err, $ret) = condor_run("bin/condor_status -long");
        if ($ret != 0) {
            $error = 1;
            $allnodedata_initialized = 1;
            return;
        }
        for (split /\n\n+/, $out) {
            my %target = condor_digest_classad($_);
            next unless defined $target{machine};
            push @allnodedata, \%target;
        }
        $allnodedata_initialized = 1;
        
    }

    #
    # Helper funtion which collects all the information about condor jobs.
    # Jobs are present twice: once by ClusterID and second, ClusterID.ProcID
    #
    sub collect_job_data {
        return if $alljobdata_initialized;
        %alljobdata = ();
        # collect data about all jobs
        my ($out, $err, $ret) = condor_run("bin/condor_q -long -global");
        if ($ret != 0) {
            $error = 1;
            $alljobdata_initialized = 1;
            return;
        }
        for my $jobdata (split /\n\n+/, $out) {
            my %job = condor_digest_classad($jobdata);
            my ($clusterid, $procid) = ($job{clusterid}, $job{procid});
            next unless defined $job{clusterid};
            # some of these might not be grid jobs
            delete $job{nordugridjob};
            $clusterid .= ".$procid" if $procid; # in case job is multi-process
            $alljobdata{$clusterid} = \%job;
        }
        # Collect data about grid jobs
        ($out, $err, $ret) = condor_run("bin/condor_q -long");
        for my $jobdata (split /\n\n+/, $out) {
            my %job = condor_digest_classad($jobdata);
            my ($clusterid, $procid) = ($job{clusterid}, $job{procid});
            next unless defined $job{clusterid};
            $job{nordugridjob} = 'Yep';
            $clusterid .= ".$procid" if $procid; # in case job is multi-process
            $alljobdata{$clusterid} = \%job;
        }

        $alljobdata_initialized = 1;
    }

    #
    # Returns the list of job IDs of all the jobs condor knows about.
    # The IDs have the same format as 'localid' in the GM.
    #
    sub lrms_get_localids {
        collect_job_data();
        return  map {"$_.condor"} keys %alljobdata;
    }

    #
    # Returns the list of job IDs of all running jobs condor knows about.
    # The IDs have the same format as 'localid' in the GM.
    #
    sub lrms_get_localids_running {
        collect_job_data();
        return  map {"$_.condor"} grep {$alljobdata{$_}{jobstatus} == 2} keys %alljobdata;
    }

    #
    # Takes two arguments:
    #
    #  1. A Condor ClassAd attribute name.
    #  2. A Condor job ID (<ClusterId>.condor).
    #
    # Returns the value of the attribute named in the first argument for the job
    # specified in the second argument.
    #
    sub condor_job_get_field {
        my $field = $_[0];
        my ($id) = $_[1] =~ /(.*)\.condor/;
        collect_job_data();
        return undef unless defined $alljobdata{$id};
        return $alljobdata{$id}{lc $field};
    }

    #
    # Count queued jobs (idle or held).  If argument is null, count only jobs
    # submitted from this host (that is, grid jobs).  If argument is not null,
    # include all jobs in the count (grid + condor local)
    #
    sub condor_queue_get_queued {
        my $sum = 0;
        my $arg = shift;
        my $qfactor = condor_queue_get_nodes() / condor_cluster_totalcpus();
        collect_job_data();
        for (values %alljobdata) {
            my %job = %$_;
            # only include nodes which are idle or held
            next unless $job{jobstatus} == 1 || $job{jobstatus} == 5;
            if (defined $job{nordugridjob}) {
                $sum += 1 if $job{nordugridqueue} eq $queuename;
            } elsif ($arg) {
                # not a grid job, we don't know which queue it belongs
                # try guessing the odds
                $sum += $qfactor;
            }
        } 
#       return $sum;
        return int($sum);
    }

    #
    # Counts all queued jobs (iddle and held) in the cluster.
    #
    sub condor_cluster_get_queued() {
        my $sum = 0;
        collect_job_data();
        do {$sum++ if $$_{jobstatus} == 1 || $$_{jobstatus} == 5} for values %alljobdata; 
        return $sum;
    }

    #
    # Counts runnung jobs (condor JobStatus == 2) submitted by Grid
    # into the current queue
    #
    sub condor_queue_get_gridrunning() {
        my $sum = 0;
        my @qnod = condor_queue_get_nodes();
        collect_job_data();
        for (values %alljobdata) {
            my %job = %$_;
            my $id = $job{clusterid};
            next unless defined $job{nordugridjob};
            next unless $job{jobstatus} == 2;
            my $host = $job{remotehost};
            $host = $job{lastremotehost} unless $host;
            next unless $host;
            # only count job if it's running in the current queue
            $sum++ if grep {$host =~ /"$_"/i} @qnod;
            
        }
        return $sum;
    }

    #
    # Counts nodes in the current queue with state 'Unclaimed'
    # Every running job is automatically included, plus nodes
    # used interactively by their owners
    #
    sub condor_queue_get_running() {
        my @free;
        my @qnod = condor_queue_get_nodes();
        collect_node_data();
        for (@allnodedata) {
            my %node = %$_;
            next unless grep {$node{name} =~ /"(vm\d+@)?$_"/i } @qnod;
            next unless $node{state} =~ /"Unclaimed"/i;
            push @free, $node{name};
        }
        return @qnod - @free;
    }

    #
    # Same as above, but for the whole cluster
    #
    sub condor_cluster_get_usedcpus() {
        my $free = 0;
        collect_node_data();
        do {$free += 1 if $$_{state} =~ /"Unclaimed"/i} for @allnodedata;
        return condor_cluster_totalcpus() - $free;
    }

    #
    # returns the total number of nodes in the cluster
    #
    sub condor_cluster_totalcpus() {
        collect_node_data();
        return scalar @allnodedata;
    }

    #
    # CPU distribution string (e.g., '1cpu:5 2cpu:1').
    #
    sub nordugrid_cluster_cpudistribution {
        # List all machines in the pool.  Machines with multiple CPUs are listed
        # one time for each CPU, with a prefix such as 'vm1@', 'vm2@', etc.
        my %machines;
        collect_node_data();
        $machines{$$_{machine}}++ for @allnodedata;
    
        # Count number of machines with one CPU, number with two, etc.
        my %dist;
        for (keys %machines) {
            $dist{$machines{$_}}++;
        }
    
        # Generate CPU distribution string.
        my $diststr = '';
        for (sort { $a <=> $b } keys %dist) {
            $diststr .= ' ' unless $diststr eq '';
            $diststr .= "${_}cpu:$dist{$_}";
        }
    
        return $diststr;
    }
}

#
# (Used in 'nordugrid-job-executionnodes'.)
# Takes one argument, the LRMS job id as represented in the GM.  (In Condor
# terms, it's <cluster>.condor.  <proc> is not included, since only one job is
# submitted at a time, so <proc> is always zero.)
#
# Returns the node the job runs on, or last ran on in case the job is not
# currently running.  Only returns one node, since we don't support MPI
# jobs.
#
sub lrms_get_job_executionnodes {
    my $tmp = condor_job_get_field('RemoteHost', $_[0]);
    if ($tmp) {
        my ($host) = $tmp =~ /^"(.+)"$/;
        return $host if $host;
    }
    $tmp = condor_job_get_field('LastRemoteHost', $_[0]);
    if ($tmp) {
        my ($host) = $tmp =~ /^"(.+)"$/;
        return $host if $host;
    }
    return 'UNKNOWN';
}

#
# Like lrms_get_job_executionnodes(), but this version looks at the
# job.ID.errors file to find out where completed jobs ran.
#
sub lrms_get_job_executionnodes_completed {
    my $gmlog = shift;
    local *GMLOG;
    open GMLOG, "<$gmlog" or return '';
    local $/;
    my $logdata = <GMLOG>;
    close GMLOG;
    my ($exechost) = $logdata =~ /.+Job executing on host: <([^:]+)/s;
    return $exechost || '';
}

sub lrms_get_jobinfo_logged {
    my ($jobinfo, $ctldir) = @_;

    my $e1 = '----- starting finish-condor-job -----';
    my $e2 = '.* Job executing on host: <.*>';
    my $e3 = 'Allocation/Run time:.*';
    my $e4 = 'Total Remote CPU Time:.*';
    my $tmp = `egrep -H '($e1|finish-condor-job: ($e2|$e3|$e4))\$' \\
                 $ctldir/job.*.errors`;

    for my $chunk (split /^.*:\Q$e1\E\n/m, $tmp) {
        # Currently only fetching the last host the job executed on.
        my ($exechost) =
          $chunk =~ /.*finish-condor-job:[^\n]+executing on host: <([^:]+)/s;
        next if !$exechost;

        # The GM job id.
        my ($id) = $chunk =~ /job\.([^.]+)\.errors:/;

        # skip jobs which not already in jobinfo
        next unless defined $jobinfo->{$id};

        # Strings in the form <days> <hours>:<minutes>:<seconds>.
        my ($walltstr) = $chunk =~ m{.*Allocation/Run time:\s+([^\n]+)}s;
        my ($cputstr) = $chunk =~ m{Total Remote CPU Time:\s+([^\n]+)};

        {   no warnings 'uninitialized';

            # Convert wallclock time string to minutes.
            my ($d, $h, $m, $s) = $walltstr =~ /(\d+) (\d\d):(\d\d):(\d\d)/;
            $m += $d * 24 * 60 + $h * 60 + $s / 60;
            $m = sprintf '%.0f', $m;
            $jobinfo->{$id}{WallTime} = $m if $m;

            # Convert CPU time string to minutes.
            ($d, $h, $m, $s) = $cputstr =~ /(\d+) (\d\d):(\d\d):(\d\d)/;
            $m += $d * 24 * 60 + $h * 60 + $s / 60;
            $m = sprintf '%.0f', $m;
            $jobinfo->{$id}{CpuTime} = $m if $m;
        }

        # Execution host.
        $jobinfo->{$id}{exec_host} = $exechost;
    }
}

my %condor_runcache;

#
# Takes one argument, which is a path to an executable (relative to
# CONDOR_LOCATION) that is appended to CONDOR_LOCATION, plus optional
# arguments.  The next time this function is called with exactly the
# same argument, the return value is fetched from %condor_runcache.
#
# Returns a list of three values:
#
# [0] String containing stdout.
# [1] String containing stderr.
# [2] Program exit code ($?) that was returned to the shell.
#
sub condor_run {
    my $condorloc = condor_location();
    unless ($ENV{CONDOR_CONFIG} && -e $ENV{CONDOR_CONFIG}) {
        $ENV{CONDOR_CONFIG} = condor_config();
    }
    my $program = "$condorloc/$_[0]";
    return @{$condor_runcache{$program}} if $condor_runcache{$program};
    my $stderr_file = "/tmp/condor_run.$$";
#   &infosys_shared::write_log("condor_run: $program");
    my $stdout = `$program 2>$stderr_file`;
    my $ret = $? >> 8;
    local *ERROR;
    open ERROR, "<$stderr_file"
      or return @{$condor_runcache{$program} = [$stdout, '', $ret]};
    local $/;
    my $stderr = <ERROR>;
    close ERROR;
    unlink $stderr_file;
    return @{$condor_runcache{$program} = [$stdout, $stderr, $ret]};
}

#
# Takes two arguments, the Condor job id and the 'controldir' attribute from
# arc.conf.  This function searches controldir for the grami file that
# belongs to the given Condor job, and extracts the location of the Condor job
# from it.  This log is parsed to see if the job has been suspended.  (condor_q
# reports 'R' for running even when the job is suspended, so we need to parse
# the log to be sure that 'R' actually means running.)
#
# Returns true if the job is suspended, and false if it's running.
#
{
    my $initialized_condor_log_db = 0;
    my %condor_log_db;

    sub condor_job_suspended {
        my ($localid, $controldir) = @_;

        # The first time condor_job_suspended() is called, the log database
        # must be initialized.
        if (!$initialized_condor_log_db) {
            $initialized_condor_log_db = 1;
            my @out = `egrep -H '^(joboption_jobid|condor_log)=' \\
                         $controldir/job.*.grami`;
            my $i = 0;
            while ($i + 1 < @out) {
                my $joboptline = $out[$i];  # joboption_jobid=...
                my $logline = $out[$i + 1]; # condor_log=...
                my ($grami, $id) =
                  $joboptline =~ /^(.+\.grami):joboption_jobid=(.*)/;
                if (!$grami || $out[$i + 1] !~ /^\Q$grami\E:/) {
                    # Grami didn't have both joboption_jobid and condor_log;
                    # Should not happen, but you never know!
                    $i++;
                    next;
                }
                my ($log) = $logline =~ /^\Q$grami\E:condor_log=(.*)/;
                $condor_log_db{$id} = $log;                
                $i += 2;
            }
        }

        my $logfile = $condor_log_db{$localid};
        return 0 if !$logfile;
        local *LOGFILE;
        open LOGFILE, "<$logfile" or return 0;
        my $suspended = 0;
        while (my $line = <LOGFILE>) {
            $suspended = 1 if $line =~ /Job was suspended\.$/;
            $suspended = 0 if $line =~ /Job was unsuspended\.$/;
        }
        close LOGFILE;
        return $suspended;
    }
}

{
    my ($progdir) = $0 =~ m#(.*)/#;

    # Cached location so that subsequent calls are free.
    my $location;

    sub condor_location {
        return $location if defined $location;

        $location = $main::config{condor_location};
        return $location if defined $location && -x "$location/bin/condor_submit";

        my $exe;
        # Search for condor_submit in PATH.
        -x ($exe = "$_/condor_submit") and last for split /:/, $ENV{PATH};
        ($location) = $exe =~ m{(.*)/bin/condor_submit$} if -x $exe;
        return $location if $location;

        # Search for CONDOR_LOCATION in /etc/sysconfig/condor.
        if (-f '/etc/sysconfig/condor') {
            $location = `. /etc/sysconfig/condor; echo -n \$CONDOR_LOCATION`;
            return $location if -x "$location/bin/condor_submit";
        }

        # Use condor_master_location, if installed.
        if (-x "$progdir/condor_master_location") {
            ($location) = `$progdir/condor_master_location` =~ m{(.*)/sbin$};
            return $location if -x "$location/bin/condor_submit";
        }

        return $location = '';
    }

    my $config;

    sub condor_config {
        return $config if defined $config;

        $config = $main::config{condor_config};
        return $config if defined $config && -r $config;

        $config = condor_location() . "/etc/condor_config";
        return $config if -r $config;

        $config = $ENV{CONDOR_LOCATION} . "/etc/condor_config";
        return $config if -r $config;

        return $config = '';
    }

}

1;
