/*
  Download files specified in job.ID.input and check if user uploaded files.
  result: 0 - ok, 1 - unrecoverable error, 2 - potentially recoverable.
  arguments: job_id control_directory session_directory
  optional arguments: -t threads , -u user
*/
#include "std.h"

#include <string>
#include <list>

#include "misc/proxy.h"
#include "files/info_types.h"
#include "files/info_files.h"
#include "jobs/job.h"
#include "jobs/users.h"
#include "misc/delete.h"
#include "misc/log_time.h"
#include "misc/stringtoint.h"
#include "datamove/datacache.h"
#include "datamove/datamovepar.h"
#include "config/config_map.h"
#include "config/environment.h"

/* check for user uploaded files every 60 seconds */
#define CHECK_PERIOD 60
/* maximum number of retries (for every source/destination) */
#define MAX_RETRIES 5
/* maximum number simultaneous downloads */
#define MAX_DOWNLOADS 5
/* maximum time for user files to upload (per file) */
#define MAX_USER_TIME 600


// globus_mutex_t list_lock;
// globus_mutex_t download_lock;
// globus_cond_t  download_cond;
JobDescription desc;
JobUser user;
char* id;
int download_done=0;
FileData::iterator next_di;
bool leaving = false;
char* cache_dir;
char* cache_data_dir;
char* cache_link_dir;
uid_t cache_uid;
gid_t cache_gid;
std::string failure_reason("");

/* if != 0 - change owner of downloaded files to this user */
uid_t file_owner = 0;
uid_t file_group = 0;

bool is_checksum(std::string str) {
  /* check if have : */
  std::string::size_type n;
  const char *s = str.c_str();
  if((n=str.find('.')) == std::string::npos) return false;
  if(str.find('.',n+1) != std::string::npos) return false;
  for(std::string::size_type i=0;i<n;i++) { if(!isdigit(s[i])) return false; };
  for(std::string::size_type i=n+1;s[i];i++) { if(!isdigit(s[i])) return false; };
  return true;
}

int clean_files(std::list<FileData> &job_files,char* session_dir) {
  std::string session(session_dir);
  if(delete_all_files(session,job_files,true) != 2) return 0;
  return 1;
}

int expand_files(std::list<FileData> &job_files,char* session_dir) {
  for(FileData::iterator i = job_files.begin();i!=job_files.end();) {
    std::string url = i->lfn;
    // Only ftp and gsiftp can be expanded to directories so far
    if(strncasecmp(url.c_str(),"ftp://",6) && 
       strncasecmp(url.c_str(),"gsiftp://",9)) { ++i; continue; };
    // user must ask explicitly
    if(url[url.length()-1] != '/') { ++i; continue; };
    std::string path(session_dir); path+="/"; path+=i->pfn;
    int l = strlen(session_dir) + 1;
    DIR *dir = opendir(path.c_str());
    if(dir) {
      struct dirent file_;
      struct dirent *file;
      for(;;) {
        readdir_r(dir,&file_,&file);
        if(file == NULL) break;
        if(!strcmp(file->d_name,".")) continue;
        if(!strcmp(file->d_name,"..")) continue;
        std::string path_ = path; path_+="/"; path+=file->d_name;
        struct stat st;
        if(lstat(path_.c_str(),&st) != 0) continue; // do not follow symlinks
        if(S_ISREG(st.st_mode)) {
          std::string lfn = url+file->d_name;
          job_files.push_back(FileData(path_.c_str()+l,lfn.c_str()));
        } else if(S_ISDIR(st.st_mode)) {
          std::string lfn = url+file->d_name+"/"; // cause recursive search
          job_files.push_back(FileData(path_.c_str()+l,lfn.c_str()));
        };
      };
      i=job_files.erase(i);
    } else {
      ++i;
    }; 
  };
}

int main(int argc,char** argv) {
  int res=0;
  time_t start_time=time(NULL);
  time_t download_timeout = 0;
  int n_threads = 1;
  int n_files = MAX_DOWNLOADS;
  bool secure = true;
  bool passive = false;
  bool userfiles_only = false;
  LogTime::Active(true);
  LogTime::Level(DEBUG);
  cache_dir=NULL;
  cache_data_dir=NULL;
  cache_link_dir=NULL;
  cache_uid=0;
  cache_gid=0;
  unsigned long long int min_speed = 0;
  time_t min_speed_time = 300;
  unsigned long long int min_average_speed = 0;
  time_t max_inactivity_time = 300;
  // process optional arguments
  for(;;) {
    opterr=0;
    int optc=getopt(argc,argv,"+hclpZn:t:u:U:s:S:a:i:d:");
    if(optc == -1) break;
    switch(optc) {
      case 'h': {
        olog<<"uploader [-t threads]  [-n files] [-c] [-u username] [-U userid] job_id control_directory session_directory"<<std::endl; 
        exit(1);
      }; break;
      case 'c': {
        secure=false;
      }; break;
      case 'l': {
        userfiles_only=true;
      }; break;
      case 'p': {
        passive=true;
      }; break;
      case 'Z': {
        central_configuration=true;
      }; break;
      case 'd': {
        LogTime::Level(NotifyLevel(atoi(optarg)));
      }; break;
      case 't': {
        n_threads=atoi(optarg);
        if(n_threads < 1) {
          olog<<"Wrong number of threads"<<std::endl; exit(1);
        };
      }; break;
      case 'n': {
        n_files=atoi(optarg);
        if(n_files < 1) {
          olog<<"Wrong number of files"<<std::endl; exit(1);
        };
      }; break;
      case 'U': {
        unsigned int tuid;
        if(!stringtoint(std::string(optarg),tuid)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        struct passwd pw_;
        struct passwd *pw;
        char buf[BUFSIZ];
        getpwuid_r(tuid,&pw_,buf,BUFSIZ,&pw);
        if(pw == NULL) {
          olog<<"Wrong user name"<<std::endl; exit(1);
        };
        file_owner=pw->pw_uid;
        file_group=pw->pw_gid;
        if((getuid() != 0) && (getuid() != file_owner)) {
          olog<<"Specified user can't be handled"<<std::endl; exit(1);
        };
      }; break;
      case 'u': {
        struct passwd pw_;
        struct passwd *pw;
        char buf[BUFSIZ];
        getpwnam_r(optarg,&pw_,buf,BUFSIZ,&pw);
        if(pw == NULL) {
          olog<<"Wrong user name"<<std::endl; exit(1);
        };
        file_owner=pw->pw_uid;
        file_group=pw->pw_gid;
        if((getuid() != 0) && (getuid() != file_owner)) {
          olog<<"Specified user can't be handled"<<std::endl; exit(1);
        };
      }; break;
      case 's': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_speed=tmpi;
      }; break;
      case 'S': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_speed_time=tmpi;
      }; break;
      case 'a': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_average_speed=tmpi;
      }; break;
      case 'i': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        max_inactivity_time=tmpi;
      }; break;
      case '?': {
        olog<<"Unsupported option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      case ':': {
        olog<<"Missing parameter for option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      default: {
        olog<<"Undefined processing error"<<std::endl;
        exit(1);
      };
    };
  };
  // process required arguments
  id = argv[optind+0];
  if(!id) { olog << "Missing job id" << std::endl; return 1; };
  char* control_dir = argv[optind+1];
  if(!control_dir) { olog << "Missing control directory" << std::endl; return 1; };
  char* session_dir = argv[optind+2];
  if(!session_dir) { olog << "Missing session directory" << std::endl; return 1; };
  cache_dir = argv[optind+3];
  if(cache_dir) {
    olog<<"Cache directory is "<<cache_dir<<std::endl;
    cache_data_dir=argv[optind+4];
    if(!cache_data_dir) {
      cache_data_dir=cache_dir;
    }
    else {
      olog<<"Cache data directory is "<<cache_data_dir<<std::endl;
      cache_link_dir=argv[optind+5];
      if(cache_link_dir) olog<<"Cache links will go to directory "<<cache_link_dir<<std::endl;
    };
  };
  if(min_speed != 0) { olog<<"Minimal speed: "<<min_speed<<" B/s during "<<min_speed_time<<" s"<<std::endl; };
  if(min_average_speed != 0) { olog<<"Minimal average speed: "<<min_average_speed<<" B/s"<<std::endl; };
  if(max_inactivity_time != 0) { olog<<"Maximal inactivity time: "<<max_inactivity_time<<" s"<<std::endl; }; 

  prepare_proxy();

  if(n_threads > 10) {
    olog<<"Won't use more than 10 threads"<<std::endl;
    n_threads=10;
  };
  UrlMapConfig url_map;
  olog<<"Uploader started"<<std::endl;

  // prepare Job and User descriptions
  desc=JobDescription(id,session_dir);
  uid_t uid;
  gid_t gid;
  if(file_owner != 0) { uid=file_owner; }
  else { uid= getuid(); };
  if(file_group != 0) { gid=file_group; }
  else { gid= getgid(); };
  cache_uid=getuid();
  cache_gid=getgid();
  desc.set_uid(uid,gid);
  user=JobUser(uid);
  user.SetControlDir(control_dir);
  user.SetSessionRoot(session_dir); /* not needed */
  if(cache_dir) {
    if(!cache_link_dir) {
      user.SetCacheDir(cache_dir,cache_data_dir);
    }
    else {
      user.SetCacheDir(cache_dir,cache_data_dir,cache_link_dir);
    };
  };
   // get the list of input files
  DataMovePar mover;
  mover.secure(secure);
  mover.passive(passive);
  if(min_speed != 0)
    mover.set_default_min_speed(min_speed,min_speed_time);
  if(min_average_speed != 0)
    mover.set_default_min_average_speed(min_average_speed);
  if(max_inactivity_time != 0)
    mover.set_default_max_inactivity_time(max_inactivity_time);
  std::list<FileData> job_files;
  bool all_data = false;
  bool transfered = true;
  bool credentials_expired = false;

  if(!job_output_read_file(desc.get_id(),user,job_files)) {
    olog << "WARNING: Can't read list of output files - whole output will be removed" << std::endl;
  };
  // remove bad files
  if(clean_files(job_files,session_dir) != 0) {
    failure_reason+="Internal error in uploader\n";
    olog << "Can't remove junk files" << std::endl; res=1; goto exit;
  };
  expand_files(job_files,session_dir);
  // initialize structures to handle download
  /* TODO: add threads=# to all urls if n_threads!=1 */
  desc.GetLocalDescription(user);

  // Push data transfer pair into mover
  for(next_di=job_files.begin();next_di!=job_files.end();++next_di) {
    if(next_di->lfn.find(":") != std::string::npos) { /* is it lfn ? */
      /* have place and file to download */
      /* define place to store */
      std::string stdlog;
      std::string source;
      JobLocalDescription* local = desc.get_local();
      if(local) stdlog=local->stdlog;
      if(stdlog.length() > 0) stdlog="/"+stdlog+"/";
      if((stdlog.length() > 0) &&
         (strncmp(stdlog.c_str(),next_di->pfn.c_str(),stdlog.length()) == 0)) {
        stdlog=next_di->pfn.c_str()+stdlog.length();
        source=std::string("file://")+control_dir+"/job."+id+"."+stdlog;
      } else {
        source=std::string("file://")+session_dir+next_di->pfn;
      };
      std::string destination = next_di->lfn;
      if(strncasecmp(destination.c_str(),"file://",7) == 0) {
        failure_reason+=std::string("User requested to store output locally ")+destination.c_str()+"\n";
        olog<<"FATAL ERROR: local destination for uploader"<<destination<<std::endl; res=1; goto exit;
      };
      if(!mover.Add(source.c_str(),destination.c_str())) {
        failure_reason+="Internal error in uploader\n";
        olog<<"Can't add data pair to list: "<<std::endl; res=1; goto exit;
      };
    };
  };
  if(!userfiles_only) {
    // Start transfer 
    if(!mover.Transfer(DataCache(),url_map,n_files)) {
      failure_reason+="Internal error in uploader\n";
      odlog(ERROR)<<"FAILED to start transfering files"<<std::endl; res=2; goto exit;
    };
    // Check if all files have been properly uploaded
    for(next_di=job_files.begin();next_di!=job_files.end();) {
      if(next_di->lfn.find(":") != std::string::npos) { /* is it lfn ? */
        // Files should be of same order as when pushed. Error if not.
        if(all_data) {
          failure_reason+="Internal error in uploader\n";
          odlog(DEBUG)<<"Data structure was destroyed."<<std::endl; res=2; goto exit;
        };
        std::string src = std::string("file://") + session_dir + next_di->pfn;
        std::string source;
        std::string destination;
        DataMovePar::result dres;
        if(!mover.Get(source,destination,dres)) { // no more data
          all_data=true; ++next_di; continue;
        };
        if(src != source) {
          failure_reason+="Internal error in uploader\n";
          odlog(ERROR)<<"Data structure was destroyed."<<std::endl; res=2; goto exit;
        };
        if(dres == DataMovePar::success) {  // remove this file from list
          next_di=job_files.erase(next_di);
          odlog(ERROR)<<"Uploaded "<<destination<<std::endl;
        }
        else {
          failure_reason+="Output file: "+next_di->lfn+" - "+
                             DataMove::get_result_string(dres)+"\n";
          if(dres == DataMove::credentials_expired_error) credentials_expired=true;
          transfered=false; ++next_di;
          odlog(ERROR)<<"Failed to upload "<<destination<<std::endl;
        };
      }
      else { ++next_di; };
    };
    if(!transfered) {
      odlog(INFO)<<"Some uploads failed"<<std::endl; res=2;
      if(credentials_expired) res=3;
    };
    /* all files left should be kept */
    for(next_di=job_files.begin();next_di!=job_files.end();) {
      next_di->lfn=""; ++next_di;
    };
    if(!job_output_write_file(desc,user,job_files)) {
      olog << "WARNING: Failed writing changed output file." << std::endl;
    };
  };
exit:
  if(cache_dir) cache_release_url(cache_dir,cache_data_dir,cache_uid,cache_gid,id,true);
  olog << "Leaving uploader ("<<res<<")"<<std::endl;
  // clean uploaded files here 
  clean_files(job_files,session_dir);
  // release cache just in case
  remove_proxy();
  if(res != 0) {
    job_failed_mark_add(desc,user,failure_reason);
  };
  return res;
}
