#include "../std.h"
#include <stdlib.h>
#include <string>
#include <list>
extern "C" {
#include <globus_common.h>
#include <globus_error.h>
#include <globus_ftp_control.h>
#include <globus_url.h>
#include <globus_object.h>
}
#include "../misc/stringtoint.h"
#include "../misc/log_time.h"
#include "lister.h"

static char* default_ftp_user    = "ftp";
static char* default_gsiftp_user = ":globus-mapping:";
static char* default_ftp_pass    = "user@";
static char* default_gsiftp_pass = "user@";

void dos_to_unix(char* s) {
  if(!s) return;
  int l = strlen(s);
  for(;l;) { l--; if((s[l]=='\r') || (s[l]=='\n')) s[l]=0; };
}

bool ListerFile::SetAttributes(const char* facts) {
  const char* name;
  const char* value;
  const char* p = facts;

  for(;*p;) {
    name=p; value=p;
    if(*p == ' ') break; // end of facts
    if(*p == ';') { p++; continue; };
    for(;*p;p++) {
      if(*p == ' ') break;
      if(*p == ';') break;
      if(*p == '=') value=p;
    };
    if(name == value) continue; // skip empty names
    value++;
    if(value == p) continue; // skip empty values
    if(((value-name-1) == 4) && (strncasecmp(name,"type",4)==0)) {
      if(((p-value) == 3) && (strncasecmp(value,"dir",3)==0)) {
        type=file_type_dir;
      } else if(((p-value) == 4) && (strncasecmp(value,"file",4)==0)) {
        type=file_type_file;
      } else {
        type=file_type_unknown;
      };
    } else if(((value-name-1) == 4) && (strncasecmp(name,"size",4)==0)) {
      std::string tmp_s(value,(int)(p-value));
      size_available=stringtoint(tmp_s,size);
    } else if(((value-name-1) == 6) && (strncasecmp(name,"modify",6)==0)) {
      std::string tmp_s(value,(int)(p-value));
      created_available=stringtoint(tmp_s,created);
    };
  };
  return true;
}

const char* ListerFile::GetLastName(void) const {
  std::string::size_type n = name.rfind('/');
  if(n == std::string::npos) return name.c_str(); 
  return name.c_str()+n+1;
}

Lister::callback_status_t Lister::wait_for_callback(void) {
    callback_status_t res;
    globus_mutex_lock(&mutex);
    while(callback_status==CALLBACK_NOTREADY) {
      globus_cond_wait(&cond,&mutex);
    };
    res=callback_status;
    callback_status=CALLBACK_NOTREADY;
    globus_mutex_unlock(&mutex);
    return res;
} 

Lister::callback_status_t Lister::wait_for_data_callback(void) {
    callback_status_t res;
    globus_mutex_lock(&mutex);
    while(data_callback_status==CALLBACK_NOTREADY) {
      globus_cond_wait(&cond,&mutex);
    };
    res=data_callback_status;
    data_callback_status=CALLBACK_NOTREADY;
    globus_mutex_unlock(&mutex);
    return res;
} 

void Lister::resp_destroy(void) {
    globus_mutex_lock(&mutex);
    if(resp_n>0) {
      globus_ftp_control_response_destroy(resp+(resp_n-1));
      resp_n--;
    };
    globus_mutex_unlock(&mutex);
}

void Lister::resp_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response) {
    Lister* it = (Lister*)arg;
    globus_mutex_lock(&(it->mutex));
    if(error != GLOBUS_SUCCESS) {
      it->callback_status=CALLBACK_ERROR;
      char* tmp = globus_object_printable_to_string(error);
      odlog(INFO)<<"Failure: "<<tmp<<std::endl; free(tmp);
      if(response) odlog(INFO)<<"Server said: "<<response->response_buffer<<std::endl;
    }
    else {
      if(it->resp_n < LISTER_MAX_RESPONSES) {
        memmove((it->resp)+1,it->resp,
                sizeof(globus_ftp_control_response_t)*(it->resp_n));
        if(response->response_buffer) {
          globus_ftp_control_response_copy(response,it->resp);
        } else {  // invalid reply causes *_copy to segfault
          it->resp->response_buffer=(globus_byte_t*)strdup("000 ");
          it->resp->response_buffer_size=5;
          it->resp->response_length=4;
          it->resp->code=0;
          it->resp->response_class=GLOBUS_FTP_UNKNOWN_REPLY;
        };
        (it->resp_n)++;
      };
      it->callback_status=CALLBACK_DONE;
      dos_to_unix((char*)(it->resp->response_buffer));
      odlog(DEBUG)<<"Response: "<<it->resp->response_buffer<<std::endl;
    };
    globus_cond_signal(&(it->cond));
    globus_mutex_unlock(&(it->mutex));
}

void Lister::list_read_callback(void *arg,globus_ftp_control_handle_t *hctrl,globus_object_t *error,globus_byte_t *buffer,globus_size_t length,globus_off_t offset,globus_bool_t eof) {
    Lister *it = (Lister*)arg;
    length+=it->list_shift;
    if(error != GLOBUS_SUCCESS) {
      /* no such file or connection error - assume no such file */
      odlog(INFO)<<"Error getting list of files (in list)"<<std::endl;
      char* tmp = globus_object_printable_to_string(error);
      odlog(INFO)<<tmp<<std::endl; free(tmp);
      odlog(INFO)<<"Assuming - file not found\n";
      globus_mutex_lock(&(it->mutex));
      it->data_callback_status=CALLBACK_ERROR;
      globus_cond_signal(&(it->cond));
      globus_mutex_unlock(&(it->mutex));
      return;
    };
    /* parse names and add to list */
    /* suppose we are receiving ordered blocks of data (no multiple streams) */
    char* name;
    (it->readbuf)[length]=0;
    name=it->readbuf;
    it->list_shift=0;
    for(;;) {
      if((*name) == 0) break;
      int nlen;
      nlen=strcspn(name,"\n\r");
      name[nlen]=0;
      odlog(DEBUG)<<"list record: "<<name<<std::endl;
      if(nlen == length) {
        if(!eof) {
          memmove(it->readbuf,name,nlen);
          it->list_shift=nlen;
          break;
        };
      };
      if(nlen == 0) { // skip empty std::string 
        if(length == 0) break;
        name++; length--; continue;
      };
      char* attrs = name;
      if(it->facts) for(;*name;) {
        nlen--; length--;
        if(*name==' ') { name++; break; };
        name++;
      };
      std::list<ListerFile>::iterator i;
      if(name[0] == '/') {
        i=it->fnames.insert(it->fnames.end(),ListerFile(name));
      } else {
        std::string name_ = it->path?it->path:"/";
        name_+="/"; name_+=name;
        i=it->fnames.insert(it->fnames.end(),ListerFile(name_.c_str()));
      };
      if(it->facts) i->SetAttributes(attrs);
      if(nlen==length) break;
      name+=(nlen+1);
      length-=(nlen+1);
      if(((*name) == '\r') || ((*name) == '\n')) {
        name++; length--;
      };
    };
    if(!eof) {
      if(globus_ftp_control_data_read(it->handle,
             (globus_byte_t*)((it->readbuf)+(it->list_shift)),
             sizeof(it->readbuf)-(it->list_shift)-1,
             &list_read_callback,arg) != GLOBUS_SUCCESS) {
        odlog(INFO)<<"Faled reading list of files\n";
        globus_mutex_lock(&(it->mutex));
        it->data_callback_status=CALLBACK_ERROR;
        globus_cond_signal(&(it->cond));
        globus_mutex_unlock(&(it->mutex));
      };
      return;
    };
    globus_mutex_lock(&(it->mutex));
    it->data_callback_status=CALLBACK_DONE;
    globus_cond_signal(&(it->cond));
    globus_mutex_unlock(&(it->mutex));
    return;
}

void Lister::list_conn_callback(void* arg,globus_ftp_control_handle_t* hctrl,
     unsigned int stripe_ndx,globus_bool_t reused,globus_object_t* error) {
    /* if(!callback_active) return; */
    Lister* it = (Lister*)arg;
    if(error != GLOBUS_SUCCESS) {
      char* tmp = globus_object_printable_to_string(error);
      odlog(INFO)<<"Failure: "<<tmp<<std::endl; free(tmp);
      globus_mutex_lock(&(it->mutex));
      it->data_callback_status=CALLBACK_ERROR;
      globus_cond_signal(&(it->cond));
      globus_mutex_unlock(&(it->mutex));
      return;
    };
    it->list_shift=0; it->fnames.clear();
    if(globus_ftp_control_data_read(hctrl,
            (globus_byte_t*)(it->readbuf),sizeof(it->readbuf)-1,
            &list_read_callback,arg) != GLOBUS_SUCCESS) {
      odlog(INFO)<<"Failed reading data"<<std::endl;
      globus_mutex_lock(&(it->mutex));
      it->data_callback_status=CALLBACK_ERROR;
      globus_cond_signal(&(it->cond));
      globus_mutex_unlock(&(it->mutex));
      return;
    };
}

globus_ftp_control_response_class_t Lister::send_command(const char* command,const char* arg,bool wait_for_response,char** sresp,char delim) {
    char* cmd = NULL;
    if(sresp) { (*sresp) = NULL; };
    if(command) { /* if no command - waiting for second reply */
      globus_mutex_lock(&mutex); 
      for(int i=0;i<resp_n;i++) globus_ftp_control_response_destroy(resp+i);
      resp_n=0;
      callback_status=CALLBACK_NOTREADY;
      globus_mutex_unlock(&mutex);
      if(arg) { cmd = (char*)malloc(strlen(arg)+strlen(command)+4); }
      else { cmd = (char*)malloc(strlen(command)+3); };
      if(cmd==NULL) {
        odlog(ERROR)<<"Memory allocation error"<<std::endl;
        return GLOBUS_FTP_UNKNOWN_REPLY;
      };
      strcpy(cmd,command); if(arg) { strcat(cmd," "); strcat(cmd,arg); };
      odlog(DEBUG)<<"Command: "<<cmd<<std::endl;
      strcat(cmd,"\r\n");
      if(globus_ftp_control_send_command(handle,cmd,resp_callback,this)
                                      != GLOBUS_SUCCESS) {
        odlog(DEBUG)<<command<<" failed"<<std::endl;
        if(cmd) free(cmd); return GLOBUS_FTP_UNKNOWN_REPLY;
      };
      odlog(VERBOSE)<<"Command is beeing sent"<<std::endl;
    };
    if(wait_for_response) {
      globus_mutex_lock(&mutex);
      while((callback_status==CALLBACK_NOTREADY) && (resp_n == 0)) {
        odlog(VERBOSE)<<"Waiting for response"<<std::endl;
        globus_cond_wait(&cond,&mutex);
      };
      free(cmd);
      if(callback_status != CALLBACK_DONE) {
        odlog(VERBOSE)<<"Callback got failure"<<std::endl;
        callback_status=CALLBACK_NOTREADY;
        if(resp_n>0) {
          globus_ftp_control_response_destroy(resp+(resp_n-1));
          resp_n--;
        };
        globus_mutex_unlock(&mutex);
        return GLOBUS_FTP_UNKNOWN_REPLY;
      };
      if((sresp) && (resp_n>0)) {
        if(delim == 0) {
          (*sresp)=(char*)malloc(resp[resp_n-1].response_length);
          if((*sresp) != NULL) {
            memcpy(*sresp,(char*)(resp[resp_n-1].response_buffer+4),
                   resp[resp_n-1].response_length-4);
            (*sresp)[resp[resp_n-1].response_length-4]=0;
            odlog(DEBUG)<<"Response: "<<*sresp<<std::endl;
          }
          else { odlog(ERROR)<<"Memory allocation error"<<std::endl; };
        } 
        else {
          /* look for pair of enclosing characters */
          odlog(DEBUG)<<"Response: "<<resp[resp_n-1].response_buffer<<std::endl;
          char* s_start = (char*)(resp[resp_n-1].response_buffer+4);
          char* s_end = NULL;
          int l = 0;
          s_start=strchr(s_start,delim);
          if(s_start) {
            s_start++; if(delim=='(') { delim=')'; }
            else if(delim=='{') { delim='}'; }
            else if(delim=='[') { delim=']'; };
            s_end=strchr(s_start,delim); if(s_end) l=s_end-s_start;
          };
          if(l>0) {
            (*sresp)=(char*)malloc(l+1);
            if((*sresp) != NULL) {
              memcpy(*sresp,s_start,l); (*sresp)[l]=0;
              odlog(DEBUG)<<"Response: "<<*sresp<<std::endl;
            };
          };
        };
      };
      globus_ftp_control_response_class_t resp_class = GLOBUS_FTP_UNKNOWN_REPLY;
      if(resp_n>0) {
        resp_class = resp[resp_n-1].response_class;
        globus_ftp_control_response_destroy(resp+(resp_n-1));
        resp_n--;
      };
      if(resp_n == 0) callback_status=CALLBACK_NOTREADY;
      globus_mutex_unlock(&mutex);
      return resp_class;
    }
    else { return GLOBUS_FTP_POSITIVE_COMPLETION_REPLY; };
    /* !!!!!!! Memory LOST - cmd !!!!!!!! */
}

Lister::Lister(void) {
    callback_status=CALLBACK_NOTREADY;
    inited=false;
    connected=false;
    port=0;
    host=NULL;
    username=NULL;
    userpass=NULL;
    path=NULL;
    resp_n=0;
    handle=NULL;
    if(globus_cond_init(&cond,GLOBUS_NULL) != GLOBUS_SUCCESS) {
      odlog(ERROR)<<"Failed initing condition"<<std::endl;
      return;
    };
    if(globus_mutex_init(&mutex,GLOBUS_NULL) != GLOBUS_SUCCESS) {
      odlog(ERROR)<<"Failed initing mutex"<<std::endl;
      globus_cond_destroy(&cond);
      return;
    };
    handle=(globus_ftp_control_handle_t*)malloc(sizeof(globus_ftp_control_handle_t));
    if(handle == NULL) {
      odlog(ERROR)<<"Failed allocating memory for handle"<<std::endl;
      globus_mutex_destroy(&mutex);
      globus_cond_destroy(&cond);
    };
    if(globus_ftp_control_handle_init(handle) != GLOBUS_SUCCESS) {
      odlog(ERROR)<<"Failed initing handle"<<std::endl;
      globus_mutex_destroy(&mutex);
      globus_cond_destroy(&cond);
      free(handle); handle=NULL;
      return;
    };
    inited=true;
}

int Lister::close_connection(void) {
    if(!connected) return 0;
    odlog(DEBUG)<<"Closing connection"<<std::endl;
    if(globus_ftp_control_quit(handle,resp_callback,this) != GLOBUS_SUCCESS) {
      if(globus_ftp_control_force_close(handle,resp_callback,this) != GLOBUS_SUCCESS) {
        odlog(INFO)<<"Failed to close connection 1"<<std::endl;
        return -1;
      };
    };
    if(wait_for_callback() != CALLBACK_DONE) {
      if(globus_ftp_control_force_close(handle,resp_callback,this) != GLOBUS_SUCCESS) {
        odlog(INFO)<<"Failed to close connection 2"<<std::endl;
        return -1;
      };
      if(wait_for_callback() != CALLBACK_DONE) {
        odlog(INFO)<<"Failed to close connection 3"<<std::endl;
        return -1;
      };
    };
    connected=false;
    odlog(DEBUG)<<"Closed successfuly"<<std::endl;
    return 0;
}

Lister::~Lister(void) {
    close_connection();
    if(host) free(host);
    if(username) free(username);
    if(userpass) free(userpass);
    if(path) free(path);
    if(inited) {
      if(globus_ftp_control_handle_destroy(handle) == GLOBUS_SUCCESS) {
        free(handle); handle=NULL;
      } else {
        odlog(DEBUG)<<"Memory leak (globus_ftp_control_handle_t)"<<std::endl;
        handle=NULL;
      };
      globus_mutex_destroy(&mutex);
      globus_cond_destroy(&cond);
    };
}

int Lister::setup_pasv(globus_ftp_control_host_port_t &pasv_addr) {
    char* sresp;
    globus_result_t res_tmp;
    if(send_command("PASV",NULL,true,&sresp,'(') != GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      odlog(INFO)<<"PASV failed: ";
      if(sresp) { odlog_(INFO)<<sresp<<std::endl; free(sresp); }
      else { odlog_(INFO)<<std::endl; };
      return -1;
    };
    pasv_addr.port=0;
    if(sresp) {
      int port_low,port_high;
      if(sscanf(sresp,"%i,%i,%i,%i,%i,%i",
           &(pasv_addr.host[0]), &(pasv_addr.host[1]),
           &(pasv_addr.host[2]), &(pasv_addr.host[3]),
           &port_high, &port_low) == 6) {
        pasv_addr.port=((port_high & 0x000FF) << 8) | (port_low & 0x000FF);
      };
    };
    if(pasv_addr.port == 0) {
      odlog(INFO)<<"Can't parse host and port in response to PASV"<<std::endl;
      if(sresp) free(sresp); return -1;
    };
    free(sresp);
    odlog(DEBUG)<<"Data channel: "<<pasv_addr.host[0]<<"."<<pasv_addr.host[1]
                         <<"."<<pasv_addr.host[2]<<"."<<pasv_addr.host[3]
                         <<" "<<pasv_addr.port<<std::endl;
    if((res_tmp=globus_ftp_control_local_port(handle,&pasv_addr)) != GLOBUS_SUCCESS) {
      odlog(INFO)<<"Obtained host and address are not acceptable"<<std::endl;
      char* tmp = globus_object_printable_to_string(globus_error_get(res_tmp));
      odlog(INFO)<<tmp<<std::endl; free(tmp);
      return -1;
    };
    return 0;
}

int Lister::retrieve_dir(const std::string &url) {
    globus_result_t res_tmp;
    /* get listing */
    fnames.clear();
    globus_url_t url_;
    globus_ftp_control_auth_info_t auth;

    if(globus_url_parse(url.c_str(),&url_) != GLOBUS_URL_SUCCESS) {
      odlog(ERROR)<<"Failed parsing url "<<url<<std::endl;
      return -1;
    };
    if((url_.scheme_type != GLOBUS_URL_SCHEME_FTP) && 
       (url_.scheme_type != GLOBUS_URL_SCHEME_GSIFTP)) { 
      odlog(ERROR)<<"Unsupported protocol in url "<<url<<std::endl;
      globus_url_destroy(&url_);
      return -1;
    };
    if(url_.port == 0) {
      if(url_.scheme_type == GLOBUS_URL_SCHEME_FTP) { url_.port=21; }
      else { url_.port=2811; };
    };

    bool reconnect = true;

    if(connected) {
      if((!strcmp(host,url_.host)) &&
         (port == url_.port) &&
         (scheme == url_.scheme_type) &&
         (
          ((username==NULL) && (url_.user==NULL)) || 
          (!strcmp(username,url_.user))
         ) &&
         (
          ((userpass==NULL) && (url_.password==NULL)) ||
          (!strcmp(userpass,url_.password))
         )
        ) {
        /* same server - check if connection alive */
        odlog(DEBUG)<<"Reusing connection"<<std::endl;
        if(send_command("NOOP",NULL,true,NULL)
                         == GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
          reconnect = false;
        };
      };
    };

    if(path) { free(path); path=NULL; };
    if(url_.url_path) {
      path=strdup(url_.url_path);
      int l = path?strlen(path):0;
      if((l != 0) && (url_.url_path[l-1] == '/')) path[l-1]=0;
    }; 
    if(reconnect) { 
      connected=false;
      if(host) { free(host); host=NULL; };
      if(username) { free(username); username=NULL; };
      if(userpass) { free(userpass); userpass=NULL; };
      port=url_.port; scheme=url_.scheme_type;
      host=strdup(url_.host);
      if(url_.user) { username=strdup(url_.user); }; 
      if(url_.password) { userpass=strdup(url_.password); };
      globus_url_destroy(&url_);
      /*
        !!!!!!!!!!!!!!!!!!!!!!!!!!!
        disconnect here ???????????
      */
      if((res_tmp=globus_ftp_control_connect(handle,host,
               port,&resp_callback,this)) != GLOBUS_SUCCESS) {
        odlog(ERROR)<<"Failed connecting to server "<<host<<":"<<port<<std::endl;
        char* tmp=globus_object_printable_to_string(globus_error_get(res_tmp));
        odlog(ERROR)<<tmp<<std::endl; free(tmp);
        return -1;
      };
      if(wait_for_callback() != CALLBACK_DONE) {
        odlog(ERROR)<<"Failed to connect to server "<<host<<":"<<port<<std::endl;
        resp_destroy();
        return -1;
      };
      resp_destroy();
      char* username_ = username;
      char* userpass_ = userpass;
      globus_bool_t use_auth;
      if(scheme == GLOBUS_URL_SCHEME_GSIFTP) {
        if(!username) username_ = default_gsiftp_user;
        if(!userpass) userpass_ = default_gsiftp_pass;
        if(globus_ftp_control_auth_info_init(&auth,GSS_C_NO_CREDENTIAL,
           GLOBUS_TRUE,username_,userpass_,GLOBUS_NULL,GLOBUS_NULL) !=
                                               GLOBUS_SUCCESS) {
          odlog(ERROR)<<"Bad authentication information"<<std::endl;
          return -1;
        };
        use_auth=GLOBUS_TRUE;
      }
      else {
        if(!username) username_ = default_ftp_user;
        if(!userpass) userpass_ = default_ftp_pass;
        if(globus_ftp_control_auth_info_init(&auth,GSS_C_NO_CREDENTIAL,
           GLOBUS_FALSE,username_,userpass_,GLOBUS_NULL,GLOBUS_NULL) !=
                                               GLOBUS_SUCCESS) {
          odlog(ERROR)<<"Bad authentication information"<<std::endl;
          return -1;
        };
        use_auth=GLOBUS_FALSE;
      };
      if(globus_ftp_control_authenticate(handle,&auth,use_auth,
                            resp_callback,this) != GLOBUS_SUCCESS) {
        odlog(ERROR)<<"Failed authenticating"<<std::endl;
        return -1;
      };
      if(wait_for_callback() != CALLBACK_DONE) {
        odlog(ERROR)<<"Failed authenticating"<<std::endl;
        resp_destroy();
        return -1;
      };
      resp_destroy();
      connected=true;
    } else {
      globus_url_destroy(&url_);
    };
    globus_ftp_control_response_class_t cmd_resp;
    char* sresp;
    if(url_.scheme_type == GLOBUS_URL_SCHEME_GSIFTP) {
      cmd_resp = send_command("DCAU","N",true,&sresp,'"');
      if((cmd_resp != GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) &&
         (cmd_resp != GLOBUS_FTP_PERMANENT_NEGATIVE_COMPLETION_REPLY)) {
        odlog(INFO)<<"DCAU failed: ";
        if(sresp) {
          odlog_(INFO)<<sresp<<std::endl; free(sresp);
        } else { odlog_(INFO)<<std::endl; };
        return -1;
      };
      free(sresp);
    };
    globus_ftp_control_dcau_t dcau;
    dcau.mode=GLOBUS_FTP_CONTROL_DCAU_NONE;
    globus_ftp_control_local_dcau(handle,&dcau,GSS_C_NO_CREDENTIAL);
    globus_ftp_control_host_port_t pasv_addr;
    /* try MLSD */
    facts=true;
    if(setup_pasv(pasv_addr) != 0) return -1;
    /* it looks like _pasv is not enough for connection - start reading
       immediately */
    data_callback_status=(callback_status_t)CALLBACK_NOTREADY;
    if(globus_ftp_control_data_connect_read(handle,&list_conn_callback,this) !=
                                        GLOBUS_SUCCESS) {
      odlog(INFO)<<"Failed to open data channel"<<std::endl;
      return -1;
    };
    cmd_resp=send_command("MLSD",path,true,&sresp);
    if(cmd_resp == GLOBUS_FTP_PERMANENT_NEGATIVE_COMPLETION_REPLY) {
      odlog(INFO)<<"MLSD is not supported - trying NLST"<<std::endl;
      /* run NLST */
      facts=false;
      cmd_resp=send_command("NLST",path,true,&sresp);
    };
    if(cmd_resp == GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      /* completion is not expected here */
      odlog(INFO)<<"Immediate completion: "<<sresp<<std::endl;
      if(sresp) free(sresp);
      return -1;
    }
    else if(cmd_resp == GLOBUS_FTP_POSITIVE_PRELIMINARY_REPLY) {  // ok
    }
    else if(cmd_resp == GLOBUS_FTP_POSITIVE_INTERMEDIATE_REPLY) { // ok
    }
    else {
      if(sresp) {
        odlog(INFO)<<"NLST/MLSD failed: "<<sresp<<std::endl;
        free(sresp);
      } else { odlog(INFO)<<"NLST/MLSD failed"<<std::endl; };
      return -1;
    };
    free(sresp);
    /* start transfer */
    for(;;) {
/* waiting for response received */
      cmd_resp=send_command(NULL,NULL,true,&sresp);
      if(cmd_resp == GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) { break; }
      if((cmd_resp != GLOBUS_FTP_POSITIVE_PRELIMINARY_REPLY) &&
         (cmd_resp != GLOBUS_FTP_POSITIVE_INTERMEDIATE_REPLY)) {
        if(sresp) {
          odlog(INFO)<<"Data transfer aborted: "<<sresp<<std::endl;
          free(sresp);
        } else { odlog(INFO)<<"Data transfer aborted"<<std::endl; };
        /*
          !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
          Destroy data connections here ?????????
        */
        return -1;
      };
      if(sresp) free(sresp);
    };
    if(sresp) free(sresp);
/* waiting for data ended */
    if(wait_for_data_callback() != CALLBACK_DONE) {
      odlog(INFO)<<"Failed to transfer data"<<std::endl;
      return -1;
    };
    /* success */
    return 0;
}

