/*
  WARNING: This module uses global variables. It should be no more than 1
  function running.
*/
#include "../std.h"

#include <string>

#include <globus_common.h>
#include <globus_error.h>
#include <globus_ftp_client.h>
#include <globus_ftp_control.h>
#include <globus_object.h>

#include "../misc/inttostring.h"
#include "../misc/log_time.h"
#include "ftpsubmit.h"

#define GLOBUS_ERROR GLOBUS_ERROR_NO_INFO
const char* ftpsubmit_cancel_req = "CANCEL";
const char* ftpsubmit_clean_req = "CLEAN";
const char* ftpsubmit_renew_req = "RENEW";

static int callback_status;
static int data_status;
#define CALLBACK_NOTREADY     0
#define CALLBACK_DONE         1
#define CALLBACK_ERROR        2
#define CALLBACK_CLOSED       3
#define CALLBACK_CLOSE_FAILED 4
#define CALLBACK_ABORTED      5
static globus_mutex_t wait_m;
static globus_cond_t wait_c;
static const char* rsl;
static int rsl_length;
static bool callback_active;
static bool use_quit = true;

static void resp_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response);
static void abort_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response);
static int wait_for_callback(globus_ftp_control_handle_t *hctrl,int timeout = -1);
globus_ftp_control_response_class_t send_command(globus_ftp_control_handle_t *hctrl,const char* command,const char* arg,char** resp = NULL,char delim = 0,int timeout = -1);
static void write_callback(void* arg,globus_ftp_control_handle_t* hctrl,
   globus_object_t* error,globus_byte_t* buffer,
   globus_size_t length,globus_off_t offset,globus_bool_t eof);
static void conn_callback(void* arg,globus_ftp_control_handle_t* hctrl,
     unsigned int stripe_ndx,globus_bool_t reused,globus_object_t* error);
static void close_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response);

bool ftpsubmit(const char* host,unsigned short port,const char* path,const char* request,char** jobid,int timeout) {
  globus_ftp_control_handle_t *hctrl = NULL;
  bool res = false;
  bool connected = false;
  bool authenticated = false;
  char* resp;
  globus_ftp_control_host_port_t pasv_addr;
  globus_ftp_control_auth_info_t auth;
  globus_ftp_control_response_class_t resp_class;
  globus_result_t gres;

  use_quit=true;
  if(jobid == NULL) return false;
  if((request == ftpsubmit_cancel_req) || 
     (request == ftpsubmit_clean_req)  ||
     (request == ftpsubmit_renew_req)
    ) {
    if((*jobid) == NULL) return false;
  };
  // else { (*jobid) = NULL; };
  data_status=CALLBACK_NOTREADY;
  callback_status=CALLBACK_NOTREADY;
  callback_active=false;
  rsl=request;
  if(rsl == NULL) return false;
  rsl_length=strlen(rsl);
  if(rsl_length == 0) return false;
  hctrl=(globus_ftp_control_handle_t*)malloc(sizeof(globus_ftp_control_handle_t));
  if(hctrl == NULL) {
    odlog(ERROR)<<"Failed to allocate memory for control handle"<<std::endl; return res;
  };
  if(globus_ftp_control_handle_init(hctrl) != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Failed to init control handle"<<std::endl;
    free(hctrl); hctrl=NULL; return res;
  };
  globus_mutex_init(&wait_m,GLOBUS_NULL);
  globus_cond_init(&wait_c,GLOBUS_NULL);
  callback_active=true;

  if((gres=globus_ftp_control_connect(
          hctrl,(char*)host,port,&resp_callback,NULL)) != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"Failed connecting to GridFTP server "<<host<<":"<<inttostring(port)<<std::endl;
    char* tmp = globus_object_printable_to_string(globus_error_get(gres));
    odlog(INFO)<<tmp<<std::endl; free(tmp);
    goto exit;
  };
  if(wait_for_callback(hctrl,timeout) != CALLBACK_DONE) {
    goto exit;
  };
  connected=true;
  globus_ftp_control_auth_info_init(&auth,GSS_C_NO_CREDENTIAL,
           GLOBUS_TRUE,"ftp","user@",GLOBUS_NULL,GLOBUS_NULL);
  if((gres=globus_ftp_control_authenticate(hctrl,&auth,GLOBUS_TRUE,
                resp_callback,NULL)) != GLOBUS_SUCCESS) {
    char* tmp = globus_object_printable_to_string(globus_error_get(gres));
    odlog(ERROR)<<"Failed authenticating: "<<tmp<<std::endl; free(tmp);
    goto exit;
  };
  if(wait_for_callback(hctrl,timeout) != CALLBACK_DONE) {
    goto exit;
  };
  authenticated=true;
  resp=NULL;
  if(send_command(hctrl,"CWD",path,&resp,'"',timeout) !=
                  GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
    odlog(INFO)<<"Failed to access base path: ";
    if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl; };
    goto exit;
  };
  free(resp);
  if(request == ftpsubmit_cancel_req) {
    resp=NULL;
    if(send_command(hctrl,"DELE",*jobid,&resp,0,timeout) !=
                     GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      odlog(INFO)<<"DELE failed: ";
      if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl; };
      goto exit;
    };
    free(resp);
  }
  else if(request == ftpsubmit_clean_req) {
    resp=NULL;
    if(send_command(hctrl,"RMD",*jobid,&resp,0,timeout) !=
                     GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      odlog(INFO)<<"RMD failed: ";
      if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl; };
      goto exit;
    };
    free(resp);
  }
  else if(request == ftpsubmit_renew_req) {
    resp=NULL;
    if(send_command(hctrl,"CWD",*jobid,&resp,0,timeout) !=
                     GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      odlog(INFO)<<"CWD failed: ";
      if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl; };      goto exit;
    };
  }
  else {
    resp=NULL;
    if((*jobid) == NULL) { // No jobid in request - allocate one
      if(send_command(hctrl,"CWD","new",&resp,'"',timeout) !=
                       GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
        odlog(INFO)<<"CWD to 'new' failed: ";
        if(resp) {
          odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl;
        };
        goto exit;
      };
      if(resp) {
        char* s = strrchr(resp,'/');
        if(s == NULL) { s=resp; }
        else { if(s[1] == 0) { s[0]=0; s = strchr(resp,'/'); }; };
        if(s == NULL) s=resp;
        (*jobid)=strdup(s+1);
      }
      else {
        odlog(ERROR)<<"Server did not return allocated id: "<<resp<<std::endl;
        free(resp); goto exit;
      };
    } else {
      (*jobid)=NULL;
    };
    free(resp); resp=NULL;
    if(send_command(hctrl,"DCAU N",NULL,&resp,'"',timeout) != GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      odlog(DEBUG)<<"DCAU failed: ";
      if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl; };
      odlog(DEBUG)<<"Going to try anyway."<<std::endl;
      // if(*jobid) free(*jobid); goto exit;
    };
    free(resp); resp=NULL;
    globus_ftp_control_dcau_t dcau;
    dcau.mode=GLOBUS_FTP_CONTROL_DCAU_NONE;
    globus_ftp_control_local_dcau(hctrl,&dcau,GSS_C_NO_CREDENTIAL);
    if(send_command(hctrl,"PASV",NULL,&resp,'(',timeout) != GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      odlog(INFO)<<"PASV failed: ";
      if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl; };
      if(*jobid) free(*jobid); goto exit;
    };
    pasv_addr.port=0;
    if(resp) {
      int port_low,port_high;
      if(sscanf(resp,"%i,%i,%i,%i,%i,%i", 
           &(pasv_addr.host[0]), &(pasv_addr.host[1]),
           &(pasv_addr.host[2]), &(pasv_addr.host[3]),
           &port_high, &port_low) == 6) {
        pasv_addr.port=((port_high & 0x000FF) << 8) | (port_low & 0x000FF);
      };
    };
    if(pasv_addr.port == 0) {
      odlog(INFO)<<"Can't parse host and port in response to PASV: ";
      if(resp) { odlog_(INFO)<<resp<<std::endl; free(resp); } else { odlog_(INFO)<<std::endl;};
      if(*jobid) free(*jobid); goto exit;
    };
    free(resp);
    if(globus_ftp_control_local_port(hctrl,&pasv_addr) != GLOBUS_SUCCESS) {
      odlog(INFO)<<"Obtained host and address are not acceptable"<<std::endl; 
      if(*jobid) free(*jobid); goto exit;
    };
    /* have host and port - now create data connection and store */
    std::string path_=path;
    path_+="/new/job";
    resp=NULL;
    resp_class=send_command(hctrl,"STOR",path_.c_str(),&resp,0,timeout);
    if(resp_class == GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) {
      /* completion is not expected here */
      odlog(INFO)<<"STOR - completion: "<<resp<<std::endl; free(resp); goto exit;
    }
    else if(resp_class == GLOBUS_FTP_POSITIVE_PRELIMINARY_REPLY) {
    }
    else if(resp_class == GLOBUS_FTP_POSITIVE_INTERMEDIATE_REPLY) {
    }
    else {
      odlog(INFO)<<"STOR - failed: "<<resp<<std::endl; free(resp);
      if(*jobid) free(*jobid); goto exit;
    };
    free(resp);

    data_status=CALLBACK_NOTREADY;
    if(globus_ftp_control_data_connect_write(hctrl,&conn_callback,NULL) !=
                                        GLOBUS_SUCCESS) {
      odlog(INFO)<<"Failed to create data connection"<<std::endl; 
      if(*jobid) free(*jobid); goto exit;
    };
    for(;;) {
/* waiting for data failure or response received */
      resp_class=send_command(hctrl,NULL,NULL,&resp,0,timeout);
      if(resp_class == GLOBUS_FTP_POSITIVE_COMPLETION_REPLY) { break; }
      if((resp_class != GLOBUS_FTP_POSITIVE_PRELIMINARY_REPLY) &&
         (resp_class != GLOBUS_FTP_POSITIVE_INTERMEDIATE_REPLY)) {
        odlog(INFO)<<"Failed to transfer data: "<<resp<<std::endl;
        free(resp); if(*jobid) free(*jobid); goto exit;
      };
    };
    if(resp) free(resp);
    /* make sure all callbacks called */
    // wait_for_callback(hctrl,timeout);
  };
  res=true;
exit:
  if(connected) {
    if(use_quit) {
      if(globus_ftp_control_quit(hctrl,&close_callback,NULL) !=
             GLOBUS_SUCCESS) { 
        odlog(INFO)<<"Failed quiting connection - forcing"<<std::endl;
        use_quit=false;
      } else { 
        for(;;) {
          int r = wait_for_callback(hctrl,timeout);
          if(r == CALLBACK_CLOSE_FAILED) { use_quit=false; break; };
          if(r == CALLBACK_ABORTED) { use_quit=false; break; };
          if(r == CALLBACK_CLOSED) break;
        };
      };
    };
    if(!use_quit) {
      if(globus_ftp_control_force_close(hctrl,&close_callback,NULL) !=
             GLOBUS_SUCCESS) { odlog(INFO)<<"Failed closing connection"<<std::endl; }
      else { 
        for(;;) {
          int r = wait_for_callback(hctrl,timeout);
          if((r == CALLBACK_CLOSED) || (r == CALLBACK_CLOSE_FAILED)) break;
        };
      };
    };
  };
  globus_mutex_lock(&wait_m);
  callback_active=false;
  globus_mutex_unlock(&wait_m);
  globus_cond_destroy(&wait_c);
  globus_mutex_destroy(&wait_m);
  if(globus_ftp_control_handle_destroy(hctrl) == GLOBUS_SUCCESS) {
    free(hctrl); hctrl=NULL;
  } else {
    odlog(DEBUG)<<"Memory leaked (globus_ftp_control_handle_t)"<<std::endl;
    hctrl=NULL;
  };
  return res;
}

static int wait_for_callback(globus_ftp_control_handle_t *hctrl,int timeout) {
  int res;
  globus_mutex_lock(&wait_m);
  while((callback_status==CALLBACK_NOTREADY) &&
        (data_status==CALLBACK_NOTREADY)) {
    if(timeout > 0) {
      globus_abstime_t tm;
      GlobusTimeAbstimeSet(tm,timeout,0);
      if(globus_cond_timedwait(&wait_c,&wait_m,&tm) != GLOBUS_SUCCESS) {
        // timeout: cancel operation, return error
        odlog(INFO)<<"Timeout: aborting operation"<<std::endl;
        if(globus_ftp_control_abort(hctrl,abort_callback,NULL)
                                               != GLOBUS_SUCCESS) {
          // probably opearation already finished (race condition) or
          // handle not valid due to some reason. Assume there will be
          // no more any callbacks called
          callback_status=CALLBACK_ERROR;
          break;
        };
        // wait for abort (pray it will be called)
        while(callback_status!=CALLBACK_ABORTED) {
          globus_cond_wait(&wait_c,&wait_m);
        };
        callback_status=CALLBACK_ERROR;
        break;
      };
    } else {
      globus_cond_wait(&wait_c,&wait_m);
    };
  };
  res=callback_status;
  if(data_status!=CALLBACK_NOTREADY) { data_status=CALLBACK_NOTREADY; };
  callback_status=CALLBACK_NOTREADY;
  data_status=CALLBACK_NOTREADY;
  globus_mutex_unlock(&wait_m);
  return res;
}

globus_ftp_control_response_class_t send_command(globus_ftp_control_handle_t *hctrl,const char* command,const char* arg,char** resp,char delim,int timeout) {
  static globus_ftp_control_response_t server_resp;
  char* cmd = NULL;

  if(resp) { (*resp) = NULL; };
  cmd=NULL;
  if(command) {
    if(arg) { cmd = (char*)malloc(strlen(arg)+strlen(command)+4); }
    else { cmd = (char*)malloc(strlen(command)+3); };
    if(cmd==NULL) {
      odlog(ERROR)<<"Memory allocation error"<<std::endl;
      return GLOBUS_FTP_UNKNOWN_REPLY;
    };
    strcpy(cmd,command); 
    if(arg) { strcat(cmd," "); strcat(cmd,arg); };
    strcat(cmd,"\r\n");
    odlog(VERBOSE)<<"Sending command: "<<cmd;
    if(globus_ftp_control_send_command(hctrl,cmd,resp_callback,&server_resp)
                != GLOBUS_SUCCESS) {
      odlog(DEBUG)<<command<<" failed"<<std::endl;
      if(cmd) free(cmd); return GLOBUS_FTP_UNKNOWN_REPLY;
    };
  };
  globus_mutex_lock(&wait_m);
  while((callback_status==CALLBACK_NOTREADY) &&
        (data_status!=CALLBACK_ERROR)) {
    if(timeout > 0) {
      globus_abstime_t tm;
      GlobusTimeAbstimeSet(tm,timeout,0);
      if(globus_cond_timedwait(&wait_c,&wait_m,&tm) != GLOBUS_SUCCESS) {
        odlog(INFO)<<"Timeout: aborting operation"<<std::endl;
        // timeout: cancel operation, return error
        if(globus_ftp_control_abort(hctrl,abort_callback,NULL)
                                               != GLOBUS_SUCCESS) {
          // probably opearation already finished (race condition) or
          // handle not valid due to some reason. Assume there will be
          // no more any callbacks called
          callback_status=CALLBACK_ERROR;
          break;
        };
        // wait for abort (pray it will be called)
        while(callback_status!=CALLBACK_ABORTED) {
          globus_cond_wait(&wait_c,&wait_m);
        };
        callback_status=CALLBACK_ERROR;
        break;
      };
    } else {
      globus_cond_wait(&wait_c,&wait_m);
    };
  };
  free(cmd);
  if(data_status==CALLBACK_ERROR) {
    data_status=CALLBACK_NOTREADY;
/* maybe have to free response here !!!!!! */
    odlog(INFO)<<"Data transfer failure detected"<<std::endl;
    globus_mutex_unlock(&wait_m);
    return GLOBUS_FTP_UNKNOWN_REPLY;
  };
  data_status=CALLBACK_NOTREADY;
  if(callback_status != CALLBACK_DONE) {
    callback_status=CALLBACK_NOTREADY;
    globus_mutex_unlock(&wait_m);
    return GLOBUS_FTP_UNKNOWN_REPLY;
  };
  callback_status=CALLBACK_NOTREADY;
  if(resp) {
    if(delim == 0) {
      (*resp)=(char*)malloc(server_resp.response_length);
      if((*resp) != NULL) {
        memcpy(*resp,(char*)(server_resp.response_buffer+4),
               server_resp.response_length-4);
        (*resp)[server_resp.response_length-4]=0;
      };
    }
    else {
      /* look for pair of enclosing characters */
      char* s_start = (char*)(server_resp.response_buffer+4);
      char* s_end = NULL;
      int l = 0;
      s_start=strchr(s_start,delim);
      if(s_start) {
        s_start++; if(delim=='(') { delim=')'; }
        else if(delim=='{') { delim='}'; }
        else if(delim=='[') { delim=']'; };
        s_end=strchr(s_start,delim); if(s_end) l=s_end-s_start;
      };
      if(l>0) {
        (*resp)=(char*)malloc(l+1);
        if((*resp) != NULL) { memcpy(*resp,s_start,l); (*resp)[l]=0; };
      };
    };
  };
  globus_ftp_control_response_class_t resp_class = server_resp.response_class;
  globus_ftp_control_response_destroy(&server_resp);
  globus_mutex_unlock(&wait_m);
  return resp_class;
}


static void close_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response) {
  if(!callback_active) return;
  globus_mutex_lock(&wait_m);
  if(error != GLOBUS_SUCCESS) {
    callback_status=CALLBACK_CLOSE_FAILED;
    char* tmp = globus_object_printable_to_string(error);
    odlog(INFO)<<"Failure(close): "<<tmp<<std::endl; free(tmp);
    if(response) odlog(INFO)<<"Server said: "<<response->response_buffer<<std::endl;
  }
  else {
    callback_status=CALLBACK_CLOSED;
  }; 
  globus_cond_signal(&wait_c);
  globus_mutex_unlock(&wait_m);
}

static void abort_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response) {
  if(!callback_active) return;
  globus_mutex_lock(&wait_m);
  callback_status=CALLBACK_ABORTED;
  if(response) odlog(VERBOSE)<<"Operation aborted"<<std::endl;
  globus_cond_signal(&wait_c);
  globus_mutex_unlock(&wait_m);
}

static void resp_callback(void *arg,globus_ftp_control_handle_t *h,globus_object_t *error,globus_ftp_control_response_t *response) {
  if(!callback_active) return;
  globus_mutex_lock(&wait_m);
  if(error != GLOBUS_SUCCESS) {
    callback_status=CALLBACK_ERROR;
    char* tmp = globus_object_printable_to_string(error);
    if(strstr (tmp, "end-of-file")) {
      odlog(INFO)<<"Failure(response): GridFTP server unexpectedly closed connection"<<std::endl;
    } else {
      odlog(INFO)<<"Failure(response): "<<tmp<<std::endl;
    }
    free(tmp);
    if(response) {
      odlog(INFO)<<"GridFTP server said: "<<response->response_buffer<<std::endl;
    } else {
      // If server said nothing that must connection problem.
      // It seem like globus_ftp_control is confused by that, so
      // do not try to close connection properly.
      use_quit=false;
    };
  }
  else {
    if(arg != 0) globus_ftp_control_response_copy(response,(globus_ftp_control_response_t*)arg);
    callback_status=CALLBACK_DONE;
  };
  if(response) odlog(VERBOSE)<<"Got response: "<<response->response_buffer<<std::endl;
  globus_cond_signal(&wait_c);
  globus_mutex_unlock(&wait_m);
}

static void conn_callback(void* arg,globus_ftp_control_handle_t* hctrl,
     unsigned int stripe_ndx,globus_bool_t reused,globus_object_t* error) {
  if(!callback_active) return;
  if(error != GLOBUS_SUCCESS) {
    char* tmp = globus_object_printable_to_string(error);
    odlog(INFO)<<"Failure(data connection): "<<tmp<<std::endl; free(tmp);
    globus_mutex_lock(&wait_m);
    callback_status=CALLBACK_ERROR;
    globus_cond_signal(&wait_c);
    globus_mutex_unlock(&wait_m);
    return;
  };
  if(globus_ftp_control_data_write(hctrl,(globus_byte_t*)rsl,rsl_length,
         0,GLOBUS_TRUE,&write_callback,NULL) != GLOBUS_SUCCESS) {
    odlog(INFO)<<"Failed sending data"<<std::endl;
    globus_mutex_lock(&wait_m);
    data_status=CALLBACK_ERROR;
    globus_cond_signal(&wait_c);
    globus_mutex_unlock(&wait_m);
    return;
  };
}

static void write_callback(void* arg,globus_ftp_control_handle_t* hctrl,
   globus_object_t* error,globus_byte_t* buffer,
   globus_size_t length,globus_off_t offset,globus_bool_t eof) {
  if(!callback_active) return;
  if(error != GLOBUS_SUCCESS) {
    char* tmp = globus_object_printable_to_string(error);
    odlog(INFO)<<"Failure(write): "<<tmp<<std::endl; free(tmp);
    globus_mutex_lock(&wait_m);
    data_status=CALLBACK_ERROR;
    globus_cond_signal(&wait_c);
    globus_mutex_unlock(&wait_m);
    return;
  };
  globus_mutex_lock(&wait_m);
  data_status=CALLBACK_DONE;
  globus_cond_signal(&wait_c);
  globus_mutex_unlock(&wait_m);
  return;
}

