// Request data transfer (and optionally wait till completed).

#include "std.h"

#include <string>
#include <list>
#include <vector>

#include <globus_ftp_client.h>

#include "datamove/datapoint.h"
#include "datamove/datahandle.h"
#include "misc/log_time.h"
#include "misc/globus_error_utils.h"
#include "misc/url_options.h"
#include "misc/condition.h"
#include <arc/certificate.h>

#include "ngdata.h"

// SSE
#include "https/se/file_soapH.h"
extern SOAP_NMAC struct Namespace file_soap_namespaces[];
#include "https/client/client.h"
#include "https/SRM/srm_client.h"


bool sse_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout);
bool srm_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout);
bool ftp_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout);
bool meta_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout);

bool ftp_ftp_replicate(const char* dest,const char* src,bool verbose,int timeout);

void arctransfer(const std::string& destination,
                 std::list<std::string>& sources,
                 int timeout) {
  LogTime::Active(false);
  bool verbose = false;

  LogTime::Level(GetNotifyLevel());
  if(LogTime::Level() > FATAL) verbose=true;

  DataPoint url(destination.c_str());
  if(!url) throw ARCCLIDataError("Unsupported destination url");
  if(url.meta()) {
    if(!meta_replicate(url,sources,verbose,timeout)) {
      throw ARCCLIDataError("Transfer failed");
    };
  } else if(strncasecmp(destination.c_str(),"se://",5) == 0) {
    // send request to SSE 
    if(!sse_replicate(url,sources,verbose,timeout)) {
      throw ARCCLIDataError("Transfer failed");
    };
  } else if(strncasecmp(destination.c_str(),"srm://",6) == 0) {
    // send request to SRM 
    if(sources.size() != 1) {
      throw ARCCLIDataError("SRM destination accepts one source only");
    };
    if(!srm_replicate(url,sources,verbose,timeout)) {
      throw ARCCLIDataError("Transfer failed");
    };
  } else if(strncasecmp(destination.c_str(),"gsiftp://",9) == 0) {
    if(!ftp_replicate(url,sources,verbose,timeout)) {
      throw ARCCLIDataError("Transfer failed");
    };
  } else {
    throw ARCCLIDataError("URL "+destination+" is not supported");
  };
  return;
}

// Destination is specified as meta-URL. Need to find proper destinations
// and initiate transfers.
bool meta_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout) {
  if(!dest.meta_resolve(false,UrlMap())) {
    odlog(ERROR)<<"Failed to resolve possible destinations"<<std::endl;
    return false;
  };
  if(!dest.have_locations()) {
    odlog(ERROR)<<"No destination locations found"<<std::endl;
    return false;
  };
  bool replication = (sources.size() == 0);
  for(;dest.have_location();dest.next_location()) {
    const std::string& loc = dest.current_location();
    if(strncasecmp(loc.c_str(),"se://",5) == 0) {
      odlog(INFO)<<"Destination chosen: "<<loc<<std::endl;
      if(!dest.meta_preregister(replication,false)) {
        odlog(INFO)<<"Failed to pre-register"<<std::endl; continue;
      };
      if(sse_replicate(dest,sources,verbose,timeout)) {
        if(!dest.meta_postregister(replication,false)) {
          odlog(INFO)<<"Failed to post-register"<<std::endl;
          dest.meta_preunregister(replication);
          continue;
        };
        return true;
      };
      if(!dest.meta_preunregister(replication)) {
        odlog(INFO)<<"WARNING: Failed to un-register failed transfer."<<std::endl;
      };
      odlog(INFO)<<"Failed - trying next destination"<<std::endl;
   } else if(strncasecmp(loc.c_str(),"srm://",6) == 0) {
      odlog(INFO)<<"Destination chosen: "<<loc<<std::endl;
      if(!dest.meta_preregister(replication,false)) {
        odlog(INFO)<<"Failed to pre-register"<<std::endl; continue;
      };
      if(srm_replicate(dest,sources,verbose,timeout)) {
        if(!dest.meta_postregister(replication,false)) {
          odlog(INFO)<<"Failed to post-register"<<std::endl;
          dest.meta_preunregister(replication);
          continue;
        };
        return true;
      };
      if(!dest.meta_preunregister(replication)) {
        odlog(INFO)<<"WARNING: Failed to un-register failed transfer."<<std::endl;
      };
      odlog(INFO)<<"Failed - trying next destination"<<std::endl;
    } else if((strncasecmp(loc.c_str(),"gsiftp://",9) == 0) ||
              (strncasecmp(loc.c_str(),"ftp://",6) == 0)) {
      odlog(INFO)<<"Destination chosen: "<<loc<<std::endl;
      if(!dest.meta_preregister(replication,false)) {
        odlog(INFO)<<"Failed to pre-register"<<std::endl; continue;
      };
      if(ftp_replicate(dest,sources,verbose,timeout)) {
        if(!dest.meta_postregister(replication,false)) {
          odlog(INFO)<<"Failed to post-register"<<std::endl;
          dest.meta_preunregister(replication);
          continue;
        };
        return true;
      };
      if(!dest.meta_preunregister(replication)) {
        odlog(INFO)<<"WARNING: Failed to un-register failed transfer."<<std::endl;
      };
    } else {
      odlog(INFO)<<"Unsupported location "<<loc<<std::endl;
    };
  };
}

bool sse_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout) {
  std::string service_url = dest.current_location();
  service_url.replace(0,2,"httpg");
  std::string service_path = "";
  {
    std::string::size_type n = service_url.find('?');
    if(n == std::string::npos) {
      odlog(ERROR)<<"Missing LFN in destination URL"<<std::endl;
      return false;
    };
    service_path=service_url.c_str()+n+1;
    service_url.resize(n);
  };
  canonic_url(service_url); // remove options
  odlog(INFO)<<"Talking to SOAP service at "<<service_url<<std::endl;
  struct soap soap;
  HTTP_ClientSOAP s(service_url.c_str(),&soap);
  soap.namespaces=file_soap_namespaces;
  if(s.connect() != 0) {
    odlog(ERROR)<<"Failed to connect to "<<service_url<<std::endl;
    return false;
  };
  ns__fileinfo info;
  ns__addResponse rr;
  int soap_err = SOAP_OK;
  info.id=(char*)(service_path.c_str());
  std::string acl("");
  {
    try {
      Certificate ci;
      acl=ci.GetIdentitySN();
    } catch(std::exception) { };
    if(acl.length()) acl="<gacl><entry><person><dn>"+acl+"</dn></person><allow><read/><write/><list/><admin/></allow></entry></gacl>";
    info.acl=(char*)(acl.c_str());
  };
  // unsigned long long int* size;
  // char* checksum;
  // char* created;
  // enum ns__filestate* state;
  // int __size_url;
  // char** url;
  int size_sources_ = sources.size();
  char** sources_;
  if(size_sources_ == 0) {
    sources_=(char**)malloc(sizeof(char*));
    sources_[0]=""; // replication
    size_sources_=1;
  } else {
    sources_=(char**)malloc(sizeof(char*)*size_sources_);
    int i = 0;
    std::list<std::string>::iterator ii = sources.begin();
    for(;i<size_sources_;++i,++ii) sources_[i]=(char*)(ii->c_str());
  };
  // Send resuest to server for new file
  if((soap_err=soap_call_ns__add(&soap,s.SOAP_URL(),"add",&info,
                           size_sources_,sources_,rr)) != SOAP_OK) {
    odlog(INFO)<<"Failed to execute remote soap call 'add' at "<<service_url<<std::endl;
    return false;
  };
  if(rr.error_code != 0) {
    odlog(INFO)<<"Failed ("<<rr.error_code<<") to create remote file "<<info.id<<" at "<<service_url<<std::endl;
    return false;
  };
  s.reset();
  // Wait till either file disappears (failed) or reaches proper state
  std::string file_url=s.SOAP_URL(); file_url+="/"; file_url+=service_path;
  bool file_detected = false;
  int retries = 5;
  time_t start = time(NULL);
  for(;;) {
    ns__infoResponse rr;
    if((soap_err=soap_call_ns__info(&soap,file_url.c_str(),"info",
                           NULL,rr)) != SOAP_OK) {
      odlog(INFO)<<"Failed to execute remote soap call 'info' at "<<service_url<<std::endl;
      retries--;
      if((--retries) <= 0) {
        odlog(ERROR)<<"Can't track state of file at "<<service_url<<std::endl;
        return false;
      };
      sleep(5);
      s.reset();
      continue;
    };
    retries=5;
    if((rr.error_code != 0) || (rr.__size_file == 0)) {
      odlog(INFO)<<"Error "<<rr.error_code<<"."<<rr.sub_error_code<<": "<<rr.error_description<<std::endl;
      if(file_detected) {
        odlog(ERROR)<<"Download failed."<<std::endl;
      } else {
        odlog(ERROR)<<"File disappeared before client could see it. Most probably download failed."<<std::endl;
      };
      return false;
    };
    if(rr.__size_file != 1) {
      odlog(ERROR)<<"Wrong number of files reported. Must be some error at  server's side."<<std::endl;
      return false;
    };
    if(verbose) {
      const char* fstate="unknown     ";
      if(rr.file[0].state) {
        switch(*(rr.file[0].state)) {
          case 0: fstate="ACCEPTED    "; break;
          case 1: fstate="COLLECTING  "; break;
          case 2: fstate="REQUESTED   "; break;
          case 3: fstate="DOWNLOADING "; break;
          case 4: fstate="COMPLETE    "; break;
          case 5: fstate="VALID       "; break;
        };
      };
      odlog(ERROR)<<"\rTime passed: "<<time(NULL)-start<<" s, remote state: "<<fstate<<std::flush;
    };
    if(rr.file[0].state && (*(rr.file[0].state) == 5)) {
      odlog(ERROR)<<"File downloaded"<<std::endl;
      return true;
    };
    s.reset();
    sleep(5);
  };
  return true;
}

bool srm_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout) {
  std::string service_url = dest.current_location();
  SRM_URL url(service_url.c_str());
  if(!url) {
    odlog(ERROR)<<"Failed to parse URL "<<service_url<<std::endl;
    return false;
  };
  if(url.FileName().empty()) {
    odlog(ERROR)<<"Missing file name in destination URL"<<std::endl;
    return false;
  };
  SRMClient client(url);
  client.Timeout(timeout);
  SRMClientRequest req;
  if(!client.copy(req,url.FileName().c_str(),url,*(sources.begin()))) {
    odlog(ERROR)<<"Failed to initiate or finish copy at "<<service_url<<std::endl;
    return false;
  };
  return true;
}

Condition<int> ftp_cond;

void ftp_replicate_callback(void *user_arg,globus_ftp_client_handle_t *handle,globus_object_t *error) {
  if(error != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"FTP operation failed: "<<error<<std::endl;
    ftp_cond.signal(1);
  } else {
    ftp_cond.signal(0);
  };
}

bool ftp_replicate(DataPoint& dest,std::list<std::string>& sources,bool verbose,int timeout) {
  if(sources.size() == 0) {
    if(!dest.meta()) return false;
    std::list<std::string> sources_;
    sources_.push_back(dest.base_url());
    return ftp_replicate(dest,sources_,verbose,timeout);
  };
  globus_module_activate(GLOBUS_FTP_CLIENT_MODULE);
  std::list<DataPoint*> srcs;
  for(std::list<std::string>::iterator s = sources.begin();s!=sources.end();++s) {
    DataPoint* url = new DataPoint(s->c_str());
    if(*url) {
      srcs.push_back(url);
    } else {
      odlog(ERROR)<<"Unsupported URL: "<<*s<<std::endl;
      delete url;
    };
  };
  for(;;) {
    bool transfer_tried = false;
    for(std::list<DataPoint*>::iterator src = srcs.begin();src!=srcs.end();++src) {
      DataPoint& source = **src;
      if(!source.meta_resolve(true,UrlMap())) {
        odlog(ERROR)<<"Failed to resolve source: "<<source<<std::endl;
        source.next_location(); /* try again */
        continue;
      };
      if(!source.have_location()) continue;
      if(source.current_location() == dest.current_location()) continue;
      if((strncasecmp(source.current_location(),"ftp://",6) == 0) ||
         (strncasecmp(source.current_location(),"gsiftp://",9) == 0)) {
        odlog(INFO)<<"Source chosen: "<<source.current_location()<<std::endl;
        transfer_tried=true;
        if(ftp_ftp_replicate(dest.current_location(),source.current_location(),verbose,timeout)) {
          for(std::list<DataPoint*>::iterator s=srcs.begin();s!=srcs.end();++s) delete *s;
          globus_module_deactivate(GLOBUS_FTP_CLIENT_MODULE);
          return true;
        } else {
          odlog(INFO)<<"Failed transfer"<<std::endl;
          source.next_location();
        };
      } else {
        odlog(INFO)<<"Can't transfer to (gsi)FTP from source "<<source.current_location()<<std::endl;
        source.next_location();
      };
    };
    if(!transfer_tried) break;
  };
  for(std::list<DataPoint*>::iterator s=srcs.begin();s!=srcs.end();++s) delete *s;
  globus_module_deactivate(GLOBUS_FTP_CLIENT_MODULE);
  return false;
}

bool ftp_ftp_replicate(const char* dest,const char* src,bool verbose,int timeout) {
  globus_ftp_client_handle_t handle;
  globus_ftp_client_handleattr_t attr;
  globus_ftp_client_operationattr_t src_attr;
  globus_ftp_client_operationattr_t dest_attr;
  globus_result_t res;

std::cerr<<"ftp_ftp_replicate: "<<dest<<" <- "<<src<<std::endl;
  globus_ftp_client_handleattr_init(&attr);
  globus_ftp_client_handle_init(&handle,&attr);
  globus_ftp_client_operationattr_init(&src_attr);
  globus_ftp_client_operationattr_init(&dest_attr);

  res=globus_ftp_client_third_party_transfer(&handle,
   src,&src_attr, dest,&dest_attr,
   GLOBUS_NULL,&ftp_replicate_callback,GLOBUS_NULL);
  if(res != GLOBUS_SUCCESS) {
    odlog(ERROR)<<"FTP transfer failed: "<<GlobusResult(res)<<std::endl;
    return false;
  };
  int r;
  if(!ftp_cond.wait(r,timeout*1000)) {
    odlog(ERROR)<<"FTP operation timed out"<<std::endl;
    globus_ftp_client_abort(&handle);
  };
  if(r != 0) return false;
  return true;
}

extern "C"
int ngtransferxx (
    const std::string& destination,
    std::vector<std::string>& sources,
    int verbosity_level,
    int timeout
) {
  try {
    SetNotifyLevel(NotifyLevel(FATAL+verbosity_level));
    std::list<std::string> sources_;
    for(std::vector<std::string>::iterator i = sources.begin();
                            i!=sources.end();++i) sources_.push_back(*i);
    arctransfer (destination,sources_,timeout);
  } catch(ARCCLIDataError& e) {
    return -1;
  } catch(std::exception& e) {
    return -1;
  };
  return 0;
}

