#include "std.h"

#include <string>
#include <list>
#include <fstream>

#include "datamove/datapoint.h"
#include "datamove/datahandle.h"
#include "misc/log_time.h"

#include "ngdata.h"

static void print_time(time_t t);

static bool get_url_list(const char* file,std::list<std::string>& urls) {
  std::ifstream f(file);
  if(!f.is_open()) return false;
  for(;!f.eof();) {
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    if(!(buf[0])) continue;
    urls.push_back(buf);
  };
  if(urls.size() == 0) return false;
  return true;
}

void arcls(const std::string& dir_url_,
           bool show_details,
           bool show_urls,
           int recursion,
           int timeout) {
  LogTime::Active(false);

  if(dir_url_[0] == '@') {
    std::list<std::string> dirs;
    if(!get_url_list(dir_url_.c_str()+1,dirs)) {
      throw ARCCLIDataError(std::string("Can't read list of locations "
                      "from file ") + (dir_url_.c_str()+1));
    };
    std::list<std::string>::iterator dir_ = dirs.begin();
    for(;dir_!=dirs.end();) {
      arcls(*dir_,show_details,show_urls,timeout);
      ++dir_;
    };
    return;
  };

  std::string dir_url(dir_url_.c_str());

  LogTime::Level(GetNotifyLevel());

  if(dir_url != "-") {
    std::string::size_type p = dir_url.find("://");
    if((p == std::string::npos) || (dir_url.find('/') < p)) {
      if(dir_url[0] == '/') {
        dir_url="file://"+dir_url;
      } else {
        char buf[1024]; buf[0]=0;
        getcwd(buf,sizeof(buf));
        dir_url=std::string("file://")+buf+"/"+dir_url;
      };
    };
  };
  DataPoint url(dir_url.c_str());
  if(!url) throw ARCCLIDataError("Unsupported url given");
  std::list<DataPoint::FileInfo> files;
  if(url.meta()) {
    if(!url.list_files(files,show_details)) {
      if(files.size() == 0) throw ARCCLIDataError("Failed listing metafiles");
      odlog(INFO)<<"Warning: Failed listing metafiles but some information is obtained"<<std::endl;
    };
  } else {
    DataHandle h(&url);
    h.secure(false);
    if(!h.list_files(files,show_details)) {
      if(files.size() == 0) throw ARCCLIDataError("Failed listing files");
      odlog(INFO)<<"Warning: Failed listing files"
                " but some information is obtained"<<std::endl;
    };
  };
  for(std::list<DataPoint::FileInfo>::iterator i = files.begin();
                                               i!=files.end();++i) {
    std::cout<<i->name;
    if(show_details) {
      switch(i->type) {
        case DataPoint::FileInfo::file_type_file: std::cout<<" file"; break;
        case DataPoint::FileInfo::file_type_dir: std::cout<<" dir"; break;
        default: std::cout<<" unknown"; break;
      };
      if(i->size_available) { std::cout<<" "<<i->size; }
      else { std::cout<<" *"; };
      if(i->created_available) { print_time(i->created); }
      else { std::cout<<" *"; };
      if(i->valid_available) { print_time(i->valid); }
      else { std::cout<<" *"; };
      if(i->checksum_available) { std::cout<<" "<<i->checksum; }
      else { std::cout<<" *"; };
    };
    std::cout<<std::endl;
    if(show_urls) {
      for(std::list<std::string>::iterator u = i->urls.begin();
                                            u!=i->urls.end();++u) {
        std::cout<<"\t"<<*u<<std::endl;
      };
    };
    // Do recursion
    if(recursion > 0) {
      if(i->type == DataPoint::FileInfo::file_type_dir) {
        std::string suburl = dir_url;
        if(suburl[suburl.length()-1] != '/') suburl+="/";
        suburl+=i->name;
        std::cout<<suburl<<" :"<<std::endl;
        arcls(suburl,
              show_details,show_urls,recursion-1,timeout);
        std::cout<<std::endl;
      };
    };
  };
  return;
}

static void print_time(time_t t) {
  char buf[1024];
  char *t_;
  t_=ctime_r(&t,buf);
  int l = strlen(t_);
  if(l>0) { if(t_[l-1] == '\n') { l--; t_[l]=0; }; };
  if((t_ == NULL) || (l==0)) { std::cout<<" *"; return; };
  std::cout<<" \""<<t_<<"\"";
  return;
}

extern "C"
int nglsxx (
    const std::string& dir_url,
    bool show_details,
    bool show_urls,
    int verbosity_level,
    int timeout
) {
  try {
    SetNotifyLevel(NotifyLevel(FATAL+verbosity_level));
    arcls (dir_url,show_details,show_urls,timeout);
  } catch(ARCCLIDataError& e) {
    return -1;
  } catch(std::exception& e) {
    return -1;
  };
  return 0;
}

