#include "std.h"

#include <string>
#include <list>
#include <fstream>

#include "datamove/datacache.h"
#include "datamove/datapoint.h"
#include "datamove/datahandle.h"
#include "datamove/datamove.h"
#include "misc/globus_modules.h"
#include "misc/log_time.h"
#include <arc/url.h>
#include "misc/url_options.h"
#include "misc/inttostring.h"

#include "ngdata.h"

static void progress(FILE* o,const char* s,unsigned int t,
       unsigned long long int all,unsigned long long int max,
       double instant,double average) {
  static int rs = 0;
  const char rs_[4] = { '|','/','-','\\' };
  if(max) {
    fprintf(o,"\r|");
    unsigned int l = (74*all+37)/max; if(l>74) l=74;
    unsigned int i = 0;
    for(;i<l;i++) fprintf(o,"=");
    fprintf(o,"%c",rs_[rs++]); if(rs>3) rs=0;
    for(;i<74;i++) fprintf(o," ");
    fprintf(o,"|\r"); fflush(o);
    return;
  };
  fprintf(o,"\r%Lu kB                    \r",all/1024);
}

static void make_url(std::string& url) {
  if(url != "-") {
    std::string::size_type p = url.find("://");
    if((p == std::string::npos) || (url.find('/') < p)) {
      if(url[0] == '/') {
        url="file://"+url;
      } else {
        char buf[1024]; buf[0]=0;
        getcwd(buf,sizeof(buf));
        url=std::string("file://")+buf+"/"+url;
      };
    };
  };
}

static bool get_url_list(const char* file,std::list<std::string>& urls) {
  std::ifstream f(file);
  if(!f.is_open()) return false;
  for(;!f.eof();) { 
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    if(!(buf[0])) continue;
    urls.push_back(buf);
  };
  if(urls.size() == 0) return false;
  return true;
}

void arcregister (
    const std::string& source_url_,
    const std::string& destination_url_,
    bool secure,
    bool passive,
    bool force_meta,
    int timeout
) {
  LogTime::Active(false);

  if((source_url_[0] == '@') && (destination_url_[0] == '@')) {
    std::list<std::string> sources;
    std::list<std::string> destinations;
    if(!get_url_list(source_url_.c_str()+1,sources)) {
      throw ARCCLIDataError(std::string("Can't read list of sources "
                      "from file ") + (source_url_.c_str()+1));
    };
    if(!get_url_list(destination_url_.c_str()+1,destinations)) {
      throw ARCCLIDataError(std::string("Can't read list of destinations "
                      "from file ") + (destination_url_.c_str()+1));
    };
    if(sources.size() != destinations.size()) {
      throw ARCCLIDataError("Numbers of sources and destinations "
                      "do not match");
    };
    std::list<std::string>::iterator source_ = sources.begin();
    std::list<std::string>::iterator destination_ = destinations.begin();
    for(;(source_!=sources.end()) && (destination_!=destinations.end());) {
      arcregister(*source_,*destination_,
            secure,passive,force_meta,timeout);
      ++source_; ++destination_;
    }; 
    return;
  };
  if(source_url_[0] == '@') {
    std::list<std::string> sources;
    if(!get_url_list(source_url_.c_str()+1,sources)) {
      throw ARCCLIDataError(std::string("Can't read list of sources "
                      "from file ") + (source_url_.c_str()+1));
    };
    std::list<std::string>::iterator source_ = sources.begin();
    for(;source_!=sources.end();) {
      arcregister(*source_,destination_url_,
            secure,passive,force_meta,timeout);
      ++source_;
    }; 
    return;
  };
  if(destination_url_[0] == '@') {
    std::list<std::string> destinations;
    if(!get_url_list(destination_url_.c_str()+1,destinations)) {
      throw ARCCLIDataError(std::string("Can't read list of destinations "
                      "from file ") + (destination_url_.c_str()+1));
    };
    std::list<std::string>::iterator destination_ = destinations.begin();
    for(;destination_!=destinations.end();) {
      arcregister(source_url_,*destination_,
            secure,passive,force_meta,timeout);
      ++destination_;
    };
    return;
  };

  std::string source_url(source_url_.c_str());
  std::string destination_url(destination_url_.c_str());

  LogTime::Level(GetNotifyLevel());

  make_url(source_url);
  make_url(destination_url);
  if(destination_url[destination_url.length()-1] == '/') 
    throw ARCCLIDataError("Fileset registration is not supported yet");
  DataPoint source(source_url.c_str());
  DataPoint destination(destination_url.c_str());
  if(!source) throw ARCCLIDataError("Unsupported source url");
  if(!destination) throw ARCCLIDataError("Unsupported destination url");
  canonic_url(destination_url);
  canonic_url(source_url);
  if(source.meta() || !destination.meta()) {
    throw ARCCLIDataError("For registration source must be ordinary URL"
                      " and destination must be indexing service");
  };
  // Obtain meta-information about source
  DataHandle source_h(&source);
  if(!source_h.check()) {
    throw ARCCLIDataError("Source probably does not exist");
  };
  // add new location
  if(!destination.meta_resolve(false)) {
    throw ARCCLIDataError("Problems resolving destination");
  };
  bool replication = destination.meta_stored();
  destination.meta(source); // pass metadata
  std::string metaname("");
  // look for similar destination
  for(destination.tries(1);destination.have_location();
                                     destination.next_location()) {
    const char* loc_url = destination.current_location();
    int l = strlen(loc_url);
    if(strncmp(loc_url,source_url.c_str(),l) == 0) {
      metaname=destination.current_meta_location();
      break;
    };
  };
  // remove locations if exist
  for(destination.tries(1);destination.remove_location();) { };
  // add new location
  if(metaname.length() == 0) {
    // HACK!!
    try {
      URL u(source_url.c_str());
      metaname=u.Protocol()+"://"+u.Host()+":"+inttostring(u.Port());
    } catch (std::exception e) {
      metaname="";
    };
  };
  if(!destination.add_location(metaname.c_str(),source_url.c_str())) {
    destination.meta_preunregister(replication);
    throw ARCCLIDataError("Failed to accept new file/destination");
  }; 
  destination.tries(1);
  if(!destination.meta_preregister(replication,force_meta)) {
    throw ARCCLIDataError("Failed to register new file/destination");
  }; 
  if(!destination.meta_postregister(replication,false)) {
    destination.meta_preunregister(replication);
    throw ARCCLIDataError("Failed to register new file/destination");
  }; 
  return;
}

void arccp (
    const std::string& source_url_,
    const std::string& destination_url_,
    bool secure,
    bool passive,
    bool force_meta,
    int recursion,
    int tries,
    bool verbose,
    int timeout
) {
  char* cache_path = NULL;
  char* cache_data_path = NULL;
  char* id = "<ngcp>";
  // bool verbose = false;
  LogTime::Active(false);
  if(timeout <= 0) timeout=300; // 5 minute default
  if(tries < 0) tries=0;

  if((source_url_[0] == '@') && (destination_url_[0] == '@')) {
    std::list<std::string> sources;
    std::list<std::string> destinations;
    if(!get_url_list(source_url_.c_str()+1,sources)) {
      throw ARCCLIDataError(std::string("Can't read list of sources "
                      "from file ") + (source_url_.c_str()+1));
    };
    if(!get_url_list(destination_url_.c_str()+1,destinations)) {
      throw ARCCLIDataError(std::string("Can't read list of destinations "
                      "from file ") + (destination_url_.c_str()+1));
    };
    if(sources.size() != destinations.size()) {
      throw ARCCLIDataError("Numbers of sources and destinations "
                      "do not match");
    };
    std::list<std::string>::iterator source_ = sources.begin();
    std::list<std::string>::iterator destination_ = destinations.begin();
    for(;(source_!=sources.end()) && (destination_!=destinations.end());) {
      arccp(*source_,*destination_,
            secure,passive,force_meta,recursion,verbose,timeout);
      ++source_; ++destination_;
    }; 
    return;
  };
  if(source_url_[0] == '@') {
    std::list<std::string> sources;
    if(!get_url_list(source_url_.c_str()+1,sources)) {
      throw ARCCLIDataError(std::string("Can't read list of sources "
                      "from file ") + (source_url_.c_str()+1));
    };
    std::list<std::string>::iterator source_ = sources.begin();
    for(;source_!=sources.end();) {
      arccp(*source_,destination_url_,
            secure,passive,force_meta,recursion,verbose,timeout);
      ++source_;
    }; 
    return;
  };
  if(destination_url_[0] == '@') {
    std::list<std::string> destinations;
    if(!get_url_list(destination_url_.c_str()+1,destinations)) {
      throw ARCCLIDataError(std::string("Can't read list of destinations "
                      "from file ") + (destination_url_.c_str()+1));
    };
    std::list<std::string>::iterator destination_ = destinations.begin();
    for(;destination_!=destinations.end();) {
      arccp(source_url_,*destination_,
            secure,passive,force_meta,recursion,verbose,timeout);
      ++destination_;
    };
    return;
  };

  std::string source_url(source_url_.c_str());
  std::string destination_url(destination_url_.c_str());

  // verbose=true;
  LogTime::Level(GetNotifyLevel());

  make_url(source_url);
  make_url(destination_url);

  if(destination_url[destination_url.length()-1] != '/') {
    if(source_url[source_url.length()-1] == '/') {
      throw ARCCLIDataError("Fileset copy to single object"
                            " is not supported yet");
    };
  } else {
    // Copy TO fileset/directory
    if(source_url[source_url.length()-1] != '/') {
      // Copy FROM single object
      std::string::size_type p = source_url.rfind('/');
      if(p == std::string::npos) {
        throw ARCCLIDataError("Can't extract object's name from source url");
      };
      destination_url+=source_url.c_str()+p+1;
    } else {
      // Fileset copy
      // Find out if source can be listed (TODO - through datapoint)
      if(strncasecmp(source_url.c_str(),"rc://",5) &&
         strncasecmp(source_url.c_str(),"rls://",6) &&
         strncasecmp(source_url.c_str(),"fireman://",6) &&
         strncasecmp(source_url.c_str(),"file://",5) &&
         strncasecmp(source_url.c_str(),"se://",5) &&
         strncasecmp(source_url.c_str(),"gsiftp://",9) &&
         strncasecmp(source_url.c_str(),"ftp://",6)) {
        throw ARCCLIDataError("Fileset copy for this kind of source"
                              " is not supported");
      };
      DataPoint source(source_url.c_str());
      if(!source) throw ARCCLIDataError("Unsupported source url");
      std::list<DataPoint::FileInfo> files;
      if(source.meta()) {
        if(!source.list_files(files,recursion>0)) {
          throw ARCCLIDataError("Failed listing metafiles");
        };
      } else {
        DataHandle h(&source);
        if(!h.list_files(files,true)) {
          throw ARCCLIDataError("Failed listing files");
        };
      };
      bool failures = false;
      // Handle transfer of files first (treat unknown like files)
      for(std::list<DataPoint::FileInfo>::iterator i = files.begin();
                                             i!=files.end();++i) {
        if((i->type != DataPoint::FileInfo::file_type_unknown) && 
           (i->type != DataPoint::FileInfo::file_type_file)) continue;
        if(verbose) olog<<"Name: "<<i->name<<std::endl;
        std::string s_url(source_url.c_str());
        std::string d_url(destination_url.c_str());
        s_url+=i->name; d_url+=i->name;
        odlog(INFO)<<"Source: "<<s_url<<std::endl;
        odlog(INFO)<<"Destination: "<<d_url<<std::endl;
        DataPoint source(s_url.c_str());
        DataPoint destination(d_url.c_str());
        if(!source) {
          olog<<"Unsupported source url "<<s_url<<std::endl; continue;
        };
        if(!destination) {
          olog<<"Unsupported destination url "<<d_url<<std::endl; continue;
        };
        DataMove mover;
        mover.secure(secure);
        mover.passive(passive);
        mover.verbose(verbose);
        mover.force_to_meta(force_meta);
        if(tries) {
          mover.retry(true); // go through all locations
          source.tries(tries); // try all locations "tries" times
          destination.tries(tries);
        };
        DataCache cache(cache_path,cache_data_path,NULL,id,getuid(),getgid());
        std::string failure;
        if(mover.Transfer(source,destination,cache,UrlMap(),
                          0,0,0,timeout,&failure) != DataMove::success) {
          if(failure.length()) {
            olog<<"Current transfer FAILED: "<<failure<<std::endl;
          } else {
            olog<<"Current transfer FAILED."<<std::endl;
          };
          destination.tries(1);
          // It is not clear how to clear half-registered file. So remove it 
          // only in case of explicit destination.
          if(!(destination.meta())) mover.Delete(destination);
          failures=true;
        } else {
          if(verbose) std::cout<<"Current transfer complete."<<std::endl;
        };
      };
      if(failures) throw ARCCLIDataError("Some transfers failed");
      // Go deeper if allowed
      if(recursion > 0) {
        // Handle directories recursively
        for(std::list<DataPoint::FileInfo>::iterator i = files.begin();
                                               i!=files.end();++i) {
          if(i->type != DataPoint::FileInfo::file_type_dir) continue;
          if(verbose) olog<<"Directory: "<<i->name<<std::endl;
          std::string s_url(source_url.c_str());
          std::string d_url(destination_url.c_str());
          s_url+=i->name; d_url+=i->name;
          s_url+="/"; d_url+="/";
          arccp(s_url,d_url,
                secure,passive,force_meta,recursion-1,verbose,timeout);
        };
      };
      return;
    };
  };

  DataPoint source(source_url.c_str());
  DataPoint destination(destination_url.c_str());
  if(!source) throw ARCCLIDataError("Unsupported source url");
  if(!destination) throw ARCCLIDataError("Unsupported destination url");

  DataMove mover;
  mover.secure(secure);
  mover.passive(passive);
  mover.verbose(verbose);
  mover.force_to_meta(force_meta);
  if(tries) { // 0 means default behavior
    mover.retry(true); // go through all locations
    source.tries(tries); // try all locations "tries" times
    destination.tries(tries);
  };
  DataCache cache(cache_path,cache_data_path,NULL,id,getuid(),getgid());
  if(verbose && (LogTime::Level() <= FATAL)) mover.set_progress_indicator(&progress);
  std::string failure;
  if(mover.Transfer(source,destination,cache,UrlMap(),
                    0,0,0,timeout,&failure) != DataMove::success) {
    if(failure.length()) {
      throw ARCCLIDataError("Transfer FAILED: "+failure);
    } else {
      throw ARCCLIDataError("Transfer FAILED.");
    };
    destination.tries(1);
    // It is not clear how to clear half-registered file. So remove it only
    // in case of explicit destination.
    if(!(destination.meta())) mover.Delete(destination);
  };
  if(verbose) std::cout<<"Transfer complete."<<std::endl;
  return;
}

extern "C"
int ngregisterxx (
    const std::string& source_url,
    const std::string& destination_url,
    bool secure,
    bool passive,
    bool force_meta,
    int verbosity_level,
    int timeout
) {
  try {
    SetNotifyLevel(NotifyLevel(FATAL+verbosity_level));
    arcregister(source_url,destination_url,
           secure,passive,force_meta,timeout);
  } catch(ARCCLIDataError& e) {
    return -1;
  } catch(std::exception& e) {
    return -1;
  };
  return 0;
}

extern "C"
int ngcpxx (
    const std::string& source_url,
    const std::string& destination_url,
    bool secure,
    bool passive,
    bool force_meta,
    int verbosity_level,
    int timeout
) {
  try {
    SetNotifyLevel(NotifyLevel(FATAL+verbosity_level));
    arccp (source_url,destination_url,
           secure,passive,force_meta,0,timeout);
  } catch(ARCCLIDataError& e) {
    return -1;
  } catch(std::exception& e) {
    return -1;
  };
  return 0;
}
