#include "../../std.h"
#include "../../misc/inttostring.h"
#include "../../misc/log_time.h"
#include "service_soap.h"
#include <iostream>

//struct Namespace namespaces[] =
//{
//  {"SOAP-ENV", "http://schemas.xmlsoap.org/soap/envelope/", "http://www.w3.org/*/soap-envelope"},
//  {"SOAP-ENC", "http://schemas.xmlsoap.org/soap/encoding/", "http://www.w3.org/*/soap-encoding"},
//  {"xsi", "http://www.w3.org/2001/XMLSchema-instance", "http://www.w3.org/*/XMLSchema-instance"},
//  {"xsd", "http://www.w3.org/2001/XMLSchema", "http://www.w3.org/*/XMLSchema"},
//  {"dummy", "http://www.dummy.org"},
//  {NULL, NULL}
//};

/* callback functions */ 
int HTTP_ServiceAdv::soap_fsend(struct soap *sp, const char* buf, size_t l) {
  HTTP_ServiceAdv *it = (HTTP_ServiceAdv*)(sp->user);
  if(it->ignore_soap_output) return SOAP_OK; /* hide gsoap response */
//  if(it->c->write(buf,l)) return SOAP_SSL_ERROR; 
  size_t fbuf_left = sizeof(it->soap_fbuf)-it->soap_fbuf_n;
  if(l<fbuf_left) {
    memcpy(it->soap_fbuf+it->soap_fbuf_n,buf,l);
    it->soap_fbuf_n+=l;
    return SOAP_OK;
  };
  if(it->soap_fbuf_n) {
    memcpy(it->soap_fbuf+it->soap_fbuf_n,buf,fbuf_left);
    buf+=fbuf_left; l-=fbuf_left;
    if(it->c->write(it->soap_fbuf,sizeof(it->soap_fbuf))) return SOAP_SSL_ERROR; 
    it->soap_fbuf_n=0;
  };
  if(l>=sizeof(it->soap_fbuf)) {
    if(it->c->write(buf,l)) return SOAP_SSL_ERROR; 
    return SOAP_OK;
  }; 
  memcpy(it->soap_fbuf,buf,l); it->soap_fbuf_n=l;
  return SOAP_OK;
}

// call it after server sent full SOAP response or do not use soap_fbuf at all
int HTTP_ServiceAdv::soap_flush(struct soap *sp) {
  HTTP_ServiceAdv *it = (HTTP_ServiceAdv*)(sp->user);
  if(it->ignore_soap_output) return SOAP_OK; /* hide gsoap response */
  if(it->soap_fbuf_n == 0) return SOAP_OK;
  if(it->c->write(it->soap_fbuf,it->soap_fbuf_n)) {
    it->soap_fbuf_n=0;
    return SOAP_SSL_ERROR; 
  };
  it->soap_fbuf_n=0;
  return SOAP_OK;
}

size_t HTTP_ServiceAdv::soap_frecv(struct soap* sp, char* buf, size_t l) {
  HTTP_ServiceAdv *it = (HTTP_ServiceAdv*)(sp->user);
  soap_flush(sp);
  int ll = it->c->read(buf,l);
  return ll;
}

// following two are not need, but to make symmetric ...
int HTTP_ServiceAdv::soap_fopen(struct soap* sp, const char*, const char*, int) {
  return SOAP_OK;
}

int HTTP_ServiceAdv::soap_fclose(struct soap* sp) {
  return SOAP_OK;
}

int HTTP_ServiceAdv::soap_parse(struct soap *sp) {
  HTTP_ServiceAdv *it = (HTTP_ServiceAdv*)(sp->user);
  char header[SOAP_HDRLEN], *s;
  *sp->endpoint = '\0';
  sp->length = 0;
  /* read rest of header */
  for (;;) {
    if (soap_getline(sp,header,SOAP_HDRLEN)) return SOAP_EOF;
    if (!*header) break; // empty line - end of header
    s=strchr(header, ':'); // parse line to key:value
    if (s) { *s = '\0'; do s++; while ((*s) && isspace(*s)); };
    sp->fparsehdr(sp,header,s);
  };
  // ????????????????????????????? 
  if(sp->keep_alive < 0) { sp->keep_alive=sp->keep_alive; }
  else if(sp->keep_alive == 1) { sp->keep_alive=0; }; /* HTTP/1.0 without keep_alive */
  /* TODO fill soap->path, soap->endpoint, etc. -  is it really needed ?
     maybe better do not use gSOAP's HTTP headers at all */
  return SOAP_OK;
}

void HTTP_ServiceAdv::soap_init(void) {
  /* SOAP */
  ignore_soap_output=false;
  ::soap_init(&sp);
  sp.user=this;
  sp.fopen=&soap_fopen;
  sp.fclose=&soap_fclose;
  sp.fsend=&soap_fsend;
  sp.frecv=&soap_frecv;
  sp.fparse=&soap_parse;
  sp.keep_alive=1;
  soap_set_imode(&sp,SOAP_IO_KEEPALIVE);
  soap_set_omode(&sp,SOAP_IO_KEEPALIVE);
}

void HTTP_ServiceAdv::soap_deinit(void) {
  soap_destroy(&sp);
  soap_end(&sp);
  soap_done(&sp);
}

/*
 * HTTP methods
 */
HTTP_Error HTTP_ServiceAdv::soap_post(const char* uri,int &keep_alive) {
  ignore_soap_output=false;
  /* keep_alive is inverted by parseheader */
  sp.keep_alive=keep_alive; /* 0,1,2 */
#ifdef OLD_GSOAP
//  soap_serve(&sp);
  soap_begin(&sp);
  if(soap_begin_recv(&sp) || 
     soap_envelope_begin_in(&sp) || 
     soap_recv_header(&sp) || 
     soap_body_begin_in(&sp)
    ) {
    soap_send_fault(&sp);
    soap_flush(&sp);
    keep_alive=0;
    soap_destroy(&sp);  // delete temporary classes
    soap_end(&sp);      // delete temporary data 
    return HTTP_FAILURE;
  } else {
    soap_methods();
    if (sp.error) {
      soap_send_fault(&sp);
    };
  };
  soap_flush(&sp);
#else
//  soap_serve_once(&sp);
  soap_begin(&sp);
  if(soap_begin_recv(&sp) ||
     soap_envelope_begin_in(&sp) ||
     soap_recv_header(&sp) ||
     soap_body_begin_in(&sp) ||
     soap_peek_element(&sp)
    ) {
    soap_fsend(&sp,"HTTP/1.1 200 OK\r\n",17);
    soap_send_fault(&sp);
    soap_flush(&sp);
    keep_alive=0;
    soap_destroy(&sp);  // delete temporary classes
    soap_end(&sp);      // delete temporary data
    return HTTP_FAILURE;
  } else {
    soap_fsend(&sp,"HTTP/1.1 200 OK\r\n",17);
    soap_methods();
    if (sp.error) {
      soap_send_fault(&sp);
    };
  };
  soap_flush(&sp);
#endif
  if(sp.keep_alive==0) keep_alive=0;
  soap_destroy(&sp);  // delete temporary classes
  soap_end(&sp);      // delete temporary data 
  ignore_soap_output=false;
  return HTTP_OK;
}

void HTTP_ServiceAdv::soap_methods(void) { // dummy service with no methods
  odlog(ERROR)<<"virtual soap_methods - running nonimplemented service ?"<<std::endl;
  sp.error = SOAP_NO_METHOD;
}

HTTP_Error HTTP_ServiceAdv::post(const char* uri,int &keep_alive) {
  HTTP_Error r = soap_post(uri,keep_alive);
  return r;
}

HTTP_Error HTTP_ServiceData::parse_header(int &keep_alive) {
  if(!c) return HTTP_ERROR;
  uint64_t range_start_,range_end_;
  char buf[1024], *s;
  unsupported_option_passed=false;
  failure_parsing=false;
  range_passed=false;
  entity_range_passed=false;
  entity_size_passed=false;
  entity_last_modified_passed=false;
  entity_expires_passed=false;
  bool keep_alive_passed=false;
  nranges=0;
  for (;;) {
    if(c->readline(buf,1023) == 0) return HTTP_ERROR; /* connection broken */
    buf[1023]=0; odlog(VERBOSE)<<"Request header: "<<buf<<std::endl;
    if(buf[0] == 0) break; /* empty line - end of header */
    char* token = buf; for(;*token;token++) if(isspace(*token)) break;
    int l = token-buf;
    for(;*token;token++) if(!isspace(*token)) break;
    if(strncasecmp("Connection:",buf,l) == 0) {
      if(strcasecmp("close",token) == 0) { keep_alive=0; }
      else if(strcasecmp("keep-alive",token) == 0) { keep_alive=2; };
    } else if(strncasecmp("Content-Length:",buf,l) == 0) {
      char *e;
      length=strtoull(token,&e,10);
      if((*e) == 0) length_passed=true;
    } else if(strncasecmp("Range:",buf,l) == 0) {
      char* p = token; for(;*p;p++) if(*p == '=') break;
      if(strncasecmp("bytes",token,p-token) == 0) {
        if(*p) p++;
        for(;*p;) {
          if(*p != '-') {
            char *e;
            range_start_=strtoull(p,&e,10);
            if((*e) == '-') {
              p=e+1; range_end_=strtoull(p,&e,10); p=e;
              if(((*e) == ',') || ((*e) == 0)) {
                if(range_start_ > range_end_) {
                  failure_parsing=true;
                } else {
                  if(nranges<MAX_RANGES) {
                    range_passed=true;
                    range_start[nranges]=range_start_;
                    range_end[nranges]=range_end_;
                    nranges++;
                  } else { failure_parsing=true; };
                };
              } else { failure_parsing=true; };
            } else { failure_parsing=true; };
          } else {
            char *e;
            range_end_=0; p++;
            range_start_=strtoull(p,&e,10); p=e;
            if(((*e) == ',') || ((*e) == 0)) {
              if(range_start_==0) { failure_parsing=true; }
              else {
                range_passed=true;
                if(nranges<MAX_RANGES) {
                  range_start[nranges]=range_start_;
                  range_end[nranges]=range_end_;
                  nranges++;
                } else { failure_parsing=true; };
              };
            } else { failure_parsing=true; };
          };
          if(failure_parsing) break;
          if((*p) == 0) break;
          p++;
        };
      } else { unsupported_option_passed=true; };
    } else if(strncasecmp("Content-Range:",buf,l) == 0) {
      char* p = token; for(;*p;p++) if(isspace(*p)) break;
      if(strncasecmp("bytes",token,p-token) == 0) {
        for(;*p;p++) if(!isspace(*p)) break;
        char *e;
        range_start_=strtoull(p,&e,10);
        if((*e) == '-') {
          p=e+1; range_end_=strtoull(p,&e,10); p=e;
          if(((*e) == '/') || ((*e) == 0)) {
            if(range_start_ <= range_end_) {
              entity_range_passed=true;
              entity_range_start=range_start_;
              entity_range_end=range_end_;
            } else { failure_parsing=true; };
            if((*p) == '/') {
              p++; entity_size=strtoull(p,&e,10);
              if((*e) == 0) { entity_size_passed=true; }
              else { failure_parsing=true; };
            };
          } else { failure_parsing=true; };
        } else { failure_parsing=true; };
      } else { unsupported_option_passed=true; };
    } else if(strncasecmp("Expires:",buf,l) == 0) {
      HTTP_Time t(token);
      if(t) {
        entity_expires_passed=true;
        entity_expires=t;
      };
    } else if(strncasecmp("Last-Modified:",buf,l) == 0) {
      HTTP_Time t(token);
      if(t) {
        entity_last_modified_passed=true;
        entity_last_modified=t;
      };
    } else if(strncasecmp("Trailer:",buf,l) == 0) {
      unsupported_option_passed=true;
    } else if(strncasecmp("Transfer-Encoding:",buf,l) == 0) {
      unsupported_option_passed=true;
    } else if(strncasecmp("Content-Encoding:",buf,l) == 0) {
      unsupported_option_passed=true;
    } else if(strncasecmp("If-Modified-Since:",buf,l) == 0) {
      unsupported_option_passed=true;
    } else if(strncasecmp("If-Unmodified-Since:",buf,l) == 0) {
      unsupported_option_passed=true;
    } else if(strncasecmp("If-None-Match:",buf,l) == 0) {
      unsupported_option_passed=true;
    } else { // skip other options

    };
  };
  if(!length_passed) {
    if(keep_alive==1) keep_alive=0; // only supported method is Content-Length
         // otherwise (1.1 or Connection:) assume there is no body
  };
  return HTTP_OK;
}

// Report error or result without body
HTTP_Error HTTP_ServiceData::send_header(int &keep_alive,int code) {
  std::string header = "HTTP/1.1 "+inttostring(code);
  switch(code) {
    case 200: header+=" OK\r\n"; break;
    case 400: header+=" Bad Request\r\n"; break;
    case 404: header+=" Not Found\r\n"; break;
    case 406: header+=" Not Acceptable\r\n"; break;
    case 500: header+=" Internal Server Error\r\n"; break;
    default: header+=" unknown\r\n"; break;
  };
  if(keep_alive) {
    header+="Connection: Keep-Alive\r\n";
  } else {
    header+="Connection: close\r\n";
  };
  header+="Content-Length: 0\r\n";
  header+="\r\n";
  odlog(VERBOSE)<<"Response header:\n"<<header<<std::endl;
  return (c->write(header.c_str(),header.length()) == 0 ? HTTP_OK : HTTP_ERROR );
}

// Data response
HTTP_Error HTTP_ServiceData::send_header(int &keep_alive,uint64_t start,uint64_t end,bool partial,uint64_t full_size,const HTTP_Time& expires,const HTTP_Time& last_modified) {
  std::string header = "HTTP/1.1 ";
  if(partial) { header+="206 Partial content\r\n"; } else { header+="200 OK\r\n"; };
  if(keep_alive) {
    header+="Connection: Keep-Alive\r\n";
  } else {
    header+="Connection: close\r\n";
  };
  uint64_t size = end-start;
  header+="Content-Range: bytes "+inttostring(start)+"-"+inttostring(end-1);
  if(full_size) { header+="/"+inttostring(full_size); };
  header+="\r\n";
  header+="Content-Length: "+inttostring(size)+"\r\n";
  if(expires) { header+="Expires: "; header+=expires.Str(); header+="\r\n"; };
  if(last_modified) { header+="Last-Modified: "; header+=last_modified.Str(); header+="\r\n"; };
  header+="\r\n";
  odlog(VERBOSE)<<"Response header:\n"<<header<<std::endl;
  return (c->write(header.c_str(),header.length()) == 0 ? HTTP_OK : HTTP_ERROR );
}

void HTTP_ServiceAdv::add_namespaces(struct Namespace* namespaces_) {
  if(!namespaces_) return;
  int n = 0;
  int n_ = 0;
  struct Namespace *ns;
  if(namespaces) for(ns=namespaces;ns->id;ns++) { n++; };
  for(ns=namespaces_;ns->id;ns++) { n_++; };
  ns=(struct Namespace*)realloc(namespaces,sizeof(struct Namespace)*(n+n_+1));
  if(ns == NULL) return;
  memcpy(ns+n,namespaces_,sizeof(struct Namespace)*(n_+1));
  sp.namespaces=(namespaces=ns);
}

HTTP_ServiceAdv::~HTTP_ServiceAdv(void) {
  if(namespaces) free(namespaces);
};

