#ifndef __CONNECTOR_H__
#define __CONNECTOR_H__

#include <globus_io.h>
#include <openssl/ssl.h>
#include <list>
#include <string>

#include "../../auth/auth.h"

class HTTP_Service;
class HTTP_Services;
//class AuthEvaluator;

class HTTP_Connector {
 protected:
  bool initialized;
  std::string base_url;
  // Identity of remote
  AuthUser user;
  // confugured authoziration rules
  std::list<AuthEvaluator*>& auths;
  // configured VOs
  std::list<AuthVO>& vos;
  // configured services
  HTTP_Services& services;
  /// Thread function which calls loop() and destroys object on exit.
  /// Called by loop_in_thread through creation of new thread.
  static void* loop_thread(void*);
 public:
  unsigned int pid;
  /// Constructor
  /// @arg url - URL to which this connector is bound
  /// @arg serv - services to be served by this connector
  /// @arg auths - authorization rules to be processed
  /// @arg vos - VOs to be matched
  HTTP_Connector(const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_Connector(void);
  operator bool(void) { return initialized; };
  virtual size_t read(char* buf, size_t l);
  virtual int write(const char* buf, size_t l);
  virtual size_t readline(char* buf, size_t l);
//  bool add_service(const char *url,HTTP_Service *service);
//  bool remove_service(const char *url);
  /// Called by loop_thread before loop(). loop_thread assumes user's 
  /// identity was already established after loop_start() finished.
  virtual void loop_start(void);
  /// Called by loop_thread second (main function for processing requests)
  virtual void loop(void);
  /// Called by loop_thread after loop()
  virtual void loop_end(void);
  /// Start new thread for processing requests. Thread function is loop_thread()
  void loop_in_thread(void);
  /// Get remote subject (deprecated)
  const char* identity_subject(void) const { return user.DN(); };
  /// Get delegated proxy (deprecated)
  const char* identity_proxy(void) const { return user.proxy(); };
  /// Get remote identity
  AuthUser& identity(void) { return user; };
  /// Get authorization rules
  std::list<AuthEvaluator*>& authorizations(void) { return auths; };
  std::list<AuthVO>& VOs(void) { return vos; };
  const char* url(void) { return base_url.c_str(); };
  int send_file(const char* fname);
  int send_response_header(int keep_alive,int code,char* type,int size);
  int send_error_response(int keep_alive,int code,char* type,char* content);
  int skip_request(int &keep_alive);
  int skip_header(int &keep_alive);
};

class HTTP_Globus_Connector: public HTTP_Connector {
 private:
  /// handle for data transfer
  globus_io_handle_t* h;
  /// Buffer is used because globus_io_read is heavy
  char r_buf[65536]; // better if it is enough for whole header
  unsigned int r_start;
  unsigned int r_end;
  // static void* loop_thread(void*);
 public:
  HTTP_Globus_Connector(globus_io_handle_t *s,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_Globus_Connector(void);
  virtual size_t read(char* buf, size_t l);
  virtual int write(const char* buf, size_t l);
  virtual size_t readline(char* buf, size_t l);
  virtual void loop_start(void);
  void identity(globus_io_handle_t* handle,const char* subject);
};

int stat_file(const char* fname,unsigned long long int &size);

class HTTP_SSL_Connector: public HTTP_Connector {
 protected:
  int s;
  SSL* con;
 public:
  HTTP_SSL_Connector(SSL_CTX *ctx,int socket,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_SSL_Connector(void);
  virtual size_t read(char* buf, size_t l);
  virtual int write(const char* buf, size_t l);
  virtual void loop_start(void);
};

class HTTP_Plain_Connector: public HTTP_Connector {
 protected:
  int s;
 public:
  HTTP_Plain_Connector(int socket,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_Plain_Connector(void);
  virtual size_t read(char* buf, size_t l);
  virtual int write(const char* buf, size_t l);
  virtual void loop_start(void);
};

class HTTP_GSSAPI_Connector: public HTTP_Plain_Connector {
 protected:
  gss_ctx_id_t context;
  gss_cred_id_t server_credentials;
  gss_buffer_desc data_tok;
  int data_tok_offset;
  int read_SSL_token(void** val);
 public:
  HTTP_GSSAPI_Connector(int socket,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_GSSAPI_Connector(void);
  virtual size_t read(char* buf, size_t l);
  virtual int write(const char* buf, size_t l);
  virtual void loop_start(void);
};

class HTTP_Listener {
 protected:
  std::string base_url;
  std::list<AuthEvaluator*>& auths;
  std::list<AuthVO>& vos;
  HTTP_Services& services;
  bool initialized;
  int (*cb)(HTTP_Listener& l,void* arg);
 public:
  HTTP_Listener(const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths_,std::list<AuthVO>& vos_):cb(NULL),initialized(false),services(serv),auths(auths_),vos(vos_) {
    if(url) base_url=url;
  };
  virtual ~HTTP_Listener() { };
  /// This method must be called on each new connection. Default
  /// behavior is to call callback if defined.
  /// Either rewrite this method or set callback.
  virtual void connect(void* arg) { if(cb) cb(*this,arg); };
  /// Set callback
  void callback(int (*cb_)(HTTP_Listener& l,void* arg)) { cb=cb_; };
  operator bool(void) { return initialized; };
};

class HTTP_Globus_Listener: public HTTP_Listener {
 private:
  class cred_info_t {
   public:
    globus_io_handle_t* h;
    std::string subject;
    cred_info_t(globus_io_handle_t* h_,const char* subject_):h(h_),subject(subject_) { };
  };
  std::list<cred_info_t> cons;
  pthread_mutex_t cons_lock;
 protected:
  unsigned short port_num;
  globus_io_handle_t s;
  static void listen_cb(void *callback_arg,globus_io_handle_t *handle,globus_result_t res);
  static void accept_cb(void *callback_arg,globus_io_handle_t *handle,globus_result_t result);
#if GLOBUS_IO_VERSION<4
  static globus_bool_t auth_cb(void* arg,globus_io_handle_t* h,globus_result_t result,char* identity,gss_ctx_id_t* context_handle);
#else
  static globus_bool_t auth_cb(void* arg,globus_io_handle_t* h,globus_result_t result,char* identity,gss_ctx_id_t context_handle);
#endif
 public:
  HTTP_Globus_Listener(int port,globus_io_attr_t* attr,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_Globus_Listener();
  int port(void) { return port_num; };
  virtual void connect(void* arg);
};

class HTTP_SSL_Listener: public HTTP_Listener {
 protected:
  unsigned short port_num;
  int s;
  SSL_CTX *ctx;
  static void* accept_thread(void*);
 public:
  HTTP_SSL_Listener(int port,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_SSL_Listener();
  int port(void) { return port_num; };
  virtual void connect(void* arg);
};


class HTTP_Plain_Listener: public HTTP_Listener {
 protected:
  unsigned short port_num;
  int s;
  static void* accept_thread(void*);
  bool new_connection(HTTP_Plain_Connector& c,int h);
 public:
  HTTP_Plain_Listener(int port,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_Plain_Listener();
  int port(void) { return port_num; };
  virtual void connect(void* arg);
};

class HTTP_GSSAPI_Listener: public HTTP_Plain_Listener {
 protected:
 public:
  HTTP_GSSAPI_Listener(int port,const char* url,HTTP_Services& serv,std::list<AuthEvaluator*>& auths,std::list<AuthVO>& vos);
  virtual ~HTTP_GSSAPI_Listener();
  virtual void connect(void* arg);
};

#endif /* __CONNECTOR_H__ */
