#include "../../std.h"

#ifdef HAVE_GLOBUS_RLS_CLIENT_H

#include <vector>

extern "C" {
#include <globus_rls_client.h>
#include <globus_rls_rpc.h>
};

#include "../../misc/time_utils.h"
#include "../../misc/inttostring.h"
#include "../../misc/escaped.h"
#include "../../misc/log_time.h"

#include "se_ns_lrc.h"


SENameServerLRC::SENameServerLRC(const char* contact,const char* se_url):SENameServer(contact,se_url),valid(false),delta(60*60) {
  for(;;) {
    std::string url;
    int n = input_escaped_string(contact,url);
    if(n==0) break;
    urls.push_back(url);
    contact+=n;
  };
  valid=true;
}

SENameServerLRC::~SENameServerLRC(void) {

}

int SENameServerLRC::Get(SEAttributes& file) {
  if(!valid) return -1;
  return 0;
}

int SENameServerLRC::Maintain(void) {
  if(!valid) return -1;
  if(!delta.test()) return 0;
  std::list<std::string>::iterator u = urls.begin();
  int registered = 0;
  for(;u!=urls.end();++u) {
    std::string url = *u;
    url.replace(0,3,"rls");
    std::string lfn("__storage_service__");
    std::string pfn(se_url());
    //std::string::size_type n = pfn.find(':'); if(n != std::string::npos) pfn.replace(0,n,"se");
    globus_result_t r = GLOBUS_SUCCESS;
    globus_rls_handle_t *h = NULL;
    char errbuf[MAXERRMSG];
    int rc;
    if((r=globus_rls_client_connect((char*)(url.c_str()),&h)) != 
                                            GLOBUS_SUCCESS) goto error1;
    odlog(VERBOSE)<<"Mapping: "<<lfn<<" -> "<<pfn<<std::endl;
    if((r=globus_rls_client_lrc_create(h,(char*)(lfn.c_str()),
                                  (char*)(pfn.c_str()))) != GLOBUS_SUCCESS){
      globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
      if(rc == GLOBUS_RLS_LFN_EXIST) { // have lfn - try to add pfn only
        if((r=globus_rls_client_lrc_add(h,(char*)(lfn.c_str()),
                                   (char*)(pfn.c_str()))) != GLOBUS_SUCCESS) {
          globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
          if(rc != GLOBUS_RLS_MAPPING_EXIST) goto error2;
        };
      } else if(rc != GLOBUS_RLS_MAPPING_EXIST) goto error2;
    };
    globus_rls_client_close(h);
    registered++;
    continue;
error1:
    globus_rls_client_error_info(r,NULL,errbuf,sizeof(errbuf),GLOBUS_FALSE);
error2:
    odlog(DEBUG)<<"SENameServerLRC::Maintain failed ("<<url<<"): "<<errbuf<<std::endl;
    globus_rls_client_close(h);
    continue;
  };
  if(registered) return 0;
  odlog(ERROR)<<"SENameServerLRC::Maintain failed: none site accepted registration"<<std::endl;
  delta.reset();
  return -1;
}


static int compare_attr(globus_rls_handle_t *h,const char* key,const char* name,const char* value) {
  globus_result_t r = GLOBUS_SUCCESS;
  globus_list_t* attr_list = GLOBUS_NULL;

  r=globus_rls_client_lrc_attr_value_get(h,(char*)key,(char*)name,
          globus_rls_obj_lrc_lfn,&attr_list);
  if(r != GLOBUS_SUCCESS) {
    int rc;
    char errbuf[MAXERRMSG];
    globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
    if(rc == GLOBUS_RLS_ATTR_NEXIST) return 1;
    odlog(ERROR)<<"attr_value_get failed: "<<errbuf<<std::endl;
    return -1;
  };
  if(attr_list == GLOBUS_NULL) return 1;
  // it should be only one value - take first
  globus_rls_attribute_t* attr =
                       (globus_rls_attribute_t*)globus_list_first(attr_list);
  if(attr->type != globus_rls_attr_type_str) { 
    globus_rls_client_free_list(attr_list);
    return -1;
  };
  const char* p = attr->val.s; for(;*p;p++) if(!isspace(*p)) break;
  if(strcmp(value,p) != 0) {
    odlog(ERROR)<<"compare_attr: mismatch: "<<name<<": "<<value<<" != "<<attr->val.s<<std::endl;
    globus_rls_client_free_list(attr_list);
    return -1;
  };
  globus_rls_client_free_list(attr_list);
  return 0;
}

static int add_attr(globus_rls_handle_t *h,const char* key,const char* name,const char* value) {
  globus_result_t r = GLOBUS_SUCCESS;
  globus_rls_attribute_t attr;
  int rc;
  char errbuf[MAXERRMSG];

  if((r=globus_rls_client_lrc_attr_create(h,(char*)name,
       globus_rls_obj_lrc_lfn,globus_rls_attr_type_str)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
    if(rc != GLOBUS_RLS_ATTR_EXIST) {
      odlog(ERROR)<<"attr_create failed: "<<errbuf<<std::endl;
      return -1;
    };
  };
  attr.objtype=globus_rls_obj_lrc_lfn;
  attr.type=globus_rls_attr_type_str;
  attr.name=(char*)name; attr.val.s=(char*)value; 
  r=globus_rls_client_lrc_attr_add(h,(char*)key,&attr);
  if(r != GLOBUS_SUCCESS){
    globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
    if(rc != GLOBUS_RLS_ATTR_EXIST) {
      odlog(ERROR)<<"attr_add failed: "<<errbuf<<std::endl;
      return -1;
    };
    rc=compare_attr(h,key,name,value);
    if(rc == -1) return -1;
  };
  return 0;
}

static int compare_size(globus_rls_handle_t *h,SEFile& file) {
  std::string s = inttostring(file.size());
  return compare_attr(h,file.id(),"size",s.c_str());
}

static int compare_checksum(globus_rls_handle_t *h,SEFile& file) {
  if(!file.checksum_available()) return 1;
  return compare_attr(h,file.id(),"checksum",file.checksum().c_str());
}

static int compare_created(globus_rls_handle_t *h,SEFile& file) {
  if(!file.created_available()) return 1;
  std::string s(""); timetostring(*file.created(),s);
  return compare_attr(h,file.id(),"created",s.c_str());
}
static int add_size(globus_rls_handle_t *h,SEFile& file) {
  std::string s = inttostring(file.size());
  return add_attr(h,file.id(),"size",s.c_str());
}

static int add_checksum(globus_rls_handle_t *h,SEFile& file) {
  if(!file.checksum_available()) return 0;
  return add_attr(h,file.id(),"checksum",file.checksum().c_str());
}

static int add_created(globus_rls_handle_t *h,SEFile& file) {
  if(!file.created_available()) return 0;
  std::string s(""); timetostring(*file.created(),s);
  return add_attr(h,file.id(),"created",s.c_str());
}

int SENameServerLRC::Register(SEFile& file,bool require_unique) {
  if(!valid) return -1;
  std::string pfn = se_url();
  //std::string::size_type n = pfn.find(':');
  //if(n == string::npos) {
  //  odlog(ERROR)<<"No protocol in url: "<<pfn<<std::endl;
  //  return -1;
  //};
  //pfn.replace(0,n,"se");
  pfn+="?"; pfn+=file.id();
  std::vector<std::string> pfns;
  for(int n = 0;;n++) {
    const char* p = access_url(n);
    if(p == NULL) break;
    std::string p_s(p); p_s+="/"; p_s+=file.id(); pfns.push_back(p_s);
  };
  //pfns.push_back(pfn);
  int registered = 0;
  for(std::list<std::string>::iterator u = urls.begin();u!=urls.end();++u) {
    std::string url = *u;
    url.replace(0,3,"rls");

    globus_result_t r = GLOBUS_SUCCESS;
    globus_rls_handle_t *h = NULL;
    char errbuf[MAXERRMSG];
    int rc;

    if((r=globus_rls_client_connect((char*)(url.c_str()),&h)) != GLOBUS_SUCCESS)
       goto error1;
    odlog(VERBOSE)<<"Mapping: "<<file.id()<<" -> "<<pfn<<std::endl;
    r=globus_rls_client_lrc_create(h,(char*)(file.id()),(char*)(pfn.c_str()));
    if(r != GLOBUS_SUCCESS){
      globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
      if(rc == GLOBUS_RLS_LFN_EXIST) { // have lfn - try to add pfn only
        r=globus_rls_client_lrc_add(h,(char*)(file.id()),(char*)(pfn.c_str()));
        if(r != GLOBUS_SUCCESS) {
          globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
          if(rc != GLOBUS_RLS_MAPPING_EXIST) goto error2;
        };
      } else if(rc != GLOBUS_RLS_MAPPING_EXIST) goto error2;
    };
    // store direct pfns
    for(std::vector<std::string>::iterator p = pfns.begin();p!=pfns.end();++p) {
      r=globus_rls_client_lrc_add(h,(char*)(file.id()),(char*)(p->c_str()));
      if(r != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
        //if(rc != GLOBUS_RLS_MAPPING_EXIST) goto error2;
      };
    };
    // store metadata
    rc=add_size(h,file); if(rc==-1) goto error3;
    rc=add_checksum(h,file); if(rc==-1) goto error3;
    rc=add_created(h,file); if(rc==-1) goto error3;
    registered++;
    globus_rls_client_close(h);
    continue;
error3:
    odlog(ERROR)<<"SENameServerLRC::Register failed"<<std::endl;
    if(h) globus_rls_client_close(h);
    continue;
error2:
    odlog(ERROR)<<"SENameServerLRC::Register failed: "<<errbuf<<std::endl;
    if(h) globus_rls_client_close(h);
    continue;
error1:
    if(r != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(r,NULL,errbuf,sizeof(errbuf),GLOBUS_FALSE);
      odlog(ERROR)<<"SENameServerLRC::Register failed: "<<errbuf<<std::endl;
    };
    if(h) globus_rls_client_close(h);
    continue;
  };
  if(registered) {
    file.state_reg(REG_STATE_ANNOUNCED);
    return 0;
  };
  return -1;
}

int SENameServerLRC::Unregister(SEFile& file,bool allow_last) {
  if(!valid) return -1;
  std::string pfn = se_url();
  //std::string::size_type n = pfn.find(':');
  //if(n == std::string::npos) {
  //  odlog(ERROR)<<"No protocol in url: "<<pfn<<std::endl;
  //  return -1;
  //};
  //pfn.replace(0,n,"se");
  pfn+="?"; pfn+=file.id();
  std::vector<std::string> pfns;
  for(int n = 0;;n++) {
    const char* p = access_url(n);
    if(p == NULL) break;
    std::string p_s(p); p_s+="/"; p_s+=file.id(); pfns.push_back(p_s);
  };
  int registered = 0;
  std::list<std::string>::iterator u = urls.begin();
  for(;u!=urls.end();++u) {
    std::string url = *u;
    url.replace(0,3,"rls");

    globus_result_t r = GLOBUS_SUCCESS;
    globus_rls_handle_t *h = NULL;
    char errbuf[MAXERRMSG];
    int rc;

    if((r=globus_rls_client_connect((char*)(url.c_str()),&h)) != GLOBUS_SUCCESS)
       goto error1;

    r=globus_rls_client_lrc_delete(h,(char*)(file.id()),(char*)(pfn.c_str()));
    if(r != GLOBUS_SUCCESS){
      globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
      if((rc != GLOBUS_RLS_LFN_NEXIST) && 
         (rc != GLOBUS_RLS_MAPPING_NEXIST) && 
         (rc != GLOBUS_RLS_PFN_NEXIST)) goto error2; 
    };
    for(std::vector<std::string>::iterator p = pfns.begin();p!=pfns.end();++p) {
      r=globus_rls_client_lrc_delete(h,(char*)(file.id()),(char*)(p->c_str()));
      if(r != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(r,&rc,errbuf,sizeof(errbuf),GLOBUS_FALSE);
        if((rc != GLOBUS_RLS_LFN_NEXIST) &&
           (rc != GLOBUS_RLS_MAPPING_NEXIST) &&
           (rc != GLOBUS_RLS_PFN_NEXIST)) goto error2;
      };
    };
    globus_rls_client_close(h);
    continue;
error2:
    registered++;
    odlog(ERROR)<<"SENameServerLRC::Unregister failed: "<<errbuf<<std::endl;
    if(h) globus_rls_client_close(h);
    continue;
error1:
    registered++;
    if(r != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(r,NULL,errbuf,sizeof(errbuf),GLOBUS_FALSE);
      odlog(ERROR)<<"SENameServerLRC::Unregister failed: "<<errbuf<<std::endl;
    };
    if(h) globus_rls_client_close(h);
    continue;
  };
  if(!registered) return 0;
  return -1;
}

int SENameServerLRC::Modify(SEFile& file) {
  return -1;
}

std::string SENameServerLRC::url(SEAttributes& file) {
  std::string url = contact();
  url.replace(0,3,"rls");
  url+="/"; url+=file.id();
  return url;  
};

#endif
