#include "../../std.h"
#include <string>
#include <list>
#include <iostream>
#include <regex.h>
#include <sys/timeb.h>

#include "../../misc/escaped.h"
#include "../../misc/stringtoint.h"
#include "../../misc/thread.h"
#include "../../misc/time_utils.h"
#include "../../misc/log_time.h"
#include "../server/httpsd.h"
#include "files.h"
#include "se_files.h"
#include "se_ns.h"
#include "se_replicator.h"
#include "se_registrator.h"
#include "se_collector.h"

// TODO: use special pointers instead of acquire/release


// one more hack
#define WITH_NOGLOBAL
#define SOAP_FMAC3 static
#include "file_soapC.cpp"
#include "file_soapClient.cpp"
#include "file_soapServer.cpp"

#define __SE_INTERNAL_INCLUDE__
#include "../se.h"

//#include "file_soap.nsmap"
extern SOAP_NMAC struct Namespace file_soap_namespaces[];

#define handleop_showincomplete 1
#define handleop_userdownloads  2
#define handleop_userregisters  4

static bool merge_urls(std::string &cfg_url,const char* contact_base) {
  if((cfg_url.length() == 0) || (cfg_url[0] == '/')) { // path
    const char* u_ = contact_base;
    u_=strchr(u_,':'); if(u_==NULL) return false; // bad url
    u_++; if((*u_) != '/') return false; // bad url
    u_++; if((*u_) != '/') return false; // bad url
    u_++; u_=strchr(u_,'/'); if(u_==NULL) u_=contact_base+strlen(contact_base);
    std::string url(contact_base,u_-contact_base);
    cfg_url=url+cfg_url;
  } else if(cfg_url[0] == ':') { // port+path
    const char* u_ = contact_base;
    u_=strchr(u_,':'); if(u_==NULL) return false; // bad url
    u_++; if((*u_) != '/') return false; // bad url
    u_++; if((*u_) != '/') return false; // bad url
    u_++; u_=strchr(u_,'/'); if(u_==NULL) u_=contact_base+strlen(contact_base);
    std::string::size_type p = cfg_url.find('/');
    if(p == std::string::npos) p=cfg_url.length();
    std::string url(contact_base,u_-contact_base);
    url+=cfg_url.c_str()+p;
    cfg_url=url;
  } else { // url

  };
  return true;
}

class HTTP_SE_Handle {
 friend class HTTP_SE;
 friend bool se_service_configurator(std::istream& f,const char* uri,HTTP_Service_Properties &prop);
 friend HTTP_Service* se_service_creator(HTTP_Connector& c,const char* uri,void* arg);
 private:
  // URL used to announce available files
  std::string base_url;
  // URL/path for which service is configured
  std::string service_url;
  // files handled by this service
  SEFiles files;
  std::list<std::string> acl_create;
  std::list<std::string> acl_replicate;
  std::list<std::string> acl_read;
  int options;
  std::string access_ssl_url;
  std::string access_gsi_url;
 public:
  HTTP_SE_Handle(const char* dirpath,const char* service_url_,const char* base_url_,int opts):files(dirpath),service_url(service_url_),base_url(base_url_),options(opts) { };
  ~HTTP_SE_Handle(void) { };
  operator bool(void) { return (bool)(files); };
  bool register_immediately(void) { return files.register_immediately(); };
  bool register_retry(void) { return files.register_retry(); };
  SENameServer* NS(void) { return files.NS(); };
  bool showincomplete(void) { return((options & handleop_showincomplete)!=0); };
  bool userdownloads(void)  { return((options & handleop_userdownloads)!=0); };
  bool userregisters(void)  { return((options & handleop_userregisters)!=0); };
  const std::string& control_url(void) { return access_gsi_url; };
  const std::string& data_url(void) { return access_ssl_url; };
};

HTTP_Service* se_service_creator(HTTP_Connector& c,const char* uri,void* arg) {
  HTTP_SE_Handle* handle = (HTTP_SE_Handle*)arg;
  if(!handle) return NULL;
  bool acl_create = false;
  bool acl_replicate = false;
  bool acl_read = false;
  std::list<AuthEvaluator*>& auths = c.authorizations();
  for(std::list<AuthEvaluator*>::iterator ii=auths.begin();ii!=auths.end();++ii){
    if(acl_create && acl_replicate && acl_read) break;

    bool for_acl_create=false;
    bool for_acl_replicate=false;
    bool for_acl_read=false;

    if(!acl_create)for(std::list<std::string>::iterator i=
           handle->acl_create.begin();i!=handle->acl_create.end();++i) {
      if((*(*ii)) == (*i)) { for_acl_create=true; break; };
    };
    if(!acl_replicate) for(std::list<std::string>::iterator i=
           handle->acl_replicate.begin();i!=handle->acl_replicate.end();++i) {
      if((*(*ii)) == (*i)) { for_acl_replicate=true; break; };
    };
    if(!acl_read) for(std::list<std::string>::iterator i=
           handle->acl_read.begin();i!=handle->acl_read.end();++i) {
      if((*(*ii)) == (*i)) { for_acl_read=true; break; };
    };
    if(for_acl_read || for_acl_replicate || for_acl_create) {
      if((*(*ii)).evaluate(c.identity()) == AAA_POSITIVE_MATCH) {
        if(for_acl_create) {
          odilog(INFO,c.pid)<<"User is granted 'create' access through acl '"<<
               (*(*ii)).get_name()<<"'"<<std::endl; acl_create=true;
        };
        if(for_acl_replicate) {
          odilog(INFO,c.pid)<<"User is granted 'replicate' access through acl '"<<
               (*(*ii)).get_name()<<"'"<<std::endl; acl_replicate=true;
        };
        if(for_acl_read) {
          odilog(INFO,c.pid)<<"User is granted 'read' access through acl '"<<
               (*(*ii)).get_name()<<"'"<<std::endl; acl_read=true;
        };
      };
    };
  };
  HTTP_SE* h = new HTTP_SE(&c,handle,uri,acl_create,acl_replicate,acl_read);
  return h;
}

void SEFilesList::add_files(SEFiles* f) {
  lock.block();
  files.push_back(f);
  lock.unblock();
}

static SEFilesList files_list;
static SERegistrator_Thread* se_thread = NULL;
static SECollector_Thread* collector_thread = NULL;
static SEReplicator_Thread* replicator_thread = NULL;

HTTP_SE::HTTP_SE(HTTP_Connector *c_,HTTP_SE_Handle *h_,const char* uri_,bool create,bool replicate,bool read):HTTP_ServiceAdv(c_),handle(h_),acl_create(create),acl_replicate(replicate),acl_read(read) {
  service_url=h_->service_url;
  files_url=h_->base_url;
  odlog(ERROR)<<"SE: new service: requested URL: "<<uri_<<std::endl;
  odlog(ERROR)<<"SE: new service: contacted URL: "<<c_->url()<<std::endl;
  odlog(ERROR)<<"SE: new service: configured URL: "<<h_->base_url<<std::endl;
  merge_urls(service_url,c_->url());
  merge_urls(files_url,c_->url());
  odlog(ERROR)<<"SE: new service: service URL: "<<service_url<<std::endl;
  odlog(ERROR)<<"SE: new service: files URL: "<<files_url<<std::endl;
  soap_init();
  sp.namespaces=file_soap_namespaces;
  sp.user=this;
}

HTTP_SE::~HTTP_SE(void) {
  soap_deinit();
}

SEFiles* HTTP_SE::files(void) {
  if(handle==NULL) return NULL;
  return &(handle->files);
}

/*
 * HTTP methods
 */
HTTP_Error HTTP_SE::get(const char* uri,int &keep_alive) {
  if((!c) || (!handle)) {
    return HTTP_NOT_IMPLEMENTED;
  };
  HTTP_Error err;
  if((err=parse_header(keep_alive)) != HTTP_OK) return err;
  if(failure_parsing || unsupported_option_passed) {
    keep_alive=0;
    return HTTP_FAILURE;
  };
  // obtain requested name
  // const char* name = uri+handle->base_url.length();
  const char* name = uri+service_url.length();
  if(*name != 0) name++; // skip slash
  odlog(DEBUG)<<"HTTP_SE::get: name: "<<name<<std::endl;
  SEFileHandle* h = SEFileHandle::open(name,true,handle->files);
  if(h==NULL) {
    odlog(INFO)<<"file not found: "<<name<<std::endl;
    if(c->send_error_response(keep_alive,404,NULL,"404 Not Found") != 0) {
      keep_alive=1;
      return HTTP_ERROR;
    };
    return HTTP_OK;
  };

  int acl_top_flags = files()->check_acl(c->identity());
  int acl_flags = h->instance().check_acl(c->identity());
  if(!(acl_read) && 
     !(acl_top_flags & FILE_ACC_READ) && 
     !(acl_flags & FILE_ACC_READ)){
//  if(!(acl_flags & FILE_ACC_READ)) {
    odlog(INFO)<<"file not allowed"<<std::endl;
    if(c->send_error_response(keep_alive,404,NULL,"404 Not Found") != 0) {
      keep_alive=1;
      return HTTP_ERROR;
    };
    return HTTP_OK;
  }; 
  // fix ranges
  uint64_t range_start_,range_end_;
  if(range_passed) {
    int n;  
    for(n=0;n<nranges;n++) { // find first suitable range
      if(range_start[n] <= range_end[n]) {
        if(range_start[n] < h->size()) {
          range_end[n]++;
          if(range_end[n] > h->size()) range_end[n]=h->size();
          break;
        };
      } else {
        if(range_start[n] > h->size()) range_start[n]=h->size();
        range_start[n]=h->size()-range_start[n];
        range_end[n]=h->size();
        break;
      };
    };
    if(n<nranges) { range_start_=range_start[n]; range_end_=range_end[n]; }
    else { range_start_=0; range_end_=0; }; // no such data
    h->seek(range_start_);
  } else {
    odlog(VERBOSE)<<"range_not_passed"<<std::endl;
    range_start_=0; range_end_=h->size();
  };
  odlog(VERBOSE)<<"range_start: "<<range_start_<<std::endl;
  odlog(VERBOSE)<<"range_end: "<<range_end_<<std::endl;
  odlog(VERBOSE)<<"size: "<<h->size()<<std::endl;
  // send reply header
  if(range_end_ <= range_start_) {
    odlog(INFO)<<"range_end <= range_start"<<std::endl;
    delete h;
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl;
      keep_alive=0;
      return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  bool partial=false;
  if((range_start_ != 0) || (range_end_ != h->size())) partial=true;
  odlog(VERBOSE)<<"partial: "<<partial<<std::endl;
  HTTP_Time expires;
  if(h->instance().created_available()) expires=*(h->instance().created());
  err=send_header(keep_alive,range_start_,range_end_,partial,h->size(),HTTP_Time(-1),expires);
  if(err != HTTP_OK) { keep_alive=0; delete h; return HTTP_ERROR; };
  // pass requested range to client
  err=HTTP_OK;
  for(;;) {
    if(h->tell() >= range_end_) break; 
    uint64_t l = range_end_-h->tell();
    char buf[65536];
    if(l > sizeof(buf)) l=sizeof(buf);
    l=h->read(buf,l);
    if(l==0) break; // eof or error
    if(c->write(buf,l) != 0) {
      odlog(ERROR)<<"HTTP_SE::get: failure while writing data to network"<<std::endl;
      keep_alive=0;
      err=HTTP_FAILURE;
    };
  };
  if(err==HTTP_OK) if(h->tell() < range_end_) { // something wrong with file
    odlog(ERROR)<<"HTTP_SE::get: failure while reading data from disc"<<std::endl;
    keep_alive=0;
    err=HTTP_FAILURE;
  };
  delete h;
  return err;
}

HTTP_Error HTTP_SE::put(const char* uri,int &keep_alive) {
  if((!c) || (!handle)) {
    return HTTP_NOT_IMPLEMENTED;
  };

  HTTP_Error err;
  if((err=parse_header(keep_alive)) != HTTP_OK) return err;
  if(failure_parsing || unsupported_option_passed) {
    odlog(ERROR)<<"HTTP_SE::put: bad header"<<std::endl;
    keep_alive=0;
    if(send_header(keep_alive,406) != HTTP_OK) keep_alive=0;
    return HTTP_FAILURE;
  };
  // obtain requested name
  // const char* name = uri+handle->base_url.length();
  const char* name = uri+service_url.length();
  if(*name != 0) name++; // skip slash
  odlog(DEBUG)<<"HTTP_SE::put: name: "<<name<<std::endl;
  // open file
  SEFileHandle* h = SEFileHandle::open(name,false,handle->files);
  if(h==NULL) {
    odlog(DEBUG)<<"HTTP_SE::put: file not found: "<<name<<std::endl;
    if(c->send_error_response(keep_alive,404,NULL,"404 Not Found") != 0) {
      keep_alive=0;
      return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  // check state (not full yet!)
  if(h->instance().state_file() != FILE_STATE_COLLECTING) {
    odlog(DEBUG)<<"HTTP_SE::put: file is not being uploaded: "<<name<<std::endl;
    if(c->send_error_response(keep_alive,404,NULL,"404 Not Found") != 0) {
      keep_alive=0;
      return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  int acl_flags = h->instance().check_acl(c->identity());
  if(!(acl_flags & FILE_ACC_WRITE)) {
    odlog(ERROR)<<"file not allowed"<<std::endl;
    if(c->send_error_response(keep_alive,404,NULL,"404 Not Found") != 0) {
      keep_alive=1;
      return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  uint64_t range_start_,range_end_;
  if(entity_range_passed) {
    range_start_=entity_range_start;
    range_end_=entity_range_end+1;
    odlog(DEBUG)<<"HTTP_SE::put: range_passed: "<<range_start_<<"-"<<range_end_<<std::endl;
    if(length_passed && ((range_end_-range_start_) != length)) { 
      keep_alive=0; delete h;
      if(send_header(keep_alive,406) != HTTP_OK) keep_alive=0;
      return HTTP_FAILURE;
    };
  } else {
    odlog(DEBUG)<<"HTTP_SE::put: range_not_passed"<<std::endl;
    range_start_=0; range_end_=h->size();
    if(length_passed) range_end_=length;
  };
  if(range_end_ > h->size()) { 
    keep_alive=0; delete h;
    if(send_header(keep_alive,500) != HTTP_OK) keep_alive=0;
    return HTTP_FAILURE;
  };
  h->seek(range_start_);
  err=HTTP_OK;
  // receive entity from network and store locally 
  for(;;) {
    if(h->tell() >= range_end_) break; 
    uint64_t l = range_end_-h->tell();
    char buf[1024*1024]; // 1MB - is it enough ?
    if(l > sizeof(buf)) l=sizeof(buf);
    // globus_io tends to return no more than SSL block, hence
    // try to fill buffer by looping
    uint64_t ll=0;
    for(;ll<l;) {
      uint64_t l_=c->read(buf+ll,l-ll);
      if(l_ == 0) break; ll+=l_;
    };
    if(ll == 0) break; // EOF
    l=h->write(buf,ll);
    if(l==0) { // error
      odlog(ERROR)<<"Failed to write to file"<<std::endl;
      err=HTTP_FAILURE;
      break;
    };
  };
  if((err!=HTTP_OK) || (h->tell() < range_end_)) { // something wrong with file
    odlog(ERROR)<<"HTTP_SE::put: failure while reading/writing data from/to network/disc"<<std::endl;
    keep_alive=0;
    if(send_header(keep_alive,500) != HTTP_OK) keep_alive=0;
    err=HTTP_FAILURE;
  };
  h->close();
  if(h->full()) {                                   // full content stored
    SEFile& f = h->instance();
    f.acquire();
    if(f.state_file() == FILE_STATE_COLLECTING) {
      f.state_file(FILE_STATE_COMPLETE);
    };
    f.release();
    // signal registrators for verification and registration
    collector_thread->attention();
  };
  delete h;
  if(err==HTTP_OK) if(send_header(keep_alive,200) != HTTP_OK) keep_alive=0;
  return err;
}

bool HTTP_SE::set_current_file(const char* uri) {
  // obtain requested name
  odlog(VERBOSE)<<"SE:set_current_file: uri: "<<uri<<std::endl;
  // current_name=uri+handle->base_url.length();
  current_name=uri+service_url.length();
  current_file=NULL;
  if(*current_name != 0) {
    current_name++; // skip slash
    odlog(VERBOSE)<<"SE:post: name: "<<current_name<<std::endl;
    // get file
    current_file=handle->files.get(current_name);
  };
  return (current_file != NULL);
}

void HTTP_SE::new_files(void) {
  if(replicator_thread) replicator_thread->attention();
}

HTTP_Error HTTP_SE::post(const char* uri,int &keep_alive) {
  if(!set_current_file(uri)) {
    odlog(VERBOSE)<<"SE:post: contacted url contains no file name"<<std::endl;
/*
    skip_header(*c,keep_alive);
    if(c->send_error_response(keep_alive,404,NULL,"404 Not Found") != 0) {
      keep_alive=0;
      return HTTP_ERROR;
    };
    return HTTP_OK;
*/
  } else {
    odlog(VERBOSE)<<"SE:post: requested file "<<current_file->id()<<std::endl;
  };
  HTTP_Error r = soap_post(uri,keep_alive);
  current_file=NULL;
  return r;
}

// called during startup of server (once for every base uri)
bool se_service_configurator(std::istream& f,const char* uri,HTTP_Service_Properties &prop) {
  std::string path("");
  std::string ns_url("");
  std::string base_url("");
  std::list<std::string> acl_create;
  std::list<std::string> acl_replicate;
  std::list<std::string> acl_read;
  int reg_type = registration_immediate;
  int handle_type = handleop_showincomplete | handleop_userdownloads;
  int timeout_collecting = -1;
  int timeout_downloading = -1;
  int retries_downloading = -1;
  int timeout_complete = -1;
  int timeout_failed = -1;
  for(;!f.eof();) {
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    char* p = buf;
    for(;*p;p++) if(!isspace(*p)) break;
    char* command = p;
    if(*command == '#') continue;
    for(;*p;p++) if(isspace(*p)) break;
    int command_len = p-command;
    if(!command_len) continue;
    if((command_len==10) && (strncmp(command,"acl_create",command_len)==0)) {
      for(;;) {
        std::string name;
        int n = input_escaped_string(p,name);
        if(n == 0) break;
        p+=n;
        acl_create.push_back(name);
      };
    } else if((command_len==13) && (strncmp(command,"acl_replicate",command_len)==0)) {
      for(;;) {
        std::string name;
        int n = input_escaped_string(p,name);
        if(n == 0) break;
        p+=n;
        acl_replicate.push_back(name);
      };
    } else if((command_len==8) && (strncmp(command,"acl_read",command_len)==0)) {
      for(;;) {
        std::string name;
        int n = input_escaped_string(p,name);
        if(n == 0) break;
        p+=n;
        acl_read.push_back(name);
      };
    } else if((command_len==4) && (strncmp(command,"path",command_len)==0)) {
      for(;*p;p++) if(!isspace(*p)) break;
      path=p;
      continue;
    } else if((command_len==7) && (strncmp(command,"storage",command_len)==0)) {
      for(;*p;p++) if(!isspace(*p)) break;
      path=p;
      continue;
    } else if((command_len==2) && (strncmp(command,"ns",command_len)==0)) {
      for(;*p;p++) if(!isspace(*p)) break;
      ns_url=p;
      continue;
    } else if((command_len==3) && (strncmp(command,"url",command_len)==0)) {
      for(;*p;p++) if(!isspace(*p)) break;
      base_url=p;
      continue;
    } else if((command_len==8) && (strncmp(command,"timeouts",command_len)==0)) {
      int i;
      std::string s;
      int n;
      if((n=input_escaped_string(p,s)) == 0) continue; p+=n;
      if(!stringtoint(s,i)) {
        odlog(ERROR)<<"warning: wrong number "<<s<<
                  " for COLLECTING timeout"<<std::endl;
      } else { if(i>=0) timeout_collecting=i; };
      if((n=input_escaped_string(p,s)) == 0) continue; p+=n;
      if(!stringtoint(s,i)) {
        odlog(ERROR)<<"warning: wrong number "<<s<<
                  " for DOWNLOADING timeout"<<std::endl;
      } else { if(i>=0) timeout_downloading=i; };
      if((n=input_escaped_string(p,s)) == 0) continue; p+=n;
      if(!stringtoint(s,i)) {
        odlog(ERROR)<<"warning: wrong number "<<s<<
                  " for DOWNLOADING retries"<<std::endl;
      } else { if(i>=0) retries_downloading=i; };
      if((n=input_escaped_string(p,s)) == 0) continue; p+=n;
      if(!stringtoint(s,i)) {
        odlog(ERROR)<<"warning: wrong number "<<s<<
                  " for COMPLETE timeout"<<std::endl;
      } else { if(i>=0) timeout_complete=i; };
      if((n=input_escaped_string(p,s)) == 0) continue; p+=n;
      if(!stringtoint(s,i)) {
        odlog(ERROR)<<"warning: wrong number "<<s<<
                  " for FAILED timeout"<<std::endl;
      } else { if(i>=0) timeout_failed=i; };
      continue;
    } else if((command_len==12) && (strncmp(command,"registration",command_len)==0)) {
      // parse options separated by ,
      for(;;) {
        for(;*p;p++) if(!isspace(*p)) break;
        if(!(*p)) break;
        char* pc = p;
        for(;*pc;pc++) if((*pc)==',') break;
        if(strncmp(p,"immediate",pc-p) == 0) {
          reg_type|=registration_immediate;
        } else if(strncmp(p,"delayed",pc-p) == 0) {
          reg_type&=~((unsigned int)(registration_immediate));
        } else if(strncmp(p,"retry",pc-p) == 0) {
          reg_type|=registration_retry;
        } else if(strncmp(p,"noretry",pc-p) == 0) {
          reg_type&=~((unsigned int)(registration_retry));
        } else if(strncmp(p,"showincomplete",pc-p) == 0) {
          handle_type=handleop_showincomplete;
        } else if(strncmp(p,"hideincomplete",pc-p) == 0) {
          handle_type=~((unsigned int)(handleop_showincomplete));
        } else if(strncmp(p,"userdownloads",pc-p) == 0) {
          handle_type=handleop_userdownloads;
        } else if(strncmp(p,"servicedownloads",pc-p) == 0) {
          handle_type=~((unsigned int)(handleop_userdownloads));
        } else if(strncmp(p,"userregisters",pc-p) == 0) {
          handle_type=handleop_userregisters;
        } else if(strncmp(p,"serviceregisters",pc-p) == 0) {
          handle_type=~((unsigned int)(handleop_userregisters));
        };
        p=pc; if(*p) p++;
      };
    } else if((command_len==3) && (strncmp(command,"end",command_len)==0)) {
      break;
    } else {
      odlog(ERROR)<<"SE(configure): skipping unknown command: "<<command<<std::endl;
    };
  };
  prop.arg=NULL;
  if(uri == NULL) uri="";
  if(base_url.length() == 0) base_url=uri;
  odlog(ERROR)<<"SE(configure): filesystem path: "<<path<<std::endl;
  odlog(ERROR)<<"SE(configure): URL: "<<base_url<<std::endl;
  //  Below is an ugly hack. Whole URL staff must be cleaned.
  const char* u;
  // Choose url for control channel, preferences - gsi, gssapi, ssl, http.
  u=base_url_by_type("gsi");
  if(!u) u=base_url_by_type("gssapi");
  if(!u) u=base_url_by_type("ssl");
  if(!u) u=base_url_by_type("plain");
  if(!u) {
    odlog(ERROR)<<"SE(configure): failed to choose URl for control channel"<<std::endl;
    return false;
  };
  std::string access_gsi_url = u;
  // Choose url for data channel, preferences - ssl, gsi, gssapi, http.
  u=base_url_by_type("ssl");
  if(!u) u=base_url_by_type("gsi");
  if(!u) u=base_url_by_type("gssapi");
  if(!u) u=base_url_by_type("plain");
  if(!u) {
    odlog(ERROR)<<"SE(configure): failed to choose URl for data channel"<<std::endl;
    return false;
  };
  std::string access_ssl_url = u;
  std::string::size_type p;
  if((p=access_ssl_url.find('/',8)) != std::string::npos) access_ssl_url.resize(p);
  if((p=access_gsi_url.find('/',8)) != std::string::npos) access_gsi_url.resize(p);
  const char* s = NULL;
  if(uri[0] == '/') { // path
    s=uri+1;
  } else if(uri[0] == 0) { // path
    s=uri;
  } else if(uri[0] == ':') { // port + path
    s=strchr(uri,'/');
  } else { // url
    s=strchr(uri,':');
    if((s != NULL) && (*(s++) == '/') && (*(s++) == '/')) {
      s=strchr(uri,'/'); if(s == NULL) s="";
    };
  };
  if(s == NULL) s="";
  access_ssl_url+="/"; access_ssl_url+=s;
  access_gsi_url+="/"; access_gsi_url+=s;
  std::string se_url = access_gsi_url;
  p=se_url.find(':'); if(p!=std::string::npos) se_url.replace(0,p,"se");
  // -----
  HTTP_SE_Handle* handle = NULL;
  if(path.length()) {
    handle = new HTTP_SE_Handle(path.c_str(),uri,base_url.c_str(),handle_type);
    if((*handle)) { prop.arg=handle; }
    else { delete handle; handle=NULL; };
  };
  if(!handle) {
    odlog(ERROR)<<"SE(configure): failed"<<std::endl;
  } else {
    handle->access_gsi_url=access_gsi_url;
    handle->access_ssl_url=access_ssl_url;
    handle->files.NS(ns_url.c_str(),se_url.c_str(),reg_type);
    handle->files.NS()->access_url(access_ssl_url.c_str());
    // handle->files.NS()->access_url(access_gsi_url.c_str());
    handle->acl_create.splice(handle->acl_create.end(),acl_create);
    handle->acl_replicate.splice(handle->acl_replicate.end(),acl_replicate);
    handle->acl_read.splice(handle->acl_read.end(),acl_read);
    // run thread to handle delayed registration and other things
    // one thread per server
    odlog(ERROR)<<"SE(configure): going to start thread"<<std::endl;
    if(se_thread == NULL) {
      odlog(ERROR)<<"SE(configure): starting thread"<<std::endl;
      se_thread = new SERegistrator_Thread(files_list);
      if(se_thread) se_thread->attention();
    };
    if(collector_thread == NULL) {
      odlog(ERROR)<<"SE(configure): starting thread"<<std::endl;
      collector_thread = new SECollector_Thread(files_list);
      if(collector_thread) collector_thread->attention();
    };
    if(replicator_thread == NULL) {
      odlog(ERROR)<<"SE(configure): starting thread"<<std::endl;
      replicator_thread = new SEReplicator_Thread(files_list);
      if(replicator_thread) replicator_thread->attention();
    };
    files_list.add_files(&(handle->files));
  };
  prop.subtree=true;
  odlog(ERROR)<<"SE(configure): serving subtree"<<std::endl;
  return true;
}

SEFiles::iterator HTTP_SE::new_file(SEAttributes& attr) {
  if((!c) || (!handle)) {
    return handle->files.end();
  };
  odlog(ERROR)<<"SE: creating new file: "<<attr.id()<<std::endl;
  SEFile* f = new SEFile(handle->files.Path(),attr,handle->files.Space());
  if(f) { if(!(*f)) { delete f; f=NULL; }; };
  if(!f) {
    odlog(ERROR)<<"SE: file creation failed"<<std::endl;
    return handle->files.end();
  };
  return handle->files.add(*f);
}

#ifdef OLD_GSOAP
#error gSOAP older than 2.5.2 is not supported.
#else
void HTTP_SE::soap_methods(void) {
  odlog(VERBOSE)<<"soap_methods: tag: "<<sp.tag<<std::endl;
  if(!soap_match_tag(&sp,sp.tag,"ns:add")) soap_serve_ns__add(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"ns:update")) soap_serve_ns__update(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"ns:info")) soap_serve_ns__info(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"ns:acl")) soap_serve_ns__acl(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"ns:del")) soap_serve_ns__del(&sp);
  else sp.error = SOAP_NO_METHOD;
}
#endif

static char* soap_strdup_l(soap* sp,const char* s,int l = -1) {
  if(l<0) l=strlen(s);
  char* s_ = (char*)soap_malloc(sp,l+1);
  if(s_ == NULL) return NULL;
  memcpy(s_,s,l); s_[l]=0;
  return s_;
}

void add_path_to_url(std::string& url,const std::string& path,bool strip_path = true) {
  int l = url.length()-1;
  if(l == 0) { url=path; return; };
  if(url[l] != '/') url+="/";
  const char* s = path.c_str();
  if(strip_path) if(s[0] == '/') s++;
  url+=s;
  return;
}

/* SOAP calls */
int ns__update(struct soap *sp, ns__fileinfo *file, struct ns__updateResponse &r) {
  // initialize response
  r.error_code=0; r.sub_error_code=0; r.error_description=NULL;
  r.file.id=NULL;
  r.file.size=NULL;
  r.file.checksum=NULL;
  r.file.acl=NULL;
  r.file.url=NULL;
  r.file.__size_url=0;
  r.file.state=NULL;
  r.file.created=NULL;
  //
  // update file attributes and check if state should not be changed 
  //
  if(!file) {
    odlog(ERROR)<<"SOAP: update: missing information about file"<<std::endl;
    r.error_code=ERR_MISSING_INPUT_DATA;
    return SOAP_OK;
  };
  if(file->url) {
    odlog(ERROR)<<"SOAP: update: update does not need contact url"<<std::endl;
    r.error_code=ERR_EXCESSIVE_INPUT_DATA;
    return SOAP_OK;
  };
  HTTP_SE* it = (HTTP_SE*)(sp->user);
  if(file->id && it->current_name && (*it->current_name) && (strcmp(it->current_name,file->id)!=0)) {
    odlog(ERROR)<<"SOAP: update: if provided file id must be same as path"<<std::endl;
    r.error_code=ERR_WRONG_INPUT_DATA;
    return SOAP_OK;
  };
  if(!(file->id) && !((it->current_name) && (*it->current_name))) {
    odlog(ERROR)<<"SOAP: update: missing file id"<<std::endl;
    r.error_code=ERR_MISSING_INPUT_DATA;
    return SOAP_OK;
  };
  if(!(file->id)) file->id=(char*)(it->current_name);
  if(file->id) odlog(INFO)<<"SOAP: update: file id: "<<file->id<<std::endl;
  if(file->size) odlog(INFO)<<"SOAP: update: file size: "<<*(file->size)<<std::endl;
  if(file->checksum) odlog(INFO)<<"SOAP: update: checksum: "<<(file->checksum)<<std::endl;
  if(file->created) odlog(INFO)<<"SOAP: update: created: "<<(file->created)<<std::endl;
  if(file->acl) odlog(INFO)<<"SOAP: update: acl: "<<(file->acl)<<std::endl;
  if(!(it->current_file)) {
    odlog(ERROR)<<"SOAP: update: there is no file to update"<<std::endl;
    r.error_code=ERR_INTERNAL_ERROR;
    return SOAP_OK;
  };
  SEFile& f = *(it->current_file);
  // updates accepted from creator only
  if(it->current_file->creator() != it->c->identity().DN()) {
    odlog(ERROR)<<"SOAP: update: not a creator is trying to update"<<std::endl;
    r.error_code=ERR_INTERNAL_ERROR;
    return SOAP_OK;
  }; 
  if(f.state_file() == FILE_STATE_VALID) {
    odlog(ERROR)<<"SOAP: update: not allowed for validated files"<<std::endl;
    r.error_code=ERR_INTERNAL_ERROR;
    return SOAP_OK;
  };
  bool previous_complete = f.complete();
  if(file->acl) {
    odlog(ERROR)<<"SOAP: update: update does not accept ACL yet"<<std::endl;
    r.error_code=ERR_EXCESSIVE_INPUT_DATA;
    return SOAP_OK;
  };
  if(file->size) {
    if(f.size() != *(file->size)) {
      odlog(ERROR)<<"SOAP: update: file size differs"<<std::endl;
      r.error_code=ERR_WRONG_INPUT_DATA;
      return SOAP_OK;
    };
  };
  if(file->checksum) {
    if(f.checksum_available()) {
      if(strcmp(file->checksum,f.checksum().c_str()) != 0) {
        odlog(ERROR)<<"SOAP: update: file checksum differs"<<std::endl;
        r.error_code=ERR_WRONG_INPUT_DATA;
        return SOAP_OK;
      };
    } else {
      f.checksum(file->checksum);
    };
  };
  if(file->created) {
    if(f.created_available()) {
      if(!f.created_compare(file->created)) {
        odlog(ERROR)<<"SOAP: update: file creation time differs"<<std::endl;
        r.error_code=ERR_WRONG_INPUT_DATA;
        return SOAP_OK;
      };
    } else {
      f.created(file->created);
      if(!f.created_available()) {
        odlog(ERROR)<<"SOAP: update: file creation uninterpretable"<<std::endl;
        r.error_code=ERR_WRONG_INPUT_DATA;
        return SOAP_OK;
      };
    };
  };
  if(f.write_attr() != 0) {
    odlog(ERROR)<<"SOAP: update: failed to write attributes"<<std::endl;
    r.error_code=ERR_INTERNAL_ERROR;
    return SOAP_OK;
  };
// TODO - created
  f.acquire();
  if((!previous_complete) && (f.complete())) { 
    if((f.state_file() == FILE_STATE_COMPLETE) ||
       (f.state_file() == FILE_STATE_VALID)) {
      collector_thread->attention();
    };
  };
  f.release();
  return SOAP_OK;
}

int ns__add(struct soap* sp,ns__fileinfo *file,int __size_source,char** source,
struct ns__addResponse &r) {
  HTTP_SE* it = (HTTP_SE*)(sp->user);

  // initialize response
  r.error_code=0; r.sub_error_code=0; r.error_description=NULL;
  r.file.id=NULL;
  r.file.size=NULL;
  r.file.checksum=NULL;
  r.file.acl=NULL;
  r.file.url=NULL;
  r.file.__size_url=0;
  r.file.state=NULL;
  r.file.created=NULL;

  if(!it->acl_create) {
    SEFiles* files = it->files();
    if(!files) {
      odlog(ERROR)<<"No files"<<std::endl;
      r.error_code=ERR_INTERNAL_ERROR; return SOAP_OK;
    };
    int acl_flags = files->check_acl(it->c->identity());
    if(!(acl_flags & FILE_ACC_CREATE)) {
      r.error_code=ERR_ACCESS;
      r.error_description="Access denied";
      return SOAP_OK;
    };
  };
  
  // check if all necessary attributes provided
  if(!file) {
    odlog(ERROR)<<"SOAP: add: missing information about file"<<std::endl;
    r.error_code=ERR_MISSING_INPUT_DATA; 
    r.error_description="Missing information about file";
    return SOAP_OK;
  };
  if(file->id) odlog(INFO)<<"SOAP: add: file id: "<<file->id<<std::endl;
  if(file->size) odlog(INFO)<<"SOAP: add: file size: "<<*(file->size)<<std::endl;
  if(file->checksum) odlog(INFO)<<"SOAP: add: checksum: "<<(file->checksum)<<std::endl;
  if(file->acl) odlog(INFO)<<"SOAP: add: acl: "<<(file->acl)<<std::endl;
  if(file->created) odlog(INFO)<<"SOAP: add: created: "<<(file->created)<<std::endl;

  r.file.id=file->id;
  if(file->url) {
    odlog(ERROR)<<"SOAP: add: add does not need contact url in input parameters"<<std::endl;
    r.error_code=ERR_EXCESSIVE_INPUT_DATA; 
    return SOAP_OK;
  };
  if(it->current_name && (*it->current_name)) {
    odlog(ERROR)<<"SOAP: add: add requires only service in path"<<std::endl;
    r.error_code=ERR_EXCESSIVE_INPUT_DATA; 
    return SOAP_OK;
  };
  
  // attributes for response
  r.file.id=file->id;
  r.file.size=file->size;
  r.file.checksum=file->checksum;
  r.file.acl=file->acl;
  if(!file->id) {
    odlog(ERROR)<<"SOAP: add: missing file id (lfn)"<<std::endl;
    r.error_code=ERR_MISSING_INPUT_DATA; 
    return SOAP_OK;
  };
  if(!file->size) {
    if(__size_source == 0) { // let se pull that information from source somehow
      odlog(ERROR)<<"SOAP: add: missing file size (needed for allocation)"<<std::endl;
      r.error_code=ERR_MISSING_INPUT_DATA; 
      return SOAP_OK;
    };
  };
  if(!file->acl) {
    odlog(ERROR)<<"SOAP: add: missing ACL (can't access file without ACL)"<<std::endl;
    r.error_code=ERR_MISSING_INPUT_DATA; 
    return SOAP_OK;
  };
  // create new file
  SEAttributes attr(file->id,it->c->identity());
  if(file->size) attr.size(*(file->size));
  if(file->checksum) attr.checksum(std::string(file->checksum));
  if(file->created) attr.created(file->created);
  if((!attr.complete()) && (it->handle->register_immediately())) {
    odlog(ERROR)<<"SOAP: add: missing file information needed for registration"<<std::endl;
    r.error_code=ERR_MISSING_INPUT_DATA; 
    return SOAP_OK;
  };
  if(__size_source) {
    for(int n=0;n<__size_source;n++) {
      if(source[n]) {
        if(source[n][0] != 0) {
          attr.source(source[n]);
        } else { // replication
          if(it->handle->NS()) {
            std::string u = it->handle->NS()->url(attr);
            attr.source(u);
          };
        };
      };
    };
  };
  SEFiles::iterator f = it->new_file(attr);
  if(!f.exists()) {
    odlog(ERROR)<<"File not created: "<<file->id<<std::endl;
    r.error_code=ERR_CREATING_NEW_FILE;  // TODO: check for reason of failure
    return SOAP_OK;
  };
  // Here file is inserted into list already

  // Store user's proxy (if delegated)
  int hh = -1;
  const char* proxy_fname = it->c->identity().proxy();
  if(proxy_fname && proxy_fname[0]) {
    int hh=::open(proxy_fname,O_RDONLY);
    if(hh != -1) {
      std::string cred;
      const char* s;
      char buf[256];
      for(;;) {
        int ll=::read(hh,buf,sizeof(buf));
        if((ll==0) || (ll==-1)) break;
        cred.append(buf,ll);
      };
      ::close(hh);
      if(f->write_credentials(cred.c_str()) != 0) {
        odlog(ERROR)<<"Can't write delegated credentials"<<std::endl; 
      };
    } else {
      odlog(ERROR)<<"Can't read delegated credentials"<<std::endl; 
    };
  } else {
    odlog(INFO)<<"No credentials delegated"<<std::endl; 
  };

  std::string url_ = it->files_url;
  add_path_to_url(url_,file->id,false);
  odlog(DEBUG)<<"SOAP: add: contact url: "<<url_<<std::endl;
  r.file.url=(char**)soap_malloc(sp,sizeof(char*)*2); // 2 urls
  r.file.__size_url=0;
  if(r.file.url == NULL) {
    odlog(ERROR)<<"SOAP: add: soap_malloc failed"<<std::endl;
    it->files()->remove(*f);
    r.error_code=ERR_INTERNAL_ERROR; 
    return SOAP_OK;
  }; 
  r.file.url[0]=(char*)soap_strdup_l(sp,url_.c_str(),url_.length()+1);
  if(r.file.url[0] == NULL) {
    odlog(ERROR)<<"SOAP: add: soap_strdup_l failed"<<std::endl;
    if(f->state_file() == FILE_STATE_ACCEPTED) it->files()->remove(*f);
    r.error_code=ERR_INTERNAL_ERROR; 
    return SOAP_OK;
  }; 
  r.file.__size_url=1;
  // hack - purpose is to use lighter SSL channel to transfer files
  if(strncmp(it->handle->control_url().c_str(),r.file.url[0],
                         it->handle->control_url().length()) == 0) {
    std::string u(r.file.url[0]);
    u.replace(0,it->handle->control_url().length(),it->handle->data_url());
    r.file.url[1]=(char*)soap_strdup_l(sp,u.c_str(),u.length()+1);
    if(r.file.url[1] == NULL) {
      odlog(ERROR)<<"SOAP: add: soap_strdup_l failed"<<std::endl;
      if(f->state_file() == FILE_STATE_ACCEPTED) it->files()->remove(*f);
      r.error_code=ERR_INTERNAL_ERROR; 
      return SOAP_OK;
    }; 
    r.file.__size_url=2;
  };
  if(f->write_acl(it->c->identity(),file->acl) != 0) {
    odlog(ERROR)<<"SOAP: add: failed to acquire/store ACL"<<std::endl;
    if(f->state_file() == FILE_STATE_ACCEPTED) it->files()->remove(*f);
    r.error_code=ERR_INTERNAL_ERROR; 
    return SOAP_OK;
  };
  r.file.state=(ns__filestate*)soap_malloc(sp,sizeof(ns__filestate));
  if(r.file.state) (*r.file.state)=(ns__filestate)(f->state_file());
  if(f->state_file() != FILE_STATE_ACCEPTED) {
    // check if new request is compatible with old one.
    if(__size_source) {
      if(f->state_file() == FILE_STATE_COLLECTING) {
        r.error_code=ERR_INTERNAL_ERROR;
      };
    };
    return SOAP_OK;
  };
  /* report information about file. */
  if(!it->register_new_file(&(*f))) {
    r.error_code=ERR_INTERNAL_ERROR; 
    return SOAP_OK;
  };
  file_state_t fst = FILE_STATE_COLLECTING;
  if(__size_source) fst=FILE_STATE_REQUESTED;
  if((file->size) && (*(file->size) == 0)) fst=FILE_STATE_COMPLETE; // empty file easy to handle
  if(!f->state_file(fst)) {
    odlog(ERROR)<<"SOAP: add: failed to set "<<file_state_str[fst]<<std::endl;
    it->files()->NS()->Unregister(*f);  // TODO: if failed set state to repeat later
    it->files()->remove(*f);
    r.error_code=ERR_INTERNAL_ERROR;
    return SOAP_OK;
  };
  if(fst == FILE_STATE_REQUESTED) if(replicator_thread) replicator_thread->attention();
  return SOAP_OK;
}

bool HTTP_SE::register_new_file(SEFile* f) {
  // register *new* file
  if(handle->register_immediately()) { 
    f->acquire();
    if(!f->state_reg(REG_STATE_REGISTERING)) {
      if(handle->register_retry()) { 
        odlog(ERROR)<<"SOAP: add: failed to set REG_STATE_REGISTERING (will retry)"
                                                             <<std::endl;
        f->state_reg(REG_STATE_LOCAL);
      } else {
        odlog(ERROR)<<"SOAP: add: failed to set REG_STATE_REGISTERING"<<std::endl;
        f->release();
        files()->remove(*f);
        return false;
      };
    } else {
      if(files()->NS()->Register(*f) != 0) {
        if(handle->register_retry()) { 
          odlog(ERROR)<<"SOAP: add: failed to register (will retry)"<<std::endl;
          f->state_reg(REG_STATE_LOCAL);
        } else {
          odlog(ERROR)<<"SOAP: add: failed to register"<<std::endl;
          f->release();
          files()->remove(*f);
          return false;
        };
      } else {
        f->state_reg(REG_STATE_ANNOUNCED);
      };
    };
    f->release();
  } else {
    if(se_thread) se_thread->attention();
  };
  return true;
}

int ns__info(soap *sp,char *pattern, ns__infoResponse &r) {
  // initialize response
  r.error_code=0; r.sub_error_code=0; r.error_description=NULL;
  r.__size_file=0; r.file=NULL;
  HTTP_SE* it = (HTTP_SE*)(sp->user);
  SEFiles* files = it->files();
  if(!files) {
    odlog(ERROR)<<"No files"<<std::endl;
    r.error_code=ERR_INTERNAL_ERROR; return SOAP_OK;
  };
  int acl_top_flags = files->check_acl(it->c->identity());
  if(pattern) {
    std::string pat = pattern;
    if(pat.length() == 0) {  // create pattern from url
      pat="^"; pat+=it->current_name; pat+=".*";
    };
    regex_t preg;
    odlog(INFO)<<"regex: "<<pat<<std::endl;
    if(regcomp(&preg,pat.c_str(),REG_EXTENDED | REG_NOSUB) != 0) {
      odlog(ERROR)<<"Failed to compile regex: "<<pat<<std::endl;
      r.error_code=ERR_INTERNAL_ERROR;
      r.error_description="Requested pattern can't be processed";
      return SOAP_OK;
    };
    SEFiles::iterator f;
    files->acquire();
    f=files->begin();
    int n_alloc = 0;
    int n = 0;
    SEFiles::iterator f_last = files->end();
    for(;f!=files->end();++f) {
      odlog(DEBUG)<<"file: "<<f->id()<<std::endl;
      if((
          it->handle->showincomplete() || 
          (f->state_file()==FILE_STATE_COMPLETE) ||
          (f->state_file()==FILE_STATE_VALID)
         ) &&
         (regexec(&preg,f->id(),0,NULL,0) == 0)
        ) {
        odlog(DEBUG)<<"matched"<<std::endl;
        if((it->acl_read) || (acl_top_flags & FILE_ACC_LIST) || (f->check_acl(it->c->identity()) & FILE_ACC_LIST)){
          odlog(DEBUG)<<"allowed"<<std::endl;
          f_last=f;
          if(n>=n_alloc) {
            n_alloc+=10;
            ns__fileinfo* f_tmp = r.file;
            r.file=(ns__fileinfo*)soap_malloc(sp,sizeof(ns__fileinfo)*n_alloc);
            if(r.file == NULL) {
              if(f_tmp != NULL) soap_dealloc(sp,f_tmp);
              r.error_code=ERR_INTERNAL_ERROR; n=0; n_alloc=0; break;
            };
            if(f_tmp != NULL) memcpy(r.file,f_tmp,sizeof(ns__fileinfo)*n);
            if(f_tmp != NULL) soap_dealloc(sp,f_tmp);
          };
          ns__fileinfo rr;
          rr.id=soap_strdup_l(sp,f->id());
          rr.size=NULL;
          rr.checksum=NULL;
          rr.acl=NULL;
          rr.__size_url=0;
          rr.url=NULL;
          rr.created=NULL;
          rr.state=(ns__filestate*)soap_malloc(sp,sizeof(ns__filestate));
          if(rr.state) (*rr.state)=(ns__filestate)(f->state_file());
          memcpy(r.file+n,&rr,sizeof(ns__fileinfo));
          n++;
        };
      };
    };
    if(n==1) {  // only one file -  give more info
      ns__fileinfo& rr = r.file[0];
      rr.size=(unsigned long long int*)soap_malloc(sp,sizeof(unsigned long long int));
      if(rr.size) (*rr.size)=f_last->size();
      rr.checksum=soap_strdup_l(sp,f_last->checksum().c_str());
      std::string acl;
      if(f_last->read_acl(it->c->identity(),acl) == 0) {
        rr.acl=soap_strdup_l(sp,acl.c_str(),acl.length());
      };
      std::string url_ = it->files_url + "/" + f_last->id();
      rr.__size_url=0;
      rr.url=(char**)soap_malloc(sp,sizeof(char*)*2); // 2 urls
      if(rr.url) {
        rr.url[0]=(char*)soap_strdup_l(sp,url_.c_str(),url_.length()+1);
        if(rr.url[0]) {
          rr.__size_url=1;
          if(strncmp(it->handle->control_url().c_str(),rr.url[0],
                             it->handle->control_url().length())==0){
            std::string u(rr.url[0]);
            u.replace(0,it->handle->control_url().length(),
                                      it->handle->data_url());
            rr.url[1]=(char*)soap_strdup_l(sp,u.c_str(),u.length()+1);
            if(rr.url[1]) rr.__size_url=2;
          };
        };
      };
      // rr.state=(ns__filestate*)soap_malloc(sp,sizeof(ns__filestate));
      // if(rr.state) (*rr.state)=(ns__filestate)(f_last->state_file());
    };
    files->release();
    r.__size_file=n;
    regfree(&preg);
    return SOAP_OK;
  } else {
    if(
       !(it->current_file) || 
       !(
         it->handle->showincomplete() || 
         (it->current_file->state_file()==FILE_STATE_COMPLETE) ||
         (it->current_file->state_file()==FILE_STATE_VALID)
        )
      ) {
      odlog(ERROR)<<"SOAP: info: there is no such file"<<std::endl;
      r.error_code=ERR_FILE_NOT_FOUND;
      r.error_description="This file does not exist (yet).";
      return SOAP_OK;
    };
    if((it->acl_read) || (acl_top_flags & FILE_ACC_LIST) || (it->current_file->check_acl(it->c->identity()) & FILE_ACC_LIST)) {
      odlog(INFO)<<"allowed"<<std::endl;
      r.file=(ns__fileinfo*)soap_malloc(sp,sizeof(ns__fileinfo));
      if(r.file == NULL) { r.error_code=ERR_INTERNAL_ERROR; return SOAP_OK; };
      ns__fileinfo rr;
      rr.id=soap_strdup_l(sp,it->current_name);
      rr.size=(unsigned long long int*)soap_malloc(sp,sizeof(unsigned long long int));
      if(rr.size) (*rr.size)=it->current_file->size();
      rr.checksum=soap_strdup_l(sp,it->current_file->checksum().c_str());
      std::string acl;
      if(it->current_file->read_acl(it->c->identity(),acl) == 0) {
        rr.acl=soap_strdup_l(sp,acl.c_str(),acl.length());
      };
      std::string url_ = it->files_url + "/" + it->current_file->id();
      rr.__size_url=0;
      rr.url=(char**)soap_malloc(sp,sizeof(char*)*2); // 2 urls
      if(rr.url) {
        rr.url[0]=(char*)soap_strdup_l(sp,url_.c_str(),url_.length()+1);
        if(rr.url[0]) {
          rr.__size_url=1;
          if(strncmp(it->handle->control_url().c_str(),rr.url[0],
                            it->handle->control_url().length()) == 0) {
            std::string u(rr.url[0]);
            u.replace(0,it->handle->control_url().length(),
                                       it->handle->data_url());
            rr.url[1]=(char*)soap_strdup_l(sp,u.c_str(),u.length()+1);
            if(rr.url[1]) rr.__size_url=2;
          };
        };
      };
      rr.state=(ns__filestate*)soap_malloc(sp,sizeof(ns__filestate));
      if(rr.state) (*rr.state)=(ns__filestate)(it->current_file->state_file());
      rr.created=NULL;
      if(it->current_file->created_available()) {
        std::string s; it->current_file->created(s);
        rr.created=soap_strdup_l(sp,s.c_str());
      };
      memcpy(r.file,&rr,sizeof(ns__fileinfo));
      r.__size_file=1;
    } else {
      r.error_code=ERR_ACCESS;
      r.error_description="No access to this file";
    };
  };
  return SOAP_OK;
}

/* report/modify ACL */
int ns__acl(soap *sp, char *acl, ns__aclResponse &r) {
  int res = -1;
  r.error_code=SUCCESS; r.error_description=NULL;
  r.sub_error_code=0;
  r.acl=NULL;
  HTTP_SE* it = (HTTP_SE*)(sp->user);
  if(it->current_file == NULL) {
    // Top ACL
    SEFiles* files = it->files();
    if(!files) {
      odlog(ERROR)<<"No files"<<std::endl;
      r.error_code=ERR_INTERNAL_ERROR; return SOAP_OK;
    };
    int acl_top_flags = files->check_acl(it->c->identity());
    if(acl) {
      if(!(acl_top_flags & FILE_ACC_WRITE_META)) {
        odlog(ERROR)<<"SOAP: acl: not allowed to write acl"<<std::endl;
        r.error_code=ERR_INSUFFICIENT_ACCESS;
        r.error_description="Not allowed to modify ACL";
        return SOAP_OK;
      };
      res=files->write_acl(it->c->identity(),acl);
    } else {
      if(!(acl_top_flags & FILE_ACC_READ_META)) {
        odlog(ERROR)<<"SOAP: acl: not allowed to read acl"<<std::endl;
        r.error_code=ERR_INSUFFICIENT_ACCESS;
        r.error_description="Not allowed to see ACL";
        return SOAP_OK;
      };
      std::string acl_;
      res=files->read_acl(it->c->identity(),acl_);
      if(res==0) {
        if((r.acl=(char*)soap_malloc(sp,acl_.length()+1)) == NULL) {
          r.error_code=ERR_INTERNAL_ERROR;
          return SOAP_OK;
        };
        strcpy(r.acl,acl_.c_str());
      };
    };    
  } else {
    int acl_flags = it->current_file->check_acl(it->c->identity());
    if(acl) {
      if(acl_flags & FILE_ACC_WRITE_META) {
        res=it->current_file->write_acl(it->c->identity(),acl);
      } else {
        r.error_code=ERR_INSUFFICIENT_ACCESS;
        r.error_description="Not allowed to modify ACL";
      };
    } else {
      if(acl_flags & FILE_ACC_READ_META) {
        std::string acl_;
        res=it->current_file->read_acl(it->c->identity(),acl_);
        if(res==0) {
          if((r.acl=(char*)soap_malloc(sp,acl_.length()+1)) == NULL) {
            r.error_code=ERR_INTERNAL_ERROR;
            return SOAP_OK;
          };
          strcpy(r.acl,acl_.c_str());
        };
      } else {
        r.error_code=ERR_INSUFFICIENT_ACCESS;
        r.error_description="Not allowed to see ACL";
      };
    };
  };
  if((res!=0) && (r.error_code==SUCCESS)) {
    r.error_code=ERR_INTERNAL_ERROR;
  };
  return SOAP_OK;
}

int ns__del(soap *sp, ns__delResponse &r) {
  // initialize response
  r.error_code=0; r.sub_error_code=0; r.error_description=NULL;
  HTTP_SE* it = (HTTP_SE*)(sp->user);
  if(it->current_file == NULL) {
    odlog(ERROR)<<"SOAP: del: file is missing"<<std::endl;
    r.error_code=ERR_OBJECT_MISSING;
    return SOAP_OK;
  };
  SEFiles* files = it->files();
  if(!files) {
    odlog(ERROR)<<"No files"<<std::endl;
    r.error_code=ERR_INTERNAL_ERROR; return SOAP_OK;
  };
  int acl_flags = it->current_file->check_acl(it->c->identity());
  int acl_top_flags = files->check_acl(it->c->identity());
  if((!(acl_flags & FILE_ACC_DELETE)) && (!(acl_top_flags & FILE_ACC_DELETE))) {
    odlog(ERROR)<<"SOAP: del: insufficient access"<<std::endl;
    r.error_code=ERR_INSUFFICIENT_ACCESS;
    r.error_description="Not allowed to delete this file";
    return SOAP_OK;
  };
  SEFile* f = it->current_file;
  if(!it->delete_file(*f)) {
    r.error_code=ERR_INTERNAL_ERROR;
  };
  return SOAP_OK;
}

bool HTTP_SE::delete_file(SEFile& f) {
  // analyze current state
  f.acquire();
  if(f.state_file() == FILE_STATE_DELETING) { // already
    odlog(ERROR)<<"SOAP: del: already deleting"<<std::endl;
    f.release(); return true;
  };
  if(!f.state_file(FILE_STATE_DELETING)) {
    odlog(ERROR)<<"SOAP: del: failed to change file state"<<std::endl;
    f.release(); return false;
  };
/* 
Race condition here. After release() unregistering thread can 
try to remove file. All SEFile* must be replaced
with SEFiles::iterator  
  if(f.state_reg() == REG_STATE_LOCAL) { // ready to by wiped out
    odlog(ERROR)<<"SOAP: del: removing directly"<<std::endl;
    f.release(); it->files()->remove(*f) return true;
  };
*/
  bool to_unregister = false;
  if((f.state_reg() == REG_STATE_ANNOUNCED) &&
     (handle->register_immediately())) {
    if(!f.state_reg(REG_STATE_UNREGISTERING)) {
      if(handle->register_retry()) {
        odlog(ERROR)<<"SOAP: del: failed to set REG_STATE_UNREGISTERING (will retry)"<<std::endl;
      } else {
        odlog(ERROR)<<"SOAP: del: failed to set REG_STATE_UNREGISTERING"<<std::endl; 
        // ???????????
        f.release(); return false;
      };
    } else {
      to_unregister=true;
    };
  };
  f.release();
  if(to_unregister) {
    if(files()->NS()->Unregister(f) != 0) {
      f.state_reg(REG_STATE_ANNOUNCED);
      if(handle->register_retry()) {
        odlog(ERROR)<<"SOAP: del: failed to unregister (will retry)"<<std::endl;
      } else {
        odlog(ERROR)<<"SOAP: del: failed to unregister"<<std::endl;
        return false;
      };
      to_unregister=false;
    } else {
      f.state_reg(REG_STATE_LOCAL);
      files()->remove(f);
    };
  };
  if(!to_unregister) {
    se_thread->attention();
  };
  return true;
}

int HTTP_SE::check_acl(void) {
  int a = 0;
  if(acl_create) a|=FILE_ACC_CREATE;
  //if(acl_replicate) a|=
  if(acl_read) a|=FILE_ACC_READ;
  return a;
}

std::string HTTP_SE::base_url(const char* proto) {
  std::string u = files_url;
  if(proto) {
    std::string::size_type p = u.find("://");
    if(p == std::string::npos) return "";
    p=u.find('/',p+3);
    if(p == std::string::npos) p=u.length();
    if(strcasecmp(proto,"https") == 0) {
      const char* u_ = base_url_by_type("ssl");
      if(!u_) return "";
      u.replace(0,p,u_);
    } else if(strcasecmp(proto,"httpg") == 0) {
      const char* u_ = base_url_by_type("gsi");
      if(!u_) u_=base_url_by_type("gssapi");
      if(!u_) return "";
      u.replace(0,p,u_);
    } else if(strcasecmp(proto,"http") == 0) {
      const char* u_ = base_url_by_type("plain");
      if(!u_) return "";
      u.replace(0,p,u_);
    };
  };
  return u;
}

