#include "../../std.h"
#include <string>
#include <list>
#include <iostream>
#include <fstream>

#include "../../misc/inttostring.h"
#include "../../misc/escaped.h"
#include "../../misc/log_time.h"
#include "../../config/environment.h"
#include "../../auth/auth.h"

#include "logger.h"
#include "logger_soap.nsmap"
#include "logger2_soap.nsmap"


#include <arc/stringconv.h>

class acl_t {
 public:
  std::list<std::string> read;
  std::list<std::string> write;
  std::list<std::string> query;
  std::string sqluser;
  std::string sqlpassword;
  std::string sqldatabase;
  std::string sqlhost;
};

HTTP_Service* logger_service_creator(HTTP_Connector& c,const char* uri,void* arg) {
  acl_t* acl = (acl_t*)arg;
  /* check authorization */
  bool acl_read=false;
  bool acl_write=false;
  bool acl_query=false;
  std::list<AuthEvaluator*>& auths = c.authorizations();
  for(std::list<AuthEvaluator*>::iterator ii=auths.begin();ii!=auths.end();++ii){
    if(!acl_read)
    for(std::list<std::string>::iterator i=acl->read.begin();i!=acl->read.end();++i) {
      if((*(*ii)) == (*i)) {
        if((*(*ii)).evaluate(c.identity()) == AAA_POSITIVE_MATCH) {
          odilog(INFO,c.pid)<<"User is granted 'read' access through acl '"<<(*(*ii)).get_name()<<"'"<<std::endl;
          acl_read=true;
        };
        break;
      };
    };
    if(!acl_write)
    for(std::list<std::string>::iterator i=acl->write.begin();i!=acl->write.end();++i) {
      if((*(*ii)) == (*i)) {
        if((*(*ii)).evaluate(c.identity()) == AAA_POSITIVE_MATCH) {
          odilog(INFO,c.pid)<<"User is granted 'write' access through acl '"<<(*(*ii)).get_name()<<"'"<<std::endl;
          acl_write=true;
        };
        break;
      };
    };
    if(!acl_query)
    for(std::list<std::string>::iterator i=acl->query.begin();i!=acl->query.end();++i) {
      if((*(*ii)) == (*i)) {
        if((*(*ii)).evaluate(c.identity()) == AAA_POSITIVE_MATCH) {
          odilog(INFO,c.pid)<<"User is granted 'full query' access through acl '"<<(*(*ii)).get_name()<<"'"<<std::endl;
          acl_query=true;
        };
        break;
      };
    };
    if(acl_read && acl_write && acl_query) break;
  };
  HTTP_Logger* h = new HTTP_Logger(&c,
              acl_read,acl_write,acl_query,
              acl->sqlhost.length()?acl->sqlhost.c_str():NULL,
              acl->sqldatabase.length()?acl->sqldatabase.c_str():NULL,
              acl->sqluser.length()?acl->sqluser.c_str():NULL,
              acl->sqlpassword.length()?acl->sqlpassword.c_str():NULL);
  return h;
}

bool logger_service_configurator(std::istream& f,const char* uri,HTTP_Service_Properties &prop) {
  acl_t* acl = new acl_t;
  acl->sqluser="nglogger";
  acl->sqldatabase="nglogger";
  acl->sqlhost="";
  for(;!f.eof();) {
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    char* p = buf;
    for(;*p;p++) if(!isspace(*p)) break;
    char* command = p;
    if(*command == '#') continue;
    for(;*p;p++) if(isspace(*p)) break;
    int command_len = p-command;
    if(!command_len) continue;
    if((command_len==3) && (strncmp(command,"end",command_len)==0)) { break; }
    else if((command_len==8) && (strncmp(command,"acl_read",command_len)==0)) {
      for(;;) {
        std::string name;
        int n = input_escaped_string(p,name);
        if(n == 0) break;
        p+=n;
        acl->read.push_back(name);
      };
    } else if((command_len==9) && (strncmp(command,"acl_write",command_len)==0)) {
      for(;;) {
        std::string name;
        int n = input_escaped_string(p,name);
        if(n == 0) break;
        p+=n;
        acl->write.push_back(name);
      };
    } else if((command_len==9) && (strncmp(command,"acl_query",command_len)==0)) {
      for(;;) {
        std::string name;
        int n = input_escaped_string(p,name);
        if(n == 0) break;
        p+=n;
        acl->query.push_back(name);
      };

    } else if((command_len==11) && (strncmp(command,"sqldatabase",command_len)==0)) {
      input_escaped_string(p,acl->sqldatabase);
      std::string::size_type pos = acl->sqldatabase.find('@');
      if(pos != std::string::npos) {
        acl->sqlhost=acl->sqldatabase.substr(pos+1);
        acl->sqldatabase.resize(pos);
      };
    } else if((command_len==10) && (strncmp(command,"sqlcontact",command_len)==0)) {
      int n;
      n=input_escaped_string(p,acl->sqluser);
      if(n) {
        p+=n;
        n=input_escaped_string(p,acl->sqlpassword);
      } else {
        acl->sqlpassword="";
      };
    } else if((command_len==16) && (strncmp(command,"sqlcontactsource",command_len)==0)) {
      std::string path;
      int n;
      n=input_escaped_string(p,path);
      acl->sqluser="";
      acl->sqlpassword="";
      if(n) {
        std::ifstream sf(path.c_str());
        if(sf) {
          char buf[1024];
          istream_readline(sf,buf,sizeof(buf));
          char* p = buf;
          n=input_escaped_string(p,acl->sqluser);
          if(n) {
            p+=n;
            n=input_escaped_string(p,acl->sqlpassword);
          };
        };
      };
    };
  };
  prop.arg=acl;
  prop.subtree=false;
  return true;
}

static char* soap_strdup_l(soap* sp,const char* s,int l = -1) {
  if(l<0) l=strlen(s);
  char* s_ = (char*)soap_malloc(sp,l+1);
  if(s_ == NULL) return NULL;
  memcpy(s_,s,l); s_[l]=0;
  return s_;
}

static std::string sql_string(const char* s) {
  std::string str(s);
  std::string::size_type p = 0;
  for(;;) {
    p=str.find_first_of("\'\\",p);
    if(p == std::string::npos) break;
    str.insert(p,"\\"); p+=2;
  };
  return str;
}

static inline std::string sql_string(const std::string& s) {
  return sql_string(s.c_str());
}

static time_t time2stamp(const char* s) {
  struct tm t;
  sscanf(s,"%u-%u-%u %u:%u:%u",&t.tm_year,&t.tm_mon,&t.tm_mday,&t.tm_hour,&t.tm_min,&t.tm_sec);
  t.tm_year-=1900; t.tm_mon--;
  time_t tl=mktime(&t);
  struct tm t2_;
  struct tm *t2;
  t2=gmtime_r(&tl,&t2_);
  time_t tl2=mktime(t2);
  tl=tl-(tl2-tl);
  return tl;
}

static std::string stamp2time(time_t t) {
  struct tm t2_;
  struct tm *t2;
  if(t==0) return std::string("EPOCH");
  if((t2=gmtime_r(&t,&t2_)) == NULL) return std::string("---"); 
  std::string s=inttostring(t2->tm_year + 1900)+"-"+
     inttostring(t2->tm_mon+1)+"-"+
     inttostring(t2->tm_mday)+" "+
     inttostring(t2->tm_hour)+":"+
     inttostring(t2->tm_min)+":"+
     inttostring(t2->tm_sec);
  return s;
} 

static void remove_hex(char* s) {
  char* p = s;
  char c[5];
  unsigned int i;
  c[0]='0'; c[1]='x'; c[4]=0;
  for(;*p;) {
    if((*p) == '%') {
      p++; if((*p) == 0) { (*s)='%'; s++; break; };
      c[2]=(*p); p++; if((*p) == 0) { (*s)='%'; s++; (*s)=c[2]; s++; break; };
      c[3]=(*p);
      if(sscanf(c,"%x",&i) != 1) {
        (*s)='%'; s++; (*s)=c[2]; s++; (*s)=c[3]; s++;
        p++;
        continue;
      };
      (*s)=(char)i; s++; p++; continue;
    };
    (*s)=(*p); if((*s) == '+') (*s)=' '; s++; p++;
  };
  (*s)=0;
};

static inline void as(std::string &s,const char* p) { if(p) s+=p; };

/* MySQL helpers */
static int mysql_field_num(const MYSQL_FIELD* fields,unsigned int num_fields,const char* name) {
  unsigned int num = 0;
  for(;num<num_fields;num++) {
    if(strcmp(fields[num].name,name)==0) break;
  };
  if(num>=num_fields) return -1;
  return num;
}

static int mysql_get_datetime(MYSQL_ROW sql_row,int n,time_t* t) {
  char* value;
  if(n>=0) {
    value=sql_row[n];
    if(value) {
      (*t)=time2stamp(value);
      return 0;
    };
  };
  (*t)=0;
  return 1;
}

static int mysql_get_datetime(MYSQL_ROW sql_row,int n,time_t** t,struct soap* sp) {
  (*t)=(time_t*)soap_malloc(sp,sizeof(time_t));
  if(*t == NULL) return 1;
  return mysql_get_datetime(sql_row,n,*t);
}

static int mysql_get_string(const MYSQL_ROW sql_row,int n,char** s,struct soap* sp = NULL) {
  char* value;
  if(n>=0) {
    value=sql_row[n];
    if(value) {
      if(sp) {
        (*s)=soap_strdup_l(sp,value);
      } else {
        (*s)=strdup(value);
      };
      return 0;
    };
  };
  (*s)=NULL;
  return 1;
}

static int mysql_get_string(const MYSQL_ROW sql_row,int n,std::string& s) {
  if(n>=0) {
    char* value=sql_row[n];
    if(value) {
      s=value;
      return 0;
    };
  };
  s="";
  return 1;
}

static int mysql_get_string(const MYSQL_ROW sql_row,int n,std::string** s,struct soap* sp = NULL) {
  if(sp) {
    (*s)=soap_new_std__string(sp,-1);
  } else {
    (*s)=new std::string;
  };
  if(!(*s)) return 1;
  return mysql_get_string(sql_row,n,**s);
}

static int mysql_get_int(const MYSQL_ROW sql_row,int n,int* i) {
  char* value;
  if(n>=0) {
    value=sql_row[n];
    if(value) {
      char* e;
      (*i)=strtol(value,&e,0);
      if((*e) == 0) return 0;
    };
  };
  (*i)=0;
  return 1;
}

static int mysql_get_int(const MYSQL_ROW sql_row,int n,int** i,struct soap* sp) {
  (*i)=(int*)soap_malloc(sp,sizeof(int));
  if(*i == NULL) return 1;
  return mysql_get_int(sql_row,n,*i);
}

static std::string create_query(const char *query,ULONG64 offset,ULONG64 size,bool acl_query,const char* identity) {
  //std::string q("SELECT * FROM jobs");
  std::string q("SELECT * FROM v_jobs");
  if((query != NULL) && ((*query) != 0)) {
    /*int l = strlen(query);
    char* query_ = (char*)malloc(2*l+1);
    if(query_ == NULL) return SOAP_OK;
    query_[0]=0;
    mysql_escape_string(query_,query,l);*/
    q+=" WHERE "; 
    //q+=query_;
    q+=query;
    if(!acl_query) {
      q+=" AND globaluserid='"; q+=sql_string(identity); q+="'";
    };
    //free(query_);
  } else {
    if(!acl_query) {
      q+=" WHERE globaluserid='"; q+=sql_string(identity); q+="'";
    };
  };
  //q+=" ORDER BY submissiontime LIMIT "; q+=inttostring(offset)+","+inttostring(size);
  q+=" LIMIT "; q+=inttostring(offset)+","+inttostring(size);
  return q;
}

HTTP_Logger::HTTP_Logger(HTTP_Connector *c_,bool acl_read_,bool acl_write_,bool acl_query_,const char* host,const char* database,const char* user,const char* password):HTTP_ServiceAdv(c_),acl_read(acl_read_),acl_write(acl_write_),acl_query(acl_query_) {
  if(!host) host="localhost";
  /* SOAP */
  soap_init();
  add_namespaces(logger_soap_namespaces);
  add_namespaces(logger2_soap_namespaces);
  sp.user=this;
  /* MySQL */
  mysql_init(&sql);
  unsigned long flag = 0L;
#ifdef CLIENT_MULTI_STATEMENTS
  flag = CLIENT_MULTI_STATEMENTS;
#endif
  if(mysql_real_connect(&sql,host,user,password,database,0,NULL,flag) == NULL) {
    odlog(ERROR)<<"Faled to connect to MySQL server: "<<mysql_error(&sql)<<std::endl;
  };
}

HTTP_Logger::~HTTP_Logger(void) {
  soap_deinit();
  mysql_close(&sql);
}

#ifdef OLD_GSOAP
#error gSOAP older than 2.5.2 is not supported.
#else
void HTTP_Logger::soap_methods(void) {
  odlog(VERBOSE)<<"soap_methods: tag: "<<sp.tag<<std::endl;
  if(!soap_match_tag(&sp,sp.tag,"nl:add")) soap_serve_nl__add(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"nl:get")) soap_serve_nl__get(&sp);
  if(!soap_match_tag(&sp,sp.tag,"nl2:add")) soap_serve___nl2__add(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"nl2:get")) soap_serve___nl2__get(&sp);
  else sp.error = SOAP_NO_METHOD;
}
#endif

/*
 * HTTP methods
 */
HTTP_Error HTTP_Logger::get(const char* uri,int &keep_alive) {
  // Plain web page is not supported anymore
  return HTTP_NOT_IMPLEMENTED;
}

HTTP_Error HTTP_Logger::put(const char* uri,int &keep_alive) {
  return HTTP_NOT_IMPLEMENTED;
}

HTTP_Error HTTP_Logger::post(const char* uri,int &keep_alive) {
  HTTP_Error r = soap_post(uri,keep_alive);
  return r;
}

void concat(std::string& str, const std::string delim, std::vector<std::string> v) {
  for (int i=0; i < v.size(); i++) {
	if (i > 0) str.append(delim);
	str.append(sql_string(v[i]));
  }
}

void split(const std::string& str, const std::string& delim, std::vector<std::string>& output) {
  std::string::size_type offset = 0;
  std::string::size_type delimIndex = 0;
    
  delimIndex = str.find(delim, offset);

  while (delimIndex != std::string::npos) {
    output.push_back(str.substr(offset, delimIndex - offset));
    offset += delimIndex - offset + delim.length();
    delimIndex = str.find(delim, offset);
  }

  output.push_back(str.substr(offset));
}

void concat_i(std::string& str, const std::string delim, std::vector<int> v) {
  for (int i=0; i < v.size(); i++) {
	if (i > 0) str.append(delim);
	str.append(sql_string(tostring(v[i])));
  }
}

void split_i(const std::string& str, const std::string& delim, std::vector<int>& output) {
  std::string::size_type offset = 0;
  std::string::size_type delimIndex = 0;
    
  delimIndex = str.find(delim, offset);

  while (delimIndex != std::string::npos) {
    try {
	  output.push_back(stringtoi(str.substr(offset, delimIndex - offset)));
    } catch(ARCLibError& e) {
      output.push_back(0);
    }
    offset += delimIndex - offset + delim.length();
    delimIndex = str.find(delim, offset);
  }

  try {
    output.push_back(stringtoi(str.substr(offset)));
  } catch(ARCLibError& e) {
    output.push_back(0);
  }
}

/*
 *  SOAP interface functions
 */
int __nl2__add(struct soap *sp,nl2__addRequest* req,nl2__addResponse* _resp) {
   HTTP_Logger* it   = (HTTP_Logger*)(sp->user);
   odilog(ERROR,it->c->pid)<<"Trying to add new job data"<<std::endl;
   if(!req) return   SOAP_OK;
   if(!_resp) return SOAP_OK;
   if(!(_resp->result=soap_new_nl2__Result(sp,-1))) return SOAP_OK;
   _resp->result->soap_default(sp);
   _resp->result->Code=nl2__ResultCode__UndefinedError;
   if(!(it->acl_write)) {
      odilog(ERROR,it->c->pid)<<"Client has no write access"<<std::endl;
      return SOAP_OK;
   };
   
   //odilog(ERROR,it->c->pid)<<"Set option: multi-statements : atemt 2"<<std::endl;
   //mysql_set_server_option(&(it->sql),MYSQL_OPTION_MULTI_STATEMENTS_ON);
   
   if(mysql_real_query(&(it->sql),"LOCK TABLES jobs WRITE, cluster WRITE, user WRITE",49) != 0)
   {
      odilog(ERROR,it->c->pid)<<"Failed to lock table: "<<mysql_error(&(it->sql))<<std::endl;
      return SOAP_OK;
   };
   int n;
   for(n=0;n<req->job.size();n++) {
      if(!((req->job)[n])) continue;
      nl2__UsageRecord& j  = *((req->job)[n]);
      if(j.globaljobid.length() == 0)  {
         odilog(ERROR,it->c->pid)<<"Missing needed argument (job id)"<<std::endl;
         break;
      };
      if(j.globaluserid.length() == 0) {
         odilog(ERROR,it->c->pid)<<"Missing needed argument (user)"<<std::endl;
         break;
      };
      std::string query =  "CALL add_job(";
      std::string params;
      //ngjobid
      params += "'"+sql_string(j.globaljobid)+"'";
      //user
      params += ",'"+sql_string(j.globaluserid)+"'";
      //cluster
      //params += ",'"+sql_string(it->c->identity_subject())+"'"; // -- reporter can be a user, not necessary a cluster
      params += ",'"+sql_string(j.cluster)+"'";
      //xrsl
      params += j.jobdescription ? ",'"+sql_string(*j.jobdescription)+"'" : ",NULL";
      //projectname
      params += j.projectname ? ",'"+sql_string(*j.projectname)+"'" : ",NULL";
      //jobname
      params += j.jobname ? ",'" + sql_string(*j.jobname)+"'" : ",NULL";
      //clienthost
      params += j.submithost ? ",'"+sql_string(*j.submithost)+"'" : ",NULL";
      //requestedcputime
      params += j.requestedcputime ? ",'"+sql_string(inttostring(*j.requestedcputime).c_str())+"'" : ",NULL";
      //requestedwalltime
      params += j.requestedwalltime ? ",'"+sql_string(inttostring(*j.requestedwalltime).c_str())+"'" : ",NULL";
      //requestedmemory
      params += j.requestedmemory ? ",'"+sql_string(inttostring(*j.requestedmemory).c_str())+"'" : ",NULL";
      //requesteddisk
      params += j.requesteddisk ? ",'"+sql_string(inttostring(*j.requesteddisk).c_str())+"'" : ",NULL";
      //submissiontime
      params += j.submissiontime ? ",'" + stamp2time(*j.submissiontime) + "'" : ",NULL";
      //localuser
      params += j.localuserid ? ",'"+sql_string(*j.localuserid)+"'" : ",NULL";
      //queue
      params += j.queue ? ",'"+sql_string(*j.queue)+"'" : ",NULL";
      //lrms
      params += j.lrms ? ",'"+sql_string(*j.lrms)+"'" : ",NULL";
      //localjobid
      params += j.localjobid ? ",'"+sql_string(*j.localjobid)+"'" : ",NULL";
      //lrmssubmissiontime
      params += j.lrmssubmissiontime ? ",'" + stamp2time(*j.lrmssubmissiontime)+"'" : ",NULL";
      //lrmsendtime
      params += j.lrmsendtime ? ",'" + stamp2time(*j.lrmsendtime)+"'" : ",NULL";
      //nodename
      if (j.nodename.size() > 0) {
		std::string tmp;
		concat(tmp, ",", j.nodename);
	    params += ",'"+ tmp +"'"; 
      } else {
		params += ",NULL";
	  }
      //nodecount
      params += j.nodecount ? ",'"+sql_string(inttostring(*j.nodecount).c_str())+"'" : ",NULL";
      //processors
      params += j.processors ? ",'"+sql_string(inttostring(*j.processors).c_str())+"'" : ",NULL";
      //exitcode
      params += j.exitcode ? ",'"+sql_string(inttostring(*j.exitcode).c_str())+"'" : ",NULL";
      //failurestring
      params += j.failurestring ? ",'"+sql_string(*j.failurestring)+"'" : ",NULL";
      //usedcputime
      params += j.usedcputime ? ",'"+sql_string(inttostring(*j.usedcputime).c_str())+"'" : ",NULL";
      //usedwalltime
      params += j.usedwalltime ? ",'"+sql_string(inttostring(*j.usedwalltime).c_str())+"'" : ",NULL";
      //usedmemory
      params += j.usedmemory ? ",'"+sql_string(inttostring(*j.usedmemory).c_str())+"'" : ",NULL";
      //useddisk
      params += j.useddisk ? ",'"+sql_string(inttostring(*j.useddisk).c_str())+"'" : ",NULL";
      //status
      params += j.status ? ",'"+sql_string(*j.status)+"'" : ",NULL";
      //endtime
      params += j.endtime ? ",'" + stamp2time(*j.endtime)+"'" : ",NULL";
      //downloadtime
      params += j.downloadtime ? ",'" + stamp2time(*j.downloadtime)+"'" : ",NULL";
      //uploadtime
      params += j.uploadtime ? ",'" + stamp2time(*j.uploadtime)+"'" : ",NULL";
      //processid
	  if (j.processid.size() > 0) {
		std::string tmp;
		concat_i(tmp, ",", j.processid);
	    params += ",'"+ tmp +"'"; 
      } else {
		params += ",NULL";
	  }
      //charge
      params += j.charge ? ",'"+sql_string(inttostring(*j.charge).c_str())+"'" : ",NULL";
      //network
      params += j.network ? ",'"+sql_string(*j.network)+"'" : ",NULL";
      //stageindata
      params += j.stageindata ? ",'"+sql_string(inttostring(*j.stageindata).c_str())+"'" : ",NULL";
      //stageoutdata
      params += j.stageoutdata ? ",'"+sql_string(inttostring(*j.stageoutdata).c_str())+"'" : ",NULL";
      //usedswap
      params += j.usedswap ? ",'"+sql_string(inttostring(*j.usedswap).c_str())+"'" : ",NULL";
      //servicelevel
      params += j.servicelevel ? ",'"+sql_string(*j.servicelevel)+"'" : ",NULL";
      //runtimeenvironment
      if (j.runtimeenvironment.size() > 0) {
		std::string tmp;
		concat(tmp, ",", j.runtimeenvironment);
	    params += ",'"+ tmp +"'"; 
      } else {
		params += ",NULL";
	  }

      query += params + ")" ;

      odilog(ERROR,it->c->pid)<<"Executing: "<<query<<std::endl;

      if(mysql_real_query(&(it->sql),query.c_str(),query.length()) !=   0) {
         odilog(ERROR,it->c->pid)<<"Failed   to query MySQL server: "<<query<<std::endl;
         odilog(ERROR,it->c->pid)<<"MySQL reported error: "<< mysql_error(&(it->sql))<<std::endl;
         break;
      };
   };
   if(mysql_real_query(&(it->sql),"UNLOCK TABLES",13) != 0) {
      odilog(ERROR,it->c->pid)<<"Warning: Failed to unlock table: "<<mysql_error(&(it->sql))<<std::endl;
   };
   if(n == req->job.size()) _resp->result->Code=nl2__ResultCode__NoError;
   return SOAP_OK;
}

int __nl2__get(struct soap *sp,class nl2__getRequest* req,nl2__getResponse* _resp) {
  HTTP_Logger* it = (HTTP_Logger*)(sp->user);
  if(!req) return SOAP_OK;
  if(!_resp) return SOAP_OK;
  if(!(_resp->result=soap_new_nl2__Result(sp,-1))) return SOAP_OK;
  _resp->result->soap_default(sp);
  _resp->result->Code=nl2__ResultCode__UndefinedError;
  if((!(it->acl_read)) && (!(it->acl_query))) return SOAP_OK;
  std::string q = create_query(req->query?req->query->c_str():NULL,req->offset,req->size,it->acl_query,it->c->identity_subject());
  odilog(ERROR,it->c->pid)<<"Executing: "<<q<<std::endl;
  if(mysql_real_query(&(it->sql),q.c_str(),q.length()) != 0) {
    odilog(ERROR,it->c->pid)<<"Failed to query MySQL server with: "<<q<<std::endl;
    odilog(ERROR,it->c->pid)<<"MySQL error: "<<mysql_error(&(it->sql))<<std::endl;
    return SOAP_OK;
  };
  MYSQL_RES* sql_res;
  if((sql_res=mysql_use_result(&(it->sql))) == NULL) {
    odilog(ERROR,it->c->pid)<<"Failed to initiate retrieving results from MySQL server: "<<mysql_error(&(it->sql))<<std::endl;
    return SOAP_OK;
  };
  unsigned int num_fields = mysql_num_fields(sql_res);
  MYSQL_FIELD* fields = mysql_fetch_fields(sql_res);
  
  unsigned int globaljobid_num = mysql_field_num(fields,num_fields,"globaljobid");     
  unsigned int globaluserid_num = mysql_field_num(fields,num_fields,"globaluserid");
  unsigned int cluster_num = mysql_field_num(fields,num_fields,"cluster");
  unsigned int jobdescription_num = mysql_field_num(fields,num_fields,"jobdescription");
  unsigned int projectname_num = mysql_field_num(fields,num_fields,"projectname");
  unsigned int jobname_num = mysql_field_num(fields,num_fields,"jobname");
  unsigned int submithost_num = mysql_field_num(fields,num_fields,"submithost");
  unsigned int requestedcputime_num = mysql_field_num(fields,num_fields,"requestedcputime");
  unsigned int requestedwalltime_num = mysql_field_num(fields,num_fields,"requestedwalltime");
  unsigned int requestedmemory_num = mysql_field_num(fields,num_fields,"requestedmemory");
  unsigned int requesteddisk_num = mysql_field_num(fields,num_fields,"requesteddisk");
  unsigned int submissiontime_num = mysql_field_num(fields,num_fields,"submissiontime");
  unsigned int localuserid_num = mysql_field_num(fields,num_fields,"localuserid");
  unsigned int queue_num = mysql_field_num(fields,num_fields,"queue");
  unsigned int lrms_num = mysql_field_num(fields,num_fields,"lrms");
  unsigned int localjobid_num = mysql_field_num(fields,num_fields,"localjobid");
  unsigned int lrmssubmissiontime_num = mysql_field_num(fields,num_fields,"lrmssubmissiontime");
  unsigned int lrmsendtime_num = mysql_field_num(fields,num_fields,"lrmsendtime");
  unsigned int nodename_num = mysql_field_num(fields,num_fields,"nodename");
  unsigned int nodecount_num = mysql_field_num(fields,num_fields,"nodecount");
  unsigned int processors_num = mysql_field_num(fields,num_fields,"processors");
  unsigned int exitcode_num = mysql_field_num(fields,num_fields,"exitcode");
  unsigned int failurestring_num = mysql_field_num(fields,num_fields,"failurestring");
  unsigned int usedcputime_num = mysql_field_num(fields,num_fields,"usedcputime");
  unsigned int usedwalltime_num = mysql_field_num(fields,num_fields,"usedwalltime");
  unsigned int usedmemory_num = mysql_field_num(fields,num_fields,"usedmemory");
  unsigned int useddisk_num = mysql_field_num(fields,num_fields,"useddisk");
  unsigned int status_num = mysql_field_num(fields,num_fields,"status");
  unsigned int endtime_num = mysql_field_num(fields,num_fields,"endtime");
  unsigned int downloadtime_num = mysql_field_num(fields,num_fields,"downloadtime");
  unsigned int uploadtime_num = mysql_field_num(fields,num_fields,"uploadtime");
  unsigned int processid_num = mysql_field_num(fields,num_fields,"processid");
  unsigned int charge_num = mysql_field_num(fields,num_fields,"charge");
  unsigned int network_num = mysql_field_num(fields,num_fields,"network");
  unsigned int stageindata_num = mysql_field_num(fields,num_fields,"stageindata");
  unsigned int stageoutdata_num = mysql_field_num(fields,num_fields,"stageoutdata");
  unsigned int usedswap_num = mysql_field_num(fields,num_fields,"usedswap");
  unsigned int servicelevel_num = mysql_field_num(fields,num_fields,"servicelevel");
  unsigned int runtimeenvironment_num = mysql_field_num(fields,num_fields,"runtimeenvironment");
  int count = 0;
  for(;;) {
    if(count >= req->size) break;
    MYSQL_ROW sql_row;
    nl2__UsageRecord* j = soap_new_nl2__UsageRecord(sp,-1);
    if(j == NULL) break;
    j->soap_default(sp);
    if((sql_row=mysql_fetch_row(sql_res)) == NULL) break;
	
	mysql_get_string(sql_row,globaljobid_num,j->globaljobid);
	mysql_get_string(sql_row,globaluserid_num,j->globaluserid);
	mysql_get_string(sql_row,cluster_num,j->cluster);
	mysql_get_string(sql_row,jobdescription_num,&(j->jobdescription),sp);
	mysql_get_string(sql_row,projectname_num,&(j->projectname),sp);
	mysql_get_string(sql_row,jobname_num,&(j->jobname),sp);
	mysql_get_string(sql_row,submithost_num,&(j->submithost),sp);
	mysql_get_int(sql_row,requestedcputime_num,&(j->requestedcputime),sp);	
	mysql_get_int(sql_row,requestedwalltime_num,&(j->requestedwalltime),sp);
	mysql_get_int(sql_row,requestedmemory_num,&(j->requestedmemory),sp);
	mysql_get_int(sql_row,requesteddisk_num,&(j->requesteddisk),sp);
	mysql_get_datetime(sql_row,submissiontime_num,&(j->submissiontime),sp);
	mysql_get_string(sql_row,localuserid_num,&(j->localuserid),sp);
	mysql_get_string(sql_row,queue_num,&(j->queue),sp);
	mysql_get_string(sql_row,lrms_num,&(j->lrms),sp);
	mysql_get_string(sql_row,localjobid_num,&(j->localjobid),sp);
	mysql_get_datetime(sql_row,lrmssubmissiontime_num,&(j->lrmssubmissiontime),sp);
	mysql_get_datetime(sql_row,lrmsendtime_num,&(j->lrmsendtime),sp);
	std::string tmp; mysql_get_string(sql_row,nodename_num,tmp); split(tmp, ",", j->nodename);
	mysql_get_int(sql_row,nodecount_num,&(j->nodecount),sp);
	mysql_get_int(sql_row,processors_num,&(j->processors),sp);
	mysql_get_int(sql_row,exitcode_num,&(j->exitcode),sp);
	mysql_get_string(sql_row,failurestring_num,&(j->failurestring),sp);
	mysql_get_int(sql_row,usedcputime_num,&(j->usedcputime),sp);
	mysql_get_int(sql_row,usedwalltime_num,&(j->usedwalltime),sp);
	mysql_get_int(sql_row,usedmemory_num,&(j->usedmemory),sp);
	mysql_get_int(sql_row,useddisk_num,&(j->useddisk),sp);
	mysql_get_string(sql_row,status_num,&(j->status),sp);
	mysql_get_datetime(sql_row,endtime_num,&(j->endtime),sp);
	mysql_get_int(sql_row,downloadtime_num,&(j->downloadtime),sp);
	mysql_get_int(sql_row,uploadtime_num,&(j->uploadtime),sp);
	tmp = ""; mysql_get_string(sql_row,processid_num,tmp); split_i(tmp, ",", j->processid);
	mysql_get_int(sql_row,charge_num,&(j->charge),sp);
	mysql_get_string(sql_row,network_num,&(j->network),sp);
	mysql_get_int(sql_row,stageindata_num,&(j->stageindata),sp);
	mysql_get_int(sql_row,stageoutdata_num,&(j->stageoutdata),sp);
	mysql_get_int(sql_row,usedswap_num,&(j->usedswap),sp);
	mysql_get_string(sql_row,servicelevel_num,&(j->servicelevel),sp);
	tmp = ""; mysql_get_string(sql_row,runtimeenvironment_num,tmp); split(tmp, ",", j->runtimeenvironment);
	
    _resp->job.push_back(j);
    count++;
  };
  _resp->result->Code=nl2__ResultCode__NoError;
  mysql_free_result(sql_res);
  return SOAP_OK;
}

// Converts 
void convert(struct soap* sp, nl__jobinfo *info,nl2__addRequest* req) {
  nl2__UsageRecord* ur = soap_new_nl2__UsageRecord(sp,-1);
  ur->soap_default(sp);
  ur->submissiontime=&(info->start);
  ur->endtime=&(info->end);
  if(info->cluster) {
    ur->cluster=std::string(info->cluster);
  } else {
    ur->cluster=std::string("");
  }
  ur->globaluserid=std::string(info->user);
  ur->globaljobid=std::string(info->id);
  if(info->name) {
    ur->jobname=new std::string(info->name);
  }
  if(info->failure) {
    ur->failurestring=new std::string(info->failure);
  }
  if(info->lrms) {
    ur->lrms=new std::string(info->lrms);
  }
  if(info->queue) {
    ur->queue=new std::string(info->queue);
  }
  if(info->rsl) {
    ur->jobdescription=new std::string(info->rsl);
  }
  if(info->ui) {
    ur->submithost=new std::string(info->ui);
  }
  ur->usedcputime=&(info->usedcpu);
  ur->usedmemory=&(info->usedmem);
  req->job.push_back(ur);
}

void convert(nl2__addResponse* _resp, int &r) {
  if (_resp->result->Code==nl2__ResultCode__NoError) {
    r=0;
  } else {
    r=1;
  }
}

// Converts old query to a new format
void convert(char *query, ULONG64 offset,nl2__getRequest* req) {
  std::string* str = new std::string(query);
  req->query = str;
  req->offset=(unsigned int)offset;
  req->size=100;
}

// Converts old query response to a new format
void convert(struct soap* sp, nl2__getResponse* _resp,array_jobinfo &r) {
  r.job = soap_new_nl__jobinfo(sp,100);
  std::vector<nl2__UsageRecord*>::iterator urIterator;
  int i = 0;
  for(urIterator=_resp->job.begin();urIterator!=_resp->job.end();++urIterator) {
    nl2__UsageRecord* ur = (nl2__UsageRecord*)(*urIterator);
    r.job[i].soap_default(sp);
    r.job[i].start = *(ur->submissiontime);
    r.job[i].end = *(ur->endtime);
    r.job[i].cluster = (char*)ur->cluster.c_str();
    r.job[i].user = (char*)ur->globaluserid.c_str();
    r.job[i].id = (char*)ur->globaljobid.c_str();
    r.job[i].name = (char*)ur->jobname->c_str();
    r.job[i].failure = (char*)ur->failurestring->c_str();
    r.job[i].lrms = (char*)ur->lrms->c_str();
    r.job[i].queue = (char*)ur->queue->c_str();
    r.job[i].rsl = (char*)ur->jobdescription->c_str();
    r.job[i].ui = (char*)ur->submithost->c_str();
    r.job[i].usedcpu = *(ur->usedcputime);
    r.job[i].usedmem = *(ur->usedmemory);
    i++;
  }
  r.__size_job=i;
}

int nl__add(struct soap *sp, nl__jobinfo *info, int &r) {
  int ret_code = SOAP_OK;
  r=1; // error by default
  HTTP_Logger* it = (HTTP_Logger*)(sp->user);
  if(!(it->acl_write)) {
    odilog(ERROR,it->c->pid)<<"Client has no write access"<<std::endl;
  } else if(info->start == 0) {
    odilog(ERROR,it->c->pid)<<"Missing needed argument (start time)"<<std::endl;
  } else if(info->user == NULL) {
    odilog(ERROR,it->c->pid)<<"Missing needed argument (user)"<<std::endl;
  } else if(info->id == NULL) {
    odilog(ERROR,it->c->pid)<<"Missing needed argument (job id)"<<std::endl;
  } else {
    nl2__addRequest* req = soap_new_nl2__addRequest(sp,-1);;
    nl2__addResponse* _resp = soap_new_nl2__addResponse(sp,-1);;
    //convert query: old -> new
    convert(sp, info, req);
    //call method
    ret_code = __nl2__add(sp,req,_resp);
    //convert result: new -> old
    convert(_resp,r);
  }
  r=0; // passed
  return ret_code;
}

/*int nl__add(struct soap *sp, nl__jobinfo *info, int &r) {
  HTTP_Logger* it = (HTTP_Logger*)(sp->user);
  //struct tm t_;
  //struct tm *t;

  r=1; // error by default

  if(!(it->acl_write)) {
    odilog(ERROR,it->c->pid)<<"Client has no write access"<<std::endl;
    return SOAP_OK;
  };
  if(info->start == 0) {
    odilog(ERROR,it->c->pid)<<"Missing needed argument (start time)"<<std::endl;
    return SOAP_OK;
  };
  //cluster is identified by subject
  //if(info->cluster == NULL) {
  //  odilog(ERROR,c->pid)<<"Missing needed argument (start time)"<<std::endl;
  //  return SOAP_OK;
  //};

  if(info->user == NULL) {
    odilog(ERROR,it->c->pid)<<"Missing needed argument (user)"<<std::endl;
    return SOAP_OK;
  };
  if(info->id == NULL) {
    odilog(ERROR,it->c->pid)<<"Missing needed argument (job id)"<<std::endl;
    return SOAP_OK;
  };
  if(mysql_real_query(&(it->sql),"LOCK TABLES jobs WRITE",22) != 0) {
    odilog(ERROR,it->c->pid)<<"Failed to lock table: "<<mysql_error(&(it->sql))<<std::endl;
    return SOAP_OK;
  };
  std::string tt;
  tt = stamp2time(info->start);
  std::string query = "UPDATE jobs SET ";
  std::string set = "start='"+tt+"'";
  if(info->end) {
    if(info->end > 86400) { // workaround for old bug
      set+=" , end='"+stamp2time(info->end)+"'";
    };
  };
//  set+=std::string(" , cluster='")+sql_string(info->cluster)+"'";
  set+=std::string(" , cluster='")+sql_string(it->c->identity_subject())+"'";
  set+=std::string(" , user='")+sql_string(info->user)+"'";
  set+=std::string(" , id='")+sql_string(info->id)+"'";
  if(info->name) set+=std::string(" , name='")+sql_string(info->name)+"'";
  if(info->failure) set+=std::string(" , failure='")+sql_string(info->failure)+"'";
  if(info->lrms) set+=std::string(" , lrms='")+sql_string(info->lrms)+"'";
  if(info->queue) set+=std::string(" , queue='")+sql_string(info->queue)+"'";
  if(info->rsl) set+=std::string(" , rsl='")+sql_string(info->rsl)+"'";
  if(info->ui) set+=std::string(" , ui='")+sql_string(info->ui)+"'";
  if(info->usedcpu) set+=std::string(" , usedcpu='")+sql_string(inttostring(info->usedcpu).c_str())+"'";
  if(info->usedmem) set+=std::string(" , usedmem='")+sql_string(inttostring(info->usedmem).c_str())+"'";
  query+=set+" WHERE cluster='"+sql_string(it->c->identity_subject())+
               "' && user='"+sql_string(info->user)+
               "' && id='"+sql_string(info->id)+
               "' && start='"+tt+"'";
  if(mysql_real_query(&(it->sql),query.c_str(),query.length()) != 0) {
    odilog(ERROR,it->c->pid)<<"Failed to query MySQL server: "<<query<<std::endl;
    odilog(ERROR,it->c->pid)<<"MySQL reported error: "<<mysql_error(&(it->sql))<<std::endl;
    return SOAP_OK;
  };
  const char* sql_info = mysql_info(&(it->sql));
  if(sql_info==NULL) {
    odilog(ERROR,it->c->pid)<<"Query did not return an information"<<std::endl;
    return SOAP_OK;
  };
  unsigned long long int matched, changed, warnings;
  if(sscanf(sql_info,"Rows matched: %Lu Changed: %Lu Warnings: %Lu",
                                          &matched,&changed,&warnings) != 3) {
    odilog(ERROR,it->c->pid)<<"Query did not return an interpreatable information: "<<info<<std::endl;
    return SOAP_OK;
  };
  if(matched == 0) {
    query = "INSERT jobs SET "+set;
    if(mysql_real_query(&(it->sql),query.c_str(),query.length()) != 0) {
      odilog(ERROR,it->c->pid)<<"Failed to query MySQL server: "<<query<<std::endl;
      odilog(ERROR,it->c->pid)<<"MySQL reported error: "<<mysql_error(&(it->sql))<<std::endl;
      return SOAP_OK;
    };
  };
  if(mysql_real_query(&(it->sql),"UNLOCK TABLES",13) != 0) {
    odilog(ERROR,it->c->pid)<<"Warning: Failed to unlock table: "<<mysql_error(&(it->sql))<<std::endl;
  };
  odilog(INFO,it->c->pid)<<"Logger: stored data from "<<it->c->identity_subject()<<" about "<<info->id<<std::endl;
  r=0; // passed
  return SOAP_OK;
}*/

int nl__get(struct soap* sp, char *query, ULONG64 offset, array_jobinfo &r) {
  int ret_code = SOAP_OK;
  HTTP_Logger* it = (HTTP_Logger*)(sp->user);
  r.job=NULL; r.__size_job=0; // kind of error. TODO - report errors properly 
  if((!(it->acl_read)) && (!(it->acl_query))) {
    odilog(ERROR,it->c->pid)<<"Client has no read/query access"<<std::endl;
  } else { 
    nl2__getRequest* req = soap_new_nl2__getRequest(sp,-1);
    nl2__getResponse* _resp = soap_new_nl2__getResponse(sp,-1);
    //convert query: old -> new
    convert(query,offset,req);
    //callmethod
    ret_code = __nl2__get(sp,req,_resp);
    //convert result: new -> old
    convert(sp,_resp,r);
  }
  return ret_code;
}

/*int nl__get(struct soap* sp, char *query, ULONG64 offset, array_jobinfo &r) {
  HTTP_Logger* it = (HTTP_Logger*)(sp->user);
  r.job=NULL; r.__size_job=0; // kind of error. TODO - report errors properly 
  if((!(it->acl_read)) && (!(it->acl_query))) return SOAP_OK;
  std::string q = create_query(query,offset,100,it->acl_query,it->c->identity_subject());
  if(mysql_real_query(&(it->sql),q.c_str(),q.length()) != 0) {
    odilog(ERROR,it->c->pid)<<"Failed to query MySQL server with: "<<q<<std::endl;
    odilog(ERROR,it->c->pid)<<"MySQL error: "<<mysql_error(&(it->sql))<<std::endl;
    return SOAP_OK;
  };
  MYSQL_RES* sql_res;
  if((sql_res=mysql_use_result(&(it->sql))) == NULL) {
    odilog(ERROR,it->c->pid)<<"Failed to initiate retrieving results from MySQL server: "<<mysql_error(&(it->sql))<<std::endl;
    return SOAP_OK;
  }; 
  unsigned int num_fields = mysql_num_fields(sql_res);
  MYSQL_FIELD* fields = mysql_fetch_fields(sql_res);
  unsigned int start_num = mysql_field_num(fields,num_fields,"start");
  unsigned int end_num = mysql_field_num(fields,num_fields,"end");
  unsigned int cluster_num = mysql_field_num(fields,num_fields,"cluster");
  unsigned int id_num = mysql_field_num(fields,num_fields,"id");
  unsigned int user_num = mysql_field_num(fields,num_fields,"user");
  unsigned int name_num = mysql_field_num(fields,num_fields,"name");
  unsigned int failure_num = mysql_field_num(fields,num_fields,"failure");
  unsigned int lrms_num = mysql_field_num(fields,num_fields,"lrms");
  unsigned int queue_num = mysql_field_num(fields,num_fields,"queue");
  unsigned int rsl_num = mysql_field_num(fields,num_fields,"rsl");
  unsigned int ui_num = mysql_field_num(fields,num_fields,"ui");
  unsigned int usedcpu_num = mysql_field_num(fields,num_fields,"usedcpu");
  unsigned int usedmem_num = mysql_field_num(fields,num_fields,"usedmem");
  int count = 0;
  // limit is always 100
  r.job = soap_new_nl__jobinfo(sp,100);
  if(r.job == NULL) {
    mysql_free_result(sql_res);
    return SOAP_OK;
  };
  for(;;) {
    if(count >= 100) break;
    r.job[count].soap_default(sp);
    MYSQL_ROW sql_row;
    if((sql_row=mysql_fetch_row(sql_res)) == NULL) break;
    mysql_get_datetime(sql_row,start_num,&(r.job[count].start));
    mysql_get_datetime(sql_row,end_num,&(r.job[count].end));
    mysql_get_string(sql_row,cluster_num,&(r.job[count].cluster),sp);
    mysql_get_string(sql_row,user_num,&(r.job[count].user),sp);
    mysql_get_string(sql_row,id_num,&(r.job[count].id),sp);
    mysql_get_string(sql_row,name_num,&(r.job[count].name),sp);
    mysql_get_string(sql_row,failure_num,&(r.job[count].failure),sp);
    mysql_get_string(sql_row,lrms_num,&(r.job[count].lrms),sp);
    mysql_get_string(sql_row,queue_num,&(r.job[count].queue),sp);
    mysql_get_string(sql_row,rsl_num,&(r.job[count].rsl),sp);
    mysql_get_string(sql_row,ui_num,&(r.job[count].ui),sp);
    mysql_get_int(sql_row,usedcpu_num,&(r.job[count].usedcpu));
    mysql_get_int(sql_row,usedmem_num,&(r.job[count].usedmem));
    count++;
  };
  r.__size_job=count;
  mysql_free_result(sql_res);
  return SOAP_OK;
}*/

