#include "../../std.h"

#include <arc/jobcontroller.h>
#include <arc/notify.h>
#include <arc/stringconv.h>

#include "../../datamove/datamove.h"

#include "jobs_client.h"

#include "jobs_soap.nsmap"

class JobHTTPControlInit {
 public:
  JobHTTPControlInit(void) {
    try {
      JobController* ctrl = JobController::GetJobController();
      JobHTTPControl* hctrl = new JobHTTPControl();
      if(ctrl && hctrl) {
        ctrl->AddProtocol("httpg",hctrl);
        ctrl->AddProtocol("https",hctrl);
      };
    } catch(std::exception) { };
  };
};

static JobHTTPControlInit hinit;

JobHTTPControl::JobHTTPControl(void):client_(NULL) {
}

JobHTTPControl::~JobHTTPControl(void) {
  if(client_) delete client_;
}

JobHTTPControl::operator bool(void) {
  return (client_ != NULL);
}

bool JobHTTPControl::operator!(void) {
  return (client_ == NULL);
}

bool JobHTTPControl::set_connection(const std::string& url,std::string& path,int timeout) {
  try {
    URL u(url);
    path=u.Path();
    return set_connection(u,timeout);
  } catch(URLError) {
    notify(ERROR)<<"Malformed URL "<<url<<std::endl;
    return false;
  };
}

bool JobHTTPControl::set_connection(const URL& url,int timeout) {
  if(!url) return false;
  if(url_) {
    if((url_.Protocol() == url.Protocol()) &&
       (url_.Host() == url.Host()) &&
       (url_.Port() == url.Port())) {
      url_=url;
    } else { // reinitialize for different server
      url_=url; if(client_) delete client_; client_=NULL;
    };
  } else { // initialize for first connection
    url_=url; if(client_) delete client_; client_=NULL;
  };
  if(client_ == NULL) {
    if((url_.Protocol() != "http") &&
       (url_.Protocol() != "https") &&
       (url_.Protocol() != "httpg")) {
      notify(ERROR)<<"Unsupported protocol in URL "<<url_<<std::endl;
      return false;
    };
    std::string soap_url = url_.Protocol()+"://"+url_.Host()+":"+
                           tostring<int>(url_.Port());
    client_ = new HTTP_ClientSOAP(soap_url.c_str(),&soap_);
    client_->AddSOAPNamespaces(jobs_soap_namespaces);
  } else {
    client_->reset();
  };
  // connect to server if not yet connected
  if(client_->connect() != 0) {
    notify(ERROR)<<"Failed to connect to "<<url_<<std::endl;
    return false;
  };
  return true;
}

std::string JobHTTPControl::Submit(const URL& url,const std::string& rsl,int timeout,bool disconnectafteruse) {
  jobid_=""; joburl_="";
  if(!set_connection(url,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  std::vector<nj__job> job;
  std::vector<nj__jobResult> jobResult;
  job.resize(1);
  job[0].soap_default(&soap_);
  //char* delegation;
  job[0].description=(char*)(rsl.c_str());;
  if((soap_err=soap_call_njf__create(&soap_,client_->SOAP_URL(url.Path().c_str()).c_str(),"create",job,jobResult))  != SOAP_OK) {
    notify(ERROR)<<"Failed to submit job"<<std::endl;
    if(GetNotifyLevel() >= INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Service responded with error");
  } else if(jobResult.size() != 1) {
    notify(ERROR)<<"Wrong number of records in response"<<std::endl;
    client_->disconnect(); client_->reset();
    throw JobControlError("Wrong response from service");
  } else if((jobResult[0].code != NULL) && (*(jobResult[0].code) != 0)) {
    if(jobResult[0].description) {
      notify(ERROR)<<"Submission failed: "<<*(jobResult[0].code)<<
                     ": "<<jobResult[0].description<<std::endl;
    } else {
      notify(ERROR)<<"Submission failed: "<<*(jobResult[0].code)<<std::endl;
    };
    client_->disconnect(); client_->reset();
    throw JobControlError("Submission failed: "+
                          tostring<int>(*(jobResult[0].code))+
                          ": "+jobResult[0].description);
  } else if(jobResult[0].job == NULL) {
    notify(ERROR)<<"Submission did not return job's information"<<std::endl;
    client_->disconnect(); client_->reset();
    throw JobControlError("Service did not return job's information");
  } else if(jobResult[0].job->id == NULL) {
    notify(ERROR)<<"Submission did not return job's ID"<<std::endl;
    client_->disconnect(); client_->reset();
    throw JobControlError("Service did not return job's ID");
  } else if(jobResult[0].job->url == NULL) {
    notify(ERROR)<<"Submission did not return job's URL"<<std::endl;
    client_->disconnect(); client_->reset();
    throw JobControlError("Service did not return job's URL");
  };
  jobid_=jobResult[0].job->id; joburl_=jobResult[0].job->url;
  if(disconnectafteruse) client_->disconnect();
  client_->reset();
  return joburl_;
}

std::string JobHTTPControl::Submit(const URL& url,const std::string& rsl,std::multimap<std::string,std::string>& files,int timeout,bool disconnectafteruse) {
  Submit(url,rsl,false,timeout);
  if(jobid_.length() == 0) return "";
  std::map<std::string,std::string>::iterator fileit;
  for(fileit=files.begin();fileit!=files.end();++fileit) {
    unsigned char buffer[1024*1024];
    unsigned long long int offset = 0;
    struct stat st;
    if(stat(fileit->first.c_str(),&st) != 0) {
      notify(ERROR)<<"Can't find file "<<fileit->first<<std::endl;
      client_->reset(); client_->disconnect(); jobid_=""; return "";
    };
    int fd = open(fileit->first.c_str(),O_RDONLY);
    if(fd == -1) {
      notify(ERROR)<<"Can't open file "<<fileit->first<<std::endl;
      client_->reset(); client_->disconnect(); jobid_=""; return "";
    };
    std::string http_path=URL(joburl_.c_str()).Path()+"/"+fileit->second;
    for(;;) {
      ssize_t l = read(fd,buffer,sizeof(buffer));
      if(l == 0) break;
      if(l == -1) {
        notify(ERROR)<<"Failed to read from file "<<fileit->first<<std::endl;
        client_->reset(); client_->disconnect(); jobid_="";
        throw JobControlError("Failed to read local file");
      };
      if(client_->connect() != 0) {
        notify(ERROR)<<"Failed to connect to "<<url_<<std::endl;
        client_->reset(); client_->disconnect(); jobid_="";
        throw JobControlError("Failed to store file");
      };
      if(client_->PUT(http_path.c_str(),offset,l,buffer,st.st_size,true) != 0) {
        notify(ERROR)<<"Failed to store file "<<fileit->first<<" at "<<joburl_<<std::endl;
        client_->reset(); client_->disconnect(); jobid_="";
        throw JobControlError("Failed to store file");
      };
      offset+=l;
    };
  };
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return joburl_;
}

void JobHTTPControl::Cancel(const std::string& jobid,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  nj__job job;
  job.soap_default(&soap_);
  nj__jobResult jobResult;
  jobResult.soap_default(&soap_);
  job.state="FINISHED";
  if((soap_err=soap_call_njc__modify(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"modify",&job,jobResult)) != SOAP_OK) {
    notify(ERROR)<<"Failed to cancel job"<<std::endl;
    if(GetNotifyLevel() >= INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to cancel job");
  } else if((jobResult.code != NULL) && (*(jobResult.code) != 0)) {
    if(jobResult.description) {
      notify(ERROR)<<"Cancel failed: "<<*(jobResult.code)<<
                     ": "<<jobResult.description<<std::endl;
    } else {
      notify(ERROR)<<"Cancel failed: "<<*(jobResult.code)<<std::endl;
    };
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to cancel job");
  };
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return;
}

void JobHTTPControl::Clean(const std::string& jobid,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  nj__job job;
  job.soap_default(&soap_);
  nj__jobResult jobResult;
  jobResult.soap_default(&soap_);
  job.state="";
  if((soap_err=soap_call_njc__modify(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"modify",&job,jobResult)) != SOAP_OK) {
    notify(ERROR)<<"Failed to clean job"<<std::endl;
    if(GetNotifyLevel() >= INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to clean job");
  } else if((jobResult.code != NULL) && (*(jobResult.code) != 0)) {
    if(jobResult.description) {
      notify(ERROR)<<"Clean failed: "<<*(jobResult.code)<<
                ": "<<jobResult.description<<std::endl;
    } else {
      notify(ERROR)<<"Clean failed: "<<*(jobResult.code)<<std::endl;
    };
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to clean job");
  };
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return;
}

void JobHTTPControl::RenewCreds(const std::string& jobid,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  nj__job job;
  job.soap_default(&soap_);
  nj__jobResult jobResult;
  jobResult.soap_default(&soap_);
  job.delegation="";
  if((soap_err=soap_call_njc__modify(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"modify",&job,jobResult)) != SOAP_OK) {
    notify(ERROR)<<"Failed to clean job"<<std::endl;
    if(GetNotifyLevel() >= INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to clean job");
  } else if((jobResult.code != NULL) && (*(jobResult.code) != 0)) {
    if(jobResult.description) {
      notify(ERROR)<<"Renew failed: "<<*(jobResult.code)<<
                ": "<<jobResult.description<<std::endl;
    } else {
      notify(ERROR)<<"Renew failed: "<<*(jobResult.code)<<std::endl;
    };
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to renew job's credentials");
  };
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return;
}

void JobHTTPControl::Resume(const std::string& jobid,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  char* log = NULL;
  if((soap_err=soap_call_njc__queryLog(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"queryFiles","local",log)) != SOAP_OK) {
    notify(ERROR)<<"Failed to query job"<<std::endl;
    if(GetNotifyLevel() > INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to query job");
  };
  std::string state;
  char* p = log;
  for(;;) {
    p=strstr(p,"failedstate=");
    if(p == NULL) break;
    if(p == log) break;
    if((*(p-1)) == '\n') break;
    if((*(p-1)) == '\r') break;
    p++;
  };
  if(p == NULL) 
    throw JobControlError("Failed to find out state at which job failed");
  p+=12;
  state.assign(p,strcspn(p,"\r\n"));
  nj__job job;
  job.soap_default(&soap_);
  nj__jobResult jobResult;
  jobResult.soap_default(&soap_);
  job.state=(char*)(state.c_str());
  if((soap_err=soap_call_njc__modify(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"modify",&job,jobResult)) != SOAP_OK) {
    notify(ERROR)<<"Failed to resume job"<<std::endl;
    if(GetNotifyLevel() > INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to resume job");
  } else if((jobResult.code != NULL) && (*(jobResult.code) != 0)) {
    if(jobResult.description) {
      notify(ERROR)<<"Resume failed: "<<*(jobResult.code)<<
                ": "<<jobResult.description<<std::endl;
    } else {
      notify(ERROR)<<"Resume failed: "<<*(jobResult.code)<<std::endl;
    };
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to resume job");
  };
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return;
}

Job JobHTTPControl::GetJobInfo(const std::string& jobid,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  nj__job job;
  job.soap_default(&soap_);
  if((soap_err=soap_call_njc__query(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"query",NULL,job)) != SOAP_OK) {
    notify(ERROR)<<"Failed to query job"<<std::endl;
    if(GetNotifyLevel() > INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to query job");
  };
  Job j;
  if(job.url) {
    j.id=job.url;
  } else {
    j.id=jobid;
  };
  if(job.state) {
    j.status=job.state;
    if(j.status == "PENDING:ACCEPTED") {
      j.status = "PREPARED";
    } else if(j.status == "PENDING:INLRMS") {
      j.status = "EXECUTED";
    } else if(j.status == "SUBMIT") {
      j.status = "SUBMITTING";
    } else if(j.status == "CANCELING") {
      j.status = "KILLING";
    };
  };
  j.exitcode=-1;
  if(job.failure_code) j.exitcode=*(job.failure_code);
  if(job.failure_reason) j.errors=job.failure_reason;
  //std::vector<nj__attribute> attribute;
  //      std::string owner;
  //      std::string cluster;
  //      std::string queue;
  //      std::string sstdout;
  //      std::string sstderr;
  //      std::string sstdin;
  //      std::string rerunable;
  //      long requested_cpu_time;
  //      long requested_wall_time;
  //      int queue_rank;
  //      std::string comment;
  //      std::string submission_ui;
  //      Time submission_time;
  //      Time completion_time;
  //      long used_cpu_time;
  //      long used_wall_time;
  //      Time erase_time;
  //      int used_memory;
  //      std::string job_name;
  //      std::list<RuntimeEnvironment> runtime_environments;
  //      int cpu_count;
  //      std::list<std::string> execution_nodes;
  //      std::string gmlog;
  //      std::string client_software;
  //      Time proxy_expire_time;
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return j;
}

void JobHTTPControl::Get(const std::string& jobid,const std::string& localdir,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  nj__jobFiles files;
  files.soap_default(&soap_);
  if((soap_err=soap_call_njc__queryFiles(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"queryFiles",NULL,files)) != SOAP_OK) {
    notify(ERROR)<<"Failed to query job files"<<std::endl;
    if(GetNotifyLevel() > INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to query job files");
  };
  if(disconnectafteruse) client_->disconnect();
  for(std::vector<char*>::iterator f = files.url.begin();
      f!=files.url.end();++f) {
    const char* url = *f;
    if(!url) continue;
    if(strncasecmp(jobid.c_str(),url,jobid.length()) != 0)
      throw JobControlError("URL of file does not match URL of job");
    const char* name = url+jobid.length();
    Get(jobid,name,localdir+name,timeout,disconnectafteruse);
  };
  client_->reset();
}

void JobHTTPControl::Get(const std::string& jobid,const std::string& name,const std::string& localfile,int timeout,bool disconnectafteruse) {
  std::string url=jobid;
  if(url.length() <= 0) throw JobControlError("Missing URL");
  if(url[url.length()-1] == '/') url.resize(url.length()-1);
  if((name.length() > 0) && (name[0] != '/')) url+="/";
  url+=name;
  DataMove mover;
  DataPoint source(url.c_str());
  DataPoint destination(("file://"+localfile).c_str());
  DataCache cache;
  UrlMap map;
  DataMove::result r = mover.Transfer(source,destination,cache,map);
  if(r != DataMove::success) throw JobControlError("Failed to dowload file");
}

std::list<FileInfo> JobHTTPControl::ListFiles(const std::string& jobid,int timeout,bool disconnectafteruse) {
  std::string path;
  if(!set_connection(jobid,path,timeout))
    throw JobControlError("Failed to connect to service");
  int soap_err = SOAP_OK;
  nj__jobFiles files_;
  files_.soap_default(&soap_);
  if((soap_err=soap_call_njc__queryFiles(&soap_,client_->SOAP_URL(path.c_str()).c_str(),"queryFiles",NULL,files_)) != SOAP_OK) {
    notify(ERROR)<<"Failed to list files of job"<<std::endl;
    if(GetNotifyLevel() >= INFO) soap_print_fault(&soap_,stderr);
    client_->disconnect(); client_->reset();
    throw JobControlError("Failed to list files of job");
  };
  std::list<FileInfo> files;
  for(std::vector<char*>::iterator f_ = files_.url.begin();
      f_!=files_.url.end();++f_) {
    if(*f_) {
      int l = strlen(*f_);
      if(l > jobid.length()) {
        FileInfo f;
        f.filename=(*f_) + jobid.length() + 1;
        f.size=0;
        f.isdir=false;
        files.push_back(f);
      };
    };
  };
  client_->reset();
  if(disconnectafteruse) client_->disconnect();
  return files;
}

