#include "../../std.h"
#include <string>
#include <list>
#include <iostream>
#include <fstream>

#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#include "../../misc/escaped.h"

#include "jobs.h"
#include "jobs_soap.nsmap"

class HTTP_JobControl_Handle {
 friend class HTTP_JobControl;
 private:
  plugin_init_t jobplugin_init;
  void* jobplugin_handle;
 public:
  HTTP_JobControl_Handle(const char* path);
  ~HTTP_JobControl_Handle(void);
};

static int* soap_new_int(struct soap* sp,int size) {
  int* i_p = (int*)soap_malloc(sp,sizeof(int));
}

HTTP_JobControl_Handle::HTTP_JobControl_Handle(const char* path):jobplugin_handle(NULL),jobplugin_init(NULL) {
  jobplugin_handle=dlopen(path,RTLD_NOW);
  if(!jobplugin_handle) {
    odlog(ERROR)<<dlerror()<<std::endl;
    odlog(ERROR)<<"Can't load plugin "<<path<<std::endl;
    return;
  };
  jobplugin_init=(plugin_init_t)dlsym(jobplugin_handle,"init");
  if(jobplugin_init == NULL) {
    olog<<"Plugin "<<path<<" is broken."<<std::endl;
    dlclose(jobplugin_handle); jobplugin_handle=NULL; return;
  };
}

HTTP_JobControl_Handle::~HTTP_JobControl_Handle(void) {
  if(jobplugin_handle) {
    dlclose(jobplugin_handle); jobplugin_handle=NULL; jobplugin_init=NULL;
  };
}

HTTP_JobControl::HTTP_JobControl(HTTP_Connector *c,const char* uri,HTTP_JobControl_Handle& handle):HTTP_ServiceAdv(c),service_url(uri) {
  /* SOAP */
  soap_init();
  sp.namespaces=jobs_soap_namespaces;
  sp.user=this;
  // Initialize plugin's instance
  userspec.fill(c->identity());
  std::string jobplugin_config = "end\n";
#ifdef HAVE_SSTREAM
  std::stringstream cfile(jobplugin_config);
#else
  std::strstream cfile;
  cfile<<jobplugin_config;
#endif
  if(handle.jobplugin_init) {
    nordugrid_config_loc="";
    unsetenv("NORDUGRID_CONFIG");
    unsetenv("ARC_CONFIG");
    jobplugin=(*(handle.jobplugin_init))(cfile,userspec);
    if(jobplugin == NULL) {
      olog<<"Job control plugin failed to initialise." << std::endl;
      return;
    };
    /*
    if(jobplugin->acquire() != 1) {
      olog<<"Job plugin's acquire failed (should never happen)." << std::endl;
      delete jobplugin; jobplugin=NULL;
      return;
    };
    */
  };
}

HTTP_JobControl::~HTTP_JobControl(void) {
  soap_deinit();
  /* if(jobplugin) if(jobplugin->release() == 0) { */
  if(jobplugin) {
    delete jobplugin; jobplugin=NULL;
  };
}

HTTP_Service* jobs_service_creator(HTTP_Connector& c,const char* uri,void* arg) {
  HTTP_JobControl_Handle& handle = *((HTTP_JobControl_Handle*)arg);
  HTTP_JobControl *h = new HTTP_JobControl(&c,uri,handle);
  return h;
}

bool jobs_service_configurator(std::istream& f,const char* uri,HTTP_Service_Properties &prop) {
  std::string jobplugin_path;
  for(;!f.eof();) {
    char buf[1024];
    istream_readline(f,buf,sizeof(buf));
    char* p = buf;
    for(;*p;p++) if(!isspace(*p)) break;
    char* command = p;
    if(*command == '#') continue;
    for(;*p;p++) if(isspace(*p)) break;
    int command_len = p-command;
    if(!command_len) continue;
    if((command_len==3) && (strncmp(command,"end",command_len)==0)) { break; }
    else if((command_len==9) && (strncmp(command,"jobplugin",command_len)==0)) {
      input_escaped_string(p,jobplugin_path);
    };
  };
  HTTP_JobControl_Handle* handle = new HTTP_JobControl_Handle(jobplugin_path.c_str());
  prop.arg=handle;
  prop.subtree=true;
  return true;
}

static char* soap_strdup_l(soap* sp,const char* s,int l = -1) {
  if(l<0) l=strlen(s);
  char* s_ = (char*)soap_malloc(sp,l+1);
  if(s_ == NULL) return NULL;
  memcpy(s_,s,l); s_[l]=0;
  return s_;
}

#ifdef OLD_GSOAP
#error gSOAP older than 2.5.2 is not supported.
#else
void HTTP_JobControl::soap_methods(void) {
  odlog(VERBOSE)<<"soap_methods: tag: "<<sp.tag<<std::endl;
  if (!soap_match_tag(&sp,sp.tag,"njf:create")) soap_serve_njf__create(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"njf:query")) soap_serve_njf__query(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"njc:query")) soap_serve_njc__query(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"njc:modify")) soap_serve_njc__modify(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"njc:queryFiles")) soap_serve_njc__queryFiles(&sp);
  else if(!soap_match_tag(&sp,sp.tag,"njc:queryLog")) soap_serve_njc__queryLog(&sp);
  else sp.error = SOAP_NO_METHOD;
}
#endif

/*
 * HTTP methods
 */
HTTP_Error HTTP_JobControl::get(const char* uri,int &keep_alive) {
  if(!jobplugin) {
    return HTTP_NOT_IMPLEMENTED;
  };
  HTTP_Error err;
  if((err=parse_header(keep_alive)) != HTTP_OK) return err;
  if(failure_parsing || unsupported_option_passed) {
    keep_alive=0; return HTTP_FAILURE;
  };
  // obtain requested name
  //cerr<<"HTTP_JobControl::get: uri: "<<uri<<std::endl;
  //cerr<<"HTTP_JobControl::get: service_url: "<<service_url<<std::endl;
  //cerr<<"HTTP_JobControl::get: requested_url: "<<requested_url<<std::endl;
  //cerr<<"HTTP_JobControl::get: requested_path: "<<requested_path<<std::endl;
  const char* name = requested_path.c_str();
  for(;*name;++name) if(*name != '/') break;
  odlog(DEBUG)<<"HTTP_JobControl::get: name: "<<name<<std::endl;
  unsigned int l = strlen(name);
  std::string jobname = name; 
  std::string::size_type p = jobname.find('/');
  if(p != std::string::npos) jobname.resize(p);
  bool dir_q = false; if((l > 0) && (name[l-1] == '/')) dir_q=true;
  std::string path_q(name,dir_q?l-1:l);
  odlog(DEBUG)<<"HTTP_JobControl::get: jobname: "<<jobname<<std::endl;
  odlog(DEBUG)<<"HTTP_JobControl::get: dir_q: "<<dir_q<<std::endl;
  odlog(DEBUG)<<"HTTP_JobControl::get: path_q: "<<path_q<<std::endl;
  std::string html;
  if(path_q.length() == 0) {
    dir_q=true;
    if(requested_url[requested_url.length()-1] != '/') requested_url+="/";
    // Top level - generate table of jobs
    html="<HTML><HEAD>J O B S</HEAD>\n<BODY>\n<TABLE>\n";
    std::list<DirEntry> dir_list;
    if(jobplugin->readdir(path_q.c_str(),dir_list,DirEntry::full_object_info) != 0) {
      if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
        odlog(VERBOSE)<<"failed to send 416"<<std::endl; keep_alive=0; return HTTP_ERROR;
      };
      return HTTP_OK;
    };
    for(std::list<DirEntry>::iterator d = dir_list.begin();d!=dir_list.end();++d) {
      if(d->name == "info") continue;
      if(d->name == "new") continue;
      html+="<TR>";
      // Status
      std::string tmp_s;
      char tmp_buf[256];
      unsigned long long int tmp_l;
      tmp_s="info/"; tmp_s+=d->name; tmp_s+="/status";
      if(jobplugin->open(tmp_s.c_str(),GRIDFTP_OPEN_RETRIEVE,0) != 0) {
        html+="<TD>UNKNOWN<TD/>";
      } else {
        tmp_l=sizeof(tmp_buf)-1;
        if(jobplugin->read((unsigned char*)tmp_buf,0,&tmp_l) != 0) {
          jobplugin->close(true);
          html+="<TD>UNKNOWN<TD/>";
        } else {
          jobplugin->close(true);
          tmp_buf[tmp_l]=0;
          html+="<TD>"; html+=tmp_buf; html+="</TD>";
        };
      };
      // Job name + link
      html+="<TD><A HREF=\""+requested_url+d->name+"/\">"+d->name+"</A></TD>";
      // Link to failure
      tmp_s="info/"; tmp_s+=d->name; tmp_s+="/failed";
      DirEntry tmp_inf;
      if(jobplugin->checkfile(tmp_s,tmp_inf,DirEntry::full_object_info) == 0) {
        html+="<TD><A HREF=\""+requested_url+tmp_s+"\">failure</A></TD>";
      } else {
        html+="<TD></TD>";
      };
      // Link to debug
      html+="<TD><A HREF=\""+requested_url+"info/"+d->name+"/errors\">debug</A></TD>";
      html+="</TR>\n";
    };
    html+="</TABLE>\n</BODY>\n</HTML>";
  } else if(dir_q && (jobname == "new")) {
    if(c->send_error_response(keep_alive,404,NULL,"Error 404") != 0) {
      odlog(VERBOSE)<<"failed to send 404"<<std::endl; keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  } else if(dir_q && (jobname == "info")) {
    if(c->send_error_response(keep_alive,404,NULL,"Error 404") != 0) {
      odlog(VERBOSE)<<"failed to send 404"<<std::endl; keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  } else if(dir_q) {
    // directory
    std::list<DirEntry> dir_list;
    if(jobplugin->readdir(path_q.c_str(),dir_list,DirEntry::full_object_info) != 0) {
      if(c->send_error_response(keep_alive,404,NULL,"Not allowed or not such directory") != 0) {
        odlog(VERBOSE)<<"failed to send 404"<<std::endl; keep_alive=0; return HTTP_ERROR;
      };
      return HTTP_OK;
    };
    // Form a page
    html="<HTML>\n<HEAD>"; html+=name; html+="</HEAD>\n<BODY>\n";
    for(std::list<DirEntry>::iterator d = dir_list.begin();d!=dir_list.end();++d) {
      html+="<BR><A HREF=\"";
      html+=requested_url; html+=d->name;
      if(!d->is_file) html+="/";
//  time_t changed;
//  time_t modified;
//  unsigned long long size;
//  uid_t uid;
//  gid_t gid;
//  bool may_rename;  //
//  bool may_delete;  //
//  bool may_create;  // for dirs
//  bool may_chdir;   // for dirs
//  bool may_dirlist; // for dirs
//  bool may_mkdir;   // for dirs
//  bool may_purge;   // for dirs
//  bool may_read;    // for files
//  bool may_append;  // for files
//  bool may_write;   // for files
      html+="\">";
      html+=d->name; html+="</A>\n";
    };
    html+="</BODY>\n</HTML>";
  };
  if(dir_q) {
    // So far always return full content
    err=send_header(keep_alive,0,html.length()-1,false,html.length(),
                                              HTTP_Time(-1),HTTP_Time(-1));
    if(err != HTTP_OK) {
      keep_alive=0; return HTTP_ERROR;
    };
    if(c->write(html.c_str(),html.length()) != 0) {
      keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  // file
  DirEntry fileinfo;
  if(jobplugin->open(name,GRIDFTP_OPEN_RETRIEVE) != 0) {
    if(c->send_error_response(keep_alive,404,NULL,"Not allowed or no such file") != 0) {
      odlog(VERBOSE)<<"failed to send 404"<<std::endl; keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  std::string name_(name);
  if(jobplugin->checkfile(name_,fileinfo,DirEntry::full_object_info) != 0) {
    jobplugin->close(false);
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl; keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  // so far only one range is effective
  uint64_t range_start_ = 0;
  uint64_t range_end_ = (uint64_t)(-1);
  if(range_passed) {
    range_start_=range_start[0];
    range_end_=range_end[0];
    if(range_end_ <= range_start_) {
      odlog(INFO)<<"range_end <= range_start"<<std::endl;
      jobplugin->close(false);
      if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
        odlog(VERBOSE)<<"failed to send 416"<<std::endl; keep_alive=0; return HTTP_ERROR;
      };
      return HTTP_OK;
    };
  };
  if(fileinfo.size && (range_start_ >= fileinfo.size)) {
    jobplugin->close(false);
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl; keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  // Simple solution: always return <= 1MB of first range
  unsigned char buf[1024*1024];
  unsigned long long int size = sizeof(buf);
  if(range_end_ < (range_start_+size)) size=range_end_-range_start_;
  if(jobplugin->read(buf,range_start_,&size) != 0) {
    jobplugin->close(false);
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl; keep_alive=0; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  jobplugin->close(false);
  range_end_=range_start_+size;
  err=send_header(keep_alive,range_start_,range_end_,true,fileinfo.size,
                                                 HTTP_Time(-1),HTTP_Time(-1));
  if(err != HTTP_OK) { keep_alive=0; return HTTP_ERROR; };
  if(c->write((const char*)buf,size) != 0) {
    keep_alive=0; return HTTP_ERROR;
  };
  return HTTP_OK;
}

HTTP_Error HTTP_JobControl::put(const char* uri,int &keep_alive) {
  if(!jobplugin) {
    return HTTP_NOT_IMPLEMENTED;
  };
  HTTP_Error err;
  if((err=parse_header(keep_alive)) != HTTP_OK) return err;
  if(failure_parsing || unsupported_option_passed) {
    keep_alive=0; return HTTP_FAILURE;
  };
  // obtain requested name
  const char* name = requested_path.c_str();
  for(;*name;++name) if(*name != '/') break;
  odlog(DEBUG)<<"HTTP_JobControl::put: name: "<<name<<std::endl;
  unsigned int l = strlen(name);
  std::string jobname = name;
  std::string::size_type p = jobname.find('/');
  if(p != std::string::npos) jobname.resize(p);
  bool dir_q = false; if((l > 0) && (name[l-1] == '/')) dir_q=true;
  std::string path_q(name,dir_q?l-1:l);
  if(dir_q) {
    // PUT directory?
    keep_alive=0;
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  if(jobname == "new") {
    keep_alive=0;
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  uint64_t range_start,range_end;
  if(entity_range_passed) {
    range_start=entity_range_start;
    range_end=entity_range_end+1;
    odlog(DEBUG)<<"jobs::put: range_passed: "<<range_start<<"-"<<range_end<<std::endl;
    if(length_passed && ((range_end-range_start) != length)) {
      keep_alive=0;
      if(send_header(keep_alive,406) != HTTP_OK) {
        odlog(VERBOSE)<<"failed to send 406"<<std::endl; return HTTP_ERROR;
      };
      return HTTP_OK;
    };
  } else {
    odlog(DEBUG)<<"jobs::put: range not passed"<<std::endl;
    range_start=0;
    if(length_passed) {
      range_end=length;
    } else {
      odlog(DEBUG)<<"jobs::put: length not passed too"<<std::endl;
      keep_alive=0;
      if(send_header(keep_alive,406) != HTTP_OK) {
        odlog(VERBOSE)<<"failed to send 406"<<std::endl; return HTTP_ERROR;
      };
      return HTTP_OK;
    };
  };
  if(jobplugin->open(name,GRIDFTP_OPEN_STORE,range_end) != 0) {
    keep_alive=0;
    if(c->send_error_response(keep_alive,416,NULL,"Error 416") != 0) {
      odlog(VERBOSE)<<"failed to send 416"<<std::endl; return HTTP_ERROR;
    };
    return HTTP_OK;
  };
  err=HTTP_OK;
  // receive entity from network and store locally
  for(;;) {
    if(range_start >= range_end) break;
    uint64_t l = range_end-range_start;
    char buf[1024*1024]; // 1MB - is it enough ?
    if(l > sizeof(buf)) l=sizeof(buf);
    // globus_io tends to return no more than SSL block, hence
    // try to fill buffer by looping
    uint64_t ll=0;
    for(;ll<l;) {
      uint64_t l_=c->read(buf+ll,l-ll);
      if(l_ == 0) break; ll+=l_;
    };
    if(ll == 0) break; // EOF
    if(jobplugin->write((unsigned char*)buf,range_start,ll) != 0) {
      odlog(ERROR)<<"Failed to write to file"<<std::endl;
      err=HTTP_FAILURE;
      break;
    };
  };
  if(err != HTTP_OK) {
    jobplugin->close(false);
    odlog(ERROR)<<"jobs::put: failure writing data from network to disc"<<std::endl;
    keep_alive=0;
    send_header(keep_alive,500);
    err=HTTP_FAILURE;
  } else {
    jobplugin->close(true);
    if(send_header(keep_alive,200) != HTTP_OK) {
      keep_alive=0;
      odlog(VERBOSE)<<"failed to send 200"<<std::endl; 
      err=HTTP_ERROR;
    };
  };
  return err;
}

HTTP_Error HTTP_JobControl::post(const char* uri,int &keep_alive) {
  const char* name = requested_path.c_str();
  for(;*name;++name) if(*name != '/') break;
  int l = strlen(name);
  for(;l;--l) { if(name[l-1] != '/') break; };
  job_id.assign(name,l); job_only=true;
  if(job_id.find('/') != std::string::npos) job_only=false;
  if(!job_only) return HTTP_NOT_FOUND;
  HTTP_Error r = soap_post(uri,keep_alive);
  return r;
}

static int make_failure(struct soap *sp,int code,const char* reason,nj__jobResult& r) {
  if(!r.code) r.code=soap_new_int(sp,-1);
  if(r.code) *r.code=code;
  r.description=(char*)reason;
  return SOAP_OK;
}

// Job manipulation

bool HTTP_JobControl::create_job(nj__job& jobdef,nj__jobResult& job) {
  int l;
  if((jobdef.description == NULL) || ((l=strlen(jobdef.description)) == 0)) {
    make_failure(&sp,1,"Missing job description",job); return false;
  };
  nj__job* j;
  if(!job.job) {
    j=soap_new_nj__job(&sp,-1);
    if(!j) return false;
    j->soap_default(&sp);
  } else {
    j=job.job;
  };
  // simulate RSL submission to jobplugin
  job_id="new";
  if(jobplugin->checkdir(job_id) != 0) {
    make_failure(&sp,1,"Internal error while allocating job ID",job); return false;
  };
  if(jobplugin->open("new/job",GRIDFTP_OPEN_STORE,l) != 0) {
    make_failure(&sp,1,jobplugin->error_description.c_str(),job); return false;
  };
  if(jobplugin->write((unsigned char*)(jobdef.description),0,l) != 0) {
    jobplugin->close(false);
    make_failure(&sp,1,jobplugin->error_description.c_str(),job); return false;
  };
  if(jobplugin->close(true) != 0) {
    make_failure(&sp,1,jobplugin->error_description.c_str(),job); return false;
  };
  j->state="ACCEPTED";
  j->id=(char*)(job_id.c_str()); // This variable is valid till next SOAP message
  std::string u(requested_url.c_str()); u+="/"; u+=job_id; 
  j->url=soap_strdup_l(&sp,u.c_str());
  j->description=jobdef.description;
  job.job=j;
  return true;
}

bool HTTP_JobControl::get_job(const std::string& id,nj__job& job) {
  if(id.empty()) return false;
  std::string s;
  if(!read_info_file("status",s)) return false;
  {
    int n = s.length();
    for(;n>0;n--) if(!isspace(s[n-1])) break;
    s.resize(n);
  };
  job.state=soap_strdup(&sp,s.c_str());
  job.failure_code=NULL; job.failure_reason=NULL;
  if(read_info_file("failure",s)) {
    job.failure_reason=soap_strdup(&sp,s.c_str());
  };
  job.id=soap_strdup(&sp,id.c_str());
  //job.url=soap_strdup(&sp,(service_url+"/"+id).c_str());
  job.url=soap_strdup(&sp,requested_url.c_str());
  job.description=NULL;
  job.attribute.resize(0);
  if(read_info_file("diag","exitcode",s)) {
    char* e;
    unsigned long i=strtoul(s.c_str(),&e,10);
    if((*e == 0) || (*e == '\n')) {
      job.failure_code=(int*)soap_malloc(&sp,sizeof(int));
      if(job.failure_code) *(job.failure_code)=i;
    };
  };
  return true;
}

bool HTTP_JobControl::get_files(const std::string& path,std::vector<char*>& url) {
  std::list<DirEntry> dir_list;
  std::string path_ = job_id + path;
  if(jobplugin->readdir(path_.c_str(),dir_list,DirEntry::full_object_info) != 0) {
    return false;
  };
  for(std::list<DirEntry>::iterator d = dir_list.begin();d!=dir_list.end();++d){
    std::string u = path+"/"+d->name;
    if(d->is_file) {
      char* u_ = soap_strdup(&sp,(requested_url+u).c_str());
      if(u_) url.push_back(u_);
      continue;
    };
    if(!get_files(u,url)) return false;
  };
  return true;
}

bool HTTP_JobControl::read_info_file(const char* name,std::string& content) {
  std::string s;
  char buf[32768];
  unsigned long long int l;
  // Status
  std::string name_("info/"); name_+=job_id; name_+="/"; name_+=name;
  if(jobplugin->open(name_.c_str(),GRIDFTP_OPEN_RETRIEVE,0) != 0) {
    return false;
  };
  l=sizeof(buf)-1;
  if(jobplugin->read((unsigned char*)buf,0,&l) != 0) {
    jobplugin->close(true); return false;
  };
  buf[l]=0; content=buf; 
  jobplugin->close(true);
  return true;
}

bool HTTP_JobControl::read_info_file(const char* name,const std::string& key,std::string& value) {
  std::string content;
  if(!read_info_file(name,content)) return false;
  std::string::size_type p = 0;
  for(;;) {
    p=content.find(key,p);
    if(p == std::string::npos) break;
    if( ((p == 0) || (content[p-1] == '\n')) &&
        (content[p+key.length()] == '=') ) {
      p+=key.length()+1;
      std::string::size_type l = content.find('\n',p);
      if(l == std::string::npos) l=content.length();
      value.assign(content.c_str()+p,l-p);
      return true;
    };
    p+=key.length()+1;
  };
  return false;
}

bool HTTP_JobControl::restart_job(void) {
  // simulate RSL submission to jobplugin
  std::string request("&(executable=/bin/echo)(action=restart)");
  if(jobplugin->open("new/job",GRIDFTP_OPEN_STORE,request.length()) != 0) return false;
  request+="(jobid="+job_id+")";
  if(jobplugin->write((unsigned char*)(request.c_str()),0,request.length()) != 0) {
    jobplugin->close(false);
    return false;
  };
  if(jobplugin->close(true) != 0) return false;
  return true;
}
  
/*
 *  SOAP interface functions
 */

// Factory methods

int njf__create(struct soap* sp,std::vector<nj__job >job,std::vector<nj__jobResult >&jobResult) {

  HTTP_JobControl* it = (HTTP_JobControl*)(sp->user);
  if(!it->job_id.empty()) { return SOAP_NO_METHOD; };
  if(it == NULL) return SOAP_OK;
  jobResult.resize(job.size());
  if(job.size() == 0) return SOAP_OK;
  for(int n = 0;n<job.size();++n) {
    jobResult[n].soap_default(sp);
    jobResult[n].code=soap_new_int(sp,-1);
    if(jobResult[n].code == NULL) return SOAP_OK;
    *(jobResult[n].code)=-1;
    jobResult[n].description="General failure";
  };
  if(!(it->jobplugin)) {
    odlog(ERROR)<<"jobplugin is not initialized"<<std::endl;
    return SOAP_OK;
  };
  for(int n = 0;n<job.size();++n) {
    if(it->create_job(job[n],jobResult[n])) {
      *(jobResult[n].code)=0;
      jobResult[n].description=NULL;
    };
  };
  return SOAP_OK;
};

int njf__query(struct soap* sp,std::vector<nj__job> pattern,std::vector<nj__job>& job) {
  HTTP_JobControl* it = (HTTP_JobControl*)(sp->user);
  if(!it->job_id.empty()) { return SOAP_NO_METHOD; };
  return SOAP_NO_METHOD;
  if(!(it->jobplugin)) {
    odlog(ERROR)<<"jobplugin is not initialized"<<std::endl;
    return SOAP_OK;
  };
  std::list<DirEntry> dir_list;
  if(it->jobplugin->readdir("",dir_list,DirEntry::minimal_object_info) != 0) {
    odlog(ERROR)<<"Can't list jobs"<<std::endl;
    return SOAP_OK;
  };
  int n = 0;
  for(std::list<DirEntry>::iterator d = dir_list.begin();d!=dir_list.end();++d){
    if(d->name == "info") continue;
    if(d->name == "new") continue;
    std::string id = d->name;
    job.resize(++n); job[n-1].soap_default(sp);
    if(!it->get_job(id,job[n-1])) --n;
  };
  job.resize(n);
  return SOAP_OK;
}


// Job methods

// Get information about job
int njc__query(struct soap* sp,nj__job *pattern,nj__job &job) {
  HTTP_JobControl* it = (HTTP_JobControl*)(sp->user);
  if(it->job_id.empty()) { return SOAP_NO_METHOD; };
  job.soap_default(sp);
  if(!(it->jobplugin)) {
    odlog(ERROR)<<"jobplugin is not initialized"<<std::endl;
    //return make_failure(sp,1,"Internal configuration error",r);
    return SOAP_OK;
  };
  it->get_job(it->job_id,job);
  return SOAP_OK;
}

int njc__modify(struct soap* sp,nj__job *modification,nj__jobResult &job) {
  HTTP_JobControl* it = (HTTP_JobControl*)(sp->user);
  if(it->job_id.empty()) { return SOAP_NO_METHOD; };
  job.soap_default(sp);
  if(!(it->jobplugin)) {
    odlog(ERROR)<<"jobplugin is not initialized"<<std::endl;
    return SOAP_OK;
  };
  make_failure(sp,-1,"General failure",job);
  if(modification) {
    if((modification->failure_code) ||
       (modification->failure_reason) ||
       (modification->id) ||
       (modification->url) ||
       (modification->attribute.size()) ||
       (modification->description)) {
      make_failure(sp,-1,"Can't modify these parameters",job);
      return SOAP_OK;
    };
  };
  job.job=soap_new_nj__job(sp,-1);
  if(!job.job) return SOAP_OK;
  job.job->soap_default(sp);
  if(!it->get_job(it->job_id,*job.job)) return SOAP_OK;
  if(!modification) { // No changes requested - kind of ping?
    odlog(DEBUG)<<"job ping"<<std::endl;
    job.code=NULL; job.description=NULL;
    return SOAP_OK;
  };

  if(modification->delegation) { // renew
    std::string dirname = it->job_id;
    odlog(DEBUG)<<"Job credentials renew"<<std::endl;
    if(it->jobplugin->checkdir(dirname) != 0) return SOAP_OK;
  };

  if(modification->state) {
    // Changing state
    if(strcasecmp(modification->state,"") == 0) {
      // Clean job
      odlog(DEBUG)<<"Job clean request"<<std::endl;
      if(it->jobplugin->removedir(it->job_id) != 0) return SOAP_OK;
      if(strcasecmp(job.job->state,"FINISHED") == 0) job.job->state="";
    } else if(strcasecmp(modification->state,"FINISHED") == 0) {
      // Cancel job
      odlog(DEBUG)<<"Job cancel request"<<std::endl;
      if(it->jobplugin->removefile(it->job_id) != 0) return SOAP_OK;
      if(strcasecmp(job.job->state,"DELETED") != 0) job.job->state="FINISHED";
    } else {
      // Rerun request
      odlog(DEBUG)<<"Job rerun request"<<std::endl;
      if(job.job->failure_reason == NULL) return SOAP_OK;
      std::string rerun_state;
      if(!it->read_info_file("local","failedstate",rerun_state))
        return SOAP_OK;
      if(rerun_state.empty()) return SOAP_OK;
      if(rerun_state != modification->state) return SOAP_OK;
      if(!it->restart_job()) return SOAP_OK;
      job.job->state=modification->state;
      job.job->failure_reason=NULL;
    };
  };
  job.code=NULL; job.description=NULL;
  return SOAP_OK;
}

int njc__queryFiles(struct soap* sp,char* pattern,nj__jobFiles& files) {
  HTTP_JobControl* it = (HTTP_JobControl*)(sp->user);
std::cerr<<"njc__queryFiles: id: "<<it->job_id<<std::endl;
  if(it->job_id.empty()) { return SOAP_NO_METHOD; };
  files.soap_default(sp);
  if(!(it->jobplugin)) {
    odlog(ERROR)<<"jobplugin is not initialized"<<std::endl;
    return SOAP_OK;
  };
  it->get_files("",files.url); 
  return SOAP_OK;
}

int njc__queryLog(struct soap* sp,char* type,char*& content) {
  HTTP_JobControl* it = (HTTP_JobControl*)(sp->user);
  if(it->job_id.empty()) { return SOAP_NO_METHOD; };
  content=NULL;
  if(!(it->jobplugin)) {
    odlog(ERROR)<<"jobplugin is not initialized"<<std::endl;
    return SOAP_OK;
  };
  if(type == NULL) return SOAP_OK;
  std::string s;
  if(!it->read_info_file(type,s)) return SOAP_OK;
  content=soap_strdup(sp,s.c_str());
  return SOAP_OK;
}

SOAP_FMAC5 int SOAP_FMAC6 ncr__delegationOffer(struct soap*, const char *intoken, const char *&outtoken) {
  outtoken=NULL;
  return SOAP_NO_METHOD;
}
