#include "../../std.h"

#include "fireman_client.h"

#include "fireman_soap.nsmap"

#include "permission_fireman.h"
#include <arc/certificate.h>
#include "../../auth/identity_dn.h"
#include "../../auth/object_access.h"
#include "../../misc/url_options.h"
#include "../../misc/log_time.h"

#ifndef SOAP_TYPE_glite__ExistsException
#define SOAP_TYPE_glite__ExistsException SOAP_TYPE__ExistsException
#define SOAP_TYPE_glite__NotExistsException SOAP_TYPE__NotExistsException
#endif

int soap_get_fault_detail_type(const struct soap* soap) {
  if(!soap->fault) {
    return 0;
  };
  if(soap->fault->SOAP_ENV__Detail) {
    if(soap->fault->SOAP_ENV__Detail->__type) 
      return (soap->fault->SOAP_ENV__Detail->__type);
    if(soap->fault->SOAP_ENV__Detail->__any) {
      const char* str = (const char*)soap->fault->SOAP_ENV__Detail->__any;
      if(strstr(str,"already exist")) return SOAP_TYPE_glite__ExistsException;
      if(strstr(str,"not exist")) return SOAP_TYPE_glite__NotExistsException;
    };
  };
  if(soap->fault->detail) {
    if(soap->fault->detail->__type)
      return (soap->fault->detail->__type);
    if(soap->fault->detail->__any) {
      const char* str = (const char*)soap->fault->detail->__any;
      if(strstr(str,"already exist")) return SOAP_TYPE_glite__ExistsException;
      if(strstr(str,"not exist")) return SOAP_TYPE_glite__NotExistsException;
    };
  };
  return 0;
}

FiremanClient::FiremanClient(const char* base_url) {
  c = new HTTP_ClientSOAP(base_url,&soap);
  if(!c) { c=NULL; return; };
  if(!*c) { delete c; c=NULL; return; };
  soap.namespaces=fireman_soap_namespaces;
  timeout=300;
  acl=NULL;
}

FiremanClient::~FiremanClient(void) {
  if(c) { c->disconnect(); delete c; };
  if(acl) delete acl;
}

bool FiremanClient::connect(void) {
  if(!c) return false;
  return (c->connect() == 0);
}

bool FiremanClient::disconnect(void) {
  if(!c) return true;
  return (c->disconnect() == 0);
}

//bool FiremanClient::add(const char* name,unsigned long long int size,const std::string& checksum,time_t time) {
//  return add(name,size,checksum,time,std::list<std::string>());
//}

bool FiremanClient::add(const char* name,unsigned long long int size,const std::string& checksum,time_t time,const std::list<std::string>& urls) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  // Do mkdir first if needed
  const char* basename = strrchr(name,'/');
  if(basename) {
    std::string dirname(name,basename-name);
    if(dirname.length() > 0) {
      ArrayOf_USCOREsoapenc_USCOREstring* directories = 
               soap_new_ArrayOf_USCOREsoapenc_USCOREstring(&soap,-1);
      if(!directories) { c->reset(); return false; };
      const char* dirs[] = { dirname.c_str() };
      directories->__ptr=(char**)dirs;
      directories->__size=1;
      struct fireman__mkdirResponse r;
      soap_err=soap_call_fireman__mkdir(&soap,c->SOAP_URL(),"",
                                           directories,true,true,r);
      if(soap_err != SOAP_OK) {
        if(soap_get_fault_detail_type(&soap) !=
                           SOAP_TYPE_glite__ExistsException) {
          odlog(INFO)<<"SOAP request failed (fireman:mkdir)"<<std::endl;
          if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
          c->disconnect();
          return false;
        };
      };
      if(!connect()) return false;
    };
  };
  // Then create file itself
  ArrayOf_USCOREtns1_USCOREFRCEntry* entries = 
            soap_new_ArrayOf_USCOREtns1_USCOREFRCEntry(&soap,-1);
  if(!entries) { c->reset(); return false; };
  glite__FRCEntry* entry[] = { soap_new_glite__FRCEntry(&soap,-1) };
  if(!entry[0]) { c->reset(); return false; };
  entries->__ptr=entry;
  entries->__size=1;
  entry[0]->guid=NULL;
  entry[0]->permission=NULL;
  if(acl) entry[0]->permission=acl->get(&soap);
  entry[0]->lfn=(char*)name;
  entry[0]->lfnStat=NULL;
  entry[0]->GUIDStat=soap_new_glite__GUIDStat(&soap,-1);
  if(!(entry[0]->GUIDStat)) { c->reset(); return false; };
  entry[0]->lfnStat=soap_new_glite__LFNStat(&soap,-1);
  if(!(entry[0]->lfnStat)) { c->reset(); return false; };
  entry[0]->lfnStat->type=1;
  entry[0]->lfnStat->validityTime=0; // ????
  entry[0]->lfnStat->modifyTime=time;
  entry[0]->lfnStat->creationTime=time;
  entry[0]->lfnStat->size=size;
  entry[0]->GUIDStat->modifyTime=time;
  entry[0]->GUIDStat->creationTime=time;
  entry[0]->GUIDStat->size=size;
  entry[0]->GUIDStat->status=0; // ?????
  entry[0]->GUIDStat->checksum=(char*)checksum.c_str();
  if(urls.size() == 0) {
    entry[0]->__sizesurlStats=0;
    entry[0]->surlStats=NULL;
  } else {
    glite__SURLEntry** surls = (glite__SURLEntry**)soap_malloc(&soap,
                                   urls.size()*sizeof(glite__SURLEntry*));
    if(!surls) { c->reset(); return false; };
    int n = 0;
    for(std::list<std::string>::const_iterator i = urls.begin();
                                                  i!=urls.end();++i,++n) {
      surls[n]=soap_new_glite__SURLEntry(&soap,-1);
      if(!surls[n]) { c->reset(); return false; };
      surls[n]->masterReplica=false; 
      surls[n]->creationTime=0;
      surls[n]->modifyTime=0;
      surls[n]->surl=(char*)i->c_str();
    };
    entry[0]->__sizesurlStats=urls.size();
    entry[0]->surlStats=surls;
  };
  struct fireman__createResponse r;
  soap_err=soap_call_fireman__create(&soap,c->SOAP_URL(),"",entries,r);
  if(soap_err != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (fireman:create)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
    c->disconnect();
    return false;
  };
  return true;
}


bool FiremanClient::add(const char* name,const std::list<std::string>& urls) {
  if(!c) return false;
  if(!connect()) return false;
  if(urls.size() <= 0) return true;
  int soap_err = SOAP_OK;
  ArrayOf_USCOREtns1_USCORESURLEntry* surls_ = 
         soap_new_ArrayOf_USCOREtns1_USCORESURLEntry(&soap,-1);
  if(!surls_) { c->reset(); return false; };
  glite__SURLEntry** surls = (glite__SURLEntry**)soap_malloc(&soap,
                                   urls.size()*sizeof(glite__SURLEntry*));
  if(!surls) { c->reset(); return false; };
  int n = 0;
  for(std::list<std::string>::const_iterator i = urls.begin();
                                                i!=urls.end();++i,++n) {
    surls[n]=soap_new_glite__SURLEntry(&soap,-1);
    if(!surls[n]) { c->reset(); return false; };
    surls[n]->masterReplica=false;
    surls[n]->creationTime=0;
    surls[n]->modifyTime=0;
    surls[n]->surl=(char*)i->c_str();
  };
  surls_->__ptr=surls;
  surls_->__size=urls.size();
  struct fireman__addReplicaResponse r;
  soap_err=soap_call_fireman__addReplica(&soap,c->SOAP_URL(),"",
                                                  (char*)name,surls_,r);
  if(soap_err != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (fireman:addReplica)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
    c->disconnect();
    return false;
  };
  return true;
}

bool FiremanClient::info(const char* name,unsigned long long int& size,std::string& checksum,time_t& time,DataPoint::FileInfo::Type& type) {
  std::list<std::string> urls;
  return info(name,size,checksum,time,type,urls);
}

bool FiremanClient::info(const char* name,unsigned long long int& size,std::string& checksum,time_t& time,DataPoint::FileInfo::Type& type,std::list<std::string>& urls) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  urls.resize(0);
  size=0;
  checksum="";
  time=0;
  ArrayOf_USCOREsoapenc_USCOREstring* lfns =
               soap_new_ArrayOf_USCOREsoapenc_USCOREstring(&soap,-1);
  if(!lfns) { c->reset(); return false; };
  const char* lfn[] = { name };
  lfns->__ptr=(char**)lfn;
  lfns->__size=1;
  struct fireman__listReplicasResponse r;
  soap_err=soap_call_fireman__listReplicas(&soap,c->SOAP_URL(),"",
                                                        lfns,false,r);
  if(soap_err != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (fireman:listReplicas)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
    c->disconnect();
    return false;
  };
  if(r._listReplicasReturn->__size != 1) {
    odlog(INFO)<<"SOAP request returned unexpected number of results (fireman:std::listReplicas)"<<std::endl;
    c->reset();
    return false;
  };
  if(r._listReplicasReturn->__ptr[0]->GUIDStat) {
    if(r._listReplicasReturn->__ptr[0]->GUIDStat->modifyTime) {
      time=r._listReplicasReturn->__ptr[0]->GUIDStat->modifyTime;
    } else if(r._listReplicasReturn->__ptr[0]->GUIDStat->creationTime) {
      time=r._listReplicasReturn->__ptr[0]->GUIDStat->creationTime;
    };
    size=r._listReplicasReturn->__ptr[0]->GUIDStat->size;
    if(r._listReplicasReturn->__ptr[0]->GUIDStat->checksum) {
      checksum=r._listReplicasReturn->__ptr[0]->GUIDStat->checksum;
    };
    if(r._listReplicasReturn->__ptr[0]->__sizesurlStats &&
       r._listReplicasReturn->__ptr[0]->surlStats) {
      for(int n = 0;n<r._listReplicasReturn->__ptr[0]->__sizesurlStats;++n) {
        glite__SURLEntry* entry = r._listReplicasReturn->__ptr[0]->surlStats[n];
        if(entry) {
          if(entry->surl) {
            urls.push_back(std::string(entry->surl));
          };
        };
      };
    };
  };
  type=DataPoint::FileInfo::file_type_unknown;
  if(r._listReplicasReturn->__ptr[0]->lfnStat) {
    switch(r._listReplicasReturn->__ptr[0]->lfnStat->type) {
      case 0: type=DataPoint::FileInfo::file_type_dir; break;
      case 1: type=DataPoint::FileInfo::file_type_file; break;
    };
  };
  return true;
}

bool FiremanClient::info(const char* name,std::list<std::string>& urls) {
  unsigned long long int size;
  std::string checksum;
  time_t time;
  DataPoint::FileInfo::Type type;
  return info(name,size,checksum,time,type,urls);
}

bool FiremanClient::remove(const char* name) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  ArrayOf_USCOREsoapenc_USCOREstring* lfns =
               soap_new_ArrayOf_USCOREsoapenc_USCOREstring(&soap,-1);
  if(!lfns) { c->reset(); return false; };
  const char* lfn[] = { name };
  lfns->__ptr=(char**)lfn;
  lfns->__size=1;
  struct fireman__removeResponse r;
  soap_err=soap_call_fireman__remove(&soap,c->SOAP_URL(),"",lfns,r);
  if(soap_err != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (fireman:remove)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
    c->disconnect();
    return false;
  };
  return true;
}

bool FiremanClient::remove(const char* name,const std::list<std::string>& urls) {
  if(urls.size() <= 0) return true;
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  ArrayOf_USCOREtns1_USCOREStringPair* lfnSurls =
               soap_new_ArrayOf_USCOREtns1_USCOREStringPair(&soap,-1);
  if(!lfnSurls) { c->reset(); return false; };
  glite__StringPair** lfnSurls_ = (glite__StringPair**)soap_malloc(&soap,
                                       urls.size()*sizeof(glite__StringPair*));
  if(!lfnSurls_) { c->reset(); return false; };
  lfnSurls->__ptr = lfnSurls_;
  lfnSurls->__size = urls.size();
  int n = 0;
  for(std::list<std::string>::const_iterator i = urls.begin();i!=urls.end();++i) {
    glite__StringPair* lfnSurl = soap_new_glite__StringPair(&soap,-1);
    if(!lfnSurl) { c->reset(); return false; };
    lfnSurls_[n]=lfnSurl;
    lfnSurl->string1=(char*)name;
    lfnSurl->string2=(char*)(i->c_str());
    n++;
  };
  struct fireman__removeReplicaResponse r;
  soap_err=soap_call_fireman__removeReplica(&soap,c->SOAP_URL(),"",
                                                   lfnSurls,false,r);
  if(soap_err != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (fireman:removeReplica)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
    c->disconnect();
    return false;
  };
  return true;
}

bool FiremanClient::list(const char* path,std::list<std::string>& names) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  names.resize(0);
  struct fireman__locateResponse r;
  r._locateReturn=NULL;
  soap_err=soap_call_fireman__locate(&soap,c->SOAP_URL(),"",(char*)path,"*",1000,r);
  if(soap_err != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (fireman:removeReplica)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap,stderr);
    c->disconnect();
    return false;
  };
  if(!r._locateReturn) return true;
  if(r._locateReturn->__size == 0) return true;
  if(r._locateReturn->__ptr == NULL) return true;
  for(int n=0;n<r._locateReturn->__size;n++) {
    if(r._locateReturn->__ptr[n])
               names.push_back(std::string(r._locateReturn->__ptr[n]));
  };
  return true;
}

bool FiremanClient::acl_set(const ObjectAccess& o) {
  if(acl) delete acl;
  acl=new ObjectAccessFireman(o);
  return (acl != NULL);
}

/* fireman://[url[|url[...]]@]server:port/path?lfn */
bool DataPointFireman::process_meta_url(void) {
  meta_service_url.resize(0); locations.clear(); meta_lfn.resize(0);
  if(strncasecmp(url.c_str(),"fireman://",10)) return false;
  std::string url_(url.c_str());
  /* find out if it contains locations */
  std::string::size_type loc_start=10;
  std::string::size_type loc_end=url_.find('@',loc_start);
  std::string urls("");
  if(loc_end!=std::string::npos) {
    urls=url_.substr(loc_start,loc_end-loc_start);
    url_.erase(loc_start,loc_end-loc_start+1);
  };
  /* get lfn */
  std::string::size_type server_start=10;
  std::string::size_type server_end=url_.find('?',server_start);
  std::string filename;
  if(server_end==std::string::npos) {
    filename=""; meta_service_url=url_;
  } else {
    filename=url_.substr(server_end+1);
    meta_service_url=url_.substr(0,server_end);
  };
  ::canonic_url(meta_service_url);
  std::string contact_url = meta_service_url;
  contact_url.replace(0,7,"https");
  c = new FiremanClient(contact_url.c_str());
  if((!c) || (!*c)) {
    if(c) { delete c; c=NULL; };
    meta_service_url.resize(0);
    return false;
  };
  extract_meta_attributes(filename);
  meta_lfn=filename;
  odlog(DEBUG)<<"LFN: "<<meta_lfn<<std::endl;
  odlog(DEBUG)<<"Fireman server: "<<meta_service_url<<std::endl;
  odlog(DEBUG)<<"Location urls: "<<urls<<std::endl;
  std::string::size_type n=0;
  for(std::string::size_type nn=0;n<urls.length();) {
    nn=urls.find('|',n); if(nn == std::string::npos) nn=urls.length();
    if(n == nn) { n++; continue; };
    std::string loc(urls.c_str()+n,nn-n);
    if(loc[0] == ';') { common_url_options+=loc; }
    else {
      locations.push_back(DataPointRLS::Location(loc.c_str(),loc.c_str()));
    };
    n=nn+1;
  };
  {
    std::string sn("");
    try {
      Certificate ci;
      sn = ci.GetIdentitySN();
    } catch (std::exception) { };
    Identity* id = NULL;
    PermissionFireman* perm = NULL;
    ObjectAccess* o = NULL;
    if(sn.length() >= 0) {
      id=new Identity;
      perm=new PermissionFireman;
    };
    if(perm && id) {
      IdentityItemDN id_i(sn.c_str());
      id->add(&id_i);
      perm->allowPermission(true);
      perm->allowRemove(true);
      perm->allowRead(true);
      perm->allowWrite(true);
      perm->allowList(true);
      perm->allowGetMetadata(true);
      perm->allowSetMetadata(true);
      o = new ObjectAccess;
      if(o) {
        o->use(id,perm);
        perm=NULL; id=NULL;
      };
    };
    if(perm) delete perm;
    if(id) delete id;
    if(o) {
      c->acl_set(*o);
      delete o;
    };
  };

  return true;
}

DataPointFireman::DataPointFireman(const char* u):DataPointMeta(u) {
  if(u == NULL) return;
  if(strncasecmp("fireman://",u,10)) return;
  if(!process_meta_url()) return;
  if(locations.size()) location=locations.begin();
  is_valid=true;
}

DataPointFireman::~DataPointFireman(void) {

}

DataPoint* DataPointFireman::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("fireman://",u,10)) return NULL;
  return new DataPointFireman(u);
}

bool DataPointFireman::meta_resolve(bool source) {
  is_resolved=false;
  is_metaexisting=false;
  if(!c) return false;
  if(source) {
    if(meta_lfn.length() == 0) {
      odlog(INFO)<<"Source must contain LFN"<<std::endl;
      return false;
    };
    unsigned long long int f_size = 0;
    std::string f_checksum("");
    time_t f_time = 0;
    std::list<std::string> f_urls;
    DataPoint::FileInfo::Type f_type;
    if(!c->info(meta_lfn.c_str(),f_size,f_checksum,f_time,f_type,f_urls)) {
      return false;
    };
    if(f_checksum.length()) meta_checksum(f_checksum.c_str());
    if(f_size) meta_size(f_size);
    if(f_time) meta_created(f_time);
    is_metaexisting=true;
    if(locations.size() == 0) {
      for(std::list<std::string>::iterator i = f_urls.begin();i!=f_urls.end();++i) {
        std::list<DataPointMeta::Location>::iterator loc =
          locations.insert(locations.end(),
                           DataPointMeta::Location(i->c_str(),i->c_str())); 
        odlog(DEBUG)<<"Adding location: "<<*i<<std::endl;
      };
    } else {
      std::list<Location>::iterator loc = locations.begin();
      for(;loc!=locations.end();) {
        std::list<std::string>::iterator i = f_urls.begin();
        for(std::list<std::string>::iterator i = f_urls.begin();i!=f_urls.end();++i) {
          if(strncmp(i->c_str(),loc->meta.c_str(),loc->meta.length()) == 0) {
            odlog(DEBUG)<<"Adding location: "<<*i<<std::endl;
            loc->meta=*i; loc->url=*i;
            break;
          };
        };
        if(i==f_urls.end()) {
          odlog(DEBUG)<<"Removing location: "<<loc->meta<<" - "<<loc->url<<std::endl;
          loc=locations.erase(loc);
        } else {
          ++loc;
        };
      };
    };
  } else {
    if(meta_lfn.length() == 0) {
      odlog(INFO)<<"Destination must contain LFN"<<std::endl;
      return false;
    };
    unsigned long long int f_size = 0;
    std::string f_checksum("");
    time_t f_time = 0;
    std::list<std::string> f_urls_src;
    std::list<std::string> f_urls_dst;
    DataPoint::FileInfo::Type f_type;
    if(c->info(meta_lfn.c_str(),f_size,f_checksum,f_time,f_type,f_urls_src)) {
      if(f_checksum.length()) meta_checksum(f_checksum.c_str());
      if(f_size) meta_size(f_size);
      if(f_time) meta_created(f_time);
      is_metaexisting=true;
    };
    if(locations.size() == 0) {
      odlog(INFO)<<"Warning: Locations are missing in destination RLS url - will use those registered with special name"<<std::endl;
      if(c->info("__storage_service__",f_urls_dst)) {
        for(std::list<std::string>::iterator i = f_urls_dst.begin();
                                   i!=f_urls_dst.end();++i) {
          std::string pfn = *i;
          if(pfn.length() <= 0) continue;
          if(pfn[pfn.length()-1] != '/') pfn+="/";
          pfn+=meta_lfn;
          std::list<DataPointMeta::Location>::iterator loc =
            locations.insert(locations.end(),
                            DataPointMeta::Location(pfn.c_str(),pfn.c_str()));
            odlog(DEBUG)<<"Adding location: "<<pfn<<std::endl;
        };
      };
    };
    if(f_urls_src.size() > 0) {
      std::list<Location>::iterator loc = locations.begin();
      for(;loc!=locations.end();) {
        std::list<std::string>::iterator i = f_urls_src.begin();
        for(std::list<std::string>::iterator i = f_urls_src.begin();
                                              i!=f_urls_src.end();++i) {
          if(strncmp(i->c_str(),loc->meta.c_str(),i->length()) == 0) break;
        };
        if(i!=f_urls_src.end()) {
          odlog(DEBUG)<<"Removing location: "<<loc->meta<<" - "<<loc->url<<std::endl;
          loc=locations.erase(loc);
        } else {
          ++loc;
        };
      };
    }; 
  };
  if(common_url_options.length() != 0) {
    std::list<Location>::iterator loc = locations.begin();
    for(;loc!=locations.end();++loc) {
      add_url_options(loc->url,common_url_options.c_str(),0);
    };
  };
  location=locations.begin();
  is_resolved=true;
  return true;
}

bool DataPointFireman::meta_preregister(bool replication,bool force) {
  if(replication) { /* replicating inside same lfn */
    if(!is_metaexisting) { /* for replication it must be there */
      odlog(ERROR)<<"LFN is missing in RLS (needed for replication)"<<std::endl;
      return false;
    };
    return true;
  };
  if(is_metaexisting) { /* algorithm require this to be new file */
    if(!force) {
      odlog(ERROR)<<"LFN already exists in replica"<<std::endl;
      return false;
    };
  };
  return true;
}

bool DataPointFireman::meta_postregister(bool replication,bool failure) {
  if(!c) return false;
  std::string pfn(location->url.c_str());
  ::canonic_url(pfn);
  const char* lfn = meta_lfn.c_str();
  std::list<std::string> urls; urls.push_back(pfn);
  if(!replication) {
    if(!c->add(lfn,meta_size(),std::string(meta_checksum()),meta_created(),urls)) {
      return false;
    };
  } else {
    if(!c->add(lfn,urls)) {
      return false;
    };
  };
  return true;
}

bool DataPointFireman::meta_preunregister(bool replication) {
  return true;
}

bool DataPointFireman::meta_unregister(bool all) {
  if(!c) return false;
  const char* lfn = meta_lfn.c_str();
  if(all) {
    if(!c->remove(lfn)) {
      return false;
    };
  } else {
    std::string pfn(location->url.c_str());
    ::canonic_url(pfn);
    std::list<std::string> urls; urls.push_back(pfn);
    if(!c->remove(lfn,urls)) {
      return false;
    };
  };
  return true;
}

bool DataPointFireman::list_files(std::list<DataPoint::FileInfo> &files,bool resolve) {
  if(!c) return false;
  std::list<std::string> names;
  const char* path = meta_lfn.c_str();
  if(!c->list(path,names)) {
    return false;
  };
  for(std::list<std::string>::iterator i = names.begin();i!=names.end();++i) {
    std::list<DataPoint::FileInfo>::iterator f =
             files.insert(files.end(),DataPoint::FileInfo(i->c_str()));
    if(resolve) {
      unsigned long long int size_f;
      std::string checksum_f;
      time_t time_f;
      DataPoint::FileInfo::Type type_f;
      std::list<std::string> urls_f;
      if(c->info(i->c_str(),size_f,checksum_f,time_f,type_f,urls_f)) {
        if(size_f) { f->size=size_f; f->size_available=true; };
        if(checksum_f.length()) { f->checksum=checksum_f; f->checksum_available=true; };
        if(time_f) { f->created=time_f; f->created_available=true; };
        f->type=type_f;
        for(std::list<std::string>::iterator u=urls_f.begin();u!=urls_f.end();++u) {
          f->urls.push_back(*u);
        };
      };
    };
  };
  return true;
}

