#include "../../std.h"

#include <fstream>

#include <arc/stringconv.h>

#include "../../misc/log_time.h"

#include "../client/client.h"
#include "srm1_soapH.h"
#include "srm2_soapH.h"

#include "srm_file_metadata.h"
#include "srm_file_status.h"
#include "srm_remote_request.h"
#include "srm_request.h"

#define MAX_REQUESTS 65536

#define skip_space(__p) for(;*__p;__p++) if(!isspace(*__p)) break;
#define skip_till_space(__p) for(;*__p;__p++) if(isspace(*__p)) break;

static std::list<SRMRemoteRequest*> RandomRequests(const std::list<SRMRemoteRequest>& requests) {
  std::list<SRMRemoteRequest*> reqs;
  int size = requests.size();
  if(!size) return reqs;
  for(int n = 0;n<size;n++) reqs.push_back(NULL);
  if(reqs.size() <= 0) return reqs;
  for(std::list<SRMRemoteRequest>::const_iterator r = requests.begin();
                                                 r!=requests.end();++r) {
    unsigned int p = random();
std::cerr<<"Random value: "<<p<<std::endl;
    for(;p>=size;) p=(p >> 1) ^ (p & 1);
std::cerr<<"New random value: "<<p<<std::endl;
    std::list<SRMRemoteRequest*>::iterator r_ = reqs.begin();
    for(;p;p--) { ++r_; if(r_ == reqs.end()) r_=reqs.begin(); };
    p=size;
    for(;p;p--) {
      if(*r_ == NULL) { (*r_)=(SRMRemoteRequest*)(&(*r)); break; };
      ++r_;
      if(r_ == reqs.end()) r_=reqs.begin();
    };
  };
  return reqs;
}

bool V1_file_state_positive(const char* state) {
  if(strcasecmp(state,"pending") == 0) return true;
  if(strcasecmp(state,"ready") == 0) return true;
  if(strcasecmp(state,"running") == 0) return true;
  if(strcasecmp(state,"done") == 0) return true;
  return false;
}


SRMRequests::SRMRequests(const char* p):path(p) {

}

bool SRMRequests::RememberRequest(SRMRequest r) {
  SRMLocalRequest& req = *r;
  if(this != &(req.parent)) return false;
  // Allocate id
  int h = -1;
  std::string fname;
  int n = 0;
  for(;n<MAX_REQUESTS;n++) {
    fname=path; fname+="/"; fname+=tostring<int>(n);
    h=open(fname.c_str(),O_CREAT | O_EXCL | O_WRONLY,S_IRUSR | S_IWUSR);
    if(h != -1) break;
  };
  if(h==-1) return false;
  req.id=tostring<int>(n);
  std::ofstream o(fname.c_str(),std::ios::trunc);
  if(!o) { close(h); ::unlink(fname.c_str()); return -1; };
  close(h);
  if(!req.command.empty()) o<<"command "<<req.command<<std::endl;
  for(std::list<SRMFile>::iterator f = req.files.begin();f!=req.files.end();++f) {
    o<<"file "<<f->name<<std::endl;
    //if(!(f->local_id.empty())) o<<"local_id "<<f->local_id<<std::endl;
    if(!(f->remote_id.empty())) o<<"remote_id "<<f->remote_id<<std::endl;
    if(f->request) {
      o<<"remote_request_id "<<f->request->id<<std::endl;
std::cerr<<"URL: host: "<<f->request->endpoint->url.Host()<<std::endl;
std::cerr<<"URL: port: "<<f->request->endpoint->url.Port()<<std::endl;
std::cerr<<"URL: path: "<<f->request->endpoint->url.Path()<<std::endl;
std::cerr<<"URL: "<<f->request->endpoint->url.str()<<std::endl;
      if(f->request->endpoint) o<<"remote_request_url "<<f->request->endpoint->url<<std::endl;
    };
  };
  return true;
}

bool SRMRequests::ForgetRequest(SRMRequest r) {
  SRMLocalRequest& req = *r;
  if(this != &(req.parent)) return false;
  if(req.id.empty()) return false;
  std::string fname = path; fname+="/"; fname+=req.id;
  if(::unlink(fname.c_str()) != 0) return false;
  return true;
}

SRMRemoteRequest* SRMRequests::FillFileRequest(SRMLocalRequest& req,const std::string file_request_id,const std::string file_request_url,const char* credentials) {
  std::list<SRMRemoteRequest>::iterator r_ = req.requests.begin();
  for(;r_!=req.requests.end();++r_) {
    if((r_->id == file_request_id) &&
       (r_->endpoint) &&
       (r_->endpoint->url == file_request_url)) {
      return &(*r_);
    };
  };
  SRMRemoteRequest* rr = NULL;
  for(std::list<SRMEndpoint>::iterator e = endpoints.begin();e!=endpoints.end();++e) {
    if(e->url == file_request_url) {
      SRMRemoteRequest rr_(&(*e),credentials);
      rr_.id=file_request_id;
      std::list<SRMRemoteRequest>::iterator rr = 
                      req.requests.insert(req.requests.end(),rr_);
      return &(*rr);
    };
  };
  return NULL;
}


bool SRMRequests::RecallRequest(SRMRequest r,const char* credentials) {
  SRMLocalRequest& req = *r;
  if(this != &(req.parent)) return false;
  if(req.id.empty()) return false;
  std::string fname = path; fname+="/"; fname+=req.id;
  std::ifstream i(fname.c_str());
  if(!i) return -1;
  char buf[1024];
  SRMFile file;
  bool have_file = false;
  std::string file_request_id;
  std::string file_request_url;
  for(;!i.eof();) {
    char buf[1024];
    istream_readline(i,buf,sizeof(buf));
    char* p = buf; skip_space(p);
    char* p_command = p;
    if(*p_command == '#') continue;
    skip_till_space(p);
    if(!(p-p_command)) continue;
    std::string command(p_command,p-p_command);
    skip_space(p);
    if(command == "command") {
      req.command=p;
      if((strcasecmp(p,"pin") == 0) || (strcasecmp(p,"unpin") == 0)) {
        file.is_turl=true;
      } else {
        file.is_turl=false;
      };
    } else if(command == "file") {
      if(have_file) {
        if((!file_request_id.empty()) && (!file_request_url.empty())) {
          file.request=FillFileRequest(req,file_request_id,file_request_url,credentials); 
        };
        req.files.push_back(file);
      };
      have_file=true;
      file.name=p;
      //file.local_id="";
      file.remote_id="";
      file.request=NULL;
      file_request_id="";
      file_request_url="";
    } else if(command == "local_id") {
      //file.local_id=p;
    } else if(command == "remote_id") {
      file.remote_id=p;
    } else if(command == "remote_request_id") {
      file_request_id=p;
    } else if(command == "remote_request_url") {
      file_request_url=p;
    };
  };
  if(have_file) {
    if((!file_request_id.empty()) && (!file_request_url.empty())) {
      file.request=FillFileRequest(req,file_request_id,file_request_url,credentials);
    };
    req.files.push_back(file);
  };
  return true;
}

SRMRequest SRMRequests::GetRequest(const std::string& reqid,const char* credentials) {
  if(reqid.empty()) return SRMRequest();
  SRMRequest r(*(new SRMLocalRequest(*this)));
  r->id=reqid;
  if(!RecallRequest(r,credentials)) return SRMRequest();
  return r;
}

SRMRequest SRMRequests::MakeRequest(const char* cmd,const std::list<std::string>& fileids,bool with_remote,const char* credentials) {
std::cerr<<"MakeRequest: cred: "<<credentials<<std::endl;
  SRMRequest r(*(new SRMLocalRequest(*this)));
  bool turl_fileids = false;
  if(cmd) {
    r->command=cmd;
    if((strcasecmp(cmd,"pin") == 0) || (strcasecmp(cmd,"unpin") == 0)) turl_fileids=true; 
  };
  int n = 0;
  for(std::list<std::string>::const_iterator i = fileids.begin();
                                              i!=fileids.end();++i) {
    SRMFile file;
    file.name=*i;
    //file.local_id=tostring<int>(n++);
    std::list<SRMFile>::iterator f = r->files.insert(r->files.end(),file);
    if(turl_fileids) f->is_turl=true;
  };
  if(!with_remote) return r;
  for(std::list<SRMEndpoint>::iterator e = endpoints.begin();
                                            e!=endpoints.end();++e) {
    std::list<SRMRemoteRequest>::iterator rr = 
     r->requests.insert(r->requests.end(),SRMRemoteRequest(&(*e),credentials));
  };
  return r;
}



// ----------------------------------------

SRMLocalRequest::~SRMLocalRequest(void) {
  // notify parent ?
}

void SRMLocalRequest::GetEndpoints(std::list<SRMEndpoint*> endpoints) {
  for(std::list<SRMFile>::iterator f = files.begin();f!=files.end();++f) {
    if(f->request) {
      if(f->request->endpoint) {
        std::list<SRMEndpoint*>::iterator e = endpoints.begin();
        for(;e!=endpoints.end();++e) {
          if((*e) == f->request->endpoint) break;
        };
        if(e!=endpoints.end()) endpoints.push_back(*e);
      };
    };
  };
}

void SRMLocalRequest::GetFiles(SRMEndpoint* endpoint,std::list<SRMFile*>& files_) {
  for(std::list<SRMFile>::iterator f = files.begin();f!=files.end();++f) {
    if(f->request) {
      if(f->request->endpoint) {
        if(f->request->endpoint == endpoint) {
          files_.push_back(&(*f));
        };
      };
    };
  };
}

const SRMFile* SRMLocalRequest::GetFile(const std::string& requestid,const std::string& fileid) {
  for(std::list<SRMFile>::iterator f = files.begin();f!=files.end();++f) {
    if(f->request) {
      if((f->request->id == requestid) && (f->remote_id == fileid)) {
        return &(*f);
      };
    };
  };
  return NULL;
}

const SRMFile* SRMLocalRequest::GetFile(const std::string& fileid) {
  int n = stringto<int>(fileid);
  if(n < 0) return NULL;
  for(std::list<SRMFile>::iterator f = files.begin();f!=files.end();++f) {
    if(n == 0) return (&(*f));
    n--;
  };
  return NULL;
}

bool SRMLocalRequest::FindFiles(bool delete_unknown) {
  bool found;
  for(std::list<SRMRemoteRequest>::iterator r = this->requests.begin();
                                             r!=this->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = this->files.begin();
                                               f!=this->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(r->FindFiles(files_)) {
      found=true;
      for(std::list<SRMFile*>::iterator f = files_.begin();
                                                 f!=files_.end();++f) {
        (*f)->request=&(*r);
      };
    };
  };
  if(!found) return false;
  if(!delete_unknown) return true;
  for(std::list<SRMFile>::iterator f = this->files.begin();
                                             f!=this->files.end();) {
    if(f->request == NULL) {
      f=files.erase(f);
    } else {
      ++f;
    };
  };
  return true;
}

// ---------------------------------------


bool SRMRequest::V1_get(std::list<std::string> protocols) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(r->V1_get(files_,protocols)) result=true;
  };
  return result;
}


bool SRMRequest::V1_put(std::list<std::string> protocols) {
  bool result = false;
  // Making full list of files to serve
  std::list<SRMFile*> files_;
  for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                      f!=(*this)->files.end();++f) files_.push_back(&(*f));
  int requests_done = 0;
  std::list<SRMRemoteRequest*> reqs = RandomRequests((*this)->requests);
  for(std::list<SRMRemoteRequest*>::iterator r = reqs.begin();
                                             r!=reqs.end();++r) {
    if(files_.size() == 0) continue;
    if(*r == NULL) continue;
    if((*r)->V1_put(files_,protocols)) result=true;
    // Remove accepted files from list being processed
    for(std::list<SRMFile*>::iterator f = files_.begin();
                                            f!=files_.end();) {
      if((*f) != NULL) {
        const SRMFileStatus* fstat = (*f)->Status();
        if(fstat != NULL) {
          if((!(fstat->state.empty())) && 
             (V1_file_state_positive(fstat->state.c_str()))) {
            f=files_.erase(f);
            continue;
          };
        };
      };
      ++f;
    };
    requests_done++;
  };
  return (files_.size()==0);
}

void SRMRequest::V1_ping(void) {}

bool SRMRequest::V1_mkPermanent(void) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(r->V1_mkPermanent(files_)) result=true;
  };
  return result;
}

bool SRMRequest::V1_copy(void) {
  bool result = false;
  // Making full list of files to serve
  std::list<SRMFile*> files_;
  for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                    f!=(*this)->files.end();++f) files_.push_back(&(*f));
  std::list<SRMRemoteRequest*> reqs = RandomRequests((*this)->requests);
  int requests_done = 0;
  for(std::list<SRMRemoteRequest*>::iterator r = reqs.begin();
                                             r!=reqs.end();++r) {
    if(files_.size() == 0) continue;
    if(*r == NULL) continue;
    if((*r)->V1_copy(files_)) result=true;
    // Remove accepted files from list being processed
    for(std::list<SRMFile*>::iterator f = files_.begin();
                                            f!=files_.end();) {
      if((*f) != NULL) {
        const SRMFileStatus* fstat = (*f)->Status();
        if(fstat != NULL) {
          if((!(fstat->state.empty())) && 
             (V1_file_state_positive(fstat->state.c_str()))) {
            f=files_.erase(f);
            continue;
          };
        };
      };
      ++f;
    };
    requests_done++;
  };
  return (files_.size()==0);
}

bool SRMRequest::V1_getEstGetTime(std::list<std::string> protocols) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(r->V1_getEstGetTime(files_,protocols)) result=true;
  };
  return result;
}

bool SRMRequest::V1_getEstPutTime(std::list<std::string> protocols) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(r->V1_getEstPutTime(files_,protocols)) result=true;
  };
  return result;
}

bool SRMRequest::V1_setFileStatus(const std::string& fileid,const std::string& state) {
  int fid = stringto<int>(fileid);
  if(fid < 0) return false;
  std::list<SRMFile>::iterator f = (*this)->files.begin();
  for(;f!=(*this)->files.end();++f) { if(fid == 0) break; --fid; };
  if(f==(*this)->files.end()) return false;
  SRMRemoteRequest* r = f->request;
  if(r == NULL) return false;
  if(!(r->V1_setFileStatus(*f,state))) return false;
  return true;
}

bool SRMRequest::V1_getRequestStatus(void) {
  bool result = true;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      if((f->request == NULL) || (f->request == (&(*r)))) {
        // TODO - check for partitioning and complete information
        files_.push_back(&(*f));
      };
    };
    if(files_.size() == 0) continue;
    if(!(r->V1_getRequestStatus(files_))) result=false;
  };
  return result;
}

bool SRMRequest::V1_pin(void) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(r->V1_pin(files_)) result=true;
  };
  return result;
}

bool SRMRequest::V1_unPin(void) {
  bool result = true;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(!(r->V1_unPin(files_))) result=false;
  };
  return result;
}

bool SRMRequest::V1_advisoryDelete(void) {
  bool result;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(!(r->V1_advisoryDelete(files_))) result=false;
  };
  return result;
}

bool SRMRequest::V1_getFileMetaData(void) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    std::list<SRMFile*> files_;
    for(std::list<SRMFile>::iterator f = (*this)->files.begin();
                                               f!=(*this)->files.end();++f) {
      // TODO - check for partitioning and complete information
      files_.push_back(&(*f));
    };
    if(files_.size() == 0) continue;
    if(r->V1_getFileMetaData(files_)) result=true; // At least something
std::cerr<<"V1_getFileMetaData: result: "<<result<<std::endl;
  };
std::cerr<<"V1_getFileMetaData: exit result: "<<result<<std::endl;
  return result;
}

bool SRMRequest::V1_getProtocols(std::list<std::string>& protocols) {
  bool result = false;
  for(std::list<SRMRemoteRequest>::iterator r = (*this)->requests.begin();
                                             r!=(*this)->requests.end();++r) {
    if(r->V1_getProtocols(protocols)) result=true;
  };
  // Remove duplicates
  for(std::list<std::string>::iterator p1 = protocols.begin();
                                              p1!=protocols.end();++p1) {
    std::list<std::string>::iterator p2 = p1;
    ++p2;
    for(;p2!=protocols.end();) {
      if(*p1 == *p2) { p2=protocols.erase(p2); } else { ++p2; };
    };
  }; 
  return result;
}


// -----------------------------------------

