#include "../../std.h"

#include "srm1_soap.nsmap"

#include "../../misc/log_time.h"

#include "srm_client.h"


SRMClient::SRMClient(const char* base_url,bool gssapi_server) {
  c = new HTTP_ClientSOAP(base_url,&soap,gssapi_server);
  if(!c) { c=NULL; return; };
  if(!*c) { delete c; c=NULL; return; };
  soap.namespaces=srm1_soap_namespaces;
  timeout=300;
}

SRMClient::SRMClient(const SRM_URL& base_url) {
  c = new HTTP_ClientSOAP(base_url.ContactURL().c_str(),&soap,base_url.GSSAPI());
  if(!c) { c=NULL; return; };
  if(!*c) { delete c; c=NULL; return; };
  soap.namespaces=srm1_soap_namespaces;
  timeout=300;
}

SRMClient::~SRMClient(void) {
  if(c) { c->disconnect(); delete c; };
}

bool SRMClient::connect(void) {
  if(!c) return false;
  return (c->connect() == 0);
}

bool SRMClient::disconnect(void) {
  if(!c) return true;
  return (c->disconnect() == 0);
}

static const char* Supported_Protocols[] = {
  "gsiftp","https","httpg","http","ftp","se"
};

bool SRMClient::getTURLs(SRMClientRequest& req,const char* name,const SRM_URL& srm_url,std::list<std::string>& urls) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  req.file_ids.resize(0);
  ArrayOfstring* SURLs = soap_new_ArrayOfstring(&soap,-1);
  ArrayOfstring* Protocols = soap_new_ArrayOfstring(&soap,-1);
  struct SRMv1Meth__getResponse r; r._Result=NULL;
  //r._Result=soap_new_SRMv1Type__RequestStatus(&soap,-1); 
  if((!SURLs) || (!Protocols)) {
    c->reset(); return false;
  };
  Protocols->__ptr=(char**)Supported_Protocols;
  Protocols->__size=sizeof(Supported_Protocols)/sizeof(Supported_Protocols[0]);
  std::string file_url = srm_url.BaseURL()+name;
  const char* surl[] = { file_url.c_str() };
  SURLs->__ptr=(char**)surl;
  SURLs->__size=1;
  if((soap_err=soap_call_SRMv1Meth__get(&soap,c->SOAP_URL(),"get",SURLs,Protocols,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (get)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->disconnect();
    return false;
  };
  if(r._Result == NULL) {  
    odlog(INFO)<<"SRM did not return any information"<<std::endl;
    return false;
  };
  char* request_state = r._Result->state;
  req.request_id = r._Result->requestId;
  SRMv1Type__RequestStatus& result = *(r._Result);
  time_t t_start = time(NULL);
  for(;;) {
    ArrayOfRequestFileStatus* fstatus = result.fileStatuses;
    if(fstatus && (fstatus->__size) && (fstatus->__ptr)) {
      for(int n=0;n<fstatus->__size;n++) {
        SRMv1Type__RequestFileStatus* fs = fstatus->__ptr[n];
        if(fs && fs->state && (strcasecmp(fs->state,"ready") == 0)) {
          if(fs->TURL) {
            urls.push_back(std::string(fs->TURL));
            req.file_ids.push_back(fs->fileId);
          };
        };
      };
    };
    if(urls.size()) break; // Have requested data
    if(!request_state) break; // No data and no state - fishy
    if(strcasecmp(request_state,"pending") != 0) break;
    if((unsigned int)(time(NULL) - t_start) > timeout) break;
    if(result.retryDeltaTime < 1) result.retryDeltaTime=1;
    if(result.retryDeltaTime > 10) result.retryDeltaTime=10;
    sleep(result.retryDeltaTime);
    SRMv1Meth__getRequestStatusResponse r;
    if((soap_err=soap_call_SRMv1Meth__getRequestStatus(&soap,c->SOAP_URL(),
             "getRequestStatus",req.request_id,r)) != SOAP_OK) {
      odlog(INFO)<<"SOAP request failed (getRequestStatus)"<<std::endl;
      if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
      c->disconnect();
      return false;
    };
    if(r._Result == NULL) {  
      odlog(INFO)<<"SRM did not return any information"<<std::endl;
      return false;
    };
    request_state = r._Result->state;
    result = *(r._Result);
  };
  if(urls.size() == 0) return false;
  return acquire(req,urls);
}

bool SRMClient::putTURLs(SRMClientRequest& req,const char* name,const SRM_URL& srm_url,std::list<std::string>& urls,unsigned long long size) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  req.file_ids.resize(0);
  // Request place for new file to put
  ArrayOfstring* src_file_names = soap_new_ArrayOfstring(&soap,-1);
  ArrayOfstring* dst_file_names = soap_new_ArrayOfstring(&soap,-1);
  ArrayOflong* sizes = soap_new_ArrayOflong(&soap,-1);
  ArrayOfboolean* wantPermanent = soap_new_ArrayOfboolean(&soap,-1);
  ArrayOfstring* protocols = soap_new_ArrayOfstring(&soap,-1);
  struct SRMv1Meth__putResponse r; r._Result=NULL;
  //r._Result=soap_new_SRMv1Type__RequestStatus(&soap,-1);
  if((!src_file_names) || (!dst_file_names) || (!sizes) || 
     (!wantPermanent) || (!protocols)) {
    c->reset(); return false;
  };
  protocols->__ptr=(char**)Supported_Protocols;
  protocols->__size=sizeof(Supported_Protocols)/sizeof(Supported_Protocols[0]);
  LONG64 sizes_[] = { size }; // TODO
  bool wantPermanent_[] = { true };
  std::string file_url = srm_url.BaseURL()+name;
  const char* surl[] = { file_url.c_str() };
  src_file_names->__ptr=(char**)surl; src_file_names->__size=1;
  dst_file_names->__ptr=(char**)surl; dst_file_names->__size=1;
  sizes->__ptr=sizes_; sizes->__size=1;
  wantPermanent->__ptr=wantPermanent_; wantPermanent->__size=1;
  if((soap_err=soap_call_SRMv1Meth__put(&soap,c->SOAP_URL(),"put",
               src_file_names,dst_file_names,sizes,
               wantPermanent,protocols,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (put)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM did not return any information"<<std::endl;
    return false;
  };
  char* request_state = r._Result->state;
  req.request_id = r._Result->requestId;
  SRMv1Type__RequestStatus* result = r._Result;
  time_t t_start = time(NULL);
  // Ask for request state in loop till SRM server returns  
  // request status !Pending and file status ready
  for(;;) {
    ArrayOfRequestFileStatus* fstatus = result->fileStatuses;
    if(fstatus && (fstatus->__size) && (fstatus->__ptr)) {
      for(int n=0;n<fstatus->__size;n++) {
        SRMv1Type__RequestFileStatus* fs = fstatus->__ptr[n];
        if(fs && fs->state && (strcasecmp(fs->state,"ready") == 0)) {
          if(fs->TURL) {
            urls.push_back(std::string(fs->TURL));
            req.file_ids.push_back(fs->fileId);
          };
        };
      };
    };
    if(urls.size()) break; // Have requested data
    if(!request_state) break; // No data and no state - fishy
    // Leave if state is not pending and no endpoints 
    if(strcasecmp(request_state,"pending") != 0) break;
    if((unsigned int)(time(NULL) - t_start) > timeout) break;
    if(result->retryDeltaTime < 1) result->retryDeltaTime=1;
    if(result->retryDeltaTime > 10) result->retryDeltaTime=10;
    sleep(result->retryDeltaTime);
    SRMv1Meth__getRequestStatusResponse r;
    if((soap_err=soap_call_SRMv1Meth__getRequestStatus(&soap,c->SOAP_URL(),
             "getRequestStatus",req.request_id,r)) != SOAP_OK) {
      odlog(INFO)<<"SOAP request failed (getRequestStatus)"<<std::endl;
      if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
      c->disconnect();
      return false;
    };
    if(r._Result == NULL) {
      odlog(INFO)<<"SRM did not return any information"<<std::endl;
      return false;
    };
    request_state = r._Result->state;
    result = r._Result;
  };
  if(urls.size() == 0) return false;
  return acquire(req,urls);
}

bool SRMClient::copy(SRMClientRequest& req,const char* name,const SRM_URL& srm_url,const std::string& source) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  req.file_ids.resize(0);
  // Request place for new file to put
  ArrayOfstring* src_file_names = soap_new_ArrayOfstring(&soap,-1);
  ArrayOfstring* dst_file_names = soap_new_ArrayOfstring(&soap,-1);
  ArrayOfboolean* bools = soap_new_ArrayOfboolean(&soap,-1);
  struct SRMv1Meth__copyResponse r; r._Result=NULL;
  if((!src_file_names) || (!dst_file_names)) {
    c->reset(); return false;
  };
  std::string file_url = srm_url.BaseURL()+name;
  const char* surl[] = { file_url.c_str() };
  const char* srcurl[] = { source.c_str() };
  bool bools_[] = { false };
  src_file_names->__ptr=(char**)srcurl; src_file_names->__size=1;
  dst_file_names->__ptr=(char**)surl; dst_file_names->__size=1;
  bools->__ptr=bools_; bools->__size=1;
  if((soap_err=soap_call_SRMv1Meth__copy(&soap,c->SOAP_URL(),"copy",
               src_file_names,dst_file_names,bools,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (copy)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM did not return any information"<<std::endl;
    return false;
  };
  char* request_state = r._Result->state;
  req.request_id = r._Result->requestId;
  SRMv1Type__RequestStatus* result = r._Result;
  time_t t_start = time(NULL);
  // Ask for request state in loop till SRM server returns
  // request status !Pending and file status Ready
  for(;;) {
    ArrayOfRequestFileStatus* fstatus = result->fileStatuses;
    if(fstatus && (fstatus->__size) && (fstatus->__ptr)) {
      for(int n=0;n<fstatus->__size;n++) {
        SRMv1Type__RequestFileStatus* fs = fstatus->__ptr[n];
        if(fs && fs->state && (strcasecmp(fs->state,"ready") == 0)) {
          req.file_ids.push_back(fs->fileId);
        };
      };
    };
    if(req.file_ids.size()) break; // Have requested data
    if(!request_state) break; // No data and no state - fishy
    if((strcasecmp(request_state,"pending") != 0) && 
       (strcasecmp(request_state,"active") != 0)) break;
    if((unsigned int)(time(NULL) - t_start) > timeout) break;
    if(result->retryDeltaTime < 5) result->retryDeltaTime=5;
    if(result->retryDeltaTime > 30) result->retryDeltaTime=30;
    sleep(result->retryDeltaTime);
    SRMv1Meth__getRequestStatusResponse r;
    if((soap_err=soap_call_SRMv1Meth__getRequestStatus(&soap,c->SOAP_URL(),
             "getRequestStatus",req.request_id,r)) != SOAP_OK) {
      odlog(INFO)<<"SOAP request failed (getRequestStatus)"<<std::endl;
      if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
      c->disconnect();
      return false;
    };
    if(r._Result == NULL) {
      odlog(INFO)<<"SRM did not return any information"<<std::endl;
      return false;
    };
    request_state = r._Result->state;
    result = r._Result;
  };
  if(req.file_ids.size() == 0) return false;
  return release(req,name,srm_url);
}

bool SRMClient::acquire(SRMClientRequest& req,std::list<std::string>& urls) {
  int soap_err = SOAP_OK;
  // Tell server to move files into "Running" state
  std::list<int>::iterator file_id = req.file_ids.begin();
  std::list<std::string>::iterator f_url = urls.begin();
  for(;file_id!=req.file_ids.end();) {
    SRMv1Meth__setFileStatusResponse r; r._Result=NULL;
    if((soap_err=soap_call_SRMv1Meth__setFileStatus(&soap,c->SOAP_URL(),
            "setFileStatus",req.request_id,*file_id,"Running",r)) != SOAP_OK) {
      odlog(INFO)<<"SOAP request failed (setFileStatus)"<<std::endl;
      if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
      file_id=req.file_ids.erase(file_id); f_url=urls.erase(f_url);
      continue;
    };
    SRMv1Type__RequestStatus* result = r._Result;
    ArrayOfRequestFileStatus* fstatus = result->fileStatuses;
    if(fstatus && (fstatus->__size) && (fstatus->__ptr)) {
      int n;
      for(n=0;n<fstatus->__size;n++) {
        SRMv1Type__RequestFileStatus* fs = fstatus->__ptr[n];
        if(!fs) continue;
        if(fs->fileId != *file_id) continue;
        if(fs->state && (strcasecmp(fs->state,"running") == 0)) {
          ++file_id; ++f_url; break;
        };
      };
      if(n<fstatus->__size) continue;
    };
    odlog(DEBUG)<<"File could not be moved to Running state: "<<*f_url<<std::endl;
    file_id=req.file_ids.erase(file_id); f_url=urls.erase(f_url);
  };
  if(urls.size() == 0) return false;
  // Do not disconnect
  return true;
}

bool SRMClient::remove(const char* name,const SRM_URL& srm_url) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = soap_new_ArrayOfstring(&soap,-1);
  if(!SURLs) {
    c->reset(); return false;
  };
  std::string file_url = srm_url.BaseURL()+name;
  const char* surl[] = { file_url.c_str() };
  SURLs->__ptr=(char**)surl;
  SURLs->__size=1;
  struct SRMv1Meth__advisoryDeleteResponse r;
  if((soap_err=soap_call_SRMv1Meth__advisoryDelete(&soap,c->SOAP_URL(),
                      "advisoryDelete",SURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (SRMv1Meth__advisoryDelete)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->disconnect();
    return false;
  };
  return true; 
}

bool SRMClient::info(const char* name,const SRM_URL& srm_url,unsigned long long int& size,std::string& checksum) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  ArrayOfstring* SURLs = soap_new_ArrayOfstring(&soap,-1);
  if(!SURLs) {
    c->reset(); return false;
  };
  std::string file_url = srm_url.BaseURL()+name;
  const char* surl[] = { file_url.c_str() };
  SURLs->__ptr=(char**)surl;
  SURLs->__size=1;
  struct SRMv1Meth__getFileMetaDataResponse r; r._Result=NULL;
  if((soap_err=soap_call_SRMv1Meth__getFileMetaData(&soap,c->SOAP_URL(),
                      "getFileMetaData",SURLs,r)) != SOAP_OK) {
    odlog(INFO)<<"SOAP request failed (getFileMetaData)"<<std::endl;
    if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
    c->disconnect();
    return false;
  };
  if(r._Result == NULL) {
    odlog(INFO)<<"SRM did not return any information"<<std::endl;
    return false;
  };
  if((r._Result->__size == 0) || 
     (r._Result->__ptr == NULL) ||
     (r._Result->__ptr[0] == NULL)) {
    odlog(INFO)<<"SRM did not return any useful information"<<std::endl;
    return false;
  };
  SRMv1Type__FileMetaData& mdata = *(r._Result->__ptr[0]);
  size=mdata.size;
  checksum="";
  if(mdata.checksumType) { checksum+=mdata.checksumType; checksum+=": "; };
  if(mdata.checksumValue) { checksum+=mdata.checksumValue; };
  return true; 
}

bool SRMClient::release(SRMClientRequest& req,const char* name,const SRM_URL& srm_url) {
  if(!c) return false;
  if(!connect()) return false;
  int soap_err = SOAP_OK;
  // Tell server to move files into "Done" state
  std::list<int>::iterator file_id = req.file_ids.begin();
  for(;file_id!=req.file_ids.end();) {
    SRMv1Meth__setFileStatusResponse r; r._Result=NULL;
    if((soap_err=soap_call_SRMv1Meth__setFileStatus(&soap,c->SOAP_URL(),
            "setFileStatus",req.request_id,*file_id,"Done",r)) != SOAP_OK) {
      odlog(INFO)<<"SOAP request failed (setFileStatus)"<<std::endl;
      if(LogTime::Level() > FATAL) soap_print_fault(&soap, stderr);
      ++file_id; continue;
    };
    SRMv1Type__RequestStatus* result = r._Result;
    ArrayOfRequestFileStatus* fstatus = result->fileStatuses;
    if(fstatus && (fstatus->__size) && (fstatus->__ptr)) {
      int n;
      for(n=0;n<fstatus->__size;n++) {
        SRMv1Type__RequestFileStatus* fs = fstatus->__ptr[n];
        if(fs->fileId != *file_id) continue;
        if(fs && fs->state && (strcasecmp(fs->state,"Done") == 0)) {
          file_id=req.file_ids.erase(file_id); break;
        };
      };
      if(n<fstatus->__size) continue;
    };
    odlog(DEBUG)<<"File could not be moved to Done state"<<std::endl;
    ++file_id;
  };
  return true; 
}
 
