/*
  Download files specified in job.ID.input and check if user uploaded files.
  result: 0 - ok, 1 - unrecoverable error, 2 - potentially recoverable.
  arguments: job_id control_directory session_directory [cache_directory]
  optional arguments: -t threads , -u user
*/
#include "std.h"
#include "misc/proxy.h"
#include "files/info_types.h"
#include "files/info_files.h"
#include "jobs/job.h"
#include "jobs/users.h"
#include "misc/delete.h"
#include "misc/log_time.h"
#include "misc/checksum.h"
#include "misc/stringtoint.h"
#include "misc/url_options.h"
#include "datamove/datacache.h"
#include "datamove/datamovepar.h"
#include "config/config_map.h"
#include "config/environment.h"

/* check for user uploaded files every 60 seconds */
#define CHECK_PERIOD 60
/* maximum number of retries (for every source/destination) */
#define MAX_RETRIES 5
/* maximum number simultaneous downloads */
#define MAX_DOWNLOADS 5
/* maximum time for user files to upload (per file) */
#define MAX_USER_TIME 600


// globus_mutex_t list_lock;
// globus_mutex_t download_lock;
// globus_cond_t  download_cond;
JobDescription desc;
JobUser user;
char* cache_dir;
char* cache_data_dir;
char* cache_link_dir;
uid_t cache_uid;
gid_t cache_gid;
char* id;
int download_done=0;
FileData::iterator next_di;
bool leaving = false;

/* if != 0 - change owner of downloaded files to this user */
uid_t file_owner = 0;
uid_t file_group = 0;

bool is_checksum(std::string str) {
  /* check if have : */
  std::string::size_type n;
  const char *s = str.c_str();
  if((n=str.find('.')) == std::string::npos) return false;
  if(str.find('.',n+1) != std::string::npos) return false;
  for(int i=0;i<n;i++) { if(!isdigit(s[i])) return false; };
  for(int i=n+1;s[i];i++) { if(!isdigit(s[i])) return false; };
  return true;
}

int clean_files(std::list<FileData> &job_files,char* session_dir) {
  std::string session(session_dir);
  /* delete only downloadable files, let user manage his/hers files */
  std::list<FileData> tmp;
  if(delete_all_files(session,job_files,false,true,false) != 2) return 0;
  return 1;
}

/*
   Check for existence of user uploadable file
   returns 0 if file exists 
           1 - it is not proper file or other error
           2 - not here yet 
*/
int user_file_exists(FileData &dt,char* session_dir,std::string* error = NULL) {
  struct stat st;
  const char *str = dt.lfn.c_str();
  if(strcmp(str,"*.*") == 0) return 0; /* do not wait for this file */
  std::string fname=std::string(session_dir) + '/' + dt.pfn;
  /* check if file does exist at all */
  if(lstat(fname.c_str(),&st) != 0) return 2;
  /* check for misconfiguration */
  /* parse files information */
  char *str_;
  unsigned long long int fsize;
  unsigned long long int fsum;
  bool have_size = false;
  bool have_checksum = false;
  fsize = strtouq(str,&str_,10);
  if((*str_) == '.') {
    if(str_ != str) have_size=true;
    str=str_+1;
    fsum = strtouq(str,&str_,10);
    if((*str_) != 0) {
      olog << "Invalid checksum in " << dt.lfn << " for " << dt.pfn << std::endl;
      if(error) (*error)="Bad information about file: checksum can't be parsed.";
      return 1;
    };
    have_checksum=true;
  }
  else {
    if(str_ != str) have_size=true;
    if((*str_) != 0) {
      olog << "Invalid file size in " << dt.lfn << " for " << dt.pfn << std::endl;
      if(error) (*error)="Bad information about file: size can't be parsed.";
      return 1;
    };
  };
  if(S_ISDIR(st.st_mode)) {
    if(have_size || have_checksum) {
      if(error) (*error)="Expected file. Directory found.";
      return 1;
    };
    return 0;
  };
  if(!S_ISREG(st.st_mode)) {
    if(error) (*error)="Expected ordinary file. Special object found.";
    return 1;
  };
  /* now check if proper size */
  if(have_size) {
    if(st.st_size < fsize) return 2;
    if(st.st_size > fsize) {
      olog << "Invalid file " << dt.pfn << " is too big." << std::endl;
      if(error) (*error)="Delivered file is bigger than specified.";
      return 1; /* too big file */
    };
  };
  if(have_checksum) {
    int h=open(fname.c_str(),O_RDONLY);
    if(h==-1) { /* if we can't read that file job won't too */
      olog << "Error accessing file " << dt.pfn << std::endl;
      if(error) (*error)="Delivered file is unreadable.";
      return 1;
    };
    CRC32Sum crc;
    char buffer[1024];
    ssize_t l;
    size_t ll = 0;
    for(;;) {
      if((l=read(h,buffer,1024)) == -1) {
        olog << "Error reading file " << dt.pfn << std::endl; return 1;
        if(error) (*error)="Could not read file to compute checksum.";
      };
      if(l==0) break; ll+=l;
      crc.add(buffer,l);
    };
    close(h);
    crc.end();
    if(fsum != crc.crc()) {
      if(have_size) { /* size was checked - it is an error to have wrong crc */ 
        olog << "File " << dt.pfn << " has wrong CRC." << std::endl;
        if(error) (*error)="Delivered file has wrong checksum.";
        return 1;
      };
      return 2; /* just not uploaded yet */
    };
  };
  return 0; /* all checks passed - file is ok */
}

int main(int argc,char** argv) {
  int res=0;
  bool new_download;
  bool not_uploaded;
  int retries=0;
  time_t start_time=time(NULL);
  time_t upload_timeout = 0;
  int n_threads = 1;
  int n_files = MAX_DOWNLOADS;
  LogTime::Active(true);
  LogTime::Level(DEBUG);
  cache_uid=0;
  cache_gid=0;
  cache_dir=NULL;
  cache_data_dir=NULL;
  cache_link_dir=NULL;
  unsigned long long int min_speed = 0;
  time_t min_speed_time = 300;
  unsigned long long int min_average_speed = 0;
  time_t max_inactivity_time = 300;
  bool secure = true;
  bool userfiles_only = false;
  bool passive = false;
  std::string failure_reason("");

  // process optional arguments
  for(;;) {
    opterr=0;
    int optc=getopt(argc,argv,"+hclpZn:t:n:u:U:s:S:a:i:d:");
    if(optc == -1) break;
    switch(optc) {
      case 'h': {
        olog<<"downloader [-t threads] [-n files] [-c] job_id control_directory session_directory [cache_directory [cache_data_directory [cache_link_directory]]]"<<std::endl; 
        exit(1);
      }; break;
      case 'c': {
        secure=false;
      }; break;
      case 'l': {
        userfiles_only=true;
      }; break;
      case 'p': {
        passive=true;
      }; break;
      case 'Z': {
        central_configuration=true;
      }; break;
      case 'd': {
        LogTime::Level(NotifyLevel(atoi(optarg)));
      }; break;
      case 't': {
        n_threads=atoi(optarg);
        if(n_threads < 1) {
          olog<<"Wrong number of threads"<<std::endl; exit(1);
        };
      }; break;
      case 'n': {
        n_files=atoi(optarg);
        if(n_files < 1) {
          olog<<"Wrong number of files"<<std::endl; exit(1);
        };
      }; break;
      case 'U': {
        unsigned int tuid;
        if(!stringtoint(std::string(optarg),tuid)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        struct passwd pw_;
        struct passwd *pw;
        char buf[BUFSIZ];
        getpwuid_r(tuid,&pw_,buf,BUFSIZ,&pw);
        if(pw == NULL) {
          olog<<"Wrong user name"<<std::endl; exit(1);
        };
        file_owner=pw->pw_uid;
        file_group=pw->pw_gid;
        if((getuid() != 0) && (getuid() != file_owner)) {
          olog<<"Specified user can't be handled"<<std::endl; exit(1);
        };
      }; break;
      case 'u': {
        struct passwd pw_;
        struct passwd *pw;
        char buf[BUFSIZ];
        getpwnam_r(optarg,&pw_,buf,BUFSIZ,&pw);
        if(pw == NULL) {
          olog<<"Wrong user name"<<std::endl; exit(1);
        };
        file_owner=pw->pw_uid;
        file_group=pw->pw_gid;
        if((getuid() != 0) && (getuid() != file_owner)) {
          olog<<"Specified user can't be handled"<<std::endl; exit(1);
        };
      }; break;
      case 's': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_speed=tmpi;
      }; break;
      case 'S': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_speed_time=tmpi;
      }; break;
      case 'a': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        min_average_speed=tmpi;
      }; break;
      case 'i': {
        unsigned int tmpi;
        if(!stringtoint(std::string(optarg),tmpi)) {
          olog<<"Bad number "<<optarg<<std::endl; exit(1);
        };
        max_inactivity_time=tmpi;
      }; break;
      case '?': {
        olog<<"Unsupported option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      case ':': {
        olog<<"Missing parameter for option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      default: {
        olog<<"Undefined processing error"<<std::endl;
        exit(1);
      };
    };
  };
  // process required arguments
  id = argv[optind+0];
  if(!id) { olog << "Missing job id" << std::endl; return 1; };
  char* control_dir = argv[optind+1];
  if(!control_dir) { olog << "Missing control directory" << std::endl; return 1; };
  char* session_dir = argv[optind+2];
  if(!session_dir) { olog << "Missing session directory" << std::endl; return 1; };
  cache_dir = argv[optind+3];
  if(cache_dir) {
    olog<<"Cache directory is "<<cache_dir<<std::endl;
    cache_data_dir=argv[optind+4];
    if(!cache_data_dir) {
      cache_data_dir=cache_dir;
    }
    else {
      olog<<"Cache data directory is "<<cache_data_dir<<std::endl;
      cache_link_dir=argv[optind+5];
      if(cache_link_dir) olog<<"Cache links will go to directory "<<cache_link_dir<<std::endl;
    };
  };
  if(min_speed != 0) { olog<<"Minimal speed: "<<min_speed<<" B/s during "<<min_speed_time<<" s"<<std::endl; };
  if(min_average_speed != 0) { olog<<"Minimal average speed: "<<min_average_speed<<" B/s"<<std::endl; };
  if(max_inactivity_time != 0) { olog<<"Maximal inactivity time: "<<max_inactivity_time<<" s"<<std::endl; };

  prepare_proxy();

  if(n_threads > 10) {
    olog<<"Won't use more than 10 threads"<<std::endl;
    n_threads=10;
  };
/*
 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  Add this to DataMove !!!!!!!!!!!!
*/
  UrlMapConfig url_map;
  olog<<"Downloader started"<<std::endl;

  // prepare Job and User descriptions
  desc=JobDescription(id,session_dir);
  uid_t uid;
  gid_t gid;
  if(file_owner != 0) { uid=file_owner; }
  else { uid= getuid(); };
  if(file_group != 0) { gid=file_group; }
  else { gid= getgid(); };
  cache_uid=getuid();
  cache_gid=getgid();
  desc.set_uid(uid,gid);
  user=JobUser(uid);
  user.SetControlDir(control_dir);
  user.SetSessionRoot(session_dir); /* not needed */
  if(cache_dir) {
    if(!cache_link_dir) {
      user.SetCacheDir(cache_dir,cache_data_dir);
    }
    else {
      user.SetCacheDir(cache_dir,cache_data_dir,cache_link_dir);
    };
  };
  // get the list of input files
  // odlog(DEBUG)<<"Cache: dir: "<<cache_dir<<std::endl;
  // odlog(DEBUG)<<"Cache: data dir: "<<cache_data_dir<<std::endl;
  // odlog(DEBUG)<<"Cache: link dir: "<<cache_link_dir<<std::endl;
  // odlog(DEBUG)<<"Cache: uid: "<<cache_uid<<std::endl;
  // odlog(DEBUG)<<"Cache: gid: "<<cache_gid<<std::endl;
  DataCache cache(cache_dir,cache_data_dir,cache_link_dir,id,cache_uid,cache_gid);
  DataMovePar mover;
  mover.secure(secure);
  mover.passive(passive);
  if(min_speed != 0)
    mover.set_default_min_speed(min_speed,min_speed_time);
  if(min_average_speed != 0)
    mover.set_default_min_average_speed(min_average_speed);
  if(max_inactivity_time != 0)
    mover.set_default_max_inactivity_time(max_inactivity_time);
  std::list<FileData> job_files;
  bool all_data = false;
  bool transfered = true;
  bool credentials_expired = false;

  if(!job_input_read_file(desc.get_id(),user,job_files)) {
    failure_reason+="Internal error in downloader\n";
    olog << "Can't read list of input files" << std::endl; res=1; goto exit;
  };
  // remove bad files
  if(clean_files(job_files,session_dir) != 0) { 
    failure_reason+="Internal error in downloader\n";
    olog << "Can't remove junk files" << std::endl; res=1; goto exit;
  };
  // initialize structures to handle download
  /* TODO: add threads=# to all urls if n_threads!=1 */

  // Push data transfer pair into mover
  for(next_di=job_files.begin();next_di!=job_files.end();++next_di) {
    if(next_di->lfn.find(":") != std::string::npos) { /* is it lfn ? */
      /* have place and file to download */
      /* define place to store */
      std::string destination=std::string("file://") + session_dir + next_di->pfn;
      std::string source = next_di->lfn;
      if(strncasecmp(source.c_str(),"file://",7) == 0) {
        failure_reason+=std::string("User requested local input file ")+source.c_str()+"\n";
        olog<<"FATAL ERROR: local source for download: "<<source<<std::endl; res=1; goto exit;
      };
      if(!mover.Add(source.c_str(),destination.c_str())) {
        failure_reason+="Internal error in downloader\n";
        olog<<"Can't add data pair to list: "<<std::endl; res=1; goto exit;
      };
    }
    else {
      upload_timeout+=MAX_USER_TIME;
    };
  };
  if(!userfiles_only) {
    // Start transfer 
    if(!mover.Transfer(cache,url_map,n_files)) {
      failure_reason+="Internal error in downloader\n";
      odlog(ERROR)<<"FAILED to start transfering files"<<std::endl; res=2; goto exit;
    };
    // Check if all files have been properly downloaded
    for(next_di=job_files.begin();next_di!=job_files.end();) {
      if(next_di->lfn.find(":") != std::string::npos) { /* is it lfn ? */
        // Files should be of same order as when pushed. Error if not.
        if(all_data) {
          failure_reason+="Internal error in downloader\n";
          odlog(ERROR)<<"Data structure was destroyed."<<std::endl; res=2; goto exit;
        };
        std::string dest=std::string("file://") + session_dir + next_di->pfn;
        std::string destination;
        std::string source;
        DataMovePar::result dres;
        if(!mover.Get(source,destination,dres)) { // no more data
          all_data=true; ++next_di; continue;
        };
        if(dest != destination) {
          failure_reason+="Internal error in downloader\n";
          odlog(ERROR)<<"Data structure was destroyed."<<std::endl; res=2; goto exit;
        };
        if(dres == DataMovePar::success) {  // remove this file from list
          odlog(INFO)<<"Downloaded "<<source<<std::endl;
          std::string value;
          if(get_url_option(next_di->lfn,"exec",0,value) == 0) {
            if(value == "yes") {
              fix_file_permissions(session_dir+next_di->pfn,true);
            };
          };
          next_di=job_files.erase(next_di);
        }
        else {
          failure_reason+="Input file: "+next_di->lfn+" - "+
                             DataMove::get_result_string(dres)+"\n";
          if(dres == DataMove::credentials_expired_error) credentials_expired=true;
          transfered=false; ++next_di;
          odlog(ERROR)<<"Failed to download "<<source<<std::endl;
        };
      }
      else { ++next_di; };
    };
    if(!transfered) {
      odlog(INFO)<<"Some downloads failed"<<std::endl; res=2;
      if(credentials_expired) res=3;
      goto exit;
    };
    if(!job_input_write_file(desc,user,job_files)) {
      olog << "WARNING: Failed writing changed input file." << std::endl;
    };
  };
  // check for user uploadable files
  // run cycle waiting for uploaded files
  for(;;) {
    not_uploaded=false;
    for(next_di=job_files.begin();next_di!=job_files.end();) {
      if(next_di->lfn.find(":") == std::string::npos) { /* is it lfn ? */
        /* process user uploadable file */
        olog<<"Check user uploadable file: "<<next_di->pfn<<std::endl;
        std::string error;
        int err=user_file_exists(*next_di,session_dir,&error);
        if(err == 0) { /* file is uploaded */
          olog<<"User has uploaded file"<<std::endl;
          next_di=job_files.erase(next_di);
          if(!job_input_write_file(desc,user,(std::list<FileData>&)job_files)) {
            olog << "WARNING: Failed writing changed input file." << std::endl;
          };
        }
        else if(err == 1) { /* critical failure */
          olog<<"Critical error for uploadable file"<<std::endl;
          failure_reason+="User file: "+next_di->pfn+" - "+error+"\n";
          res=1; goto exit;
        }
        else {
          not_uploaded=true; ++next_di;
        };
      }
      else {
        ++next_di;
      };
    };
    if(!not_uploaded) break;
    // check for timeout
    if((time(NULL)-start_time) > upload_timeout) {
      for(next_di=job_files.begin();next_di!=job_files.end();++next_di) {
        if(next_di->lfn.find(":") == std::string::npos) { /* is it lfn ? */
          failure_reason+="User file: "+next_di->pfn+" - Timeout waiting\n";
        };
      };
      olog<<"Uploadable files timed out"<<std::endl; res=2; break;
    };
    sleep(CHECK_PERIOD);
  };
  if(!job_input_write_file(desc,user,job_files)) {
    olog << "WARNING: Failed writing changed input file." << std::endl;
  };
exit:
  olog << "Leaving downloader ("<<res<<")"<<std::endl;
  // clean unfinished files here 
  clean_files(job_files,session_dir);
  // release cache just in case
  if(res != 0) {
    if(cache_dir) cache_release_url(cache_dir,cache_data_dir,cache_uid,cache_gid,id,true);
  };
  remove_proxy();
  if(res != 0) {
    job_failed_mark_add(desc,user,failure_reason);
  };
  return res;
}
