#include "../std.h"

#include "../config/config_map.h"
#include "../transfer/rls.h"
#include "../misc/url_options.h"
#include "../misc/checksum.h"
#include "../misc/stringtoint.h"
#include "../misc/inttostring.h"
#include "../misc/time_utils.h"
#include "../misc/log_time.h"
#include "../misc/guid.h"

#include "datapoint.h"

#ifdef HAVE_GLOBUS_RLS_CLIENT_H
#define globus_rls_free_result(err) globus_rls_client_error_info(err,NULL,NULL,0,GLOBUS_FALSE)
#endif

DataPointRLS::DataPointRLS(const char* u):DataPointMeta(u),guid_enabled(false) {
  if(u == NULL) return;
  if(strncasecmp("rls://",u,6)) return;
  if(!process_meta_url()) return;
  if(locations.size()) location=locations.begin();
  is_valid=true;
}

DataPoint* DataPointRLS::CreateInstance(const char* u) {
  if(u == NULL) return NULL;
  if(strncasecmp("rls://",u,6)) return NULL;
  return new DataPointRLS(u);
}

DataPointRLS::~DataPointRLS(void) {

}

#ifdef HAVE_GLOBUS_RLS_CLIENT_H
static globus_result_t globus_rls_client_lrc_attr_put(globus_rls_handle_t* h,char* key,globus_rls_attribute_t *attr,int overwrite) {
  globus_result_t err;
  int errcode;
  err = globus_rls_client_lrc_attr_add(h,key,attr);
  if(err != GLOBUS_SUCCESS) {
    err = globus_rls_client_error_info(err,&errcode,NULL,0,GLOBUS_TRUE);
    if((overwrite) && (errcode == GLOBUS_RLS_DBERROR)) { /* guess this can mean duplicate entry */
      globus_result_t err_;
      err_ = globus_rls_client_lrc_attr_remove(h,key,attr);
      globus_rls_free_result(err_);
      if(err_ != GLOBUS_SUCCESS) return err;
      return globus_rls_client_lrc_attr_put(h,key,attr,0);
    };
    if(errcode != GLOBUS_RLS_ATTR_NEXIST) return err;
    globus_rls_free_result(err);
    err = globus_rls_client_lrc_attr_create(h,attr->name,
                                               attr->objtype,attr->type);
    if(err != GLOBUS_SUCCESS) return err;
    err = globus_rls_client_lrc_attr_add(h,key,attr);
  };
  return err;
}
#endif


#ifdef HAVE_GLOBUS_RLS_CLIENT_H
typedef class meta_resolve_rls_t {
 public:
  DataPointRLS& url;
  bool source;
  bool success;
  bool locations_empty;
  bool obtained_info;
  std::string guid;
  meta_resolve_rls_t(DataPointRLS& u,bool s):url(u),source(s),success(false),locations_empty(false),obtained_info(false),guid("") {
  };
};

bool DataPointRLS::meta_resolve_callback(globus_rls_handle_t* h,const char* url,void* arg) {
  meta_resolve_rls_t* arg_ = (meta_resolve_rls_t*)arg;
  DataPointRLS& it = ((meta_resolve_rls_t*)arg)->url;
//  string& lfn = ((meta_resolve_rls_t*)arg)->lfn;
  std::string& lfn = it.meta_lfn;
  bool source = arg_->source;
  char errmsg[MAXERRMSG];
  globus_result_t err;
  int errcode;

  // Ask LRC if it contains file of interest

  if(it.guid_enabled && source && !arg_->guid.length()) {
    // map lfn->guid (only once)
    globus_rls_attribute_t opr;
    opr.type=globus_rls_attr_type_str;
    opr.val.s=(char*)it.meta_lfn.c_str();
    int off = 0;
    globus_list_t* guids = NULL;
    err = globus_rls_client_lrc_attr_search(h,"lfn",globus_rls_obj_lrc_lfn,
                  globus_rls_attr_op_eq,&opr,NULL,&off,1,&guids);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      odlog(INFO)<<"Failed to find GUID for specified LFN in "<<url<<" : "<<errmsg<<std::endl;
      return true;
    };
    if(!guids) {
      odlog(INFO)<<"There is no GUID for specified LFN in "<<url<<std::endl;
      return true;
    };
    globus_rls_attribute_object_t* obattr =
                 (globus_rls_attribute_object_t*)globus_list_first(guids);
    arg_->guid=obattr->key;
    globus_rls_client_free_list(guids);
  };
  globus_list_t* pfns_list = NULL;
  if(source) {
    if(arg_->guid.length()) {
      err=globus_rls_client_lrc_get_pfn(h,(char*)(arg_->guid.c_str()),0,0,&pfns_list);
    } else {
      err=globus_rls_client_lrc_get_pfn(h,(char*)(lfn.c_str()),0,0,&pfns_list);
    };
  } else {
    err=globus_rls_client_lrc_get_pfn(h,(char*)("__storage_service__"),0,0,&pfns_list);
  };
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
    if(errcode == GLOBUS_RLS_INVSERVER) {
      return true;
    } else if(errcode == GLOBUS_RLS_LFN_NEXIST) {
      return true;
    } else { // do not know
      odlog(INFO)<<"Warning: can't get PFNs from server "<<url<<" : "<<errmsg<<std::endl;
      return true;
    };
  };
  if(!(arg_->success)) {
    arg_->success=true; // got something
    if(source) it.is_metaexisting=true;
    arg_->locations_empty=(it.locations.size() == 0);
  };
  if(arg_->locations_empty) {
    globus_list_t* list_p;
    for(list_p=pfns_list;list_p;list_p=globus_list_rest(list_p)) {
      globus_rls_string2_t* str2=
                       (globus_rls_string2_t*)globus_list_first(list_p);
      //char* lfn = str2->s1;
      char* pfn = str2->s2;
      std::list<DataPointRLS::Location>::iterator loc =
        it.locations.insert(it.locations.end(),DataPointRLS::Location(url,pfn));
      loc->arg=(void*)1; // marker
      odlog(DEBUG)<<"Adding location: "<<url<<" - "<<pfn<<std::endl;
    };
  } else {
    std::list<Location>::iterator loc = it.locations.begin();
    for(;loc!=it.locations.end();++loc) {
      if(loc->arg != NULL) continue;
      globus_list_t* list_p;
      for(list_p=pfns_list;list_p;list_p=globus_list_rest(list_p)) {
        globus_rls_string2_t* str2=
                         (globus_rls_string2_t*)globus_list_first(list_p);
        //char* lfn = str2->s1;
        char* pfn = str2->s2;
        // for RLS URLs are used instead of metanames
        if(strncmp(pfn,loc->meta.c_str(),loc->meta.length()) == 0) {
          odlog(DEBUG)<<"Adding location: "<<url<<" - "<<pfn<<std::endl;
          if(source) {
            loc->meta=url; loc->url=pfn;
          } else {
            loc->meta=url;
          };
          loc->arg=(void*)1; break;
        };
      };
    };
  };
  globus_rls_client_free_list(pfns_list);
  if(!arg_->obtained_info) {
    /* obtain metadata - assume it is same everywhere */
    globus_list_t* attr_list;
    if(arg_->guid.length()) {
      err = globus_rls_client_lrc_attr_value_get(h,(char*)(arg_->guid.c_str()),
                             NULL,globus_rls_obj_lrc_lfn,&attr_list);
    } else {
      err = globus_rls_client_lrc_attr_value_get(h,(char*)(lfn.c_str()),NULL,
                                  globus_rls_obj_lrc_lfn,&attr_list);
    };
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      if(errcode == GLOBUS_RLS_ATTR_NEXIST) {
        return true;
      };
      odlog(INFO)<<"Warning: Failed to obtain attributes from "<<url<<" : "<<errmsg<<std::endl;
      return true;
    };
    it.is_metaexisting=true; // even for destination
    for(globus_list_t* list_pa=attr_list;list_pa;
                                       list_pa=globus_list_rest(list_pa)) {
      unsigned long long int i;
      globus_rls_attribute_t* attr =
                          (globus_rls_attribute_t*)globus_list_first(list_pa);
      if(attr->type != globus_rls_attr_type_str) continue;
      odlog(DEBUG)<<"Attribute: "<<attr->name<<" - "<<attr->val.s<<std::endl;
      if(strcmp(attr->name,"filechecksum") == 0) {
        if(!it.meta_checksum_valid) 
          it.meta_checksum(attr->val.s);
          // if(stringtoint(attr->val.s,i))it.meta_checksum(i);
      } else if(strcmp(attr->name,"size") == 0) {
        if(!it.meta_size_valid) 
          if(stringtoint(attr->val.s,i)) it.meta_size(i);
      } else if(strcmp(attr->name,"modifytime") == 0) {
        if(!it.meta_created_valid) 
          if(stringtoint(attr->val.s,i)) it.meta_created(i);
      } else if(strcmp(attr->name,"created") == 0) {
        if(!it.meta_created_valid) 
          if(stringtoint(attr->val.s,i)) it.meta_created(i);
      };
    };
    globus_rls_client_free_list(attr_list);
    arg_->obtained_info=true;
  };
  return true;
}
#endif

/* perform resolve operation, which can take long time */
bool DataPointRLS::meta_resolve(bool source) {
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  is_resolved=false;
  is_metaexisting=false;
  if(source) {
    if(meta_lfn.length() == 0) {
      odlog(INFO)<<"Source must contain LFN"<<std::endl;
      return false;
    };
    std::list<std::string> rlis;
    std::list<std::string> lrcs;
    rlis.push_back(meta_service_url);
    lrcs.push_back(meta_service_url);
    meta_resolve_rls_t arg(*this,source);
    bool res = rls_find_lrcs(rlis,lrcs,true,false,&meta_resolve_callback,(void*)(&arg));
    if(!arg.success) return false;
    // Remove unresolved locations
    std::list<Location>::iterator loc = locations.begin();
    for(;loc!=locations.end();) {
      if(loc->arg==NULL) {
        odlog(DEBUG)<<"Removing location: "<<loc->meta<<" - "<<loc->url<<std::endl;
        loc=locations.erase(loc);
      } else {
        odlog(DEBUG)<<"Using location: "<<loc->meta<<" - "<<loc->url<<std::endl;
        loc->arg=(void*)1; ++loc;
      };
    };
  } else { 
    if(meta_lfn.length() == 0) {
      odlog(INFO)<<"Destination must contain LFN"<<std::endl;
      return false;
    };
    std::list<std::string> rlis;
    std::list<std::string> lrcs;
    rlis.push_back(meta_service_url);
    lrcs.push_back(meta_service_url);
    if(locations.size() == 0) {
      odlog(INFO)<<"Warning: Locations are missing in destination RLS url - will use those registered with special name"<<std::endl;
    };
    meta_resolve_rls_t arg(*this,source);
    bool res = rls_find_lrcs(rlis,lrcs,true,false,&meta_resolve_callback,(void*)(&arg));
    if(!arg.success) return false;
    if(locations.size() == 0) {
      odlog(INFO)<<"No locations found for destination"<<std::endl;
      return false;
    };
    // Make pfns
    std::list<Location>::iterator loc = locations.begin();
    std::list<std::string>::iterator lrc_p = lrcs.begin();
    for(;loc!=locations.end();) {
      bool se_uses_lfn=false;
      if(strncasecmp(loc->url.c_str(),"se://",5) == 0) { loc->url+="?"; se_uses_lfn=true; }
      else { loc->url+="/"; };
      if(guid_enabled) {
        std::string guid;
        GUID(guid);
        if((!se_uses_lfn) && (!pfn_path.empty())) { loc->url+=pfn_path; }
        else { loc->url+=guid; };
      } else {
        if((!se_uses_lfn) && (!pfn_path.empty())) { loc->url+=pfn_path; }
        else { loc->url+=meta_lfn; };
      };
      if(loc->arg!=NULL) {
        odlog(DEBUG)<<"Using location: "<<loc->meta<<" - "<<loc->url<<std::endl;
        ++loc;
      } else { // Use arbitrary lrc
        if(lrc_p == lrcs.end()) { // no LRC
          odlog(DEBUG)<<"Removing location: "<<loc->meta<<" - "<<loc->url<<std::endl;
          loc=locations.erase(loc);
        } else {
          loc->meta=*lrc_p; ++lrc_p; if(lrc_p==lrcs.end()) lrc_p=lrcs.begin();
          odlog(DEBUG)<<"Using location: "<<loc->meta<<" - "<<loc->url<<std::endl;
          loc->arg=(void*)1; ++loc;
        };
      };
    };
  };
  odlog(DEBUG)<<"meta_get_data: checksum: "<<meta_checksum()<<std::endl;
  odlog(DEBUG)<<"meta_get_data: size: "<<meta_size()<<std::endl;
  odlog(DEBUG)<<"meta_get_data: created: "<<meta_created()<<std::endl;
  if(common_url_options.length() != 0) {
    std::list<Location>::iterator loc = locations.begin();
    for(;loc!=locations.end();++loc) {
      add_url_options(loc->url,common_url_options.c_str(),0);
    };
  };
  location=locations.begin();
  is_resolved=true;
  return true;
#else
  return false;
#endif
}

bool DataPointRLS::meta_preregister(bool replication,bool force) {
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  if(replication) { /* replicating inside same lfn */
    if(!is_metaexisting) { /* for replication it must be there */
      odlog(ERROR)<<"LFN is missing in RLS (needed for replication)"<<std::endl;
      return false;
    };
    return true;
  };
  if(is_metaexisting) { /* algorithm require this to be new file */
    if(!force) {
      odlog(ERROR)<<"LFN already exists in replica"<<std::endl;
      return false;
    };
  };
  /* RLS does not support LFN only in database - hence doing nothing here */
  return true;
#else
  return false;
#endif
}

bool DataPointRLS::meta_postregister(bool replication,bool failure) {
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  globus_rls_handle_t* h;
  char errmsg[MAXERRMSG];
  globus_result_t err;
  int errcode;

  err = globus_rls_client_connect((char*)(meta_service_url.c_str()),&h);
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,NULL,errmsg,MAXERRMSG,GLOBUS_FALSE);
    odlog(INFO)<<"Failed to connect to RLS server: "<<errmsg<<std::endl;
    return false;
  };
  // assume that is RLI and try to resolve for special/any name 

  std::string pfn(location->url.c_str());
  ::canonic_url(pfn); // it is always better to register pure url
  std::string guid;
  const char* rls_lfn = meta_lfn.c_str();
  if(!replication) {
    if(guid_enabled) {
      for(;;) {
        // generate guid
        GUID(guid);
        // store in LRC
        if((err = globus_rls_client_lrc_create(h,(char*)(guid.c_str()),
                  (char*)(pfn.c_str()))) != GLOBUS_SUCCESS) {
          err=globus_rls_client_error_info(err,&errcode,NULL,0,GLOBUS_TRUE);
          if(errcode == GLOBUS_RLS_LFN_EXIST) {
            globus_rls_free_result(err);
            continue;
          };
        };
        rls_lfn = guid.c_str();
        break;
      };
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        odlog(INFO)<<"Failed to create GUID in RLS: "<<errmsg<<std::endl;
        globus_rls_client_close(h);
        return false;
      };
      // Check if there is no same LFN
      globus_rls_attribute_t opr;
      opr.type=globus_rls_attr_type_str;
      opr.val.s=(char*)meta_lfn.c_str();
      int off = 0;
      globus_list_t* guids = NULL;
      err = globus_rls_client_lrc_attr_search(h,"lfn",globus_rls_obj_lrc_lfn,
                  globus_rls_attr_op_eq,&opr,NULL,&off,1,&guids);
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE)
;
        if((errcode != GLOBUS_RLS_LFN_NEXIST) &&
           (errcode != GLOBUS_RLS_ATTR_NEXIST) &&
           (errcode != GLOBUS_RLS_ATTR_VALUE_NEXIST)) {
          odlog(INFO)<<"Failed to check for existing LFN in "<<url<<" : "<<errmsg<<std::endl;
          globus_rls_client_close(h);
          return false;
        };
      };
      if(guids) {
        globus_rls_client_free_list(guids);
        odlog(INFO)<<"There is same LFN in "<<url<<std::endl;
        globus_rls_client_close(h);
        return false;
      };
      // add LFN
      globus_rls_attribute_t attr;
      attr.objtype=globus_rls_obj_lrc_lfn;
      attr.type=globus_rls_attr_type_str;
      attr.name="lfn";
      attr.val.s=(char*)meta_lfn.c_str();
      err=globus_rls_client_lrc_attr_put(h,(char*)(rls_lfn),&attr,0);
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        odlog(INFO)<<"Failed to add LFN-GUID to RLS: "<<errmsg<<std::endl;
        globus_rls_client_close(h);
        return false;
      };
    } else {
      if((err = globus_rls_client_lrc_create(h,(char*)(meta_lfn.c_str()),
                  (char*)(pfn.c_str()))) != GLOBUS_SUCCESS) {
        err=globus_rls_client_error_info(err,&errcode,NULL,0,GLOBUS_TRUE);
        if(errcode == GLOBUS_RLS_LFN_EXIST) {
           globus_rls_free_result(err);
           err = globus_rls_client_lrc_add(h,(char*)(meta_lfn.c_str()),
                                           (char*)(pfn.c_str()));
        };
      };
    };
  }
  else {
    if(guid_enabled) {
      // get guid
      globus_rls_attribute_t opr;
      opr.type=globus_rls_attr_type_str;
      opr.val.s=(char*)meta_lfn.c_str();
      int off = 0;
      globus_list_t* guids = NULL;
      err = globus_rls_client_lrc_attr_search(h,"lfn",globus_rls_obj_lrc_lfn,
                  globus_rls_attr_op_eq,&opr,NULL,&off,1,&guids);
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        odlog(INFO)<<"Failed to find GUID for specified LFN in "<<url<<" : "<<errmsg<<std::endl;
        globus_rls_client_close(h);
        return false;
      };
      if(!guids) {
        odlog(INFO)<<"There is no GUID for specified LFN in "<<url<<std::endl;
        globus_rls_client_close(h);
        return false;
      };
      globus_rls_attribute_object_t* obattr =
                 (globus_rls_attribute_object_t*)globus_list_first(guids);
      guid=obattr->key;
      globus_rls_client_free_list(guids);
      rls_lfn = guid.c_str();
    };
    err = globus_rls_client_lrc_add(h,(char*)(rls_lfn),
                                      (char*)(pfn.c_str()));
  };
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
    if(errcode != GLOBUS_RLS_MAPPING_EXIST) {
      odlog(INFO)<<"Failed to create/add LFN-PFN mapping: "<<errmsg<<std::endl;
      globus_rls_client_close(h);
      return false;
    };
  };
  globus_rls_attribute_t attr;
  std::string attr_val;
  attr.objtype=globus_rls_obj_lrc_lfn;
  attr.type=globus_rls_attr_type_str;
  attr.name="filetype";
  attr.val.s="file";
  err=globus_rls_client_lrc_attr_put(h,(char*)(rls_lfn),&attr,0);
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
    if(errcode != GLOBUS_RLS_ATTR_EXIST) {
      odlog(INFO)<<"Warning: failed to add attribute to RLS: "<<errmsg<<std::endl;
    };
  };
  if(meta_size_valid) {
    attr.name="size";
    attr_val=inttostring(meta_size_);
    attr.val.s=(char*)(attr_val.c_str());
    err=globus_rls_client_lrc_attr_put(h,(char*)(rls_lfn),&attr,0);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      if(errcode != GLOBUS_RLS_ATTR_EXIST) {
        odlog(INFO)<<"Warning: failed to add attribute to RLS: "<<errmsg<<std::endl;
      };
    };
  };
  if(meta_checksum_valid) {
    attr.name="filechecksum";
    attr_val=meta_checksum_;
    // attr_val=inttostring(meta_checksum_);
    attr.val.s=(char*)(attr_val.c_str());
    err=globus_rls_client_lrc_attr_put(h,(char*)(rls_lfn),&attr,0);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      if(errcode != GLOBUS_RLS_ATTR_EXIST) {
        odlog(INFO)<<"Warning: failed to add attribute to RLS: "<<errmsg<<std::endl;
      };
    };
  };
  if(meta_created_valid) {
    attr.name="modifytime";
    attr_val=inttostring(meta_created_);
    attr.val.s=(char*)(attr_val.c_str());
    err=globus_rls_client_lrc_attr_put(h,(char*)(rls_lfn),&attr,0);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      if(errcode != GLOBUS_RLS_ATTR_EXIST) {
        odlog(INFO)<<"Warning: failed to add attribute to RLS: "<<errmsg<<std::endl;
      };
    };
  };
  if(meta_attributes.size()>0) {
    std::map<std::string,std::string>::iterator pos;
    for( pos = meta_attributes.begin(); pos != meta_attributes.end(); ++pos) {
      attr.name= (char*)(pos->first.c_str());
      attr.val.s = (char*)(pos->second.c_str());
      err=globus_rls_client_lrc_attr_put(h,(char*)(rls_lfn),&attr,0);
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        if(errcode != GLOBUS_RLS_ATTR_EXIST) {
          odlog(INFO)<<"Warning: failed to add attribute to RLS: "<<errmsg<<std::endl;
        };
      };
    };
  };
  globus_rls_client_close(h);
  return true;
#else
  return false;
#endif
}

bool DataPointRLS::meta_preunregister(bool replication) {
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  if(replication) return true;
  return true;
#else
  return false;
#endif
}

#ifdef HAVE_GLOBUS_RLS_CLIENT_H
typedef class meta_unregister_rls_t {
 public:
  DataPointRLS& url;
  bool all;
  bool failure;
  std::string guid;
  meta_unregister_rls_t(bool a,DataPointRLS& u):all(a),url(u),failure(false),guid("") {
  };
};

bool DataPointRLS::meta_unregister_callback(globus_rls_handle_t* h,const char* url,void* arg) {
  meta_unregister_rls_t* arg_ = (meta_unregister_rls_t*)arg;
  DataPointRLS& it = ((meta_unregister_rls_t*)arg)->url;
  int lrc_offset = 0;
  int lrc_limit = 0;
  globus_result_t err;
  int errcode;
  char errmsg[MAXERRMSG];
  globus_list_t* pfns_list;
  const char* lfn = it.meta_lfn.c_str();
  if(it.guid_enabled && !arg_->guid.length()) {
    // map lfn->guid (only once)
    globus_rls_attribute_t opr;
    opr.type=globus_rls_attr_type_str;
    opr.val.s=(char*)it.meta_lfn.c_str();
    int off = 0;
    globus_list_t* guids = NULL;
    err = globus_rls_client_lrc_attr_search(h,"lfn",globus_rls_obj_lrc_lfn,
                  globus_rls_attr_op_eq,&opr,NULL,&off,1,&guids);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      odlog(VERBOSE)<<"Warning: failed to find GUID for specified LFN in "<<url<<" : "<<errmsg<<std::endl;
      return true;
    };
    if(!guids) {
      odlog(VERBOSE)<<"Warning: there is no GUID for specified LFN in "<<url<<std::endl;
      return true;
    };
    globus_rls_attribute_object_t* obattr =
                 (globus_rls_attribute_object_t*)globus_list_first(guids);
    arg_->guid=obattr->key;
    globus_rls_client_free_list(guids);
    lfn=arg_->guid.c_str();
  };
  if(arg_->all) {
    err = globus_rls_client_lrc_get_pfn(h,(char*)lfn,&lrc_offset,lrc_limit,&pfns_list);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      odlog(INFO)<<"Warning: Failed to retrieve LFN/PFNs from "<<url<<" : "<<errmsg<<std::endl;
      arg_->failure=true; return true;
    };
    globus_list_t* list_p;
    for(list_p=pfns_list;list_p;list_p=globus_list_rest(list_p)) {
      globus_rls_string2_t* str2=
                         (globus_rls_string2_t*)globus_list_first(list_p);
      char* pfn = str2->s1;
      if(strncasecmp(pfn,"se://",5) == 0) {
        odlog(DEBUG)<<"SE location will be unregistered automatically"<<std::endl;
      } else {
        err = globus_rls_client_lrc_delete(h,(char*)lfn,pfn);
        if(err != GLOBUS_SUCCESS) {
          globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
          if((errcode != GLOBUS_RLS_MAPPING_NEXIST) && 
             (errcode != GLOBUS_RLS_LFN_NEXIST) && 
             (errcode != GLOBUS_RLS_PFN_NEXIST)) {
            odlog(INFO)<<"Warning: Failed to delete LFN/PFN from "<<url<<" : "<<errmsg<<std::endl;
            arg_->failure=true; continue;
          };
        };
      };
    };
    globus_rls_client_free_list(pfns_list);
  } else { // ! all
    err = globus_rls_client_lrc_delete(h,(char*)lfn,(char*)(it.location->url.c_str()));
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      if((errcode != GLOBUS_RLS_MAPPING_NEXIST) &&
         (errcode != GLOBUS_RLS_LFN_NEXIST) &&
         (errcode != GLOBUS_RLS_PFN_NEXIST)) {
        odlog(INFO)<<"Warning: Failed to delete LFN/PFN from "<<url<<" : "<<errmsg<<std::endl;
        arg_->failure=true;
      };
    };
  };
  return true;
}
#endif

bool DataPointRLS::meta_unregister(bool all) {
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  if(!all) {
    if(location == locations.end()) { 
      odlog(ERROR)<<"Location is missing"<<std::endl;
      return false;
    };
    if(strncasecmp(location->url.c_str(),"se://",5) == 0) {
      odlog(DEBUG)<<"SE location will be unregistered automatically"<<std::endl;
      return true;
    };
  };
  if(!guid_enabled) {
  globus_rls_handle_t* h;
  char errmsg[MAXERRMSG];
  globus_result_t err;
  int errcode;
  globus_list_t* pfns_list;

  err = globus_rls_client_connect((char*)(meta_service_url.c_str()),&h);
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,NULL,errmsg,MAXERRMSG,GLOBUS_FALSE);
    odlog(INFO)<<"Failed to connect to RLS server: "<<errmsg<<std::endl;
    return false;
  };
  // first find all LRC servers storing required information
  globus_list_t* lrcs = NULL;
  globus_rls_string2_t lrc_direct;
  globus_bool_t free_lrcs = GLOBUS_FALSE;
  lrc_direct.s1=(char*)(meta_lfn.c_str());
  lrc_direct.s2=NULL; // for current connection
  int lrc_offset = 0;
  int lrc_limit = 0;
  err = globus_rls_client_rli_get_lrc(h,(char*)(meta_lfn.c_str()),&lrc_offset,lrc_limit,&lrcs);
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
    if(errcode == GLOBUS_RLS_LFN_NEXIST) {
      odlog(INFO)<<"LFN must be already deleted, try LRC anyway"<<std::endl;
      lrcs=NULL;
    } else if(errcode != GLOBUS_RLS_INVSERVER) {
      odlog(INFO)<<"Failed to retrieve LFN/LRC: "<<errmsg<<std::endl;
      globus_rls_client_close(h);
      return false;
    };
    // Probably that is LRC server only.
    globus_list_insert(&lrcs,&lrc_direct);
  } else { free_lrcs=GLOBUS_TRUE; };
  globus_list_t* p;
  err=GLOBUS_SUCCESS;
  // TODO: sort by lrc and cache connections
  bool failure = false;
  for(p=lrcs;p;p=globus_list_rest(p)) {
    globus_rls_string2_t* str2 = (globus_rls_string2_t*)globus_list_first(p);
    char* lrc = str2->s2;
    globus_rls_handle_t* h_;
    if(lrc) {
      err = globus_rls_client_connect(lrc,&h_);
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        odlog(INFO)<<"Warning: Failed to connect to LRC at "<<lrc<<" : "<<errmsg<<std::endl;
        failure=true; continue;
      };
    } else { h_=h; }; // This server is already connected
    if(all) {
      err = globus_rls_client_lrc_get_pfn(h_,(char*)(meta_lfn.c_str()),&lrc_offset,lrc_limit,&pfns_list);
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        if((errcode != GLOBUS_RLS_MAPPING_NEXIST) && (errcode != GLOBUS_RLS_LFN_NEXIST) && (errcode != GLOBUS_RLS_PFN_NEXIST)) {
          odlog(INFO)<<"Warning: Failed to retrieve LFN/PFNs from "
                  <<(lrc?lrc:meta_service_url.c_str())<<" : "<<errmsg
                  <<std::endl;
          if(lrc) globus_rls_client_close(h_);
          failure=true; continue;
        };
        // Probably no such LFN - good, less work to do
        pfns_list=NULL;
      };
      globus_list_t* list_p;
      for(list_p=pfns_list;list_p;list_p=globus_list_rest(list_p)) {
        globus_rls_string2_t* str2=
                         (globus_rls_string2_t*)globus_list_first(list_p);
        char* pfn = str2->s1;
        if(strncasecmp(pfn,"se://",5) == 0) {
          odlog(DEBUG)<<"SE location will be unregistered automatically"<<std::endl;
        } else {
          err = globus_rls_client_lrc_delete(h_,(char*)(meta_lfn.c_str()),pfn);
          if(err != GLOBUS_SUCCESS) {
            globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
            if((errcode != GLOBUS_RLS_MAPPING_NEXIST) && (errcode != GLOBUS_RLS_LFN_NEXIST) && (errcode != GLOBUS_RLS_PFN_NEXIST)) {
              odlog(INFO)<<"Warning: Failed to delete LFN/PFN from "<<(lrc?lrc:meta_service_url.c_str())<<" : "<<errmsg<<std::endl;
              if(lrc) globus_rls_client_close(h_);
              failure=true; continue;
            };
          };
        };
      };
      if(pfns_list) globus_rls_client_free_list(pfns_list);
    } else { // ! all
      err = globus_rls_client_lrc_delete(h_,(char*)(meta_lfn.c_str()),(char*)(location->url.c_str()));
      if(err != GLOBUS_SUCCESS) {
        globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
        if((errcode != GLOBUS_RLS_MAPPING_NEXIST) && 
           (errcode != GLOBUS_RLS_LFN_NEXIST) &&
           (errcode != GLOBUS_RLS_PFN_NEXIST)) {
          odlog(INFO)<<"Warning: Failed to delete LFN/PFN from "<<lrc<<" : "<<errmsg<<std::endl;
          if(lrc) globus_rls_client_close(h_);
          failure=true; continue;
        };
      };
    };
    if(lrc) globus_rls_client_close(h_);
  };
  globus_rls_client_close(h);
  if(free_lrcs) { globus_rls_client_free_list(lrcs); }
  else { globus_list_free(lrcs); };
  if(!failure) fix_unregistered(all);
  return !failure;
  } else { // guid_enabled
    std::list<std::string> rlis;
    std::list<std::string> lrcs;
    rlis.push_back(meta_service_url);
    lrcs.push_back(meta_service_url);
    meta_unregister_rls_t arg(all,*this);
    bool res = rls_find_lrcs(rlis,lrcs,true,false,&meta_unregister_callback,(void*)(&arg));
    if(!arg.failure) fix_unregistered(all);
    return !arg.failure;
  };
#else
  return false;
#endif
}

/* rls://[url[|url[...]]@]server/lfn */
bool DataPointRLS::process_meta_url(void) {
  if(!strncasecmp(url.c_str(),"rls://",6)) {
    meta_service_url.resize(0); locations.clear(); meta_lfn.resize(0);
    std::string url_(url.c_str());
    /* find out if it contains locations */
    std::string::size_type loc_start=6;
    std::string::size_type loc_end=url_.find('@',loc_start);
    std::string urls("");
    if(loc_end!=std::string::npos) {
      urls=url_.substr(loc_start,loc_end-loc_start);
      url_.erase(loc_start,loc_end-loc_start+1);
    };
    /* get lfn */
    std::string::size_type server_start=6;
    std::string::size_type server_end=url_.find('/',server_start);
    std::string filename;
    if(server_end==std::string::npos) {
//      odlog(INFO)<<"LFN is missing in url "<<url_<<std::endl;
//      return false;
//      std::string::size_type port_start = url_.find(':',server_start);
//      if(port_start==std::string::npos) {
//        url_+=":"; url_+=inttostring(PORT_DEFAULT_RLS);
//      };
      filename=""; meta_service_url=url_;
    } else {
      filename=url_.substr(server_end+1);
//      std::string::size_type port_start = url_.find(':',server_start);
//      if((port_start==std::string::npos) || (port_start > server_end)) {
//        std::string port(":"); port+=inttostring(PORT_DEFAULT_RLS); 
//        url_.insert(server_end,port); server_end+=port.length();
//      };
      meta_service_url=url_.substr(0,server_end);
    };
    std::string guid_val;
    if(get_url_option(meta_service_url,"guid",guid_val) == 0) {
      if((guid_val == "yes") || (guid_val == "")) {
        guid_enabled=true;
      };
    };
    get_url_option(meta_service_url,"pfnpath",pfn_path);
    ::canonic_url(meta_service_url);
    extract_meta_attributes(filename);
    meta_lfn=filename;
    odlog(DEBUG)<<"LFN: "<<meta_lfn<<std::endl;
    odlog(DEBUG)<<"RLS server: "<<meta_service_url<<std::endl;
    odlog(DEBUG)<<"Location urls: "<<urls<<std::endl;
    std::string::size_type n=0;
    for(std::string::size_type nn=0;n<urls.length();) {
      nn=urls.find('|',n); if(nn == std::string::npos) nn=urls.length();
      if(n == nn) { n++; continue; };
      std::string loc(urls.c_str()+n,nn-n);
      if(loc[0] == ';') { common_url_options+=loc; }
      else {
        locations.push_back(DataPointRLS::Location(loc.c_str(),loc.c_str()));
      };
      n=nn+1;
    };
    return true;
  };
  return false;
};

#ifdef HAVE_GLOBUS_RLS_CLIENT_H
static bool get_attributes(globus_rls_handle_t* h,const char* lfn,DataPoint::FileInfo &f) {
  globus_list_t* attr_list;
  char errmsg[MAXERRMSG];
  globus_result_t err;
  int errcode;
  err = globus_rls_client_lrc_attr_value_get(h,(char*)(lfn),NULL,
                                  globus_rls_obj_lrc_lfn,&attr_list);
 if(err != GLOBUS_SUCCESS) {
   globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
   if(errcode != GLOBUS_RLS_ATTR_NEXIST) {
     odlog(INFO)<<"Warning: Failed to retrieve attributes: "<<errmsg<<std::endl;
     return false;
   };
   return true;
 };
 for(globus_list_t* pa=attr_list;pa;pa=globus_list_rest(pa)) {
   globus_rls_attribute_t* attr =
                        (globus_rls_attribute_t*)globus_list_first(pa);
   if(attr->type != globus_rls_attr_type_str) continue;
   odlog(DEBUG)<<"Attribute: "<<attr->name<<" - "<<attr->val.s<<std::endl;
   if(strcmp(attr->name,"filechecksum") == 0) {
     f.checksum=attr->val.s; f.checksum_available=true;
     // if(stringtoint(attr->val.s,f.checksum)) f.checksum_available=true;
   } else if(strcmp(attr->name,"size") == 0) {
     if(stringtoint(attr->val.s,f.size)) f.size_available=true;
   } else if(strcmp(attr->name,"modifytime") == 0) {
     if(stringtotime(f.created,std::string(attr->val.s))==0)f.created_available=true;
   } else if(strcmp(attr->name,"created") == 0) {
     if(stringtotime(f.created,std::string(attr->val.s))==0)f.created_available=true;
   };
 };
 globus_rls_client_free_list(attr_list);
 return true;
}

typedef class list_files_rls_t {
 public:
  std::list<DataPoint::FileInfo> &files;
  DataPointRLS& url;
  bool success;
  bool resolve;
  std::string guid;
  list_files_rls_t(std::list<DataPoint::FileInfo> &f,DataPointRLS& u,bool r):files(f),url(u),resolve(r),success(false),guid("") {
  };
};

bool DataPointRLS::list_files_callback(globus_rls_handle_t* h,const char* url,void* arg) {
  list_files_rls_t* arg_ = (list_files_rls_t*)arg;
  DataPointRLS& it = ((list_files_rls_t*)arg)->url;
  std::list<DataPoint::FileInfo>& files = ((list_files_rls_t*)arg)->files; 
  int lrc_offset = 0;
  globus_result_t err;
  int errcode;
  char errmsg[MAXERRMSG];
  globus_list_t* pfns = NULL;
  if(it.guid_enabled && it.meta_lfn.length() && !arg_->guid.length()) {
    // looking gor guid only once
    // looking for guid only if lfn specified
    globus_rls_attribute_t opr;
    opr.type=globus_rls_attr_type_str;
    opr.val.s=(char*)it.meta_lfn.c_str();
    int off = 0;
    globus_list_t* guids = NULL;
    err = globus_rls_client_lrc_attr_search(h,"lfn",globus_rls_obj_lrc_lfn,
                  globus_rls_attr_op_eq,&opr,NULL,&off,1,&guids);
    if(err != GLOBUS_SUCCESS) {
      globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
      odlog(INFO)<<"Failed to find GUID for specified LFN in "<<url<<" : "<<errmsg<<std::endl;
      return true;
    };
    if(!guids) {
      odlog(INFO)<<"There is no GUID for specified LFN in "<<url<<std::endl;
      return true;
    };
    globus_rls_attribute_object_t* obattr = 
                 (globus_rls_attribute_object_t*)globus_list_first(guids);
    arg_->guid=obattr->key;
    globus_rls_client_free_list(guids);
  };
  if(arg_->guid.length()) {
    err = globus_rls_client_lrc_get_pfn(h,(char*)(arg_->guid.c_str()),
                          &lrc_offset,1000,&pfns);
  } else if(it.meta_lfn.length()) {
    err = globus_rls_client_lrc_get_pfn(h,(char*)(it.meta_lfn.c_str()),
                          &lrc_offset,1000,&pfns);
  } else {
    err = globus_rls_client_lrc_get_pfn_wc(h,"*",rls_pattern_unix,
                          &lrc_offset,1000,&pfns);
  };
  if(err != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(err,&errcode,errmsg,MAXERRMSG,GLOBUS_FALSE);
    if(errcode == GLOBUS_RLS_LFN_NEXIST) {
      odlog(DEBUG)<<"No LFNs found in "<<url<<std::endl;
      ((list_files_rls_t*)arg)->success=true;
      return true;
    };
    odlog(INFO)<<"Failed to retrieve list of LFNs/PFNs from "<<url<<" : "<<errmsg<<std::endl;
    return true;
  };
  ((list_files_rls_t*)arg)->success=true;
  std::string last_lfn = "";
  const char* last_guid = "";
  globus_list_t* p;
  for(p=pfns;p;p=globus_list_rest(p)) {
    globus_rls_string2_t* str2=
                   (globus_rls_string2_t*)globus_list_first(p);
    char* lfn = str2->s1;
    char* pfn = str2->s2;
    if(it.guid_enabled) {
      if(strcmp(lfn,last_guid) != 0) {
        last_guid=lfn; last_lfn="";
        // get real lfn
        globus_list_t* lfn_list = NULL;
        err = globus_rls_client_lrc_attr_value_get(h,lfn,"lfn",
                                  globus_rls_obj_lrc_lfn,&lfn_list);
        if(err != GLOBUS_SUCCESS) {
          globus_rls_client_error_info(err,&errcode,errmsg,
                                             MAXERRMSG,GLOBUS_FALSE);
          continue;
        };
        if(lfn_list == NULL) continue;
        globus_rls_attribute_t* attr =
           (globus_rls_attribute_t*)globus_list_first(lfn_list);
        if(attr->type != globus_rls_attr_type_str) {
          globus_rls_client_free_list(lfn_list);
          continue;
        };
        // use only first lfn (TODO: all lfns)
        last_lfn=attr->val.s;
        globus_rls_client_free_list(lfn_list);
      };
      if(last_lfn.length()) {
        odlog(DEBUG)<<"lfn: "<<last_lfn<<"("<<last_guid<<") - "<<" pfn: "<<pfn<<std::endl;
        std::list<DataPoint::FileInfo>::iterator f;
        for(f=files.begin();f != files.end();++f) if(f->name == last_lfn) break;
        if(f == files.end()) {
          f=files.insert(files.end(),DataPoint::FileInfo(last_lfn.c_str()));
          if(((list_files_rls_t*)arg)->resolve) get_attributes(h,last_guid,*f);
        };
        f->urls.push_back(std::string(pfn));
      };
    } else { // !guid_enabled
      odlog(DEBUG)<<"lfn: "<<lfn<<" - "<<" pfn: "<<pfn<<std::endl;
      std::list<DataPoint::FileInfo>::iterator f;
      for(f=files.begin();f != files.end();++f) if(f->name == lfn) break;
      if(f == files.end()) {
        f=files.insert(files.end(),DataPoint::FileInfo(lfn));
        if(((list_files_rls_t*)arg)->resolve) get_attributes(h,lfn,*f);
      };
      f->urls.push_back(std::string(pfn));
    };
  };
  globus_rls_client_free_list(pfns);
  return true;
}
#endif

bool DataPointRLS::list_files(std::list<DataPoint::FileInfo> &files,bool resolve) {
#ifdef HAVE_GLOBUS_RLS_CLIENT_H
  std::list<std::string> rlis;
  std::list<std::string> lrcs;
  rlis.push_back(meta_service_url);
  lrcs.push_back(meta_service_url);
  list_files_rls_t arg(files,*this,resolve);
  bool res = rls_find_lrcs(rlis,lrcs,true,false,&list_files_callback,(void*)(&arg));
  return arg.success;
#else
  return false;
#endif
}


