#include "../std.h"
#include <iostream>

#include "../misc/url_options.h"
#include "../misc/stringtoint.h"
#include "../misc/globus_error_utils.h"
#include "../transfer/lister.h"
#include "../misc/log_time.h"
#include <arc/certificate.h>
#include "datapoint.h"
#include "databufferpar.h"
#include "datahandle_ftp.h"

#define RESULT_SUCCESS 0
#define RESULT_ERROR 1
#define RESULT_CREDENTIALS_EXPIRED 2

bool DataHandleFTP::check_credentials(void) {
  try {
    Certificate ci(PROXY);
    if(!ci.IsExpired()) return true;
  } catch (std::exception) { // proxy not available
  };
  try {
    Certificate ci(USERCERT);
    if(!ci.IsExpired()) return true;
  } catch (std::exception) { // proxy not available
  };
  odlog(ERROR)<<"proxy/credentials expired"<<std::endl;
  failure_description="credentials expired";
  failure_code=credentials_expired_failure;
  return false;
}

void DataHandleFTP::ftp_complete_callback(void* arg,globus_ftp_client_handle_t* handle,globus_object_t* error) {
  DataHandleFTP* it = (DataHandleFTP*)arg;
  if(error == GLOBUS_SUCCESS) {
    odlog(VERBOSE)<<"ftp_complete_callback: success"<<std::endl;
    it->ftp_completed.signal(RESULT_SUCCESS);
  }
  else {
    char* tmp = globus_object_printable_to_string(error);
    odlog(INFO)<<"ftp_complete_callback: error: "<<tmp<<std::endl; free(tmp);
    if(it->is_secure) {
      if(it->check_credentials()) {
        it->ftp_completed.signal(RESULT_ERROR);
      } else {
        it->ftp_completed.signal(RESULT_CREDENTIALS_EXPIRED);
      };
    } else {
      it->ftp_completed.signal(RESULT_ERROR);
    };
  };
}

void DataHandleFTP::ftp_check_callback(void* arg,globus_ftp_client_handle_t* handle,globus_object_t* error,globus_byte_t* buffer,globus_size_t length,globus_off_t offset,globus_bool_t eof) {
  odlog(DEBUG)<<"ftp_check_callback"<<std::endl;
  DataHandleFTP* it = (DataHandleFTP*)arg;
  if(error != GLOBUS_SUCCESS) { 
     odlog(DEBUG)<<"Globus error: "<<error<<std::endl;
     return;
  };
  if(eof) return;
  globus_result_t res=globus_ftp_client_register_read(&(it->ftp_handle),(globus_byte_t*)(it->ftp_buf),sizeof(it->ftp_buf),&ftp_check_callback,it);
  if(res != GLOBUS_SUCCESS) {
    odlog(INFO)<<"Registration of Globus FTP buffer failed - cancel check"<<std::endl;
    odlog(DEBUG)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    globus_ftp_client_abort(&(it->ftp_handle));
    return;
  };
  return;
}

/* check if user is allowed to get specified file. */
bool DataHandleFTP::check(void) {
  if(!DataHandleCommon::check()) return false;
  globus_result_t res;
  globus_off_t size = 0;
  globus_abstime_t gl_modify_time;
  time_t modify_time;
  int modify_utime;
  int c_res;
  bool size_available = false;
  res = globus_ftp_client_size(&ftp_handle,c_url.c_str(),&ftp_opattr,
                     &size,&ftp_complete_callback,this);
  if(res != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"check_ftp: globus_ftp_client_size failed"<<std::endl;
    odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl;
    // return false;
  } else if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
    odlog(INFO)<<"check_ftp: timeout waiting for size"<<std::endl;
    globus_ftp_client_abort(&ftp_handle);
    ftp_completed.wait(c_res,-1);
    // return false;
  } else if(c_res != 0) {
    odlog(INFO)<<"check_ftp: failed to get file's size"<<std::endl;
    // return false;
  } else {
    url->meta_size(size);
    size_available=true;
  };
  res = globus_ftp_client_modification_time(&ftp_handle,c_url.c_str(),
                     &ftp_opattr,&gl_modify_time,&ftp_complete_callback,this);
  if(res != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"check_ftp: globus_ftp_client_modification_time failed"<<std::endl;
    odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl;
    // return false;
  } else if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
    odlog(INFO)<<"check_ftp: timeout waiting for modification_time"<<std::endl;
    globus_ftp_client_abort(&ftp_handle);
    ftp_completed.wait(c_res,-1);
    // return false;
  } else if(c_res != 0) {
    odlog(INFO)<<"check_ftp: failed to get file's modification time"<<std::endl;
    // return false;
  } else {
    GlobusTimeAbstimeGet(gl_modify_time,modify_time,modify_utime);
    url->meta_created(modify_time);
  };
  // Do not use partial_get for ordinary ftp. Stupid globus tries to 
  // use non-standard commands anyway.
  if(is_secure) {
    res=globus_ftp_client_partial_get(&ftp_handle,c_url.c_str(),&ftp_opattr,
                         GLOBUS_NULL,0,1,&ftp_complete_callback,this);
    if(res != GLOBUS_SUCCESS) {
      odlog(DEBUG)<<"check_ftp: globus_ftp_client_get failed"<<std::endl;
      odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl;
      return false;
    };
    ftp_eof_flag=false; /* use eof_flag to pass result from callback */
    odlog(DEBUG)<<"check_ftp: globus_ftp_client_register_read"<<std::endl;
    res=globus_ftp_client_register_read(&ftp_handle,(globus_byte_t*)ftp_buf,sizeof(ftp_buf),&ftp_check_callback,this);
    if(res != GLOBUS_SUCCESS) { 
      globus_ftp_client_abort(&ftp_handle);
      ftp_completed.wait(c_res,-1);
      return false;
    };
    if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
      odlog(INFO)<<"check_ftp: timeout waiting for partial get"<<std::endl;
      globus_ftp_client_abort(&ftp_handle);
      ftp_completed.wait(c_res,-1);
      return false;
    };
    return (c_res==0);
  } else {
    // Do not use it at all. It does not give too much usefull 
    // information anyway. But request at least existence of file.
    if(!size_available) return false;
    return true;
  };
}

bool DataHandleFTP::remove(void) {
  if(!DataHandleCommon::remove()) return false;
  //char buf[32];
  globus_result_t res;
  res=globus_ftp_client_delete(&ftp_handle,c_url.c_str(),&ftp_opattr,
                         &ftp_complete_callback,this);
  if(res != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"delete_ftp: globus_ftp_client_delete failed"<<std::endl;
    odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl;
    return false;
  };
  int c_res;
  if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
    odlog(INFO)<<"delete_ftp: globus_ftp_client_delete timeout"<<std::endl;
    globus_ftp_client_abort(&ftp_handle);
    ftp_completed.wait(c_res,-1);
    return false;
  };
  return (c_res==0);
}

static bool remove_last_dir(std::string &dir) {
  // dir also contains proto and server
  std::string::size_type nn = std::string::npos;
  if(!strncasecmp(dir.c_str(),"ftp://",6)) { nn=dir.find('/',6); }
  else if(!strncasecmp(dir.c_str(),"gsiftp://",9)) { nn=dir.find('/',9); };
  if(nn == std::string::npos) return false;
  std::string::size_type n;
  if((n=dir.rfind('/')) == std::string::npos) return false;
  if(n < nn) return false;
  dir.resize(n);
  return true;
}

static bool add_last_dir(std::string &dir,std::string &path) {
  int l = dir.length();
  std::string::size_type n = path.find('/',l+1);
  if(n == std::string::npos) return false;
  dir=path; dir.resize(n);
  return true;
}

bool DataHandleFTP::mkdir_ftp() {
  ftp_dir_path=c_url;
  for(;;) if(!remove_last_dir(ftp_dir_path)) break;
  /*
  for(;;) {
    if(!remove_last_dir(ftp_dir_path)) {
      return false;
    };
    odlog(DEBUG)<<"mkdir_ftp: checking for "<<ftp_dir_path<<std::endl;
    globus_result_t res=globus_ftp_client_exists(&ftp_handle,ftp_dir_path.c_str(),&ftp_opattr,&ftp_complete_callback,this);
    if(res != GLOBUS_SUCCESS) {
      odlog(INFO)<<"Globus error: "<<GlobusResult(res)<<std::endl; return false;
    };
    int c_res;
    if(!ftp_completed.wait(c_res,300000)) {
      odlog(INFO)<<"mkdir_ftp: timeout waiting for exists"<<std::endl;
      // timeout - have to cancel operation here
      globus_ftp_client_abort(&ftp_handle);
      ftp_completed.wait(c_res,-1);
      return false;
    };
    if(c_res == RESULT_CREDENTIALS_EXPIRED) return false;
    if(c_res == RESULT_SUCCESS) break;
  };
  // have this object - go back with creating directories 
  */
  bool result = false;
  for(;;) {
    if(!add_last_dir(ftp_dir_path,c_url)) break;
    odlog(DEBUG)<<"mkdir_ftp: making "<<ftp_dir_path<<std::endl;
    globus_result_t res=globus_ftp_client_mkdir(&ftp_handle,ftp_dir_path.c_str(),&ftp_opattr,&ftp_complete_callback,this);
    if(res != GLOBUS_SUCCESS) {
      odlog(INFO)<<"Globus error: "<<GlobusResult(res)<<std::endl; return false;
    };
    int c_res;
    if(!ftp_completed.wait(c_res,300000)) {
      odlog(INFO)<<"mkdir_ftp: timeout waiting for mkdir"<<std::endl;
      /* timeout - have to cancel operation here */
      globus_ftp_client_abort(&ftp_handle);
      ftp_completed.wait(c_res,-1);
      return false;
    };
    if(c_res == RESULT_CREDENTIALS_EXPIRED) return false;
    // if(c_res != 0) return false;
    result = result || (c_res == 0);
  };
  return result;
  // return true;
}

bool DataHandleFTP::start_reading(DataBufferPar &buf) {
  if(!DataHandleCommon::start_reading(buf)) return false;
  buffer=&buf;
  /* size of file first */
  globus_off_t size = 0;
  bool limit_length = false;
  unsigned long long int range_length;
  if(range_end > range_start) {
    range_length = range_end - range_start;
    limit_length = true;
  };
  odlog(DEBUG)<<"start_reading_ftp"<<std::endl;
  ftp_completed.reset(); ftp_eof_flag=false;
  globus_ftp_client_handle_cache_url_state(&ftp_handle,c_url.c_str());
  globus_result_t res;
  int c_res;      
  if((!no_checks) && (!(url->meta_size_available()))) {
    odlog(DEBUG)<<"start_reading_ftp: size: url: "<<c_url.c_str()<<std::endl;
    res = globus_ftp_client_size(&ftp_handle,c_url.c_str(),
                     &ftp_opattr,&size,&ftp_complete_callback,this);
    if(res != GLOBUS_SUCCESS) {
      odlog(ERROR)<<"start_reading_ftp: failure"<<std::endl;
      odlog(INFO)<<"Globus error: "<<GlobusResult(res)<<std::endl;
      // globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
      // buffer->error_read(true);
      // DataHandleCommon::stop_reading();
      // return false;
    } else if(!ftp_completed.wait(c_res,300000)) {
      odlog(ERROR)<<"start_reading_ftp: timeout waiting for file size"<<std::endl;
      /* timeout - have to cancel operation here */
      odlog(INFO)<<"Timeout waiting for FTP file size - cancel transfer"<<std::endl;
      globus_ftp_client_abort(&ftp_handle);
      /* have to do something in addition if complete callback will be called */
      ftp_completed.wait(c_res,-1);
      // globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
      // buffer->error_read(true);
      // DataHandleCommon::stop_reading();
      // return false;
    } else if(c_res != 0) {
      odlog(INFO)<<"start_reading_ftp: failed to get file's size"<<std::endl;
      //buffer->error_read(true);
      // DataHandleCommon::stop_reading();
      //return false;
    } else {
      /* provide some metadata */
      odlog(INFO)<<"start_reading_ftp: obtained size: "<<size<<std::endl;
      url->meta_size(size);
    };
  };
  if((!no_checks) && (!(url->meta_created_available()))) {
    globus_abstime_t gl_modify_time;
    res = globus_ftp_client_modification_time(&ftp_handle,c_url.c_str(),
                     &ftp_opattr,&gl_modify_time,&ftp_complete_callback,this);
    if(res != GLOBUS_SUCCESS) {
      odlog(DEBUG)<<"start_reading_ftp: globus_ftp_client_modification_time failed"<<std::endl;
      odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl;
      //buffer->error_read(true);
      // DataHandleCommon::stop_reading();
      //return false;
    } else if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
      odlog(INFO)<<"start_reading_ftp: timeout waiting for modification_time"<<std::endl;
      globus_ftp_client_abort(&ftp_handle);
      ftp_completed.wait(c_res,-1);
      //globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
      //buffer->error_read(true);
      // DataHandleCommon::stop_reading();
      //return false;
    } if(c_res != 0) {
      odlog(INFO)<<"start_reading_ftp: failed to get file's modification time"<<std::endl;
      //buffer->error_read(true);
      // DataHandleCommon::stop_reading();
      //return false;
    } else {
      time_t modify_time;
      int modify_utime;
      GlobusTimeAbstimeGet(gl_modify_time,modify_time,modify_utime);
      odlog(DEBUG)<<"start_reading_ftp: creation time: "<<modify_time<<std::endl;
      url->meta_created(modify_time);
    };
    if(limit_length) {
      if(size < range_end) {
        if(size <= range_start) { // report eof immediately
          odlog(DEBUG)<<"start_reading_ftp: range is out of size"<<std::endl;
          buffer->eof_read(true);
          ftp_completed.signal(RESULT_SUCCESS);
          return true;
        };
        range_length=size-range_start;
      };
    };
  };
  odlog(DEBUG)<<"start_reading_ftp: globus_ftp_client_get"<<std::endl;
  if(limit_length) {
    res=globus_ftp_client_partial_get(&ftp_handle,c_url.c_str(),&ftp_opattr,
                         GLOBUS_NULL,range_start,range_start+range_length+1,&ftp_get_complete_callback,this);
  } else {
    res=globus_ftp_client_get(&ftp_handle,c_url.c_str(),&ftp_opattr,
                         GLOBUS_NULL,&ftp_get_complete_callback,this);
  };
  if(res != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"start_reading_ftp: globus_ftp_client_get failed"<<std::endl;
    odlog(INFO)<<"Globus error: "<<GlobusResult(res)<<std::endl;
    globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
    buffer->error_read(true);
    DataHandleCommon::stop_reading();
    return false;
  };
  if(globus_thread_create(&ftp_control_thread,GLOBUS_NULL,&ftp_read_thread,this) != 0) {
    odlog(DEBUG)<<"start_reading_ftp: globus_thread_create failed"<<std::endl;
    globus_ftp_client_abort(&ftp_handle);
    ftp_completed.wait(c_res,-1);
    globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
    buffer->error_read(true);
    DataHandleCommon::stop_reading();
    return false;
  };
  /* make sure globus has thread for handling network/callbacks */
  globus_thread_blocking_will_block();
  return true;
}

bool DataHandleFTP::stop_reading(void) {
  if(!DataHandleCommon::stop_reading()) return false;
  if(!buffer->eof_read()) {
    odlog(DEBUG)<<"stop_reading_ftp: aborting connection"<<std::endl;
    globus_ftp_client_abort(&ftp_handle);
  };
  int c_res;
  odlog(DEBUG)<<"stop_reading_ftp: waiting for transfer to finish"<<std::endl;
  ftp_completed.wait(c_res,-1); 
  odlog(DEBUG)<<"stop_reading_ftp: exiting: "<<c_url<<std::endl;
  globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
  return true;
}

void* DataHandleFTP::ftp_read_thread(void *arg) {
  DataHandleFTP* it = (DataHandleFTP*)arg;
  int h;
  unsigned int l;
  globus_result_t res;
  int registration_failed = 0;
  odlog(INFO)<<"ftp_read_thread: get and register buffers"<<std::endl;
  int n_buffers = 0;
  for(;;) {
    if(it->buffer->eof_read()) break;
    if(!it->buffer->for_read(h,l,true)) { /* eof or error */
      if(it->buffer->error()) { /* error -> abort reading */
        odlog(DEBUG)<<"ftp_read_thread: for_read failed - aborting: "<<it->c_url<<std::endl;
        globus_ftp_client_abort(&(it->ftp_handle));
      };
      break;
    };
    res=globus_ftp_client_register_read(&(it->ftp_handle),
              (globus_byte_t*)((*(it->buffer))[h]),l,
              &(it->ftp_read_callback),it);
    if(res != GLOBUS_SUCCESS) {
      odlog(VERBOSE)<<"ftp_read_thread: Globus error: "<<GlobusResult(res)<<std::endl;
      globus_object_t* err = globus_error_get(res);
      registration_failed++;
      if(registration_failed>=10) {
        it->buffer->is_read(h,0,0);
        it->buffer->error_read(true);
        it->buffer->eof_read(true); // can set eof here because no callback will be called (I guess).
        odlog(VERBOSE)<<"ftp_read_thread: too many registration failures - abort: "<<it->c_url<<std::endl;
      } else {
        odlog(VERBOSE)<<"ftp_read_thread: failed to register globus buffer - will try later: "<<it->c_url<<std::endl;
        it->buffer->is_read(h,0,0); sleep(1);
      };
    }
    else { n_buffers++; };
  };
  /* make sure complete callback is called */
  odlog(DEBUG)<<"ftp_read_thread: waiting for eof"<<std::endl;
  it->buffer->wait_eof_read();
  odlog(DEBUG)<<"ftp_read_thread: exiting"<<std::endl;
  it->ftp_completed.signal(it->buffer->error_read()?1:0);
  return NULL;
}

void DataHandleFTP::ftp_read_callback(void* arg,globus_ftp_client_handle_t* handle,globus_object_t* error,globus_byte_t* buffer,globus_size_t length,globus_off_t offset,globus_bool_t eof) {
  DataHandleFTP* it = (DataHandleFTP*)arg;
  if(error != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"ftp_read_callback: failure"<<std::endl;
    it->buffer->is_read((char*)buffer,0,0);
    return;
  };
  odlog(VERBOSE)<<"ftp_read_callback: success"<<std::endl;
  it->buffer->is_read((char*)buffer,length,offset);
  if(eof) it->ftp_eof_flag=true;
  return;
}

void DataHandleFTP::ftp_get_complete_callback(void* arg,globus_ftp_client_handle_t* handle,globus_object_t* error) {
  odlog(DEBUG)<<"ftp_get_complete_callback"<<std::endl;
  DataHandleFTP* it = (DataHandleFTP*)arg;
  /* data transfer finished */
  if(error != GLOBUS_SUCCESS) {
    odlog(INFO)<<"Failed to get ftp file."<<std::endl;
    globus_object_to_string(error,it->failure_description);
    odlog(DEBUG)<<"Globus error: "<<it->failure_description<<std::endl;
    if(it->is_secure) {
      it->check_credentials();
    };
    it->buffer->error_read(true);
    return;
  };
  it->buffer->eof_read(true);
  return;
}

bool DataHandleFTP::start_writing(DataBufferPar &buf,DataCallback *space_cb) {
  if(!DataHandleCommon::start_writing(buf)) return false;
  buffer=&buf;
  /* size of file first */
  //globus_off_t size;
  bool limit_length = false;
  unsigned long long int range_length;
  if(range_end > range_start) {
    range_length = range_end - range_start;
    limit_length = true;
  };
  odlog(DEBUG)<<"start_writing_ftp"<<std::endl;
  ftp_completed.reset(); ftp_eof_flag=false; ftp_counter.reset();
  globus_result_t res;
  //int c_res;
/* !!!! TODO - preallocate here */

  globus_ftp_client_handle_cache_url_state(&ftp_handle,c_url.c_str());
  if(!no_checks) {
    odlog(DEBUG)<<"start_writing_ftp: mkdir"<<std::endl;
    if(!DataHandleFTP::mkdir_ftp()) {
      odlog(DEBUG)<<"start_writing_ftp: mkdir failed - still trying to write"<<std::endl;
    };
  };
  odlog(DEBUG)<<"start_writing_ftp: put"<<std::endl;
  if(limit_length) {
    res=globus_ftp_client_partial_put(&ftp_handle,c_url.c_str(),&ftp_opattr,
                          GLOBUS_NULL,range_start,range_start+range_length,&ftp_put_complete_callback,this);

  } else {
    res=globus_ftp_client_put(&ftp_handle,c_url.c_str(),&ftp_opattr,
                         GLOBUS_NULL,&ftp_put_complete_callback,this);
  };
  if(res != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"start_writing_ftp: put failed"<<std::endl;
    GlobusResult(res).get(failure_description);
    odlog(INFO)<<"Globus error: "<<failure_description<<std::endl;
    globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
    buffer->error_write(true);
    DataHandleCommon::stop_writing();
    return false;
  };
  if(globus_thread_create(&ftp_control_thread,GLOBUS_NULL,&ftp_write_thread,this) != 0) {
    odlog(DEBUG)<<"start_writing_ftp: globus_thread_create failed"<<std::endl;
    globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
    buffer->error_write(true);
    DataHandleCommon::stop_writing();
    return false;
  };
  /* make sure globus has thread for handling network/callbacks */
  globus_thread_blocking_will_block();
  return true;
}

bool DataHandleFTP::stop_writing(void) {
  if(!DataHandleCommon::stop_writing()) return false;
  if(!buffer->eof_write()) {
    globus_ftp_client_abort(&ftp_handle);
  };
  int c_res;
  ftp_completed.wait(c_res,-1);
  globus_ftp_client_handle_flush_url_state(&ftp_handle,c_url.c_str());
  return true;
}

void* DataHandleFTP::ftp_write_thread(void *arg) {
  DataHandleFTP* it = (DataHandleFTP*)arg;
  int h;
  unsigned int l;
  unsigned long long int o;
  globus_result_t res;
  globus_bool_t eof = GLOBUS_FALSE;
  odlog(INFO)<<"ftp_write_thread: get and register buffers"<<std::endl;
  //int n_buffers = 0;
  for(;;) {
    if(!it->buffer->for_write(h,l,o,true)) {
      if(it->buffer->error()) {
        odlog(DEBUG)<<"ftp_write_thread: for_write failed - aborting"<<std::endl;
        globus_ftp_client_abort(&(it->ftp_handle));
        break;
      };
      // no buffers and no errors - must be pure eof
      eof = GLOBUS_TRUE;
      char dummy;
      o = it->buffer->eof_position();
      res=globus_ftp_client_register_write(&(it->ftp_handle),
                   (globus_byte_t*)(&dummy),0
                   ,o,eof,&ftp_write_callback,it);
      break;
      // if(res == GLOBUS_SUCCESS) break;
      // sleep(1); continue;
    };
    res=globus_ftp_client_register_write(&(it->ftp_handle),
                   (globus_byte_t*)((*(it->buffer))[h]),l
                   ,o,eof,&ftp_write_callback,it);
    if(res != GLOBUS_SUCCESS) { 
      it->buffer->is_notwritten(h); sleep(1);
    };
  };
  /* make sure complete callback is called */
  it->buffer->wait_eof_write();
  it->ftp_completed.signal(it->buffer->error_write()?1:0);
  return NULL;
}

void DataHandleFTP::ftp_write_callback(void* arg,globus_ftp_client_handle_t* handle,globus_object_t* error,globus_byte_t* buffer,globus_size_t length,globus_off_t offset,globus_bool_t eof) {
  DataHandleFTP* it = (DataHandleFTP*)arg;
  // if(it->ftp_eof_flag) return; /* eof callback */
  if(error != GLOBUS_SUCCESS) {
    odlog(DEBUG)<<"ftp_write_callback: failure"<<std::endl;
    it->buffer->is_written((char*)buffer);
    return;
  };
  odlog(VERBOSE)<<"ftp_write_callback: success"<<std::endl;
  it->buffer->is_written((char*)buffer);
  return;
}

void DataHandleFTP::ftp_put_complete_callback(void* arg,globus_ftp_client_handle_t* handle,globus_object_t* error) {
  odlog(DEBUG)<<"ftp_put_complete_callback"<<std::endl;
  DataHandleFTP* it = (DataHandleFTP*)arg;
  /* data transfer finished */
  if(error != GLOBUS_SUCCESS) {
    odlog(INFO)<<"Failed to store ftp file."<<std::endl;
    globus_object_to_string(error,it->failure_description);
    odlog(DEBUG)<<"Globus error: "<<it->failure_description<<std::endl;
    if(it->is_secure) {
      it->check_credentials();
    };
    it->buffer->error_write(true);
    return;
  };
  it->buffer->eof_write(true);
  return;
}

bool DataHandleFTP::list_files(std::list<DataPoint::FileInfo> &files,bool resolve) {
  if(!DataHandleCommon::list_files(files,resolve)) return false;
  Lister lister; 
  if(lister.retrieve_dir(c_url) != 0) {
    odlog(ERROR)<<"Failed to obtain listing from ftp: "<<c_url<<std::endl;
    return false;
  };
  lister.close_connection();
  std::string base_url = c_url;
  {
    std::string::size_type n = base_url.find("://");
    if(n == std::string::npos) { n=0; } else { n+=3; };
    n=base_url.find('/',n);
    if(n != std::string::npos) base_url.resize(n);
  };
  bool result = true;
  for(std::list<ListerFile>::iterator i = lister.begin();i!=lister.end();++i) {
    std::list<DataPoint::FileInfo>::iterator f = 
           files.insert(files.end(),DataPoint::FileInfo(i->GetLastName()));
    if(resolve) {
      globus_result_t res;
      globus_off_t size = 0;
      globus_abstime_t gl_modify_time;
      time_t modify_time;
      int modify_utime;
      int c_res;
      // Lister should always return full path to file
      std::string f_url = base_url+i->GetName();
      f->type=(DataPoint::FileInfo::Type)(i->GetType());
      if(i->CheckSize()) {
        f->size=i->GetSize(); f->size_available=true;
      } else if(i->GetType() != ListerFile::file_type_dir) {
        odlog(VERBOSE)<<"list_files_ftp: looking for size of "<<f_url<<std::endl;
        res = globus_ftp_client_size(&ftp_handle,f_url.c_str(),&ftp_opattr,
                       &size,&ftp_complete_callback,this);
        if(res != GLOBUS_SUCCESS) {
          odlog(DEBUG)<<"list_files_ftp: globus_ftp_client_size failed"<<std::endl;
          odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl; result=false;
        } else if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
          odlog(INFO)<<"list_files_ftp: timeout waiting for size"<<std::endl;
          globus_ftp_client_abort(&ftp_handle);
          ftp_completed.wait(c_res,-1); result=false;
        } else if(c_res != 0) {
          odlog(INFO)<<"list_files_ftp: failed to get file's size"<<std::endl;
          result=false;
          // Guessing - directories usually have no size
          f->type=DataPoint::FileInfo::file_type_dir;
        } else {
          f->size=size; f->size_available=true;
          // Guessing - only files usually have no size
          f->type=DataPoint::FileInfo::file_type_file;
        };
      };
      if(i->CheckCreated()) {
        f->created=i->GetCreated(); f->created_available=true;
      } else {
        odlog(VERBOSE)<<"list_files_ftp: looking for modification time of "<<f_url<<std::endl;
        res = globus_ftp_client_modification_time(&ftp_handle,f_url.c_str(),
                     &ftp_opattr,&gl_modify_time,&ftp_complete_callback,this);
        if(res != GLOBUS_SUCCESS) {
          odlog(DEBUG)<<"list_files_ftp: globus_ftp_client_modification_time failed"<<std::endl;
          odlog(INFO)<<"Globus error"<<GlobusResult(res)<<std::endl; result=false;
        } else if(!ftp_completed.wait(c_res,300000)) { /* 5 minutes timeout */
          odlog(INFO)<<"list_files_ftp: timeout waiting for modification_time"<<std::endl;
          globus_ftp_client_abort(&ftp_handle);
          ftp_completed.wait(c_res,-1); result=false;
        } else if(c_res != 0) {
          odlog(INFO)<<"list_files_ftp: failed to get file's modification time"<<std::endl;
          result=false;
        } else {
          GlobusTimeAbstimeGet(gl_modify_time,modify_time,modify_utime);
          f->created=modify_time; f->created_available=true;
        };
      };
    };
  };
  return result;
}

DataHandle* DataHandleFTP::CreateInstance(DataPoint* url_) {
  if((!url_) || (!*url_)) return NULL;
  const char* cur_url = url_->current_location();
  if(strncasecmp("ftp://",cur_url,6) &&
     strncasecmp("gsiftp://",cur_url,9)) return NULL;
  return new DataHandleFTP(url_);
}

DataHandleFTP::DataHandleFTP(DataPoint* url_):DataHandleCommon(url_) {
  ftp_active=false;
  if(!ftp_mod.active()) {
    odlog(ERROR)<<"GLOBUS_FTP_CLIENT_MODULE activation failed"<<std::endl;
    url=NULL;
  };
}

DataHandleFTP::~DataHandleFTP(void) {
  stop_reading();
  stop_writing();
  deinit_handle();
}

bool DataHandleFTP::init_handle(void) {
  if(!DataHandleCommon::init_handle()) return false;
  const char* cur_url = url->current_location();
  const std::string cur_url_s(cur_url);
  std::string value;
  if((!strncasecmp("ftp://",cur_url,6)) ||
     (!strncasecmp("gsiftp://",cur_url,9))) {
    is_secure=false;
    if(!strncasecmp("gsiftp://",cur_url,9)) is_secure=true;
    if(!ftp_active) {
      globus_result_t res;
      if((res=globus_ftp_client_handle_init(
                           &ftp_handle,GLOBUS_NULL)) != GLOBUS_SUCCESS) {
        odlog(ERROR)<<"init_handle: globus_ftp_client_handle_init failed"<<std::endl;
        odlog(ERROR)<<"Globus error: "<<GlobusResult(res)<<std::endl;
        ftp_active=false; return false;
      };
      if((res=globus_ftp_client_operationattr_init(
                           &ftp_opattr)) != GLOBUS_SUCCESS) {
        odlog(ERROR)<<"init_handle: globus_ftp_client_operationattr_init failed"<<std::endl;
        odlog(ERROR)<<"Globus error: "<<GlobusResult(res)<<std::endl;
        globus_ftp_client_handle_destroy(&ftp_handle);
        ftp_active=false; return false;
      };
    };
    ftp_active=true; ftp_threads=1;
    if(allow_out_of_order) {
      if(get_url_option(cur_url_s,"threads",0,value) == 0) {
        if(stringtoint(value,ftp_threads)) {
          if(ftp_threads < 1) ftp_threads = 1;
          if(ftp_threads > MAX_PARALLEL_STREAMS) ftp_threads = MAX_PARALLEL_STREAMS;
        }
        else { ftp_threads = 1; };
      };
    };
    globus_ftp_control_parallelism_t paral;
    if(ftp_threads > 1) {
      paral.fixed.mode=GLOBUS_FTP_CONTROL_PARALLELISM_FIXED;
      paral.fixed.size=ftp_threads;
    }
    else {
      paral.fixed.mode=GLOBUS_FTP_CONTROL_PARALLELISM_NONE;
      paral.fixed.size=1;
    };
    globus_ftp_client_operationattr_set_parallelism(&ftp_opattr,&paral);
    globus_ftp_client_operationattr_set_striped(&ftp_opattr,GLOBUS_FALSE);
    /*   globus_ftp_client_operationattr_set_layout         */
    /*   globus_ftp_client_operationattr_set_tcp_buffer     */
    globus_ftp_client_operationattr_set_type(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_TYPE_IMAGE);
    if(!is_secure) { // plain ftp protocol
      globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_STREAM);
      globus_ftp_client_operationattr_set_data_protection(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_PROTECTION_CLEAR);
      globus_ftp_client_operationattr_set_control_protection(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_PROTECTION_CLEAR);
      // need to set dcau to none in order Globus libraries not to send
      // it to pure ftp server
      globus_ftp_control_dcau_t dcau;
      dcau.mode=GLOBUS_FTP_CONTROL_DCAU_NONE;
      globus_ftp_client_operationattr_set_dcau(&ftp_opattr,&dcau);
    }
    else { // gridftp protocol

      char* subj = getenv("SUBJECT");
      if(subj) {
        globus_ftp_client_operationattr_set_authorization(&ftp_opattr,
          GSS_C_NO_CREDENTIAL,NULL,NULL,NULL,subj);
      };
      if(get_url_option(cur_url_s,"secure",0,value) == 0) {
        if(value == "yes") {
          globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_EXTENDED_BLOCK);
          globus_ftp_client_operationattr_set_data_protection(&ftp_opattr,
                                      GLOBUS_FTP_CONTROL_PROTECTION_PRIVATE);
          odlog(DEBUG)<<"Using secure data transfer (requested in url)"<<std::endl;
        } else {
          if(force_passive) {
            globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_STREAM);
          } else {
            globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_EXTENDED_BLOCK);
          };
          globus_ftp_client_operationattr_set_data_protection(&ftp_opattr,
                                      GLOBUS_FTP_CONTROL_PROTECTION_CLEAR);
          odlog(DEBUG)<<"Using insecure data transfer (requested in url)"<<std::endl;
        };
      } else {
        if(force_secure) {
          globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_EXTENDED_BLOCK);
          globus_ftp_client_operationattr_set_data_protection(&ftp_opattr,
                                      GLOBUS_FTP_CONTROL_PROTECTION_PRIVATE);
        } else {
          if(force_passive) {
            globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_STREAM);
          } else {
            globus_ftp_client_operationattr_set_mode(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_MODE_EXTENDED_BLOCK);
          };
          globus_ftp_client_operationattr_set_data_protection(&ftp_opattr,
                                      GLOBUS_FTP_CONTROL_PROTECTION_CLEAR);
          odlog(DEBUG)<<"Using insecure data transfer"<<std::endl;
        };
      };
      globus_ftp_client_operationattr_set_control_protection(&ftp_opattr,
                                    GLOBUS_FTP_CONTROL_PROTECTION_PRIVATE);
    };
    /*   globus_ftp_client_operationattr_set_dcau                         */
    /*   globus_ftp_client_operationattr_set_resume_third_party_transfer  */
    /*   globus_ftp_client_operationattr_set_authorization                */
    globus_ftp_client_operationattr_set_append(&ftp_opattr,GLOBUS_FALSE);
    return true;
  };
  return false;
}

bool DataHandleFTP::deinit_handle(void) {
  if(!DataHandleCommon::deinit_handle()) return false;
  if(ftp_active) {
    odlog(DEBUG)<<"DataHandle::deinit_handle: destroy ftp_handle"<<std::endl;
    globus_ftp_client_handle_destroy(&ftp_handle);
    globus_ftp_client_operationattr_destroy(&ftp_opattr);
  };
  return true;
}

bool DataHandleFTP::analyze(analyze_t &arg) {
  return DataHandleCommon::analyze(arg);
}

bool DataHandleFTP::out_of_order(void) {
  return true;
}
