#include "../std.h"

#include "../misc/url_options.h"
#include "../misc/stringtoint.h"
#include "../misc/globus_error_utils.h"
#include "../misc/globus_modules.h"
#include "../misc/log_time.h"
#include "datapoint.h"
#include "datahandle.h"
#include "datahandle_file.h"
#include "datahandle_ftp.h"
#include "datahandle_httpg.h"
#include "datahandle_srm.h"

// --------------------------------------------------------------------

std::list<DataHandle::constructor_t> DataHandle::protocols;
LockSimple DataHandle::protocols_lock;

class DataHandleRegistrator {
 public:
  DataHandleRegistrator(void) {
    DataHandle::AddProtocol(&(DataHandleFile::CreateInstance));
    DataHandle::AddProtocol(&(DataHandleFTP::CreateInstance));
    DataHandle::AddProtocol(&(DataHandleHTTPg::CreateInstance));
    DataHandle::AddProtocol(&(DataHandleSRM::CreateInstance));
  };
};

static DataHandleRegistrator registrator;

bool DataHandle::AddProtocol(constructor_t constructor) {
  protocols_lock.block();
  protocols.push_back(constructor);
  protocols_lock.unblock();
  return true;
}

DataHandle* DataHandle::CreateInstance(DataPoint* url_) {
  if((!url_) || (!*url_)) return NULL;
  DataHandle* handle = NULL;
  protocols_lock.block();
  for(std::list<constructor_t>::const_iterator i = protocols.begin();
                                          i!=protocols.end();++i) {
    handle = (*(*i))(url_);
    if(handle) {
      break;
      //if(*handle) break;
      //delete handle; handle=NULL;
    };
  };
  protocols_lock.unblock();
  return handle;
}

// --------------------------------------------------------------------

DataHandle::DataHandle(DataPoint* url_):instance(NULL) {
  instance=CreateInstance(url_);
}

bool DataHandle::analyze(analyze_t &arg) {
  if(!instance) return false; return instance->analyze(arg);
}

bool DataHandle::deinit_handle(void) {
  if(!instance) return false; return instance->deinit_handle();
}

bool DataHandle::init_handle(void) {
  if(!instance) return false; return instance->init_handle();
}

DataHandle::~DataHandle(void) {
  if(instance) delete instance;
}

bool DataHandle::start_reading(DataBufferPar &buffer) {
  if(instance) return instance->start_reading(buffer);
  odlog(DEBUG)<<"DataHandle::start_reading: unknown protocol"<<std::endl;
  return false;
}

bool DataHandle::stop_reading(void) {
  if(instance) return instance->stop_reading();
  return false;
}

bool DataHandle::start_writing(DataBufferPar &buffer,DataCallback *space_cb) {
  if(instance) return instance->start_writing(buffer);
  odlog(DEBUG)<<"DataHandle::start_writing: unknown protocol"<<std::endl;
  return false;
}

bool DataHandle::stop_writing(void) {
  if(instance) return instance->stop_writing();
  return false;
}

bool DataHandle::check(void) {
  if(instance) return instance->check();
  odlog(DEBUG)<<"DataHandle::check: unknown protocol"<<std::endl;
  return false;
}

bool DataHandle::remove(void) {
  if(instance) return instance->remove();
  odlog(DEBUG)<<"DataHandle::remove: unknown protocol"<<std::endl;
  return false;
}

bool DataHandle::list_files(std::list<DataPoint::FileInfo> &files,bool resolve) {
  if(instance) return instance->list_files(files,resolve);
  odlog(DEBUG)<<"DataHandle::list_files: unknown protocol"<<std::endl;
  return false;
}

bool DataHandle::out_of_order(void) {
  if(instance) return instance->out_of_order(); return false;
}

void DataHandle::out_of_order(bool val) {
  if(instance) instance->out_of_order(val);
}

bool DataHandle::secure(void) {
  if(instance) return instance->secure(); return false;
}

void DataHandle::secure(bool val) {
  if(instance) instance->secure(val);
}

void DataHandle::passive(bool val) {
  if(instance) instance->passive(val);
}

void DataHandle::additional_checks(bool val) {
  if(instance) instance->additional_checks(val);
}

bool DataHandle::additional_checks(void) {
  if(instance) return instance->additional_checks(); return false;
}

DataHandle::failure_reason_t DataHandle::failure_reason(void) { 
  if(instance) return instance->failure_reason();
  return common_failure;
}

std::string DataHandle::failure_text(void) { 
  if(instance) return instance->failure_text();
  return "Protocol not supported";
}

void DataHandle::range(unsigned long long int start,unsigned long long end) {
  if(instance) instance->range(start,end);
}

// --------------------------------------------------------------------

DataHandleCommon::DataHandleCommon(DataPoint* url_):DataHandle(),url(url_) {
  /* connection is not initialised here - only if request is made */
  no_checks=false;
  reading=false;
  writing=false;
  is_secure=false; force_secure=true; force_passive=false;
  allow_out_of_order=false; // safe setting
  failure_code=common_failure;
  range_start=0;
  range_end=0;
}

DataHandleCommon::~DataHandleCommon(void) {
  /*  Stop any transfer - use this as an example in derived methods */
  stop_reading();
  stop_writing();
  /* !!!!!!!!!!! destroy activated handles, etc here !!!!!!!! */
  deinit_handle();
}

bool DataHandleCommon::analyze(analyze_t &arg) {
  if(!url) return false;
  const std::string cur_url_s(url->current_location());
  std::string value;
  unsigned int threads;
  if(get_url_option(cur_url_s,"threads",0,value) == 0) {
    if(stringtoint(value,threads)) {
      if(threads < 1) threads = 1;
      if(threads > MAX_PARALLEL_STREAMS) threads = MAX_PARALLEL_STREAMS;
    }
    else { threads = 1; };
    arg.bufnum=threads;
  };
  if(get_url_option(cur_url_s,"blocksize",0,value) == 0) {
    int n;
    if(stringtoint(value,n)) {
      if(n<0) n=0;
      if(n > MAX_BLOCK_SIZE) n = MAX_BLOCK_SIZE;
      arg.bufsize=n;
    };
  };
  arg.cache=true;
  if(get_url_option(cur_url_s,"cache",0,value) == 0) {
    if(strcasecmp(value.c_str(),"no") == 0) arg.cache=false;
  };
  arg.readonly=true;
  if(get_url_option(cur_url_s,"readonly",0,value) == 0) {
    if(strcasecmp(value.c_str(),"no") == 0) arg.readonly=false;
  };
  arg.local=false;
  return true;
}

bool DataHandleCommon::init_handle(void) {
  if(!url) return false;
  const char* cur_url = url->current_location();
  const std::string cur_url_s(cur_url);
  std::string value;
  cacheable=true;
  linkable=true;
  if(get_url_option(cur_url_s,"cache",0,value) == 0) {
    if(strcasecmp("no",value.c_str()) == 0) cacheable=false;
  };
  if(get_url_option(cur_url_s,"readonly",0,value) == 0) {
    if(strcasecmp("no",value.c_str()) == 0) linkable=false;
  };
  out_of_order(out_of_order());
  transfer_streams=1;
  if(allow_out_of_order) {
    if(get_url_option(cur_url_s,"threads",0,value) == 0) {
      if(stringtoint(value,transfer_streams)) {
        if(transfer_streams < 1) transfer_streams = 1;
        if(transfer_streams > MAX_PARALLEL_STREAMS)
          transfer_streams = MAX_PARALLEL_STREAMS;
      }
      else { transfer_streams=1; };
    };
  };
  c_url=cur_url;
  // Temporary workaround
  if(strcmp("-",cur_url)) if(::canonic_url(c_url) != 0) return false;
  return true;
}

bool DataHandleCommon::deinit_handle(void) {
  return true;
}

bool DataHandleCommon::start_reading(DataBufferPar &buffer) {
  failure_code=common_failure;
  failure_description="";
  if(reading || writing || (url==NULL)) return false;
  if(!init_handle()) return false;
  reading=true;
  return true;
}

bool DataHandleCommon::stop_reading(void) {
  if(!reading) return false;
  reading=false;
  return true;
}

bool DataHandleCommon::start_writing(DataBufferPar &buffer,DataCallback *space_cb) {
  failure_code=common_failure;
  failure_description="";
  if(reading || writing || (url==NULL)) return false;
  if(!init_handle()) return false;
  writing=true;
  return true;
}

bool DataHandleCommon::stop_writing(void) {
  if(!writing) return false;
  writing=false;
  return true;
} 

bool DataHandleCommon::list_files(std::list<DataPoint::FileInfo> &files,bool resolve) {
  failure_code=common_failure;
  failure_description="";
  if(reading || writing || (url==NULL)) return false;
  if(!init_handle()) return false;
  return true;
}

bool DataHandleCommon::check(void) {
  failure_code=common_failure;
  failure_description="";
  if(reading || writing || (url==NULL)) return false;
  if(!init_handle()) return false;
  return true;
}

bool DataHandleCommon::remove(void) {
  failure_code=common_failure;
  if(reading || writing || (url==NULL)) return false;
  if(!init_handle()) return false;
  return true;
}

bool DataHandleCommon::out_of_order(void) {
  return false;
}

void DataHandleCommon::additional_checks(bool val) { no_checks=!val; }

bool DataHandleCommon::additional_checks(void) { return !no_checks; };

void DataHandleCommon::passive(bool val) { force_passive=val; }

void DataHandleCommon::secure(bool val) { force_secure=val; }

bool DataHandleCommon::secure(void) { return is_secure; }

void DataHandleCommon::out_of_order(bool val) { allow_out_of_order=val; }

DataHandle::failure_reason_t DataHandleCommon::failure_reason(void) { 
  return failure_code;
}

std::string DataHandleCommon::failure_text(void) { 
  return failure_description;
}

void DataHandleCommon::range(unsigned long long int start,unsigned long long end) {
  range_start=start; range_end=end;
}

DataHandle* DataHandleCommon::CreateInstance(DataPoint* url_) {
  return NULL;
}

// --------------------------------------------------------------------

