#include "../std.h"
#include "../misc/mkdir_recursive.h"
#include "../misc/stringtoint.h"
#include "../misc/inttostring.h"
#include "../misc/checkfile.h"
#include "../misc/log_time.h"
#include "datacache.h"

DataCache::DataCache(void) {
  cache_path="";
  cache_data_path="";
  have_url=false;
  cache_file="";
  cache_uid=0; cache_gid=0;
  id="";
}

DataCache::DataCache(const DataCache &cache) {
  odlog(VERBOSE)<<"DataCache: constructor with copy"<<std::endl;
  have_url=false;
  if(cache.cache_path.length() == 0) { cache_path=""; return; };
  cache_path=cache.cache_path;
  cache_data_path=cache.cache_data_path;
  cache_link_path=cache.cache_link_path;
  cache_uid=cache.cache_uid;
  cache_gid=cache.cache_gid;
  id=cache.id;
  if(cache.have_url) {
    odlog(VERBOSE)<<"DataCache: constructor with copy: calling start"<<std::endl;
    bool available;
    start(cache.cache_url.c_str(),available);
  };
}

DataCache::DataCache(const char* cache_path_,const char* cache_data_path_,const char* cache_link_path_,const char* id_,uid_t cache_uid_,gid_t cache_gid_) {
  cache_path="";
  if(cache_path_) { 
    cache_path=cache_path_;
    if(cache_data_path_) { cache_data_path=cache_data_path_; }
    else { cache_data_path=cache_path; };
    if(cache_link_path_) { cache_link_path=cache_link_path_; }
    else { cache_link_path=cache_data_path; };
  };
  have_url=false;
  cache_uid=cache_uid_;
  cache_gid=cache_gid_;
  if(id_) id=id_;
  cache_file="";
}

DataCache::~DataCache(void) {
  //if(have_url) stop(true,false);
  if(have_url) stop(file_download_failed);
}

bool DataCache::start(const char* base_url,bool &available) {
  if(have_url) return false;
  available=false; cache_file="";
  std::string url_options="";
  /* Find or create new url in cache. This also claims url */
  std::string fname;
  if(cache_find_url(cache_path.c_str(),cache_data_path.c_str(),cache_uid,cache_gid,base_url,id,url_options,fname) != 0) {
    return false;
  };
  cache_url=base_url;
  /* options contain 2 numbers: creation and expiration timestamps */
  have_creation_time=false;
  have_expiration_time=false;
  if(url_options.length() != 0) {
    std::string::size_type n = url_options.find(' ');
    if(n==std::string::npos) n=url_options.length();
    std::string s = url_options.substr(0,n);
    if(s != ".") {
      if(stringtoint(s,creation_time)) have_creation_time=true;
    };
    s = url_options.substr(n+1);
    if((s.length() != 0) && (s != ".")) {
      if(stringtoint(s,expiration_time)) have_expiration_time=true;
    };
    if((have_creation_time) && (!have_expiration_time)) {
      expiration_time=creation_time+24*60*60;
      have_expiration_time=true;
    };
    if(!have_expiration_time) {
      expiration_time=time(NULL)+24*60*60;
    };
  };
  switch(cache_download_file_start(cache_path.c_str(),cache_data_path.c_str(),cache_uid,cache_gid,fname.c_str(),id,cdh)) {
    case 1: { // error
      odlog(ERROR)<<"Error while locking file in cache"<<std::endl;
      cache_release_file(cache_path.c_str(),cache_data_path.c_str(),
                              cache_uid,cache_gid,fname.c_str(),id,true);
      return false;
    };
    case 2: {  // already have it
      // reread options because they could be changed by process which
      // just downloded that file
      url_options="";
      std::string u;
      if(cache_find_file(cache_path.c_str(),cache_data_path.c_str(),cache_uid,cache_gid,fname.c_str(),u,url_options) == 0) {
        have_creation_time=false;
        have_expiration_time=false;
        if(url_options.length() != 0) {
          std::string::size_type n = url_options.find(' ');
          if(n==std::string::npos) n=url_options.length();
          std::string s = url_options.substr(0,n);
          if(s != ".") {
            if(stringtoint(s,creation_time)) have_creation_time=true;
          };
          s = url_options.substr(n+1);
          if((s.length() != 0) && (s != ".")) {
            if(stringtoint(s,expiration_time)) have_expiration_time=true;
          };
          if((have_creation_time) && (!have_expiration_time)) {
            expiration_time=creation_time+24*60*60;
            have_expiration_time=true;
          };
          if(!have_expiration_time) {
            expiration_time=time(NULL)+24*60*60;
          };
        };
      };
      available=true;
    }; break;
    case 0: {  // have to download
      available=false;
    }; break;
    default: {
      odlog(ERROR)<<"Unknown error while locking file in cache"<<std::endl;
      cache_release_file(cache_path.c_str(),cache_data_path.c_str(),
                              cache_uid,cache_gid,fname.c_str(),id,true);
      return false;
    }; 
  };
  cache_file=cdh.name();
  have_url=true;
  return true;
}

//bool DataCache::stop(bool failure,bool invalidate) {
bool DataCache::stop(int file_state) {
  if(!have_url) return false; /* this object was not initialized */
  const char* url_opt = NULL;
  std::string url_options;
  //if((!failure) && (!invalidate)) {  /* compose url with options */
  if(!(file_state & (file_download_failed | file_not_valid))) {
    /* compose url with options */
    url_options=cache_url+"\n"+
        (have_creation_time?inttostring(creation_time):".")+" "+
        (have_expiration_time?inttostring(expiration_time):".");
    url_opt=url_options.c_str();
  };
  cache_download_url_end(cache_path.c_str(),cache_data_path.c_str(),
        cache_uid,cache_gid,url_opt,cdh,!(file_state & file_download_failed));
  //                   cache_uid,cache_gid,url_opt,cdh,!failure);
  //if(invalidate) {  
  if(file_state & file_not_valid) {  
    cache_invalidate_url(cache_path.c_str(),cache_data_path.c_str(),cache_uid,cache_gid,cdh.cache_name().c_str());
  };
  //if(failure || invalidate) {  /* failed to download or outdated file */
  if((file_state & (file_download_failed | file_not_valid)) &&
     !(file_state & file_keep)) {  /* failed to download or outdated file */
    cache_release_file(cache_path.c_str(),cache_data_path.c_str(),
                       cache_uid,cache_gid,cdh.cache_name().c_str(),id,true);
  };
  have_url=false; cache_file="";
  return true;
}

bool DataCache::clean(unsigned long long int size) {
  odlog(INFO)<<"Cache cleaning requested: "<<cache_path<<", "<<size<<" bytes"<<std::endl;
  unsigned long long int freed=cache_clean(cache_path.c_str(),cache_data_path.c_str(),cache_uid,cache_gid,size);
  odlog(DEBUG)<<"Cache cleaned: "<<cache_path<<", "<<freed<<" bytes"<<std::endl;
  if(freed < size) return false;
  return true;
}

bool DataCache::link(const char* link_path) {
  uid_t uid = get_user_id();
  gid_t gid = get_user_group(uid);
  return link(link_path,uid,gid);
}

bool DataCache::copy(const char* link_path) {
  uid_t uid = get_user_id();
  gid_t gid = get_user_group(uid);
  return copy(link_path,uid,gid);
}

bool DataCache::link(const char* link_path,uid_t uid,gid_t gid) {
  {
    std::string dirpath = link_path;
    std::string::size_type n = dirpath.rfind('/');
    if(n == std::string::npos) {
      dirpath="/";
    }
    else {
      dirpath.erase(n,dirpath.length()-n+1);
    };
    if(mkdir_recursive(NULL,dirpath.c_str(),S_IRWXU,uid,gid) != 0) {
      if(errno != EEXIST) {
        odlog(ERROR)<<"Failed to create/find directory "<<dirpath<<std::endl;
        return false;
      };
    };
  };
  if(cache_link_path==".") {  /* special link - copy request */
    return copy_file(link_path,uid,gid);
  };
  return link_file(link_path,uid,gid);
}

bool DataCache::link_file(const char* link_path,uid_t uid,gid_t gid) {
  std::string fname(cache_file.c_str()+cache_data_path.length());
  fname=cache_link_path+fname;
  if(symlink(fname.c_str(),link_path) == -1) {
    perror("symlink");
    odlog(ERROR)<<"Failed to make symbolic link "<<link_path<<" to "<<fname<<std::endl;
    return false;
  };
  lchown(link_path,uid,gid);
  return true;
}

bool DataCache::copy(const char* link_path,uid_t uid,gid_t gid) {
  {
    std::string dirpath = link_path;
    std::string::size_type n = dirpath.rfind('/');
    if(n == std::string::npos) {
      dirpath="/";
    }
    else {
      dirpath.erase(n,dirpath.length()-n+1);
    };
    if(mkdir_recursive(NULL,dirpath.c_str(),S_IRWXU,uid,gid) != 0) {
      if(errno != EEXIST) {
        odlog(ERROR)<<"Failed to create/find directory "<<dirpath<<std::endl;
        return false;
      };
    };
  };
  return copy_file(link_path,uid,gid);
}

bool DataCache::copy_file(const char* link_path,uid_t uid,gid_t gid) {
  char buf[65536];
  int fd = open64(link_path,O_WRONLY | O_CREAT | O_EXCL,S_IRUSR | S_IWUSR);
  if(fd == -1) {
    perror("open64");
    odlog(ERROR)<<"Failed to create file for writing: "<<link_path<<std::endl;
    return false;
  };
  fchown(fd,uid,gid);
  int fd_ = open64(cache_file.c_str(),O_RDONLY);
  if(fd_==-1) {
    close(fd);
    perror("open64");
    odlog(ERROR)<<"Failed to open file for reading: "<<cache_file<<std::endl;
    return false;
  };
  for(;;) {
    int l = read(fd_,buf,sizeof(buf));
    if(l == -1) {
      close(fd); close(fd_);
      perror("read");
      odlog(ERROR)<<"Failed to read file: "<<cache_file<<std::endl;
      return false;
    };
    if(l == 0) break;
    for(int ll = 0;ll<l;) {
      int l_=write(fd,buf+ll,l-ll);
      if(l_ == -1) {
        close(fd); close(fd_);
        perror("write");
        odlog(ERROR)<<"Failed to write file: "<<link_path<<std::endl;
        return false;
      };
      ll+=l_;
    };
  };
  close(fd); close(fd_);
  return true;
}

bool DataCache::cb(unsigned long long int size) {
  if(size == 0) size=1;
  return clean(size);
}
