#include "../std.h"

#include <iostream>
#include <pthread.h>

#include "../misc/checksum.h"
#include "../misc/log_time.h"

#include "databufferpar.h"

bool DataBufferPar::set(CheckSum *cksum,unsigned int size,int blocks) {
  pthread_mutex_lock(&lock); 
  if(blocks < 0) { pthread_mutex_unlock(&lock); return false; };
  if(bufs != NULL) {
    for(int i=0;i<bufs_n;i++) {
      if(bufs[i].start) free(bufs[i].start);
    };
    free(bufs); bufs_n=0; bufs=NULL; set_counter++;
    pthread_cond_broadcast(&cond); /* make all waiting loops to exit */
  };
  if((size == 0) || (blocks == 0)) {
    pthread_mutex_unlock(&lock);
    return true;
  };
  bufs=(buf_desc*)malloc(sizeof(buf_desc)*blocks);
  if(bufs == NULL) {
    pthread_mutex_unlock(&lock);
    return false;
  };
  bufs_n=blocks;
  for(int i=0;i<blocks;i++) {
    bufs[i].start=NULL; 
    bufs[i].taken_for_read=false; bufs[i].taken_for_write=false;
    bufs[i].size=size; bufs[i].used=0; bufs[i].offset=0;
  };
  checksum=cksum;
  checksum_offset=0;
  checksum_ready=true;
  if(checksum) checksum->start();
  pthread_mutex_unlock(&lock);
  return true;
}

DataBufferPar::DataBufferPar(unsigned int size,int blocks) {
  bufs_n=0; bufs=NULL; set_counter=0;
  eof_read_flag=false; eof_write_flag=false;
  error_read_flag=false; error_write_flag=false; error_transfer_flag=false;
  pthread_mutex_init(&lock,NULL);
  pthread_cond_init(&cond,NULL);
  set(NULL,size,blocks);
  eof_pos=0;
}

DataBufferPar::DataBufferPar(CheckSum *cksum,unsigned int size,int blocks) {
  bufs_n=0; bufs=NULL; set_counter=0;
  eof_read_flag=false; eof_write_flag=false;
  error_read_flag=false; error_write_flag=false; error_transfer_flag=false;
  pthread_mutex_init(&lock,NULL);
  pthread_cond_init(&cond,NULL);
  set(cksum,size,blocks);
  eof_pos=0;
}

DataBufferPar::~DataBufferPar(void) {
  set(NULL,0,0);  
  pthread_cond_destroy(&cond);
  pthread_mutex_destroy(&lock);
}

bool DataBufferPar::eof_read(void) { return eof_read_flag; };

bool DataBufferPar::eof_write(void) { return eof_write_flag; };

bool DataBufferPar::error_transfer(void) { return error_transfer_flag; };

bool DataBufferPar::error_read(void) { return error_read_flag; };

bool DataBufferPar::error_write(void) { return error_write_flag; };

void DataBufferPar::eof_read(bool eof_) { 
  pthread_mutex_lock(&lock); 
  if(eof_) if(checksum) checksum->end();
  eof_read_flag=eof_;
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock); 
};

void DataBufferPar::eof_write(bool eof_) { 
  pthread_mutex_lock(&lock); 
  eof_write_flag=eof_;
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock); 
};

bool DataBufferPar::error(void) {
  return (error_read_flag || error_write_flag || error_transfer_flag);
};

void DataBufferPar::error_read(bool error_) {
  pthread_mutex_lock(&lock); 
  // error_read_flag=error_;
  if(error_) {
    if(!(error_write_flag || error_transfer_flag)) error_read_flag=true;
    if(checksum) checksum->end();
    eof_read_flag=true;
  } else {
    error_read_flag=false;
  };
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock); 
};

void DataBufferPar::error_write(bool error_) {
  pthread_mutex_lock(&lock); 
  // error_write_flag=error_;
  if(error_) {
    if(!(error_read_flag || error_transfer_flag)) error_write_flag=true;
    eof_write_flag=true;
  } else {
    error_write_flag=false;
  };
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock); 
};

bool DataBufferPar::wait_eof_read(void) {
  pthread_mutex_lock(&lock);
  for(;;) {
    if(eof_read_flag) break;
    pthread_cond_wait(&cond,&lock);
  };
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::wait_read(void) {
  pthread_mutex_lock(&lock);
  for(;;) {
    if(eof_read_flag) break;
    if(error_read_flag) break;
    pthread_cond_wait(&cond,&lock);
  };
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::wait_eof_write(void) {
  pthread_mutex_lock(&lock);
  for(;;) {
    if(eof_write_flag) break;
    pthread_cond_wait(&cond,&lock);
  };
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::wait_write(void) {
  pthread_mutex_lock(&lock);
  for(;;) {
    if(eof_write_flag) break;
    if(error_write_flag) break;
    pthread_cond_wait(&cond,&lock);
  };
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::wait_eof(void) {
  pthread_mutex_lock(&lock);
  for(;;) {
    if(eof_read_flag && eof_write_flag) break;
    pthread_cond_wait(&cond,&lock);
  };
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::cond_wait(void) {
  // Wait for any event
  int tmp = set_counter;
  bool eof_read_flag_tmp = eof_read_flag;
  bool eof_write_flag_tmp = eof_write_flag;
/*
  pthread_cond_wait(&cond,&lock);
*/
  int err = -1;
  for(;;) {
    if(!speed.transfer()) {
      if((!(error_read_flag || error_write_flag)) && 
         (!(eof_read_flag && eof_write_flag))) error_transfer_flag=true;
    };
    if(eof_read_flag && eof_write_flag) { // there wil be no more events
      pthread_mutex_unlock(&lock);
      pthread_yield(); 
      pthread_mutex_lock(&lock);
      return true;
    };
    if(eof_read_flag_tmp != eof_read_flag) return true;
    if(eof_write_flag_tmp != eof_write_flag) return true;
    if(error()) return false; // useless to wait for - better fail
    if(set_counter != tmp) return false;
    if(err == 0) break; // Some event
    int t = 60; 
    struct timeval stime;
    gettimeofday(&stime,NULL);
    struct timespec etime;
    etime.tv_sec = stime.tv_sec + t;
    etime.tv_nsec = stime.tv_usec*1000;
    // Using timeout to workaround lost signal
    err=pthread_cond_timedwait(&cond,&lock,&etime);
  };
  return true;
};

bool DataBufferPar::for_read(void) {
  if(bufs == NULL) return false;
  pthread_mutex_lock(&lock); 
  for(int i=0;i<bufs_n;i++) {
    if((!bufs[i].taken_for_read) && (!bufs[i].taken_for_write) &&
       (bufs[i].used == 0)) {
      pthread_mutex_unlock(&lock);
      return true;
    };
  };
  pthread_mutex_unlock(&lock);
  return false;
}

bool DataBufferPar::for_read(int &handle,unsigned int &length,bool wait) {
  pthread_mutex_lock(&lock); 
  if(bufs == NULL) { pthread_mutex_unlock(&lock); return false; };
  for(;;) {
    if(error()) {   /* errors detected/set - any continuation is unusable */
      pthread_mutex_unlock(&lock); return false;
    };
    for(int i=0;i<bufs_n;i++) {
      if((!bufs[i].taken_for_read) && (!bufs[i].taken_for_write) &&
         (bufs[i].used == 0)) {
        if(bufs[i].start == NULL) {
          bufs[i].start=(char*)malloc(bufs[i].size);
          if(bufs[i].start == NULL) continue;
        };
        handle=i; bufs[i].taken_for_read=true; length=bufs[i].size;
        pthread_cond_broadcast(&cond);
        pthread_mutex_unlock(&lock);
        return true;
      };
    };
    /* suitable block not found - wait for changes or quit */
    if(eof_write_flag) {   /* writing side quited, no need to wait */
      pthread_mutex_unlock(&lock); return false;
    };
    if(!wait) { pthread_mutex_unlock(&lock); return false; };
    if(!cond_wait()) { pthread_mutex_unlock(&lock); return false; };
  };
  pthread_mutex_unlock(&lock);
  return false;
}

bool DataBufferPar::is_read(char* buf,unsigned int length,unsigned long long int offset) {
  pthread_mutex_lock(&lock);
  for(int i=0;i<bufs_n;i++) {
    if(bufs[i].start == buf) {
      pthread_mutex_unlock(&lock);
      return is_read(i,length,offset);
    };
  };
  pthread_mutex_unlock(&lock);
  return false;
}

bool DataBufferPar::is_read(int handle,unsigned int length,unsigned long long int offset) {
  pthread_mutex_lock(&lock);
  if(bufs == NULL) { pthread_mutex_unlock(&lock); return false; };
  if(handle >= bufs_n) { pthread_mutex_unlock(&lock); return false; };
  if(!bufs[handle].taken_for_read) {pthread_mutex_unlock(&lock); return false;};
  if(length>bufs[handle].size) { pthread_mutex_unlock(&lock); return false; };
  bufs[handle].taken_for_read=false;
  bufs[handle].used=length; bufs[handle].offset=offset;
  if((offset+length) > eof_pos) eof_pos=offset+length;
  /* checksum on the fly */
  if((checksum != NULL) && (offset == checksum_offset)) {
    for(int i=handle;i<bufs_n;i++) {
      if(bufs[i].used != 0) {
        if(bufs[i].offset == checksum_offset) {
          checksum->add(bufs[i].start,bufs[i].used);
          checksum_offset+=bufs[i].used;
          i=-1; checksum_ready=true;
        }
        else {
          if(checksum_offset<bufs[i].offset) checksum_ready=false;
        };
      };
    };
  };
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::for_write(void) {
  if(bufs == NULL) return false;
  pthread_mutex_lock(&lock); 
  for(int i=0;i<bufs_n;i++) {
    if((!bufs[i].taken_for_read) && (!bufs[i].taken_for_write) &&
       (bufs[i].used != 0)) {
      pthread_mutex_unlock(&lock);
      return true;
    };
  };
  pthread_mutex_unlock(&lock);
  return false;
}

/* return true + buffer with data,
   return false in case of failure, or eof + no buffers claimed for read */
bool DataBufferPar::for_write(int &handle,unsigned int &length,unsigned long long int &offset,bool wait) {
  pthread_mutex_lock(&lock); 
  if(bufs == NULL) { pthread_mutex_unlock(&lock); return false; };
  for(;;) {
    if(error()) { /* internal/external errors - no need to continue */
      pthread_mutex_unlock(&lock); return false;
    };
    bool have_for_read = false;
    bool have_unused = false;
    unsigned long long int min_offset = (unsigned long long int)(-1);
    handle=-1;
    for(int i=0;i<bufs_n;i++) {
      if(bufs[i].taken_for_read) have_for_read=true;
      if((!bufs[i].taken_for_read) && (!bufs[i].taken_for_write) &&
         (bufs[i].used != 0)) {
        if(bufs[i].offset<min_offset) {
          min_offset=bufs[i].offset;
          handle=i;
        };
      };
      if(bufs[i].taken_for_read || (bufs[i].used==0)) have_unused=true;
    };
    if(handle != -1) {
      if((!checksum_ready) && (bufs[handle].offset>=checksum_offset)) {
        /* try to keep buffers as long as possible for checksuming */
        if(have_unused && (!eof_read_flag)) { 
          /* still have chances to get that block */
          if(!wait) { pthread_mutex_unlock(&lock); return false; };
          if(!cond_wait()) { pthread_mutex_unlock(&lock); return false; };
          continue;
        };
      };
      bufs[handle].taken_for_write=true;
      length=bufs[handle].used;
      offset=bufs[handle].offset;
      pthread_cond_broadcast(&cond);
      pthread_mutex_unlock(&lock); 
      return true;
    };
    if(eof_read_flag && (!have_for_read)) {
      pthread_mutex_unlock(&lock); return false;
    };
    /* suitable block not found - wait for changes or quit */
    if(!wait) { pthread_mutex_unlock(&lock); return false; };
    if(!cond_wait()) { pthread_mutex_unlock(&lock); return false; };
  };
  pthread_mutex_unlock(&lock); 
  return false;
}

bool DataBufferPar::is_written(char* buf) {
  pthread_mutex_lock(&lock);
  for(int i=0;i<bufs_n;i++) {
    if(bufs[i].start == buf) {
      pthread_mutex_unlock(&lock);
      return is_written(i);
    };
  };
  pthread_mutex_unlock(&lock);
  return false;
}

bool DataBufferPar::is_notwritten(char* buf) {
  pthread_mutex_lock(&lock);
  for(int i=0;i<bufs_n;i++) {
    if(bufs[i].start == buf) {
      pthread_mutex_unlock(&lock);
      return is_notwritten(i);
    };
  };
  pthread_mutex_unlock(&lock);
  return false;
}

bool DataBufferPar::is_written(int handle) {
  pthread_mutex_lock(&lock); 
  if(bufs == NULL) { pthread_mutex_unlock(&lock); return false; };
  if(handle >= bufs_n) { pthread_mutex_unlock(&lock); return false; };
  if(!bufs[handle].taken_for_write) {pthread_mutex_unlock(&lock);return false;};
  /* speed control */
  if(!speed.transfer(bufs[handle].used)) { 
    if((!(error_read_flag || error_write_flag)) &&
       (!(eof_read_flag && eof_write_flag))) error_transfer_flag=true;
  };
  bufs[handle].taken_for_write=false;
  bufs[handle].used=0; bufs[handle].offset=0;
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock); 
  return true;
}

bool DataBufferPar::is_notwritten(int handle) {
  pthread_mutex_lock(&lock);
  if(bufs == NULL) { pthread_mutex_unlock(&lock); return false; };
  if(handle >= bufs_n) { pthread_mutex_unlock(&lock); return false; };
  if(!bufs[handle].taken_for_write) {pthread_mutex_unlock(&lock);return false;};
  bufs[handle].taken_for_write=false;
  pthread_cond_broadcast(&cond);
  pthread_mutex_unlock(&lock);
  return true;
}

char* DataBufferPar::operator[](int block) {
  pthread_mutex_lock(&lock);
  if((block<0) || (block>=bufs_n)) { pthread_mutex_unlock(&lock); return NULL; };
  char* tmp = bufs[block].start;
  pthread_mutex_unlock(&lock); 
  return tmp;
}

bool DataBufferPar::wait(void) {
  pthread_mutex_lock(&lock); 
  bool res = cond_wait();
  pthread_mutex_unlock(&lock); 
  return res;
}

bool DataBufferPar::wait_used(void) {
  pthread_mutex_lock(&lock); 
  for(int i=0;i<bufs_n;i++) {
    if((bufs[i].taken_for_read) || (bufs[i].taken_for_write) ||
         (bufs[i].used != 0)) {
      if(!cond_wait()) { pthread_mutex_unlock(&lock); return false; };
      i=-1;
    };
  }; 
  pthread_mutex_unlock(&lock);
  return true;
}

bool DataBufferPar::checksum_valid(void) {
  return checksum_ready;
}

const CheckSum* DataBufferPar::checksum_object(void) {
  return checksum;
}

unsigned int DataBufferPar::buffer_size(void) {
  if(bufs == NULL) return 65536;
  unsigned int size = 0;
  for(int i=0;i<bufs_n;i++) {
    if(size < bufs[i].size) size=bufs[i].size;
  };
  return size;
}

