// Utility to register/unregister content of cache to Indexing Service (RC,RLS)
// TODO: sorting of records, recognize indexing service which do no allow to write, ...
#include "../std.h"

#include <string>
#include <list>

#include <globus_common.h>

#include "../cache/cache.h"
#include "../config/config_file.h"
#include "../config/environment.h"
#include "../jobs/users.h"
#include "../datamove/datapoint.h"
#include "../misc/log_time.h"


int main(int argc,char* argv[]) {
  LogTime::Active(true);
  LogTime::Level(FATAL);
  for(;;) {
    opterr=0;
    int optc=getopt(argc,argv,"+hZd:c:");
    if(optc == -1) break;
    switch(optc) {
      case 'h': {
        olog<<"cache-register [-h] [-d level] [-Z] [-c configuration_file]"<<std::endl;
        olog<<" -h - print help"<<std::endl;
        olog<<" -d - set debug level (use 0 for verbose)"<<std::endl;
        olog<<" -c - path to grid-manager's configuration file"<<std::endl;
        olog<<" -Z - use central configuration file"<<std::endl;
        exit(1);
      }; break;
      case 'v': {
        olog<<"cache-register: version "<<VERSION<<std::endl;
        exit(0);
      };
      case 'd': {
        char* p;
        int i = strtol(optarg,&p,10);
        if(((*p) != 0) || (i<0)) {
          olog<<"Improper debug level '"<<optarg<<"'"<<std::endl;
          exit(1);
        };
        LogTime::Level(NotifyLevel(FATAL+i));
      }; break;
      case 'c': {
        nordugrid_config_loc=optarg;
      }; break;
      case 'Z': {
        central_configuration=true;
      }; break;
      case '?': {
        olog<<"Unsupported option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      case ':': {
        olog<<"Missing parameter for option '"<<(char)optopt<<"'"<<std::endl;
        exit(1);
      }; break;
      default: {
        olog<<"Undefined processing error"<<std::endl;
        exit(1);
      };
    };
  };

  char hostname[256];
  if(globus_libc_gethostname(hostname,sizeof(hostname)) != 0) {
    odlog(ERROR)<<"Failed to obtain own host name"<<std::endl;
    return 1;
  };
  std::string cache_name(hostname);
  std::string cache_url(hostname);
  cache_name=cache_name+":cache";
  cache_url="cache://"+cache_url+"/";
  // Process configuration (like in GM)
  if(!read_env_vars()) exit(1);
  JobUsers users;
  std::string my_username("");
  uid_t my_uid=getuid();
  JobUser *my_user = NULL;
  {
    struct passwd pw_;
    struct passwd *pw;
    char buf[BUFSIZ];
    getpwuid_r(my_uid,&pw_,buf,BUFSIZ,&pw);
    if(pw != NULL) { my_username=pw->pw_name; };
  };
  if(my_username.length() == 0) {
    olog << "Can't recognize myself." << std::endl; exit(1);
  };
  my_user = new JobUser(my_username);
  if(!configure_serviced_users(users,my_uid,my_username,*my_user)) {
    olog<<"Error processing configuration."<<std::endl; exit(1);
  };
  if(users.size() == 0) {
    olog<<"No suitable users found in configuration."<<std::endl; exit(1);
  };


  // Process all configured caches
  for(JobUsers::const_iterator user = users.begin();user != users.end();++user) {
    if(user->CachePrivate()) continue; // only public caches made public
    std::string cache_dir = user->CacheDir();
    if(cache_dir.length() == 0) continue; // cache not configured
    odlog(INFO)<<"Cache directory: "<<cache_dir<<std::endl;

    // Get list of old and new files (snapshot)
    std::list<std::string> old_records;
    std::list<std::string> new_records;
    if(cache_history_lists(cache_dir.c_str(),old_records,new_records) != 0) {
      odlog(ERROR)<<"Failed to get lists from cache"<<std::endl;
      continue;
    };
    std::list<std::string> old_records_o = old_records;
    std::list<std::string> new_records_o = new_records;
    // Remove duplicates
    // "old" records have priority ?!
    for(std::list<std::string>::iterator i = old_records.begin();
                                          i!=old_records.end();++i) {
      std::list<std::string>::iterator i_ = i; ++i_;
      for(;i_!=old_records.end();) {
        if((*i) == (*i_)) { i_=old_records.erase(i_); continue; }; ++i_;
      };
    };
    for(std::list<std::string>::iterator i = new_records.begin();
                                          i!=new_records.end();++i) {
      std::list<std::string>::iterator i_ = i; ++i_;
      for(;i_!=new_records.end();) {
        if((*i) == (*i_)) { i_=new_records.erase(i_); continue; }; ++i_;
      };
    };
    for(std::list<std::string>::iterator i_o = old_records.begin();
                                          i_o!=old_records.end();++i_o) {
      for(std::list<std::string>::iterator i_n=new_records.begin();
                                            i_n!=new_records.end();) {
        if((*i_n) == (*i_o)) { i_n=new_records.erase(i_n); continue; }; ++i_n;
      };
    };

    for(int tries = 5;tries;tries--) {
      // Process "new"
      for(std::list<std::string>::iterator i=new_records.begin();i!=new_records.end();) {
        odlog(DEBUG)<<"Url (new): "<<*i<<std::endl;
        DataPoint url(i->c_str()); // take url
        if(url.meta()) { // ordinary urls are not registrable
          odlog(INFO)<<"Url to register: "<<*i<<std::endl;
          // Add new location. This also limits resolved locations 
          // to those specified
          if(!url.add_location(cache_name.c_str(),
                                            (cache_url+url.lfn()).c_str())) {
            odlog(ERROR)<<"Failed to add location: "<<url<<std::endl; 
            /* i=new_records.erase(i); */ ++i; continue;
          };
          //  Resolve for destination to check if name is registered
          //  and for metadata (would be nice to get already available data)
          if(!url.meta_resolve(false)) {
            odlog(ERROR)<<"Failed to resolve: "<<url<<std::endl;
            /* i=new_records.erase(i); */ ++i; continue;
          };
          // preregister for replication
          if(!url.meta_preregister(true)) {
            odlog(ERROR)<<"Failed to preregister: "<<url<<std::endl;
            /* i=new_records.erase(i); */ ++i; continue;
          };
          // postregister to announce existence of file
          if(!url.meta_postregister(true,false)) {
            odlog(ERROR)<<"Failed to postregister: "<<url<<std::endl;
            /* i=new_records.erase(i); */ ++i; continue;
          };
        }; 
        i=new_records.erase(i); /* ++i; */
      };  
      // Process "old"
      for(std::list<std::string>::iterator i=old_records.begin();i!=old_records.end();) {
        odlog(DEBUG)<<"Url (old): "<<*i<<std::endl;
        DataPoint url(i->c_str()); // take url
        if(url.meta()) { // ordinary urls are not registrable
          odlog(INFO)<<"Url to unregister: "<<*i<<std::endl;
          // Add new location. This also limits resolved locations 
          // to those specified
          if(!url.add_location(cache_name.c_str(),
                                            (cache_url+url.lfn()).c_str())) {
            odlog(ERROR)<<"Failed to add location"<<std::endl;
            i=old_records.erase(i); continue;
          };
          //  Resolve for source to check if name is registered
          if(!url.meta_resolve(true)) {
            odlog(ERROR)<<"Failed to resolve: "<<url<<std::endl;
            i=old_records.erase(i); continue;
          };
          if(!url.have_locations()) {
            odlog(INFO)<<"Location is already missing (that's ok)"<<std::endl;
          } else {
            if(!url.meta_unregister(false)) {
              odlog(ERROR)<<"Failed to unregister"<<std::endl;
              i=old_records.erase(i); continue;
            };
          };
        };
        ++i;
      };
      if((new_records.size() == 0) && (old_records.size() == 0)) break;
      sleep(300); // wait 5 minutes before retry
    };

    // Remove processed records
    if(cache_history_remove(cache_dir.c_str(),old_records_o,new_records_o) != 0) {
      odlog(ERROR)<<"Failed to remove lists from cache"<<std::endl;
      return 1;
    };
  };
  return 0;
}


