#include <list>
#include <string>
#include <iostream>
#include <map>
#include <fstream>
#include <algorithm>

#include "arccli.h"

#include <arc/certificate.h>
#include <arc/xrsl.h>
#include <arc/url.h>
#include <arc/ldapquery.h>
#include <arc/notify.h>
#include <arc/mdsquery.h>
#include <arc/stringconv.h>
#include <arc/giis.h>
#include <arc/mdsdiscovery.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif

void FindVOMembers(const std::string& attr, const std::string& value, void* arg);


void arctest(const unsigned int testjob,
             const bool certificateinfo,
             const bool voinfo,
             const bool resourceinfo,
             const bool configuration,
             const unsigned int runtime,
             const std::list<std::string>& clusterselect,
             const std::list<std::string>& clusterreject,
             const std::list<std::string>& giisurls,
             const std::string& joblistfile,
             const bool dryrun,
             const bool dumpxrsl,
             const int timeout,
             const bool anonymous) {

	if (testjob>0) {
		Xrsl xrsl;
		if (testjob<1 || testjob>3)
			throw ARCCLIError(_("Illegal testjob-id given"));

		Certificate proxy(PROXY);
		if (proxy.IsExpired())
			throw ARCLibError(_("The proxy has expired"));

		std::string runstr = tostring(runtime+1);
		std::string reqstr = tostring(runtime+3);

		if (testjob==1) xrsl = Xrsl("&(executable=run.sh)(arguments=" + runstr + ")(inputfiles=(\"run.sh\" \"http://www.nordugrid.org;cache=no/data/run.sh\")(\"Makefile\" \"rls://rls.nordugrid.org/Makefile\")(\"prime.cpp\" \"ftp://ftp.nordugrid.org;cache=no/applications/test/prime.cpp\"))(stderr=primenumbers)(outputfiles=(\"primenumbers\" \"\"))(jobname=\"ARC testjob 1\")(stdout=stdout)(gmlog=gmlog)(CPUTime=" + reqstr + ")");

		if (testjob==2) xrsl = Xrsl("&(executable=/bin/echo)(arguments=\"hello, grid\")(jobname=\"ARC testjob 2\")(stdout=stdout)(cputime=5)");

		if (testjob==3) xrsl = Xrsl("&(executable=md5sum.sh)(arguments=downloaded[0-7])(inputfiles=(\"downloaded0\" \"http://www.nordugrid.org;cache=yes/data/echo.sh\")(\"downloaded1\" \"http://www.nordugrid.org;cache=yes/data/somefile\")(\"downloaded2\" \"gsiftp://hathi.hep.lu.se;cache=yes/public/ngtest/echo.sh\")(\"downloaded3\" \"gsiftp://hathi.hep.lu.se;cache=yes/public/ngtest/somefile\")(\"downloaded4\" \"ftp://ftp.nordugrid.org;cache=yes/applications/test/echo.sh\")(\"downloaded5\" \"ftp://ftp.nordugrid.org;cache=yes/applications/test/somefile\")(\"downloaded6\" \"rls://rls.nordugrid.org/echo.sh\")(\"downloaded7\" \"rls://rls.nordugrid.org/somefile\")(\"md5sum.sh\" \"ftp://ftp.nordugrid.org;cache=yes/applications/test/md5sum.sh\"))(jobname=\"ARC testjob 3\")(stdout=stdout)(stderr=stderr)(gmlog=gmlog)(CPUTime=5)");

		std::cout << _("Submitting test-job") << " " << testjob << ": " << std::endl << xrsl.str() << std::endl;

		std::cout << _("Client middleware") << ": nordugrid-arc-" << VERSION << ", globus-";
		// Find globus middleware
		std::string globusloc = GetEnv("GLOBUS_LOCATION");

		bool written = false;
		if (!globusloc.empty()) {
			std::string globusversion = globusloc + "/bin/globus-version";
			std::ifstream versionfile(globusversion.c_str());

			if (versionfile) {
				std::string line;
				while (versionfile >> line) {
					if (line.substr(0, 14)!="GLOBUS_VERSION") continue;

					std::string version = line.substr(16);
					version = version.substr(0, version.size()-1);
					std::cout << version << std::endl;
					written = true;
					break;
				}
			}
		}

		if (!written) std::cout << _("unknown") << std::endl;

		std::list<std::string> xrslfiles, xrslstrings;
		xrslstrings.push_back(xrsl.str());

		// submit test-job
		std::list<std::string> jobids = arcsub(xrslfiles, xrslstrings, clusterselect, clusterreject, giisurls, joblistfile, dryrun, dumpxrsl, false, timeout, anonymous);

		if (!jobids.empty()) {
			URL joburl(*jobids.begin());
			URL ldapurl("ldap://" + joburl.Host() + ":2135/o=grid/mds-vo-name=local");
			Cluster cluster = GetClusterInfo(ldapurl);
			std::cout << _("Cluster") << ": " << cluster.hostname << "; " << _("middleware") << ": ";

			std::string nordugridmw("nordugrid-arc-unknown");
			std::string globusmw("globus-unknown");
			std::list<RuntimeEnvironment> mws = cluster.middlewares;
			std::list<RuntimeEnvironment>::iterator it;
			for (it = mws.begin(); it != mws.end(); it++) {
				if (it->Name()=="nordugrid" || it->Name()=="nordugrid-arc")
					nordugridmw = it->str();
				if (it->Name().substr(0, 6)=="globus") globusmw = it->str();
			}

			std::cout << nordugridmw << ", " << globusmw << std::endl;
		}

		return;
	}

	if (certificateinfo) {
		SetNotifyLevel(ERROR);

		// Print certificate info
		std::cout << _("Certificate information") << ":" << std::endl;
		std::cout << std::endl;

		std::string issuersn;
		try { // user-certificate
			Certificate user;
			std::cout << _("Certificate") << ": " << user.GetCertFilename() << std::endl;
			std::cout << _("Subject name") << ": " << user.GetSN() << std::endl;

			Time::SetFormat(ASCTime);
			std::cout << _("Valid until") << ": ";
			if (user.IsExpired()) {
				std::cout << _("Expired") << std::endl;
			} else {
				std::cout << user.ExpiryTime() << std::endl;
			}

			issuersn = user.GetIssuerSN();
		} catch(CertificateError e) {
			std::cout << _("No user-certificate found") << std::endl;
		}

		std::cout << std::endl;

		try { // user-proxy
			Certificate proxy(PROXY);
			std::cout << _("Proxy") << ": " << proxy.GetCertFilename() << std::endl;
			std::cout << _("Proxy-subject") << ": " << proxy.GetSN() << std::endl;

			std::cout << _("Valid for") << ": ";
			if (proxy.IsExpired()) {
				std::cout << _("Expired") << std::endl;
			} else {
				std::cout << proxy.ValidFor() << std::endl;
			}
		} catch(CertificateError e) {
			std::cout << _("No proxy found") << std::endl;
		}

		std::cout << std::endl;

		if (!issuersn.empty())
			std::cout << _("Certificate issuer") << ": " << issuersn << std::endl << std::endl;

		// CA-certificates
		std::cout << _("CA-certificates installed") << ":" << std::endl;
		std::list<Certificate> calist = GetCAList();
		if (calist.size()==0)
			std::cout << _("None") << std::endl;

		std::list<Certificate>::iterator it;
		for (it = calist.begin(); it != calist.end(); it++)
			std::cout << it->GetSN() << std::endl;

		if (!issuersn.empty() && !CheckIssuer(issuersn)) {
			std::cout << std::endl << _("Warning: Your issuer's certificate is not installed") << std::endl;
		}

		return;
	}

	if (voinfo) {
		SetNotifyLevel(ERROR);

		std::cout << _("VO information") << ":" << std::endl;
		std::cout << std::endl;

		std::string usersn;
		Certificate user;
		usersn = user.GetSN();
		std::cout << _("Certificate-subject") << ": " << usersn << std::endl << std::endl;

		std::list<std::pair<std::string, URL> > vos;

		URL NGVO("ldap://grid-vo.nordugrid.org/dc=org/dc=nordugrid/ou=people");
		std::string NGVOdesc("NorduGrid VO");

		URL NGguestVO("ldap://grid-vo.nordugrid.org/dc=org/dc=nordugrid/ou=guest");
		std::string NGguestVOdesc("NorduGrid guest VO");

		//URL ARCVO("ldap://grid-vo.nordugrid.org/dc=org/dc=nordugrid/ou=people");
		//string ARCVOdesc("ARC Community VO");

		vos.push_back(make_pair(NGVOdesc, NGVO));
		vos.push_back(make_pair(NGguestVOdesc, NGguestVO));
		//vos.push_back(make_pair(ARCVOdesc, ARCVO));

		std::list<std::pair<std::string, URL> >::iterator it;
		for (it = vos.begin(); it != vos.end(); it++) {
			std::cout << it->first << ": ";
			LdapQuery ldapq(it->second.Host(), it->second.Port());
			std::list<std::string> vomembers;
			try {
				ldapq.Query(it->second.BaseDN());
				ldapq.Result(FindVOMembers, (void*)&vomembers);

				if (std::find(vomembers.begin(), vomembers.end(), usersn)==vomembers.end()) {
					std::cout << _("Not a member") << std::endl;
				} else {
					std::cout << _("Member") << std::endl;
				}
			} catch (LdapQueryError e) {
				std::cout << _("Can't contact LDAP-server") << std::endl;
			}
		}

		return;
	}

	if (resourceinfo) {
		SetNotifyLevel(ERROR);

		std::cout << _("Resource authorizations") << ":" << std::endl;
		std::cout << std::endl;

		try {
			Certificate cert = GetEffectiveCredential();
			if (cert.GetCertificateType()==USERCERT && cert.IsExpired())
				throw CertificateError(_("Your certificate has expired!"));

		} catch (CertificateError e) {
			std::cout << e.what() << std::endl;
			return;
		}

		std::string sn = GetEffectiveSN();

		std::cout << _("Certificate-subjectname") << ": " << std::endl << sn << std::endl << std::endl;
		std::cout << _("Retrieving information") << "..." << std::endl << std::endl;

		// clusters
		std::list<URL> giises = ConvertToURLs(giisurls);
		std::list<URL> urls = GetClusterResources(giises, anonymous, "", timeout);
		std::list<Queue> queueinfo;
		queueinfo = GetQueueInfo(urls, MDS_FILTER_CLUSTERINFO, anonymous, "", timeout);

		std::list<std::string> authclusters;
		std::list<Queue>::iterator it;
		for (it = queueinfo.begin(); it != queueinfo.end(); it++) {
			if (it->users.size()>0)
				authclusters.push_back(it->cluster.hostname);
		}

		authclusters.sort();
		authclusters.unique();

		std::cout << _("You are authorized at the following clusters") << ":" << std::endl;
		if (authclusters.empty())
			std::cout << _("None");

		std::list<std::string>::iterator strit;
		for (strit = authclusters.begin(); strit != authclusters.end(); strit++)
			std::cout << *strit << "  ";

		std::cout << std::endl << std::endl;

		// storage-elements
		urls = GetSEResources(giises, anonymous, "", timeout);
		std::list<StorageElement> seinfo = GetSEInfo(urls, MDS_FILTER_SEINFO, anonymous, "", timeout);
		std::list<StorageElement>::iterator seit;
		std::cout << _("You are authorized at the following storage-elements") << ":" << std::endl;
		if (seinfo.empty())
			std::cout << _("None") << std::endl;

		for (seit = seinfo.begin(); seit != seinfo.end(); seit++) {
			std::list<std::string> authusers = seit->auth_users;
			if (find(authusers.begin(), authusers.end(), sn)!=authusers.end())
				std::cout << seit->name << "  ";
		}

		std::cout << std::endl;
		return;
	}

	if (configuration) {
		std::cout << _("Configuration") << ":" << std::endl;
		std::cout << std::endl;

		std::string nglocation = GetEnv("ARC_LOCATION");
		if (nglocation.empty()) nglocation = GetEnv("NORDUGRID_LOCATION");
		std::string globuslocation = GetEnv("GLOBUS_LOCATION");
		std::cout << "ARC_LOCATION: ";
		if (nglocation.empty()) {
			std::cout << "(" << _("undefined") << ")" << std::endl;
		} else {
			std::cout << nglocation << std::endl;
		}

		std::cout << "GLOBUS_LOCATION: ";
		if (globuslocation.empty()) {
			std::cout << "(" << _("undefined") << ")" << std::endl << std::endl;
		} else {
			std::cout << globuslocation << std::endl << std::endl;
		}

		std::cout << _("Top-level GIIS's used") << ":" << std::endl;
		std::list<URL> giislist = GetGIISList();
		std::list<URL>::iterator it;
		for (it = giislist.begin(); it != giislist.end(); it++)
			std::cout << it->str() << std::endl;

		std::cout << std::endl << _("User certificate") << ": ";
		try {
			Certificate cert;
			std::cout << cert.GetSN() << std::endl;
			std::cout << _("Valid until") << ": ";
			if (cert.IsExpired()) {
				std::cout << "Expired" << std::endl;
			} else {
				std::cout << cert.ExpiryTime() << std::endl;
			}
		} catch (CertificateError e) {
			std::cout << "(" << _("not found") << ")" << std::endl;
		}
		std::cout << std::endl;

		try {
			Certificate proxy(PROXY);
			std::cout << _("Proxy") << ": " << proxy.GetSN() << std::endl;
			std::cout << _("Valid for") << ": ";
			if (proxy.IsExpired()) {
				std::cout << "Expired" << std::endl;
			} else {
				std::cout << proxy.ValidFor() << std::endl;
			}
		} catch (CertificateError e) {
			std::cout << _("Proxy") << ": (" << _("not found") << ")" << std::endl;
		}
	}
}


void FindVOMembers(const std::string& attr, const std::string& value, void* arg) {
	if (attr!="description") return;
	std::list<std::string>* vomembers = (std::list<std::string>*)arg;

	std::string val(value.substr(8));
	while (val[0]==' ') val = val.substr(1);
	vomembers->push_back(val);
}
