#include <fstream>
#include <iostream>
#include <list>
#include <string>
#include <algorithm>

#include "arccli.h"

#include <arc/common.h>
#include <arc/ftpcontrol.h>
#include <arc/giis.h>
#include <arc/jobftpcontrol.h>
#include <arc/joblist.h>
#include <arc/jobsubmission.h>
#include <arc/mdsdiscovery.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/standardbrokers.h>
#include <arc/target.h>
#include <arc/url.h>
#include <arc/xrsl.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


void arcresub(const std::list<std::string>& jobs,
              const std::list<std::string>& clusterselect,
              const std::list<std::string>& clusterreject,
              const std::list<std::string>& status,
              const std::list<std::string>& klusterselect,
              const std::list<std::string>& klusterreject,
              const std::list<std::string>& giisurls,
              const std::string& joblistfile,
              const bool dryrun,
              const bool dumpxrsl,
              const bool unknownattr,
              const bool keep,
              const int timeout,
              const bool anonymous) {

	std::list<std::string> jobids =
		GetJobIDsList(jobs, clusterselect, clusterreject);

	if (jobs.empty() && clusterselect.empty() && jobids.empty())
		throw ARCCLIError(_("No jobs"));

	if (jobids.empty())
		throw ARCCLIError(_("No valid jobnames/jobids given"));

	std::list<Job> joblist =
		GetJobInfo(jobids, MDS_FILTER_JOBINFO, anonymous, "", timeout);

	// find available clusters

	std::list<URL> clusterurllist;

	for (std::list<std::string>::const_iterator it = klusterselect.begin();
	     it != klusterselect.end(); it++) {
		bool found = false;
		for (std::list<URL>::iterator cli = clusterurllist.begin();
		     !found && cli != clusterurllist.end(); cli++)
			if (*it == cli->Host()) found = true;
		if (!found)
			clusterurllist.push_back("ldap://" + *it +
			                         ":2135/O=Grid/Mds-Vo-Name=local");
	}

	if (clusterurllist.empty()) {
		std::list<URL> giisurllist = ConvertToURLs(giisurls);
		clusterurllist =
			GetClusterResources(giisurllist, anonymous, "", timeout);
		if (clusterurllist.empty())
			throw ARCCLIError(_("Could not retrieve cluster list from GIIS"));
	}

	for (std::list<std::string>::const_iterator it = klusterreject.begin();
	     it != klusterreject.end(); it++)
		for (std::list<URL>::iterator cli = clusterurllist.begin();
		     cli != clusterurllist.end(); cli++)
			if (cli->Host() == *it) {
				notify(INFO) << _("Rejecting cluster")
				             << ": " << *it << std::endl;
				clusterurllist.erase (cli);
				break;
			}

	std::list<Queue> queuelist =
		GetQueueInfo (clusterurllist, MDS_FILTER_CLUSTERINFO,
		              anonymous, "", timeout);

	FTPControl ctrl;
	JobFTPControl jctrl;

	for (std::list<Job>::iterator jli = joblist.begin();
	     jli != joblist.end(); jli++) {

		if (jli->status.empty()) {
			notify(WARNING) << _("Job information not found")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (!status.empty() &&
		    std::find(status.begin(), status.end(), jli->status) == status.end())
			continue;

		std::string filename("/tmp/arcresub.XXXXXX");
		int tmp_h = mkstemp((char*)filename.c_str());
		if (tmp_h == -1)
			throw ARCCLIError(_("Could not create temporary file"));
		close(tmp_h);

		std::string url(jli->id);
		url.insert(url.rfind('/'), "/info");
		url += "/description";

		ctrl.Download(url, filename, timeout);

		std::ifstream xrslfile(filename.c_str());
		xrslfile.seekg(0, std::ios::end);
		std::streamsize length = xrslfile.tellg();
		xrslfile.seekg(0, std::ios::beg);
		char* buffer = new char[length + 1];
		xrslfile.read(buffer, length);
		buffer[length] = '\0';
		Xrsl x(buffer);
		delete[] buffer;
		xrslfile.close();

		unlink(filename.c_str());

		Xrsl xrsl(x.GetRelation("clientxrsl").GetSingleValue());

		PerformXrslValidation(xrsl, unknownattr);

		std::list<Target> targetlist = ConstructTargets(queuelist, xrsl);

		PerformStandardBrokering(targetlist);

		JobSubmission submit(xrsl, targetlist, dryrun);

		std::string jobid = submit.Submit(timeout);

		submit.RegisterJobsubmission(queuelist);

		std::string jobname;
		try {
			jobname = xrsl.GetRelation("jobname").GetSingleValue();
		}
		catch (XrslError e) {}

		AddJobID(jobid, jobname);

		if (!joblistfile.empty()) {
			LockFile(joblistfile);
			std::ofstream jobs(joblistfile.c_str(), std::ios::app);
			jobs << jobid << std::endl;
			jobs.close();
			UnlockFile(joblistfile);
		}

		std::string histfilename = GetEnv("HOME");
		histfilename.append ("/.arc/history");
		LockFile(histfilename);
		std::ofstream nghist (histfilename.c_str(), std::ios::app);
		nghist << TimeStamp() << "  " << jobid << std::endl;
		nghist.close();
		UnlockFile(histfilename);

		std::cout << _ ("Job submitted with jobid")
				  << ": " << jobid << std::endl;

		if (!dryrun) {
			if (jli->status != "FINISHED" && jli->status != "FAILED" &&
				jli->status != "KILLED" && jli->status != "DELETED") {
				notify(INFO) << _("Killing job")
				                << ": " << jli->id << std::endl;

				jctrl.Cancel(jli->id, timeout);
			}

			if (!keep) {
				notify(INFO) << _("Deleting job")
				                << ": " << jli->id << std::endl;

				jctrl.Clean(jli->id, timeout);
				RemoveJobID (jli->id);
			}	
		}
	}
}
