#include <iostream>
#include <list>
#include <string>
#include <algorithm>

#include "arccli.h"

#include <arc/certificate.h>
#include <arc/common.h>
#include <arc/jobftpcontrol.h>
#include <arc/joblist.h>
#include <arc/mdsquery.h>
#include <arc/notify.h>
#include <arc/resource.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arccli", (A))
#else
#define _(A) (A)
#endif


void arcrenew (const std::list<std::string>& jobs,
               const std::list<std::string>& clusterselect,
               const std::list<std::string>& clusterreject,
               const std::list<std::string>& status,
               const int timeout,
               const bool anonymous) {

	std::list<std::string> jobids =
		GetJobIDsList(jobs, clusterselect, clusterreject);

	if (jobs.empty() && clusterselect.empty() && jobids.empty())
		throw ARCCLIError(_("No jobs"));

	if (jobids.empty())
		throw ARCCLIError(_("No valid jobnames/jobids given"));

	std::list<Job> joblist =
		GetJobInfo(jobids, MDS_FILTER_JOBINFO, anonymous, "", timeout);

	std::list<Queue> queues;
	JobFTPControl ctrl;

	joblist.sort(SortByHost());

	int jobs_processed = 0;
	int jobs_renewed = 0;
	for (std::list<Job>::iterator jli = joblist.begin();
	     jli != joblist.end(); jli++) {

		jobs_processed++;

		if (jli->status.empty()) {
			notify(WARNING) << _("Job information not found")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (!status.empty() &&
		    std::find(status.begin(), status.end(), jli->status) == status.end())
			continue;

		if (jli->status == "DELETED") {
			notify(WARNING) << _("Job has already been deleted")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (jli->status == "FINISHED" || jli->status == "KILLED") {
			notify(WARNING) << _("Job has already finished")
			                << ": " << jli->id << std::endl;
			continue;
		}

		if (jli->proxy_expire_time != UNDEFINED &&
			jli->proxy_expire_time >= Certificate(PROXY).Expires()) {
			notify(WARNING) << _("Remote proxy not older than local")
			                << ": " << jli->id << std::endl;
			continue;
		}

		notify(INFO) << _("Renewing credentials")
		             << ": " << jli->id << std::endl;

		try {
			ctrl.RenewCreds(jli->id, timeout, false);
			jobs_renewed++;
		} catch(FTPControlError e) {
			std::string queuestatus;

			if(WARNING <= GetNotifyLevel())
				queuestatus = QueueStatusByJob(*jli, queues, anonymous, timeout);

			if (queuestatus.find("gridftp is down") == std::string::npos) {
				notify(WARNING) << e.what() << std::endl;;
			} else {
				notify(WARNING) << _("Gridftp server of job is down at the "
				                     "moment") << ": " << jli->id << std::endl;
			}
 		}
 
	}

	notify(WARNING) << _("Jobs processed") << ": " << jobs_processed << ", "
	                << _("successfuly renewed credentials") << ": " << jobs_renewed
	                << std::endl;
}
