#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <regex.h>

#include <algorithm>
#include <fstream>
#include <list>
#include <map>
#include <string>

#include <arc/common.h>
#include <arc/joblist.h>
#include <arc/notify.h>
#include <arc/url.h>
#include <arc/datetime.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


std::multimap<std::string, std::string> GetJobIDs(
                                const std::list<std::string>& jobs,
                                const std::list<std::string>& clusterselect,
                                const std::list<std::string>& clusterreject) {

	std::multimap<std::string, std::string> jobids;

	std::string filename = GetEnv("HOME");
	filename.append("/.ngjobs");

	LockFile(filename);

	std::ifstream ngjobs(filename.c_str());

	if (!ngjobs) {
		notify(WARNING) << _("Local job list file can not be read")
		                << std::endl;
	}

	if (jobs.empty() || !clusterselect.empty()) {
		std::string jobidname;
		while (getline(ngjobs, jobidname)) {
			size_t pos = jobidname.find('#');
			std::string jobid = jobidname.substr(0, pos);
			std::string jobname = jobidname.substr(pos + 1);
			if (!clusterselect.empty()) {
				URL joburl(jobid);
				std::string clustername = joburl.Host();
				if (find(clusterselect.begin(), clusterselect.end(),
				         clustername) == clusterselect.end())
					continue;
			}
			jobids.insert(std::make_pair(jobname, jobid));
		}
	}

	for (std::list<std::string>::const_iterator vsci = jobs.begin();
	     vsci != jobs.end(); vsci++) {
		bool found = false;
		ngjobs.clear();
		ngjobs.seekg(std::ios::beg);
		std::string jobidname;
		while (getline(ngjobs, jobidname)) {
			int pos = jobidname.find('#');
			std::string jobid = jobidname.substr(0, pos);
			std::string jobname = jobidname.substr(pos + 1);
			bool regmatch = false;
			if (vsci->find_first_of("*?") != std::string::npos) {
				std::string reg = '^' + *vsci + '$';
				std::string::size_type pos = 0;
				while ((pos = reg.find('.', pos)) != std::string::npos) {
					reg.insert(pos, "\\");
					pos += 2;
				}
				pos = 0;
				while ((pos = reg.find('*', pos)) != std::string::npos) {
					reg.insert(pos, ".");
					pos += 2;
				}
				pos = 0;
				while ((pos = reg.find('?', pos)) != std::string::npos) {
					reg.replace(pos, 1, ".");
					pos++;
				}
				regex_t regx;
				regcomp(&regx, reg.c_str(), 0);
				if (regexec(&regx, jobid.c_str(), 0, NULL, 0) == 0 ||
					regexec(&regx, jobname.c_str(), 0, NULL, 0) == 0)
					regmatch = true;
				regfree(&regx);
			}
			if (regmatch || jobid == *vsci || jobname == *vsci) {
				jobids.insert(std::make_pair(jobname, jobid));
				found = true;
			}
		}
		if (!found) {
			jobids.insert(std::make_pair(std::string(), *vsci));
		}
	}

	ngjobs.close();

	UnlockFile(filename);

	if (!clusterreject.empty()) {
		std::multimap<std::string, std::string> filteredjobids;
		for (std::multimap<std::string, std::string>::iterator vsi =
		    jobids.begin(); vsi != jobids.end(); vsi++) {
			URL joburl(vsi->second);
			std::string clustername = joburl.Host();
			if (find(clusterreject.begin(), clusterreject.end(),
			         clustername) == clusterreject.end())
				filteredjobids.insert(std::make_pair(vsi->first, vsi->second));
		}
		jobids = filteredjobids;
	}

	return jobids;
}


std::list<std::string> GetJobIDsList(
                                const std::list<std::string>& jobs,
                                const std::list<std::string>& clusterselect,
                                const std::list<std::string>& clusterreject) {

	std::multimap<std::string, std::string> jobids =
		GetJobIDs(jobs, clusterselect, clusterreject);

	std::list<std::string> jobidslist;

	for (std::multimap<std::string, std::string>::iterator i = jobids.begin();
	     i != jobids.end(); i++)
		jobidslist.push_back(i->second);

	return jobidslist;
}


void AddJobID(const std::string& jobid, const std::string& jobname) {

	std::string filename = GetEnv("HOME");
	filename.append("/.ngjobs");

	LockFile(filename);

	std::ifstream oldngjobs(filename.c_str());
	std::string newfilename = filename + ".tmp";
	std::ofstream newngjobs(newfilename.c_str());
	std::string jobidname;
	bool written = false;
	while (getline (oldngjobs, jobidname)) {
		int pos = jobidname.find('#');
		std::string jobname_ = jobidname.substr (pos + 1);
		if (!written && jobname_ > jobname) {
			newngjobs << jobid << '#' << jobname << std::endl;
			written = true;
		}
		newngjobs << jobidname << std::endl;
	}
	if (!written) newngjobs << jobid << '#' << jobname << std::endl;
	oldngjobs.close();
	newngjobs.close();
	remove (filename.c_str());
	rename (newfilename.c_str(), filename.c_str());

	UnlockFile(filename);
}


void RemoveJobID(const std::string& jobid) {

	std::string filename = GetEnv("HOME");
	filename.append("/.ngjobs");

	LockFile(filename);

	std::ifstream oldngjobs(filename.c_str());
	std::string newfilename = filename + ".tmp";
	std::ofstream newngjobs(newfilename.c_str());
	std::string jobidname;
	while (getline(oldngjobs, jobidname)) {
		int pos = jobidname.find('#');
		std::string jobid_ = jobidname.substr(0, pos);
		if (jobid_ != jobid) newngjobs << jobidname << std::endl;
	}
	oldngjobs.close();
	newngjobs.close();
	remove(filename.c_str());
	rename(newfilename.c_str(), filename.c_str());

	UnlockFile(filename);
}


std::map<std::string, Time> GetJobHistory() {

	std::string filename = GetEnv("HOME");
	filename.append("/.arc/history");

	std::map<std::string, Time> jobhistory;

	std::list<std::string> lines = ReadFile(filename);
	std::list<std::string>::iterator line;
	for (line = lines.begin(); line != lines.end(); line++) {

		std::string::size_type pos = line->find("gsiftp");
		std::string jobid = line->substr(pos);
		std::string timestr = line->substr(0, pos-2);

		try {
			Time time(timestr);
			jobhistory[jobid] = time;
		} catch(TimeError& e) {
			notify(DEBUG) << "Illegal time-specification in .nghistory file "
			                 "for job " << jobid << std::endl;
			continue;
		}
	}

	return jobhistory;
}
