#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <arc/jobftpcontrol.h>
#include <arc/stringconv.h>
#include <arc/notify.h>
#include <arc/common.h>

#include <globus_ftp_control.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


JobFTPControl::JobFTPControl() throw(FTPControlError) : FTPControl() { }


JobFTPControl::~JobFTPControl() { }


std::string JobFTPControl::Submit(const URL& url,
                                  const std::string& rsl,
                                  int timeout,
                                  bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	// dump rsl into temporary file
	TmpFile file("rsl");
	int fd = file.Open();
	if (fd==-1)
		throw JobFTPControlError(_("Could not create temporary file"));
	write(fd, (void*)rsl.c_str(), rsl.size());
	file.Close();

	std::string resp;

	try {
		Connect(url, timeout);
		resp = SendCommand("CWD " + url.Path(), timeout);
		resp = SendCommand("CWD new", timeout);
	} catch(FTPControlError e) {
		throw JobFTPControlError(e.what());
	}

	std::string::size_type pos;
	if ((pos = resp.find('"')) == std::string::npos)
		throw JobFTPControlError(_("Could not parse server response"));
	resp = resp.substr(pos+1);

	if ((pos = resp.find('"')) == std::string::npos)
		throw JobFTPControlError(_("Could not parse server response"));
	jobid = resp.substr(0, pos);

	pos = jobid.rfind('/');
	if (pos == std::string::npos)
		throw JobFTPControlError(_("Invalid jobid returned by server"));
	jobid = jobid.substr(pos+1);

	std::string uploadurl = url.str();
	if (uploadurl[uploadurl.length()-1] == '/') // remove trailing slash
		uploadurl.resize(uploadurl.length()-1);

	Upload(file.Name(), uploadurl + "/new/job", timeout, false);
	if (disconnectafteruse) Disconnect(url, timeout);

	file.Destroy();

	return uploadurl + "/" + jobid;
}


std::string JobFTPControl::Submit(const URL& url,
                                  const std::string& rsl,
                                  std::multimap<std::string, std::string>& files,
                                  int timeout,
                                  bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	Connect(url, timeout);

	std::string uploadurl = url.str();
	if (uploadurl[uploadurl.length()-1] == '/') // remove trailing slash
		uploadurl.resize(uploadurl.length()-1);

	// Submit RSL and obtain ID
	Submit(URL(uploadurl), rsl, timeout, false);

	// Submit local files
	std::multimap<std::string, std::string>::iterator fileit;
	for (fileit = files.begin(); fileit != files.end(); fileit++) {

		notify(DEBUG) << _("Uploading local file from") << " " << fileit->first
		              << " " << _("to") << " " << fileit->second <<std::endl;

		for (unsigned int tries = 1; tries<=3; tries++) {
			try {
				Upload(fileit->first,
				       uploadurl + "/" + jobid + "/" + fileit->second,
				       timeout,
				       false);
				break;
			} catch(FTPControlError e) {
				notify(INFO) << e.what() << std::endl;

				if (tries==3) { // out of tries
					throw JobFTPControlError(_("Failed to upload input file")
					                       + (" " + fileit->second) + ": " + e.what());
				}

				// Reconnection often helps
				try {
					Disconnect(url, timeout);
					Connect(url, timeout);
				} catch(FTPControlError) { }
			}
		}
	}

	if (disconnectafteruse) Disconnect(url, timeout);

	return uploadurl + "/" + jobid;
}


void JobFTPControl::Cancel(const std::string& jobid,
                           int timeout,
                           bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	URL url(jobid);
	std::string path = url.Path();

	std::string::size_type pos = path.rfind('/');
	if (pos==std::string::npos || pos==0)
		throw JobFTPControlError(_("Illegal jobid specified"));

	std::string jobpath = path.substr(0, pos);
	std::string jobidnum = path.substr(pos+1);

	Connect(url, timeout);
	SendCommand("CWD " + jobpath, timeout);
	SendCommand("DELE " + jobidnum, timeout);
	if (disconnectafteruse) Disconnect(url, timeout);
}


void JobFTPControl::Clean(const std::string& jobid,
                          int timeout,
                          bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	URL url(jobid);
	std::string path = url.Path();

	std::string::size_type pos = path.rfind('/');
	if (pos==std::string::npos || pos==0)
		throw JobFTPControlError(_("Illegal jobid specified"));

	std::string jobpath = path.substr(0, pos);
	std::string jobidnum = path.substr(pos+1);

	Connect(url, timeout);
	SendCommand("CWD " + jobpath, timeout);
	SendCommand("RMD " + jobidnum, timeout);
	if (disconnectafteruse) Disconnect(url, timeout);
}


void JobFTPControl::Resume(const std::string& jobid,
                           int timeout,
                           bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	URL url(jobid);
	std::string urlstr = url.str();

	std::string::size_type pos = urlstr.rfind('/');
	if (pos==std::string::npos || pos==0)
		throw JobFTPControlError(_("Illegal jobid specified"));
	std::string jobnr = urlstr.substr(pos+1);
	urlstr = urlstr.substr(0, pos) + "/new/action";

	std::string rsl("&(action=restart)(jobid=" + jobnr + ")");

	// dump rsl into temporary file
	TmpFile file("rsl");
	int fd = file.Open();
	if (fd==-1)
		throw JobFTPControlError(_("Could not create temporary file"));
	write(fd, (void*)rsl.c_str(), rsl.size());
	file.Close();

	try {
		Connect(url, timeout);
		Upload(file.Name(), urlstr, timeout, false);
		if (disconnectafteruse) Disconnect(url, timeout);
	} catch(FTPControlError e) {
		throw;
	}

	file.Destroy();
}


void JobFTPControl::RenewCreds(const std::string& jobid,
                               int timeout,
                               bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	URL url(jobid);
	std::string path = url.Path();

	std::string::size_type pos = path.rfind('/');
	if (pos==std::string::npos || pos==0)
		throw JobFTPControlError(_("Illegal jobid specified"));

	std::string jobpath = path.substr(0, pos);
	std::string jobidnum = path.substr(pos+1);

	Connect(url, timeout);
	SendCommand("CWD " + jobpath, timeout);
	SendCommand("CWD " + jobidnum, timeout);
	if (disconnectafteruse) Disconnect(url, timeout);
}


Job JobFTPControl::GetJobInfo(const std::string& jobid,
                              int timeout,
                              bool disconnectafteruse)
throw(JobFTPControlError, FTPControlError) {

	std::string infourl = jobid;
	std::string::size_type pos = infourl.rfind('/');
	if (pos==std::string::npos || pos==0)
		throw JobFTPControlError(_("Illegal jobid specified"));
	infourl.insert(pos,"/info");

	Job job;
	job.id = jobid;
	// job.job.owner =   etc

	// temporary file for job's information
	TmpFile file("info");
	int fd = file.Open();
	if (fd==-1) throw JobFTPControlError(_("Could not create temporary file"));
	file.Close();

	URL url(infourl + "/status");
	try {
		Connect(url, timeout);
		unlink(file.Name().c_str());
		Download(url, file.Name(), timeout, false);
   	} catch(FTPControlError e) {
		Disconnect(url, timeout);
		throw;
	}

	std::list<std::string> lines = ReadFile(file.Name());
	if (lines.size() != 1)
		throw JobFTPControlError(_("Job status looks wrong"));
		
	std::string status = *(lines.begin());
	pos = status.find(':');
	if (status == "PENDING:ACCEPTED") {
		status = "PREPARED";
	} else if (status == "PENDING:INLRMS") {
		status = "EXECUTED";
	} else if (status == "SUBMIT") {
		status = "SUBMITTING";
	} else if (status == "CANCELING") {
		status = "KILLING";
	}

	job.status = status;

	std::list<std::string>::iterator line;
	job.exitcode = -1;
	if (job.status=="FINISHED") {
		URL diagurl(infourl + "/diag");
		try {
			Connect(diagurl, timeout);
			unlink(file.Name().c_str());
			Download(diagurl, file.Name(), timeout, false);

			std::list<std::string> lines = ReadFile(file.Name());
			for (line = lines.begin(); line != lines.end(); line++) {
				if (line->substr(0,9)=="exitcode=") {
					try {
						job.exitcode = stringtoi(line->substr(9));
					} catch(StringConvError e) { }
					break;
				}
			}
		} catch(FTPControlError e) {
			Disconnect(diagurl, timeout);
		}

		URL failurl(infourl + "/failed");
		try {
			Connect(failurl, timeout);
			unlink(file.Name().c_str());
			Download(failurl, file.Name(), timeout, false);
			job.status = "FAILED";

			std::list<std::string> lines = ReadFile(file.Name());
			for (line = lines.begin(); line!=lines.end(); line++)
				job.errors += *line;

		} catch(FTPControlError e) {
			Disconnect(failurl, timeout);
		}
	}

	if (disconnectafteruse) Disconnect(URL(infourl), timeout);
	file.Destroy();

	return job;
}


bool JobFTPControl::operator==(const URL& url) {
	if(!isconnected) return false;
	if(connected_url.Port() != url.Port()) return false;
	if(connected_url.Host() != url.Host()) return false;
	if(connected_url.Protocol() != url.Protocol()) return false;
	return true;
}


bool JobFTPControl::Connected(void) {
	return isconnected;
}

void CancelJob(const std::string& jobid)
throw(JobFTPControlError, FTPControlError) {
	JobFTPControl ftpc;
	ftpc.Cancel(jobid);
}


void CleanJob(const std::string& jobid)
throw(JobFTPControlError, FTPControlError) {
	JobFTPControl ftpc;
	ftpc.Clean(jobid);
}


void ResumeJob(const std::string& jobid)
throw(JobFTPControlError, FTPControlError) {
	JobFTPControl ftpc;
	ftpc.Resume(jobid);
}


void RenewCreds(const std::string& jobid)
throw(JobFTPControlError, FTPControlError) {
	JobFTPControl ftpc;
	ftpc.RenewCreds(jobid);
}


Job GetJobInfoDirect(std::string jobid)
throw(JobFTPControlError, FTPControlError) {
	JobFTPControl ftpc;
	return ftpc.GetJobInfo(jobid);
}


std::list<Job> GetJobInfoDirect(std::list<std::string> jobids) 
throw(JobFTPControlError, FTPControlError) {
	std::list<Job> jobs;
	std::list<std::string>::iterator it;	
// TODO: parallel queries
	for (it = jobids.begin(); it != jobids.end(); it++) {
		Job jobinfo = GetJobInfoDirect(*it);
		jobs.push_back(jobinfo);
	}

	return jobs;
}


