#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <fstream>
#include <iostream>

#include <arc/common.h>
#include <arc/ftpcontrol.h>
#include <arc/notify.h>
#include <arc/stringconv.h>

#include <globus_ftp_control.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


FTPControl::FTPControl() throw(FTPControlError) : isconnected(false) {

	control_handle = (globus_ftp_control_handle_t*)
	                              malloc(sizeof(globus_ftp_control_handle_t));
	if (control_handle == NULL) 
		throw FTPControlError(
		    _("Failed to allocate globus ftp control handle"));

	if (globus_ftp_control_handle_init(control_handle) != GLOBUS_SUCCESS)
		throw FTPControlError(
		    _("Failed to initialize globus ftp control handle"));
}


FTPControl::~FTPControl() {
	try {
		Disconnect();

		if (globus_ftp_control_handle_destroy(control_handle)==GLOBUS_SUCCESS) {
			free(control_handle);
		} else {
			notify(VERBOSE) << _("Could not destroy control handle. "
			                     "Leaking it.") << std::endl;
		}
	} catch (FTPControlError e) {
		notify(DEBUG) << _("Failed to close ftp-connection. Leaking "
		                   "globus_ftp_control handle.") << std::endl;
	}
}


void FTPControl::Upload(const std::string& localfile,
                        const URL& url,
                        int timeout,
                        bool disconnectafteruse) throw(FTPControlError) {

	if (url.Protocol() != "gsiftp")
		throw FTPControlError(_("Bad url passed to FTPControl"));

	Connect(url, timeout);

	int fd = open(localfile.c_str(), O_RDONLY);
	if (fd == -1)
		throw FTPControlError(_("File does not exist") + (": " + localfile));

	notify(VERBOSE) << _("Opened file for reading") << ": " << localfile
	                << std::endl;

	SetupReadWriteOperation(timeout);
	SendCommand("STOR " + url.Path(), timeout);

	data_resp = false;
	control_resp = false;
	if (globus_ftp_control_data_connect_write(control_handle,
	                                          DataConnectCallback,
	                                          (void*)this) != GLOBUS_SUCCESS)
		throw FTPControlError(
			_("Failed to create data connection for writing"));

	WaitForCallback(timeout);
	if (!data_resp) {
		close(fd);
		throw FTPControlError(
		    std::string(_("Unexpected response from server")) + ": "
		                + server_resp);
	}

	notify(DEBUG) << _("Uploading file") << ": " << localfile << std::endl;

	const unsigned int maxsize = 65536;
	char filebuffer[maxsize];

	bool eof = GLOBUS_FALSE;
	unsigned long long offset = 0;
	int len = 0;

	do {
		len = read(fd, filebuffer, maxsize);
		if (len == -1) {
			close(fd);
			throw FTPControlError(_("Error reading local file during upload"));
		}

		notify(VERBOSE) << _("Read buffer-length") << ": " << len << std::endl;

		if (len == 0) eof = GLOBUS_TRUE;
		data_resp = false;

		if (globus_ftp_control_data_write(control_handle,
		                                  (globus_byte_t*)filebuffer,
		                                  len,
										  offset,
		                                  eof,
		                                  &DataReadWriteCallback,
		                                  this) != GLOBUS_SUCCESS) {
			close(fd);
			throw FTPControlError(
				_("Failed writing data to data connection"));
		}

		do {
			WaitForCallback(timeout);
		} while (!data_resp); // Wait for data event
		
		offset += len;
	} while (len != 0);

	close(fd);

	/* Globus sometimes comes with spontaneous callbacks
	   on the data channel. So let's wait for the real callback
	   on the control channel */
	while (!control_resp) WaitForCallback(timeout);

	if (disconnectafteruse) Disconnect(url, timeout);

	notify(INFO) << _("File uploaded") << ": " << localfile << std::endl;
}


void FTPControl::Download(const URL& url,
                          const std::string& localfile,
                          int timeout,
                          bool disconnectafteruse) throw(FTPControlError) {
	Download(url, 0, (size_t)(-1), localfile, timeout, disconnectafteruse);
}

void FTPControl::Download(const URL& url,
                          size_t offset,
                          size_t length,
                          const std::string& localfile,
                          int timeout,
                          bool disconnectafteruse) throw(FTPControlError) {

	if (url.Protocol() != "gsiftp")
		throw FTPControlError(_("Bad url passed to FTPControl"));

	std::string file(localfile);
	if (file.empty()) {
		std::string::size_type p = url.str().rfind("/");
		file = url.str().substr(p+1);
	}

	int fd = open(file.c_str(),
	              O_RDWR | O_CREAT,
	              S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	if (fd == -1)
		throw FTPControlError(_("File could not be created") + (": " + file));

	notify(VERBOSE) << _("Opened file for writing") << ": " << localfile
	                << std::endl;

	try {

		Connect(url, timeout);
		SetupReadWriteOperation(timeout);
		if (offset > 0)
			SendCommand("REST " + tostring<unsigned long long>(offset), timeout);

		SendCommand("RETR " + url.Path(), timeout);

		data_resp = false;
		control_resp = false;
		if (globus_ftp_control_data_connect_read(control_handle,
		                                         DataConnectCallback,
		                                         (void*)this) != GLOBUS_SUCCESS) {
			throw FTPControlError(
				_("Failed to create data connection for reading"));
		}

		WaitForCallback(timeout);
		if (!data_resp) {
			/* On very fast connections response over control channel
			   may come before any of data callbacks are called by Globus.
			   So wait for data callback now */
			WaitForCallback(timeout);
		}
		if (!data_resp) {
			throw FTPControlError(
				std::string(_("Unexpected response from server")) + ": "
			                + server_resp);
		}

		notify(DEBUG) << _("Downloading file") << ": " << url << std::endl;

		const unsigned int maxsize = 65536;
		char filebuffer[maxsize];
		unsigned long long filesize = 0;

		eof = GLOBUS_FALSE;

		do {
			if (length != (size_t)(-1) && filesize >= length) {
				AbortOperation();
				break;
			}
			data_resp = false;
			buffer_length = 0;
			/* reset buffer_length so that the same chunk is not written
			   twice. globus_ftp_control can apparantly call the same
			   callback more than once (stupid) */
			if (globus_ftp_control_data_read(control_handle,
			                                 (globus_byte_t*)filebuffer,
			                                 maxsize,
			                                 &DataReadWriteCallback,
			                                 this) != GLOBUS_SUCCESS) {
				throw FTPControlError(
					_("Failed reading data from data connection"));
			}

			do {
				WaitForCallback(timeout);
			} while (!data_resp); // Wait for data event

			if (buffer_length > 0) {
				filesize += buffer_length;
				if (length != (size_t)(-1) && filesize > length) {
					buffer_length -= (filesize - length);
					filesize = length;
				}
				int len = write(fd, filebuffer, buffer_length);
				if (len == -1) {
					throw FTPControlError(
						_("Error writing local file during download"));
				}

				notify(VERBOSE) << _("Wrote buffer - length") << ": " << len
				                << std::endl;

			}

		} while (eof != GLOBUS_TRUE);

	} catch (ARCLibError& e) {
		close(fd);
		throw;
	}

	close(fd);

	/* Globus sometimes comes with spontaneous callbacks
	   on the data channel. So let's wait for the real callback
	   on the control channel */
	while (!control_resp) WaitForCallback(timeout);

	if (disconnectafteruse) Disconnect(url, timeout);

	notify(INFO) << _("File downloaded") << ": " << url << std::endl;
}


std::list<FileInfo> FTPControl::ListDir(const URL& url,
                                        int timeout,
                                        bool disconnectafteruse)
throw(FTPControlError) {

	if (url.Protocol() != "gsiftp")
		throw FTPControlError(_("Bad url passed to FTPControl"));

	Connect(url, timeout);

	SetupReadWriteOperation(timeout);
	if (!url.Path().empty())
		SendCommand("MLSD " + url.Path(), timeout);
	else
		SendCommand("MLSD", timeout);

	data_resp = false;
	control_resp = false;
	if (globus_ftp_control_data_connect_read(control_handle,
	                                         DataConnectCallback,
	                                         (void*)this) != GLOBUS_SUCCESS) {
		throw FTPControlError(
			_("Failed to create data connection for reading"));
	}

	WaitForCallback(timeout);
	if (!data_resp) {
		/* On very fast connections response over control channel
		   may come before any of data callbacks are called by Globus.
		   So wait for data callback now */
		WaitForCallback(timeout);
	}
	if(!data_resp) 
		throw FTPControlError(
		    std::string(_("Unexpected response from server")) + ": "
		                + server_resp);

	std::string dirlisting;

	const unsigned int maxsize = 65536;
	char filebuffer[maxsize+1];
	filebuffer[maxsize] = 0;
	eof = GLOBUS_FALSE;

	do {
		data_resp = false;
		buffer_length = 0;
		if (globus_ftp_control_data_read(control_handle,
		                                 (globus_byte_t*)filebuffer,
		                                 maxsize,
		                                 &DataReadWriteCallback,
		                                 this) != GLOBUS_SUCCESS) {
			throw FTPControlError(_("Failed reading data from data connection"));
		}

		do {
			WaitForCallback(timeout);
		} while (!data_resp); // Wait for data event

		if (buffer_length > 0) {
			if (buffer_length<maxsize) filebuffer[buffer_length] = 0;
			dirlisting.append(filebuffer);
		}
	} while (eof != GLOBUS_TRUE);

	/* Globus sometimes comes with spontaneous callbacks
	   on the data channel. So let's wait for the real callback
	   on the control channel */
	while (!control_resp) WaitForCallback(timeout);

	notify(VERBOSE) << _("Directory listing") << ": " << std::endl
	                << dirlisting << std::endl;

	std::list<FileInfo> dirlist;
	std::string::size_type pos = 0;
	std::string::size_type pos2 = 0;

	while ((pos = dirlisting.find("\r\n", pos2)) != std::string::npos) {
		std::string fileinfo = dirlisting.substr(pos2, pos-pos2);

		FileInfo info;
		std::string::size_type p = fileinfo.find(" ");
		info.filename = url.Path() + "/" + fileinfo.substr(p+1);

		info.isdir = false;
		p = fileinfo.find("type=");
		if (fileinfo.substr(p+5, 3)=="dir") info.isdir = true;

		p = fileinfo.find("size=");
		std::string::size_type q = fileinfo.find(";", p+1);
		info.size = stringtoull(fileinfo.substr(p+5, q-p-5));	

		pos2 = pos + 2;

		dirlist.push_back(info);
	}

	if (disconnectafteruse) Disconnect(url, timeout);
	return dirlist;
}


std::list<FileInfo> FTPControl::RecursiveListDir(const URL& url,
                                                 int timeout,
                                                 bool disconnectafteruse)
throw(FTPControlError) {

	if (url.Protocol() != "gsiftp")
		throw FTPControlError(_("Bad url passed to FTPControl"));

	Connect(url, timeout);
	std::list<FileInfo> files = ListDir(url, timeout, false);

	std::string ustr = url.Protocol() + "://" + url.Host();
	if (url.Port() > 0) ustr += ":" + tostring(url.Port());

	std::list<FileInfo>::iterator it = files.begin();
	while (it != files.end()) {
		if (it->isdir) {
			URL newurl(ustr + it->filename);
			std::list<FileInfo> more = ListDir(newurl, timeout, false);
			files.insert(files.end(), more.begin(), more.end());
		}
		it++;
	}

	if (disconnectafteruse) Disconnect(url, timeout);
	return files;
}


void FTPControl::DownloadDirectory(const URL& url,
                                   const std::string& localdir,
                                   int timeout,
                                   bool disconnectafteruse)
throw(FTPControlError) {

	mode_t dirmode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
	
	std::list<FileInfo> allfiles = RecursiveListDir(url, timeout, false);

	std::list<FileInfo>::iterator it;
	for (it = allfiles.begin(); it != allfiles.end(); it++) {
		if (!it->isdir) continue;

		std::string path = it->filename;
		// remove leading path in dirname
		path = path.substr(url.Path().size()+1);
		if (!localdir.empty()) path = localdir + "/" + path;

		int err = mkdir(path.c_str(), dirmode);
		if (err==-1)
			throw FTPControlError(_("Could not create the necessary directory "
			                        "structure for downloading the files"));
	}

	std::string newstr = url.Protocol() + "://" + url.Host();
	if (url.Port() > 0) newstr += ":" + tostring(url.Port());

	for (it = allfiles.begin(); it != allfiles.end(); it++) {
		if (it->isdir) continue;

		std::string localfilename = it->filename;
		// remove leading path in filename
		localfilename = localfilename.substr(url.Path().size()+1);
		if (!localdir.empty()) localfilename = localdir + "/" + localfilename;

		URL newurl(newstr + it->filename);
		Download(newurl, localfilename, timeout, false);
	}

	if (disconnectafteruse) Disconnect(url, timeout);
}


unsigned long long FTPControl::Size(const URL& url,
                                    int timeout,
                                    bool disconnectafteruse)
throw(FTPControlError) {

	if (url.Protocol() != "gsiftp")
		throw FTPControlError(_("Bad url passed to FTPControl"));

	Connect(url, timeout);
	std::string resp = SendCommand("SIZE " + url.Path());
	if (disconnectafteruse) Disconnect(url, timeout);

	notify(DEBUG) << resp << std::endl;
	if (resp.empty())
		throw FTPControlError(_("Server returned nothing"));

	return stringtoull(resp);
}


void FTPControl::Connect(const URL& url,
                         int timeout) throw(FTPControlError) {

	if (isconnected) {
		if ((url.Host() == connected_url.Host()) &&
		    (url.Port() == connected_url.Port())) return;
		try {
			Disconnect();
		} catch(FTPControlError e) { }
	};
	control_resp = false;

	notify(DEBUG) << _("Connecting to server") << ": " << url.Host()
	              << std::endl;
	if (globus_ftp_control_connect(control_handle,
	                               (char*)url.Host().c_str(),
	                               url.Port(),
	                               &FTPControlCallback,
	                               this) != GLOBUS_SUCCESS) {
		throw FTPControlError(
			_("Failed to connect to server") + (": " + url.Host()));
	}

	isconnected = true;
	try {
		while (!control_resp) WaitForCallback(timeout);
	} catch(FTPControlError e) {
		std::string errstr = errorstring;
		try {
			Disconnect(url, timeout);
		} catch(FTPControlError e) { }
		throw FTPControlError(
			_("Failed to connect to server") + (": " + errstr));
	}

	connected_url = url;

	notify(DEBUG) << _("Authenticating to server") << ": " << url.Host()
	              << std::endl;
	globus_ftp_control_auth_info_t auth;
	globus_ftp_control_auth_info_init(&auth,
	                                  GSS_C_NO_CREDENTIAL,
	                                  GLOBUS_TRUE,
	                                  "ftp",
	                                  "user@",
	                                  GLOBUS_NULL,
	                                  GLOBUS_NULL);

	// authenticate to server
	if (globus_ftp_control_authenticate(control_handle,
	                                    &auth,
	                                    GLOBUS_TRUE,
	                                    FTPControlCallback,
	                                    this) != GLOBUS_SUCCESS) {
		try {
			Disconnect(url, timeout);
		} catch(FTPControlError e) { }
		throw FTPControlError(
			_("Failed to authenticate to server") + (": " + url.Host()));
	}

	control_resp = false;
	try {
		while (!control_resp) WaitForCallback(timeout);
	} catch(FTPControlError e) {
		std::string errstr = errorstring;
		std::string response = server_resp;
		try {
			Disconnect(url, timeout);
		} catch(FTPControlError e) { }
		if(response.empty()) {
			throw FTPControlError(
				_("Failed to authenticate to server") + (": " + errstr));
		} else {
			throw FTPControlError(
				_("Failed to authorize at server") + (": " + errstr + ": ") +
				_("Server responded") + (": " + response));
		}
	}

	notify(DEBUG) << _("Connection established to") << ": " << url.Host()
	              << std::endl;
}


void FTPControl::Disconnect(int timeout) throw(FTPControlError) {

	if (!isconnected) return;
	
	Disconnect(connected_url, timeout);
}

void FTPControl::Disconnect(const URL& url,
                            int timeout) throw(FTPControlError) {

	if (!isconnected) return;

	notify(DEBUG) << _("Closing connection to") << ": " << url.Host()
	              << std::endl;

	std::string errstr = url.Host();
	control_resp = false;
	bool closed = false;
	if (globus_ftp_control_quit(control_handle,
	                            &FTPControlCallback,
	                            this)==GLOBUS_SUCCESS) {
		try {
			while (!control_resp) WaitForCallback(timeout);
			closed = true;
		} catch(FTPControlError e) {
			errstr = errorstring;
		}
	}

	control_resp = false;
	if (!closed) {
		// Ordinary close failed. Let's force close.
		notify(DEBUG) << _("Forcing closed connection to") << ": "
					  << url.Host() << std::endl;
		if (globus_ftp_control_force_close(control_handle,
		                                   &FTPControlCallback,
		                                   this) != GLOBUS_SUCCESS) {
			notify(DEBUG) << _("Failed forcing closed connection to") +
								  (": " + url.Host());
		}

		try {
			while (!control_resp) WaitForCallback(timeout);
			closed = true;
		} catch(FTPControlError e) {
			errstr = errorstring;
		}
	}

	isconnected = false; // It is useless to repeat disconnect
	if(!closed) 
		throw FTPControlError(
			_("Failed closing connection to server") + (": " + errstr));
	notify(DEBUG) << _("Connection closed to") << ": " << url.Host()
	              << std::endl;
}


std::string FTPControl::SendCommand(const std::string& command,
                                    int timeout) throw(FTPControlError) {

	control_resp = false;

	if (!command.empty()) {
		notify(VERBOSE) << _("Sending command") << ": " << command
		                << std::endl;
		std::string newcommand = command + "\r\n";

		if (globus_ftp_control_send_command(control_handle,
		                                    newcommand.c_str(),
		                                    FTPControlCallback,
		                                    this) != GLOBUS_SUCCESS) {
			throw FTPControlError(
				_("Sending command failed") + (": " + command));
		}
	}

	/* Globus sometimes comes with spontaneous callbacks
	   on the data channel. So let's wait for the real callback
	   on the control channel */
	while (!control_resp) WaitForCallback(timeout);

	return server_resp;
}


void FTPControl::SetupReadWriteOperation(int timeout) throw(FTPControlError) {

	SendCommand("DCAU N", timeout);
	SendCommand("TYPE I", timeout);

	std::string resp = SendCommand("PASV", timeout);
	std::string::size_type pos;
	if ((pos = resp.find('(')) == std::string::npos)
		throw FTPControlError(_("Could not parse server response"));
	resp = resp.substr(pos+1);
	
	if ((pos = resp.find(')')) == std::string::npos)
		throw FTPControlError(_("Could not parse server response"));
	resp = resp.substr(0, pos);
	
	globus_ftp_control_host_port_t passive_addr;
	passive_addr.port = 0;
	unsigned short port_low, port_high;

	if (sscanf(resp.c_str(),
			   "%i,%i,%i,%i,%hu,%hu",
			   &passive_addr.host[0],
			   &passive_addr.host[1],
			   &passive_addr.host[2],
			   &passive_addr.host[3],
			   &port_high,
			   &port_low)==6) {
		passive_addr.port = 256*port_high + port_low;
	}

	if (passive_addr.port == 0) {
		throw FTPControlError(
			_("Could not parse host and port in PASV response")
			+ (": " + resp));
	}

	if (globus_ftp_control_local_port(control_handle,
	                                  &passive_addr) != GLOBUS_SUCCESS) {
		throw FTPControlError(
			_("The received PASV host and address values are not acceptable")
			+ (": " + resp));
	}

	if (globus_ftp_control_local_type(control_handle,
	                                  GLOBUS_FTP_CONTROL_TYPE_IMAGE,
	                                  0) != GLOBUS_SUCCESS) {
		throw FTPControlError(_("Setting data type to IMAGE failed"));
	}
}


void FTPControl::WaitForCallback(int timeout, bool abort)
throw(FTPControlError) {

	notify(VERBOSE) << _("Waiting for callback") << "(" << _("timeout") << " "
	                << timeout << ")" << std::endl;

	bool valid;
	if (!cond.Wait(valid, 1000*timeout)) {
		// timeout -- abort operation and return error
		notify(DEBUG) << _("Timeout: Aborting operation") << std::endl;
		if (abort) AbortOperation();
		valid = false;
	}

	cond.Reset();
	if (!valid) {
		if (!errorstring.empty()) throw FTPControlError(errorstring);
		if (!server_resp.empty())
			throw FTPControlError(_("Server responded") + (": " + server_resp));
		throw FTPControlError(_("Unknown error"));
	}
}


void FTPControl::AbortOperation() {

	notify(VERBOSE) << _("Aborting operation") << std::endl;
	if (globus_ftp_control_abort(control_handle,
	                             FTPControlCallback,
	                             this) != GLOBUS_SUCCESS) {
		/* if the operation cannot be aborted here, probably the operation
		   already finished (race-condition?). We return an error
		   assuming no more callbacks will be called. */
		errorstring = _("Aborting operation failed");
		return;
	}

	/* Wait until AbortCallback is called (no timeout).
	   Pray it will be called! */
	WaitForCallback(20, false);
}


void FTPControl::DataReadWriteCallback(void* arg,
                                       globus_ftp_control_handle_t* handle,
                                       globus_object_t* error,
                                       globus_byte_t* buffer,
                                       globus_size_t length,
                                       globus_off_t offset,
                                       globus_bool_t eof) {

	notify(VERBOSE) << _("DataReadWriteCallback called") << std::endl;
	FTPControl* it = (FTPControl*)arg;

	/* Sometimes this callback is called spontaneously
	   with EOF and zero length. */
	if (eof == GLOBUS_TRUE) it->eof = GLOBUS_TRUE;
	if (length > 0) it->buffer_length = length;
	it->data_resp = true;

	FTPControlCallback(arg, handle, error, NULL);
}


void FTPControl::DataConnectCallback(void* arg,
                                     globus_ftp_control_handle_t* handle,
                                     unsigned int stripe_ndx,
                                     globus_bool_t reused,
                                     globus_object_t* error) {

	notify(VERBOSE) << _("DataControlCallback called") << std::endl;
	FTPControl* it = (FTPControl*)arg;
	it->data_resp = true;

	FTPControlCallback(arg, handle, error, NULL);
}


void FTPControl::FTPControlCallback(void* arg,
                                    globus_ftp_control_handle_t* handle,
                                    globus_object_t* error,
                                    globus_ftp_control_response_t* response) {

	notify(VERBOSE) << _("FTPControlCallback called") << std::endl;

	FTPControl* it = (FTPControl*)arg;
	it->server_resp.clear();
	globus_ftp_control_response_class_t responseclass =
		GLOBUS_FTP_POSITIVE_COMPLETION_REPLY;

	if (response) {
		it->server_resp.clear();
		it->control_resp = true;

		if (response->response_buffer) {
			responseclass = response->response_class;

			it->server_resp.assign((char*)response->response_buffer,
			                       response->response_length);
			if (it->server_resp[it->server_resp.size() - 1] == '\0')
				it->server_resp.resize(it->server_resp.size() - 1);

			std::string::size_type pos = 0;
			while ((pos = it->server_resp.find("\r\n",pos))!=std::string::npos)
				it->server_resp.erase(pos, 2);

			it->server_resp = it->server_resp.substr(4);
			notify(VERBOSE) << "Server-response: " << it->server_resp
			                << std::endl;
		}
	}

	if (error == GLOBUS_SUCCESS) { // normal callback
		/* If response class is GLOBUS_FTP_TRANSIENT_NEGATIVE_COMPLETION_REPLY
		   or GLOBUS_FTP_PERMANENT_NEGATIVE_COMPLETION_REPLY i.e. >= 4, we
		   have an error. */
		if (responseclass >= 4) {
			it->cond.Signal(false);
		} else {
			it->cond.Signal(true);
		}
	} else { // callback-error
		it->errorstring = GlobusErrorString(error);
		std::string::size_type pos = 0;
		while ((pos = it->errorstring.find("\r\n", pos)) != std::string::npos)
			it->errorstring.erase(pos, 2);

		if (it->errorstring.find("end-of-file") != std::string::npos)
			it->errorstring = _("Server unexpectedly closed connection");
		if (it->errorstring.find("GSS failure") != std::string::npos)
			it->errorstring = _("Problem with GSI credential");

		it->cond.Signal(false);
	}
}
