#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstdio>
#include <iomanip>
#include <sstream>

#include <arc/datetime.h>
#include <arc/notify.h>
#include <arc/stringconv.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#define __(A, B, C) dngettext("arclib", (A), (B), (C))
#else
#define _(A) (A)
#define __(A, B, C) ((C) == 1 ? (A) : (B))
#endif


#ifndef HAVE_TIMEGM
time_t timegm (struct tm *tm) {
	char *tz = getenv("TZ");
	setenv("TZ", "", 1);
	tzset();
	time_t ret = mktime(tm);
	if (tz)
		setenv("TZ", tz, 1);
	else
		unsetenv("TZ");
	tzset();
	return ret;
}
#endif


TimeFormat Time::time_format = UserTime;


Time::Time() {
	gtime = time(NULL);
}


Time::Time(time_t newtime) {
	gtime = newtime;
}


Time::Time(std::string timestring) {

	if (timestring.size()==15) {
		// might be MDSTime
		tm timestr;
		if (sscanf(timestring.c_str(),
		           "%4d%2d%2d%2d%2d%2dZ",
		           &timestr.tm_year,
		           &timestr.tm_mon,
		           &timestr.tm_mday,
		           &timestr.tm_hour,
		           &timestr.tm_min,
		           &timestr.tm_sec)!=6)
			throw TimeError("Illegal format for time-string.");

		timestr.tm_year -= 1900;
		timestr.tm_mon--;

		time_t newtime = timegm(&timestr);
		if (newtime==-1)
			throw TimeError("Cannot convert string to a time_t.");

		gtime = newtime;
		time_format = MDSTime;
		return;
	}

	if (timestring.size()==19) {
		// might be UserTime
		tm timestr;

		if (sscanf(timestring.c_str(),
		           "%4d-%2d-%2d %2d:%2d:%2d",
		           &timestr.tm_year,
		           &timestr.tm_mon,
		           &timestr.tm_mday,
		           &timestr.tm_hour,
		           &timestr.tm_min,
		           &timestr.tm_sec)==6) {
			notify(VERBOSE) << "Valid time-string of format "
			                << "YYYY-MM-DD HH:MM:SS" << std::endl;
		} else if (sscanf(timestring.c_str(),
						"%4d %2d %2d %2d:%2d:%2d",
						&timestr.tm_year,
						&timestr.tm_mon,
						&timestr.tm_mday,
						&timestr.tm_hour,
						&timestr.tm_min,
						&timestr.tm_sec)==6) {
			notify(VERBOSE) << "Valid time-string of format "
			                << "YYYY MM DD HH:MM:SS" << std::endl;
		} else if (sscanf(timestring.c_str(),
					   "%4d.%2d.%2d %2d:%2d:%2d",
					   &timestr.tm_year,
					   &timestr.tm_mon,
					   &timestr.tm_mday,
					   &timestr.tm_hour,
					   &timestr.tm_min,
					   &timestr.tm_sec)==6) {
			notify(VERBOSE) << "Valid time-string of format "
			                << "YYYY.MM.DD HH:MM:SS" << std::endl;
		} else {
			throw TimeError("Illegal format for time-string.");
		}

		timestr.tm_year -= 1900;
		timestr.tm_mon--;

		time_t newtime = mktime(&timestr);
		if (newtime==-1)
			throw TimeError("Cannot convert string to a time_t.");

		gtime = newtime;
		time_format = UserTime;
		return;
	}

	if (timestring.size()==24) {
		// might be ASCTime
		tm timestr;
		char day[4];
		char month[4];

		if (sscanf(timestring.c_str(),
		           "%3s %3s %2d %2d:%2d:%2d %4d",
		           day,
		           month,
		           &timestr.tm_mday,
		           &timestr.tm_hour,
		           &timestr.tm_min,
		           &timestr.tm_sec,
		           &timestr.tm_year)!=7)
			throw TimeError("Illegal format for time-string.");

		timestr.tm_year -= 1900;

		if (strncmp(month, "Jan", 3)==0) {
			timestr.tm_mon = 0;
		} else if (strncmp(month, "Feb", 3)==0) {
			timestr.tm_mon = 1;
		} else if (strncmp(month, "Mar", 3)==0) {
			timestr.tm_mon = 2;
		} else if (strncmp(month, "Apr", 3)==0) {
			timestr.tm_mon = 3;
		} else if (strncmp(month, "May", 3)==0) {
			timestr.tm_mon = 4;
		} else if (strncmp(month, "Jun", 3)==0) {
			timestr.tm_mon = 5;
		} else if (strncmp(month, "Jul", 3)==0) {
			timestr.tm_mon = 6;
		} else if (strncmp(month, "Aug", 3)==0) {
			timestr.tm_mon = 7;
		} else if (strncmp(month, "Sep", 3)==0) {
			timestr.tm_mon = 8;
		} else if (strncmp(month, "Oct", 3)==0) {
			timestr.tm_mon = 9;
		} else if (strncmp(month, "Nov", 3)==0) {
			timestr.tm_mon = 10;
		} else if (strncmp(month, "Dec", 3)==0) {
			timestr.tm_mon = 11;
		} else {
			throw TimeError("Illegal format for month given");
		}

		time_t newtime = mktime(&timestr);
		if (newtime==-1)
			throw TimeError("Cannot convert string to a time_t.");

		gtime = newtime;
		time_format = ASCTime;
		return;
	}

	throw TimeError("Illegal format for given string");
}


void Time::SetTime(time_t timet) {
	gtime = timet;
}


time_t Time::GetTime() const {
	return gtime;
}


void Time::SetFormat(const TimeFormat& format) {
	time_format = format;
}


TimeFormat Time::GetFormat() {
	return time_format;
}


std::string Time::str() const {

	switch (time_format) {

		case ASCTime: {

			char timestr[26];
			ctime_r(&gtime, timestr);
			// chop the trailing \n
			return std::string(timestr, 24);
		}

		case UserTime: {

			tm tmtime;
			localtime_r(&gtime, &tmtime);

			std::stringstream ss;

			ss << std::setfill('0');

			ss << std::setw(4) << tmtime.tm_year + 1900 << '-'
			   << std::setw(2) << tmtime.tm_mon + 1 << '-'
			   << std::setw(2) << tmtime.tm_mday << ' '
			   << std::setw(2) << tmtime.tm_hour << ':'
			   << std::setw(2) << tmtime.tm_min << ':'
			   << std::setw(2) << tmtime.tm_sec;

			return ss.str();
		}

		case MDSTime: {

			tm tmtime;
			gmtime_r(&gtime, &tmtime);

			std::stringstream ss;

			ss << std::setfill('0');

			ss << std::setw(4) << tmtime.tm_year + 1900
			   << std::setw(2) << tmtime.tm_mon + 1
			   << std::setw(2) << tmtime.tm_mday
			   << std::setw(2) << tmtime.tm_hour
			   << std::setw(2) << tmtime.tm_min
			   << std::setw(2) << tmtime.tm_sec << 'Z';

			return ss.str();
		}
	}
	return "";
}


bool Time::operator<(const Time& othertime) const {
	return gtime<othertime.GetTime();
}


bool Time::operator>(const Time& othertime) const {
	return gtime>othertime.GetTime();
}


bool Time::operator<=(const Time& othertime) const {
	return gtime<=othertime.GetTime();
}


bool Time::operator>=(const Time& othertime) const {
	return gtime>=othertime.GetTime();
}


bool Time::operator==(const Time& othertime) const {
	return gtime==othertime.GetTime();
}


bool Time::operator!=(const Time& othertime) const {
	return gtime!=othertime.GetTime();
}


Time& Time::operator=(time_t newtime) {
	gtime = newtime;
	return *this;
}


std::ostream& operator<<(std::ostream& out, Time& dt) {
	return ( out << dt.str() );
}


std::string TimeStamp(const TimeFormat& tmformat) {
	Time now;
	TimeFormat oldformat = Time::GetFormat();

	Time::SetFormat(tmformat);
	std::string timestring = now.str();
	Time::SetFormat(oldformat);

	return timestring;
}


std::string TimeStamp(Time newtime, const TimeFormat& tmformat) {
	TimeFormat oldformat = Time::GetFormat();

	Time::SetFormat(tmformat);
	std::string timestring = newtime.str();
	Time::SetFormat(oldformat);

	return timestring;
}


std::string Period(unsigned long seconds) {

	if (seconds == 0) {
		std::string str("0");
		return str;
	}

	int years = 0;
	int weeks = 0;
	int days = 0;
	int hours = 0;
	int minutes = 0;

	if (seconds >= 60*60*24*365) {
		years = seconds/(60*60*24*365);
		seconds = seconds - years*60*60*24*365;
	}
	if (seconds >= 60*60*24*7) {
		weeks = seconds/(60*60*24*7);
		seconds = seconds - weeks*60*60*24*7;
	}
	if (seconds >= 60*60*24) {
		days = seconds/(60*60*24);
		seconds = seconds - days*60*60*24;
	}
	if (seconds >= 60*60) {
		hours = seconds/(60*60);
		seconds = seconds - hours*60*60;
	}
	if (seconds >= 60) {
		minutes = seconds/60;
		seconds = seconds - minutes*60;
	}

	std::stringstream ss;
	bool first = true;

	if (years > 0) {
		if (!first) ss << ", ";
		first = false;
		ss << years << " " << __("year", "years", years);
	}
	if (weeks > 0) {
		if (!first) ss << ", ";
		first = false;
		ss << weeks << " " << __("week", "weeks", weeks);
	}
	if (days > 0) {
		if (!first) ss << ", ";
		first = false;
		ss << days << " " << __("day", "days", days);
	}
	if (hours > 0) {
		if (!first) ss << ", ";
		first = false;
		ss << hours << " " << __("hour", "hours", hours);
	}
	if (minutes > 0) {
		if (!first) ss << ", ";
		first = false;
		ss << minutes << " " << __("minute", "minutes", minutes);
	}
	if (seconds > 0) {
		if (!first) ss << ", ";
		first = false;
		ss << seconds << " " << __("second", "seconds", seconds);
	}

	return ss.str();
}


long Seconds(const std::string& period,enum PeriodBase base) throw(TimeError) {

	long seconds = 0;

	std::string::size_type pos = std::string::npos;
	int len = 0;

	try {
		for (unsigned int i=0; i!=period.length(); i++) {
			if (isdigit(period[i])) {
				if (pos==std::string::npos) {
					pos = i;
					len = 0;
				}
				len++;
			} else if (pos!=std::string::npos) {
				switch (period[i]) {
				case 'w':
				case 'W':
					seconds += stringtoi(period.substr(pos, len).c_str())*60*60*24*7;
					pos = std::string::npos;
					break;
				case 'd':
				case 'D':
					seconds += stringtoi(period.substr(pos, len).c_str())*60*60*24;
					pos = std::string::npos;
					break;
				case 'h':
				case 'H':
					seconds += stringtoi(period.substr(pos, len).c_str())*60*60;
					pos = std::string::npos;
					break;
				case 'm':
				case 'M':
					seconds += stringtoi(period.substr(pos, len).c_str())*60;
					pos = std::string::npos;
					break;
				case 's':
				case 'S':
					seconds += stringtoi(period.substr (pos, len).c_str());
					pos = std::string::npos;
					break;
				case ' ':
					break;
				default:
					throw TimeError("Invalid period-string passed");
					break;
				}
			}
		}

		if (pos!=std::string::npos) {
			long tmp = stringtoi(period.substr(pos, len).c_str());
			switch(base) {
				case PeriodSeconds: 
					seconds += tmp;
					break;
				case PeriodMinutes: 
					seconds += tmp*60;
					break;
				case PeriodHours: 
					seconds += tmp*(60*60);
					break;
				case PeriodDays: 
					seconds += tmp*(60*60*24);
					break;
				case PeriodWeeks: 
					seconds += tmp*(60*60*24*7);
					break;
			}
		}
	} catch(StringConvError e) { // for stringto*()
		throw TimeError(e.what());
	}

	return seconds;
}
