#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>

#include <cerrno>
#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <map>
#include <string>

#include <arc/common.h>
#include <arc/error.h>
#include <arc/notify.h>

#ifdef HAVE_LIBINTL_H
#include <libintl.h>
#define _(A) dgettext("arclib", (A))
#else
#define _(A) (A)
#endif


std::string GetEnv(const std::string& var) {

	char* env = getenv(var.c_str());
	return (env ? env : "");
}


std::string GlobusErrorString(globus_object_t* error) {

	std::string errstr;

	for (globus_object_t* err = error; err;
	     err = globus_error_base_get_cause(err)) {
		char* tmp = globus_object_printable_to_string(error);
		if (tmp) {
			if (!errstr.empty()) errstr += "; ";
			errstr += tmp;
			free(tmp);
		}
	}

	return errstr;
}


std::list<std::string> ReadFile(const std::string& filename) {

	static std::map<std::string, std::list<std::string> > filecache;

	if (filecache.find(filename) != filecache.end()) {
		notify(VERBOSE) << _("Using cached configuration") << ": "
		                << filename << std::endl;
		return filecache[filename];
	}

	notify(VERBOSE) << _("Reading configuration file") << ": " << filename
	                << std::endl;

	std::list<std::string> lines;
	std::string line;

	std::ifstream readin(filename.c_str());
	while(getline(readin, line)) {
		// Find line delimiters - remove external whitespaces
		std::string::size_type pos1 = line.find_first_not_of(" \t");
		std::string::size_type pos2 = line.find_last_not_of(" \t");

		// Drop comments and empty lines
		if (pos1 == std::string::npos || line[pos1] == '#')
			continue;

		lines.push_back(line.substr(pos1, pos2 - pos1 + 1));
	}

	readin.close();
	filecache[filename] = lines;
	return lines;
}


void LockFile(const std::string& filename, unsigned int timeout) {

	notify(DEBUG) << _("Locking file") << ": " << filename << std::endl;

	std::string jobfilelock = filename + ".lock";
	int fd = open(jobfilelock.c_str(), O_WRONLY|O_CREAT|O_EXCL);
	time_t start_time = time(NULL);
	while ((fd == -1) && (errno == EEXIST)) {
		if ((timeout >= 0) &&
		    (((unsigned int)(time(NULL) - start_time)) > timeout)) break;
		if (fd != -1) close(fd);
		notify(VERBOSE) << _("Waiting for file lock") << std::endl;
		usleep (10000);
		fd = open(jobfilelock.c_str(), O_WRONLY|O_CREAT|O_EXCL);
	}
	if (fd != -1) close(fd);
}


void UnlockFile(const std::string& filename) {

	notify(DEBUG) << _("Unlocking file") << ": " << filename << std::endl;

	std::string jobfilelock = filename + ".lock";
	remove(jobfilelock.c_str());
}


void MkDir(const std::string& dir, bool failifexists) {

	bool exists;
	std::string::size_type pos = 0;
	while (pos != std::string::npos) {
		pos = dir.find('/', pos + 1);
		exists = false;
		if (mkdir(dir.substr(0, pos).c_str(), 0777)) {
			if (errno == EEXIST)
				exists = true;
			else
				throw ARCLibError(strerror(errno) +
				                  (": " + dir.substr(0, pos)));
		}

		struct stat st;
		stat(dir.substr(0, pos).c_str(), &st);
		if (!S_ISDIR(st.st_mode))
			throw ARCLibError(_("Not a directory") +
			                  (": " + dir.substr(0, pos)));
	}

	if (exists && failifexists)
		throw ARCLibError(_("Directory already exists") + (": " + dir));
}


void RmDir(const std::string& dir, bool failifnotempty) {

	if (rmdir(dir.c_str()) != 0) {

		if (errno == ENOENT) return;

		if ((errno != ENOTEMPTY) || failifnotempty)
			throw ARCLibError(strerror(errno) + (": " + dir));

		DIR* d = opendir(dir.c_str());
		if (d == NULL)
			throw ARCLibError(strerror(errno) + (": " + dir));

		while (true) {
			dirent* de = readdir(d);
			if (de == NULL) break;
			std::string name = dir + "/" + de->d_name;
			struct stat st;
			if (lstat(name.c_str(), &st) != 0) {
				closedir(d);
				throw ARCLibError(strerror(errno) + (": " + name));
			}

			if (S_ISDIR(st.st_mode)) {
				try {
					RmDir(name, false);
				} catch (ARCLibError& e) {
					closedir(d);
					throw e;
				}
			} else {
				if (unlink(name.c_str()) != 0) {
					closedir(d);
					throw ARCLibError(strerror(errno) + (": " + name));
				}
			}
		}

		closedir(d);
		if (rmdir(dir.c_str()) != 0)
			throw ARCLibError(strerror(errno) + (": " + dir));
	}
}

int MakeTmpFile(std::string& filename) {
    // Find directory for temporary files
    const char* tmpdir = NULL;
    tmpdir = getenv("TMPDIR");
    if((!tmpdir) || (!tmpdir[0])) tmpdir = getenv("TMP");
    if((!tmpdir) || (!tmpdir[0])) tmpdir = getenv("TEMP");
    if((!tmpdir) || (!tmpdir[0])) tmpdir = "/tmp";

    // Make per-user directory for temporary files
    std::string dir = tmpdir;
    struct passwd pwd_;
    struct passwd *pwd;
    char buf[BUFSIZ];
    getpwuid_r(geteuid(),&pwd_,buf,BUFSIZ,&pwd);
    if(pwd && (pwd->pw_name)) {
        dir+="/"; dir+=pwd->pw_name;
        if(mkdir(dir.c_str(),S_IRWXU) != 0) {
            if(errno == EEXIST) {
                struct stat st;
                if((stat(dir.c_str(),&st) != 0) ||
                   (st.st_uid != geteuid())) {
                    dir=tmpdir;
                }
            } else {
                dir = tmpdir;
            }
        }
    }

    // Create a file
    filename = dir + "/" + filename + ".XXXXXX";
    return mkstemp((char*)filename.c_str());
}

TmpFile::TmpFile(const std::string& id) {
    filename = id;
    fd = MakeTmpFile(filename);
    if(fd == -1) filename.resize(0);
}

TmpFile::~TmpFile(void) {
    Destroy();
    Close();
}

int TmpFile::Open(void) {
    if(fd != -1) return fd;
    fd = open(filename.c_str(),O_RDWR);
    return fd;
}

void TmpFile::Close(void) {
    if(fd != -1) close(fd);
    fd = -1;
}

void TmpFile::Destroy(void) {
    if(!filename.empty()) unlink(filename.c_str());
}

const std::string& TmpFile::Name(void) {
    return filename;
}

