#ifndef ARCLIB_STRINGCONV
#define ARCLIB_STRINGCONV

#include <iomanip>
#include <sstream>
#include <string>
#include <typeinfo>

#include <arc/error.h>


/** Exception class thrown when a string-to-number or number-to-string
 *  conversion fails.
 */
class StringConvError : public ARCLibError {
	public:
		/** Standard exception class constructor. */
		StringConvError(std::string message) : ARCLibError(message) {}
};


std::string StringConvErrorString(const std::type_info& t, bool empty = false);


/** This method converts a string to any type -- throwing an exception
 *  if not possible.
 */
template<typename T>
T stringto(const std::string& s) throw(StringConvError) {
	T t;
	if (s.empty())
		throw StringConvError(StringConvErrorString(typeid(t), true));
	std::stringstream ss(s);
	ss >> t;
#if __GNUC__ < 3
	if (!ss.good())
#else
	if (!ss.eof())
#endif
		throw StringConvError(StringConvErrorString(typeid(t)) + ": " + s);
	return t;
}


#define stringtoi(A) stringto<int>((A))
#define stringtoui(A) stringto<unsigned int>((A))
#define stringtol(A) stringto<long>((A))
#define stringtoll(A) stringto<long long>((A))
#define stringtoul(A) stringto<unsigned long>((A))
#define stringtoull(A) stringto<unsigned long long>((A))
#define stringtof(A) stringto<float>((A))
#define stringtod(A) stringto<double>((A))
#define stringtold(A) stringto<long double>((A))


/** This method converts a long to any type of the width given. */
template<typename T>
std::string tostring(T t, const int width = 0) {
	std::stringstream ss;
	ss << std::setw(width) << t;
	return ss.str();
}


#endif // ARCLIB_STRINGCONV
