#ifndef ARCLIB_TIME
#define ARCLIB_TIME

#ifdef HAVE_CONFIG_H
#include "config.h" // to get HAVE_TIMEGM
#endif

#include <ctime>
#include <iostream>
#include <string>

#include <arc/error.h>

#ifndef HAVE_TIMEGM
time_t timegm (struct tm *tm);
#endif

/** Class to represent errors thrown by the Time class. */
class TimeError : public ARCLibError {
	public:
		/** Standard exception class constructor. */
		TimeError(std::string message) : ARCLibError(message) { }
};


/** An enumeration that contains the possible textual timeformats. */
enum TimeFormat {
	MDSTime,   // YYYYMMDDHHMMSSZ  (size 15)
	ASCTime,   // Day Month DD HH:MM:SS YYYY
	UserTime   // YYYY-MM-DD HH:MM:SS  (size 19)
};


/** A class for storing and manipulating times. */
class Time
{
	public:
		/** Default constructor. The time is put equal the current time. */
		Time();

		/** Constructor that takes a time_t variable and stores it. */
		Time(time_t);

		/**
		 * Constructor that tries to convert a string into a time_t variable
		 * using either one of the three TimeFormat's above.
		 */
		Time(std::string);

		/** sets the time */
		void SetTime(time_t);

		/** gets the time */
		time_t GetTime() const;

		/** Sets the TimeFormat to some format. */
		static void SetFormat(const TimeFormat&);

		/** Gets the TimeFormat. */
		static TimeFormat GetFormat();

		/** operator for setting a DateTime object equal to a time_t. */
		Time& operator= (time_t);

		/** Returns a string representation of the time -- after the time
			format wanted. */
		std::string str() const;

		/** Comparing two Time objects. */
		bool operator<(const Time&) const;

		/** Comparing two Time objects. */
		bool operator>(const Time&) const;

		/** Comparing two Time objects. */
		bool operator<=(const Time&) const;

		/** Comparing two Time objects. */
		bool operator>=(const Time&) const;

		/** Comparing two Time objects. */
		bool operator==(const Time&) const;

		/** Comparing two Time objects. */
		bool operator!=(const Time&) const;

	private:
		/** The time stored -- by default it is equal to the current time. */
		time_t gtime;

		/** The time-format stored. By default it is equal to CTime */
		static TimeFormat time_format;
};

/** Prints a Time-object to the given ostream -- typically cout. */
std::ostream& operator<<(std::ostream&, Time&);

/** Returns a time-stamp of the current time in some format. */
std::string TimeStamp(const TimeFormat& = UserTime);

/** Returns a time-stamp of some specified time in some format. */
std::string TimeStamp(Time, const TimeFormat& = UserTime);

enum PeriodBase {
	PeriodSeconds,
	PeriodMinutes,
	PeriodHours,
	PeriodDays,
	PeriodWeeks
};

/** Returns a textual representation of a period of seconds. */
std::string Period(unsigned long);

/** Converts a textual representation to a period of seconds. */
long Seconds(const std::string&,enum PeriodBase base = PeriodMinutes) throw(TimeError);

#endif // ARCLIB_TIME
