#ifndef ARCLIB_CONDITION
#define ARCLIB_CONDITION

#include <pthread.h>
#include <sys/time.h>

#include <cerrno>

#include <arc/common.h>

/** Common purpose conditional variable implemented as a template. 
 *  Can pass arbitrary objects from 'signal' to 'wait'. Well-suited for
 *  waiting for callbacks and reporting errors.
 */
template<class T> class Condition {
	public:
		/** Constructor with default timeout of 20 seconds. */
		Condition(int timeo = TIMEOUT*1000);

		/** Standard destructor. */
		~Condition();

		/** Reset the conditional variable. */
		void Reset();

		/** Lock the mutex. */
		void Block();

		/** Unlock the mutex. */
		void Unblock();

		/** Signal the conditional variable but don't lock the
		 *  mutex meanwhile.
		 */
		void SignalNonBlock(T val);

		/** Lock the mutex and signal the conditional variable with the
		 *  value passed. */
		void Signal(T val);

		/** Wait for the conditional variable to get signaled. */
		bool Wait(T& val);

		/** Wait timeo milliseconds for the conditional variable to get
		 *  signaled. Returns true if it gets signaled and false if not --
		 *  i.e. if there's a timeout. At the same time, the value val is
		 *  replaced by the signaled value.
		 */
		bool Wait(T& val, int timeo);

		/** Has the condition been signaled? */
		bool Check();

	private:
		/** The default timeout. */
		int timeout;

		/** mutex. */
		pthread_mutex_t lock;

		/** conditional variable. */
		pthread_cond_t cond;

		/** The value passed between signal and wait. */
		T value;

		/** Is the conditional variable signaled? */
		bool signaled;
};


template<class T>
Condition<T>::Condition(int timeo) {
	signaled = false;
	timeout = timeo;

	pthread_mutex_init(&lock, NULL);
	pthread_cond_init(&cond, NULL);
}


template<class T>
Condition<T>::~Condition() {
	pthread_cond_broadcast(&cond);
	pthread_cond_destroy(&cond);
	pthread_mutex_destroy(&lock);
}


template<class T>
void Condition<T>::Reset() {
	signaled = false;
}


template<class T>
void Condition<T>::Block() {
	pthread_mutex_lock(&lock);
}


template<class T>
void Condition<T>::Unblock() {
	pthread_mutex_unlock(&lock);
}


template<class T>
void Condition<T>::SignalNonBlock(T val) {
	if (!signaled) {
		value = val;
		signaled = true;
		pthread_cond_signal(&cond);
	}
}


template<class T>
void Condition<T>::Signal(T val) {
	Block();
	SignalNonBlock(val);
	Unblock();
}


template<class T>
bool Condition<T>::Wait(T& val) {
	return Wait(val, timeout);
}


template<class T>
bool Condition<T>::Wait(T& val, int timeo) {

	Block();
	if (timeo<0) {
		while (!signaled) {
			int err = pthread_cond_wait(&cond, &lock);
			if (err==EINTR) continue;
			if (err!=0) {
				Unblock();
				return false;
			}
		}
	} else {
		struct timeval curtime;
		gettimeofday(&curtime, NULL);

		struct timespec endtime;
		endtime.tv_sec = curtime.tv_sec + timeo/1000;
		endtime.tv_nsec = 1000*(curtime.tv_usec + (timeo%1000)*1000);
		endtime.tv_sec += endtime.tv_nsec/1000000000;
		endtime.tv_nsec = endtime.tv_nsec % 1000000000;

		while (!signaled) {
			int err = pthread_cond_timedwait(&cond, &lock, &endtime);
			if (err == EINTR) continue;
			if (err != 0) {
				Unblock();
				return false;
			}
		}
	}
    
	val = value;
	signaled = false;
	Unblock();
	return true;
}


template<class T>
bool Condition<T>::Check() {

	return signaled;
}

#endif // ARCLIB_CONDITION
