/**
 * Contains functionality which is common for ARCLib, and is believed
 * to be of general usefulness.
 *
 * It should not contain functionality that just didn't belong anywhere else.
 */

#ifndef ARCLIB_COMMON
#define ARCLIB_COMMON

#include <list>
#include <string>

#include <globus_error.h>

#define UNDEFINED -1
#define TIMEOUT 20

/**
 *  Version macro for defining module versions. A module can declare its
 *  version number like  #define ARCLIB_VERSION ARCVERSION(0,1,0)
 *  and clients can use it like
 *  if ARCLIB_VERSION > ARCVERSION(0,1,0)  etc.
 *
 *  Updates to modules can be reflected by bumping the version number
 *  according to the rule:
 *  c : Minor changes only. No API change.
 *  b : Minor API changes.
 *  a : Major API changes.
 */
#define ARCVERSION(a,b,c) (((a) << 16) + ((b) << 8) + (c))

/**
 * Returns a string containing the value of the specified environment
 * variable. In case that no such environment variable exist, the empty
 * string is return.
 */
std::string GetEnv(const std::string& var);

/**
 * Given a file name, this function return a list where each element contains
 * a line of file (in logical order).
 */
std::list<std::string> ReadFile(const std::string& filename);

/** Converts a globus error object to a string. */
std::string GlobusErrorString(globus_object_t*);

/** Creates the .lock file for the given file. */
void LockFile(const std::string& filename, unsigned int timeout = TIMEOUT);

/** Removes the .lock file for the given file. */
void UnlockFile(const std::string& filename);

/** Creates a directory recursively. If failifexists is true an exception
 *	is thrown if the directory already exists. */ 
void MkDir(const std::string& dir, bool failifexists);

/** Deletes a directory recursively. If failifnotempty is true an exception
 *	is thrown if the directory is not empty. */ 
void RmDir(const std::string& dir, bool failifnotempty);

/** Creaate temporary file in standard location (TMPDIR, TMP, TEMP or /tmp)
 *	under subdirectory named after username. File will have name 
 *  filename.XXXXXX. Full path to new file name is returned in filename 
 *  variable. */ 
int MakeTmpFile(std::string& filename);

class TmpFile {
	public:
        /** Constructor - creates temporary file and opens it. */
		TmpFile(const std::string& id);

		/** Destructor - closes and removes file. */
		~TmpFile(void);

		/** Opens (if not oleady open) file and returns handler. */
		int Open(void);

		/** Closes file (if open). */
		void Close(void);

		/** Removes file. */
		void Destroy(void);

		/** Returns full path to file. Empty if no file was created. */
		const std::string& Name(void);

	private:
		std::string filename;
		int fd;
};

#endif // ARCLIB_COMMON
