#ifndef ARCLIB_BROKER
#define ARCLIB_BROKER

#include <list>
#include <string>

#include <arc/target.h>
#include <arc/xrsl.h>

/**
 * Virtual base-class for the brokering-framework. A broker is simply
 * a method that takes a list of Targets and erases and sort in this list.
 * After brokering, the valid targets are left in the order for jobsubmission.
 */
class Broker {
	public:
		virtual ~Broker() { };

		/** This method performs the actual brokering. It is virtual and
		 *  should be implemented in all brokers.
		 */
		virtual void DoBrokering(std::list<Target>& targets) = 0;

	protected:
		/** This is a helper method of general usefulness to many brokers
		 *  that only depend on one Xrsl-attribute. It loops over all targets
		 *  and all subxrsls in the targets and calls RelationCheck with the
		 *  XrslRelation corresponding to the attribute. Brokers that only
		 *  depend on one attribute can thus call this method from
		 *  DoBrokering and implement RelationCheck instead. Note this
		 *  method is not virtual.
		 */
		void SetupAttributeBrokering(const std::string& attribute,
                                     std::list<Target>& targets);

		/** This method is a helper method that performs a sorting of
		 *  the targets after the criteria specified in the user-supplied
		 *  Compare() method. A standard sorting broker only needs to
		 *  implement Compare() and call SetupSorting.
		 */
		void SetupSorting(std::list<Target>& targets);

		/** This method is called from SetupBrokering and should return
		 *  true or false according whether the Target is acceptable or not.
		 *  The default version in broker.cpp always returns true and thus
		 *  does not do anything.
		 */
		virtual bool RelationCheck(Target& target, XrslRelation& rel);

		/** This method compares two targets and returns true or false
		 *  according to which target is the preferred one. A sorting broker
		 *  needs to implement this and call SetupSorting.
		 */
		virtual bool Compare(Target& target1, Target& target2);
};


/** This method performs the actual brokering. It accepts a list of Broker-
 * class-pointers and calls DoBrokering for each of the classes.
 */
void PerformBrokering(std::list<Broker*> brokers, std::list<Target>& targets);

#endif // ARCLIB_BROKER
