#!/usr/bin/env python
#
# Submit an xrsl file multiple times with given parameters
#

import sys
import thread
import os
import socket
import re
import utils
import time
import curses

def printJobStates(stdscr, problemSet, args):
    simple = args.simple
    stdscr.erase()
    stdscr.addstr("Status of current jobs:\n", curses.color_pair(0))

    for p in range(1, curses.COLORS):
        curses.init_pair(p, p, 0)


    for key in problemSet.processed:
        for el in problemSet.processed[key]:
            try:
                if el.status != "sent":
                    if (simple):
                        stdscr.addstr("X", curses.color_pair(2))
                    else:
                        stdscr.addstr("[", curses.color_pair(0))
                        stdscr.addstr("ok  ", curses.color_pair(2))
                        stdscr.addstr("] %s %02d\n" % (el.name, el.seed),
                            curses.color_pair(0))
                elif (time.time() - el.time) > el.timeout:
                    if (simple):
                        stdscr.addstr("X", curses.color_pair(1))
                    else:
                        stdscr.addstr("[", curses.color_pair(0))
                        stdscr.addstr("bad ", curses.color_pair(1))
                        stdscr.addstr("] %s %02d\n" % (el.name, el.seed),
                            curses.color_pair(0))
                else:
                    if (simple):
                        stdscr.addstr("X", curses.color_pair(3))
                    else:
                        stdscr.addstr("[", curses.color_pair(0))
                        stdscr.addstr("sent", curses.color_pair(3))
                        stdscr.addstr("] %s %02d\n" % (el.name, el.seed),
                            curses.color_pair(0))
            except curses.error:
                pass
    stdscr.refresh()


# Check the initial state of the jobs.  This could be a costly operation,
# so it is not done on every pass
def checkinitial(problemset, args, logwin):
    try:
        dirs = os.listdir(args.finals)
    except os.error:
        os.mkdir(os.path.join(args.finals))
        return

    dirs = filter(lambda x: (x[0] >= '0' and x[0] <= '9'), dirs)

    total_results = 0

    for dir in dirs:
        output = os.path.join(args.finals, dir, "std.out")

        if os.path.isfile(output):
            total_results += 1
            ms = utils.ms_parse(output)

            if ms.name in problemset.processed:
                found = False
                for el in problemset.processed[ms.name]:
                    if el.seed == ms.seed:
                        el.status = ms.status
                        found = True
                if not found:
                    pr = ProblemInstance(ms.name, ms.seed, time.time(), \
                        0, ms.status)
                    problemset.processed[ms.name].append(pr)
            else:
                name = ms.name
                seed = ms.seed
                t = time.time()
                timeout = 0
                status = ms.status
                pr = ProblemInstance(name, seed, t, timeout, status)
                problemset.processed[ms.name] = [pr]

    logwin.addstr("Updated %d existing results\n" % total_results)
    logwin.refresh()

# Given a downloaded directory, extract the information in the directory
# and store it in problemset

def checkseeds(dir, problemset, args, logwin):

    output = os.path.join(dir, "std.out")

    if os.path.isfile(output):

        logwin.addstr("Found job %s\n" % dir)

        ms = utils.ms_parse(output)
        if ms.name in problemset.processed:
            found = False
            for el in problemset.processed[ms.name]:
                if el.seed == ms.seed:
                    el.status = ms.status
                    found = True
            if not found:
                pr = ProblemInstance(ms.name, ms.seed, time.time(), \
                    0, ms.status)
                problemset.processed[ms.name].append(pr)
        else:
            name = ms.name
            seed = ms.seed
            t = time.time()
            timeout = 0
            status = ms.status
            pr = ProblemInstance(name, seed, t, timeout, status)
            problemset.processed[ms.name] = [pr]

        # Move the files away to keep the routine fast
        dirbase = output.split("/")[-2]
        os.rename(dir, os.path.join(args.finals, dirbase))

    else:
        logwin.addstr("No job in %s\n" % dir)

    logwin.refresh()

# Go through the problemset and remove jobs that have been sent but no
# result has been received before the timeout

def cleartimeouts(problemset, args, logwin):

    to_remove = []
    for key in problemset.processed:
        l = problemset.processed[key]
        for el in l:
            if el.status == "sent" and \
                (time.time() - el.time > el.timeout + 600):
                to_remove.append((key, el.seed))

    if len(to_remove) > 0:
        logwin.addstr("Clearing %d old jobs\n" % len(to_remove))
        logwin.refresh()

    for tuple in to_remove:
        key = tuple[0]
        seed = tuple[1]
        for idx in range(0, len(problemset.processed[key])):
            el = problemset.processed[key][idx]
            if el.seed == seed:
                del problemset.processed[key][idx]
                break # go to next tuple

class XrslBase:
    def __init__(self, template, timeout):
        self.template = template
        self.timeout = timeout
        self.uniq = 0 # A unique identifier for job names

    def xrslconv(self, file):
        filepath = os.path.dirname(file)
        basefile = os.path.basename(file)
        name = "%s_%02d_%d" % (basefile, self.uniq, self.uniq)
        self.uniq += 1
        return (self.template % \
            (self.timeout, self.uniq, basefile, name, basefile, filepath, \
                basefile, filepath, filepath, (self.timeout/60+1)), self.uniq)

class ProblemInstance:
    def __init__(self, name, seed, time, timeout, status):
        self.name = name
        self.seed = seed
        self.time = time
        self.timeout = timeout
        self.status = status


class ProblemSet:
    # A dictionary indexed by file names, containing ProblemInstances
    processed = {}
    def __init__(self, dir, repeat, xrslbase):
        self.dir = dir
        self.repeat = repeat
        self.xrslbase = xrslbase

    def getProblem(self):
        for file in os.listdir(self.dir):
            if not (file[-3:] == ".in"):
                continue
            fullpath = os.path.join(self.dir, file)
            if not (file in self.processed):
                self.processed[file] = []
            num_processed = len(self.processed[file])
            if num_processed < self.repeat:
                # xrslbase gives the seed
                (xrsl, seed) = self.xrslbase.xrslconv(fullpath)
                self.processed[file].append(ProblemInstance(file, seed,
                    time.time(), self.xrslbase.timeout, "sent"))
                return xrsl
        return ""

def communicate(stdscr, host, port, problemSet, args):
    stdscr.scrollok(True)

    logwin = curses.newwin(10, 70, 0, 0)
    logwin.scrollok(True)
    logwin.erase()

    statewin = curses.newwin(40, 80, 10, 0)
    statewin.scrollok(True)
    statewin.erase()

    freewin = curses.newwin(2, 10, 0, 70)
    freewin.scrollok(True)
    freewin.erase()

    checkinitial(problemSet, args, logwin) # A kludge
    printJobStates(statewin, problemSet, args)

    # AF_INET
    sock = socket.socket()
    sock.connect((host, port))

    # The data read from the socket
    data = ""

    # The outmost loop, where output from manager is processed
    while True:
        received = True
        while received:
            try:
                buf = sock.recv(4096)
            except socket.error, (errno, strerror):
                logwin.addstr("Socket error: (%d) %s\n" % (errno, strerror))
                logwin.refresh()
                time.sleep(1)
                pass
            data = "".join([data, buf])
            if len(buf) < 4096:
                received = False

        # At this point we assume we have the most up-to-date
        # information about the grid state, and act based on that;
        # Read commands from data until the last # free n message.

        curr_space = 0

        while len(data) > 0:
            matched = False
            mo = re.search("^#[ \t]*free[ \t]*([0-9][0-9]*)$", data, re.M)
            if mo:
                curr_space = int(mo.group(1))
                data = "".join([data[:mo.start()], data[mo.end():]])
                matched = True
                freewin.addstr("free %d\n" % curr_space)
                freewin.refresh()

            mo = re.search("^#[ \t]*dlfinish[ \t]*(.*)$", data, re.M)
            if mo:
                checkseeds(mo.group(1), problemSet, args, logwin)
                data = "".join([data[:mo.start()], data[mo.end():]])
                matched = True

            if not matched:
                # The rest of the string is nonsense
                data = ""

        if (curr_space > 0):
            logwin.addstr("Job request: ")
            xrsl_pr = problemSet.getProblem()
            if len(xrsl_pr) == 0:
                logwin.addstr("nothing to send!\n")
                pass
            else:
                sock.sendall(xrsl_pr + "\xff")
                logwin.addstr("sent a job\n")

        logwin.refresh()
        time.sleep(1)

        printJobStates(statewin, problemSet, args)
        cleartimeouts(problemSet, args, logwin)

        if len(buf) == 0:
            logwin.addstr("Lost connection\n")
            return
        data = "".join([data, buf])

class Arguments:
    def __init__(this, finals, simple):
        this.finals = finals
        this.simple = simple

if __name__ == '__main__':
    if len(sys.argv) != 9:
        print """
This is an example program for submitting las vegas -type algorithms
to grid.  The algorithms accept a seed, and you must specify a timeout
for them.  The timeout is a time by which the script assumes that the
job has disappeared and it is resubmitted.  This should be a fairly large
value in high-delay grids (say, in order of tens of minutes of hours).

xrsltemplate    a template xrsl file containing the basic form.
                You need to leave open the arguments, job name and input files.
                See the example

timeout         Specify here a large timeout

dir             Give the _full path_ to the input files.  Each must end in
                the suffix `.in'

name            This is the base of the name of the job to be sent.

maxseed         The maximum repetitions or the max seed value - 1

port            Port where gridjm is listening

finals          A path where to move the final results from the grid

simple | full   the visualization format.  Try and see.


  Usage: %s xrsltemplate timeout dir name maxseed port finals [simple | full]
""" % sys.argv[0]
        sys.exit(1)

    xrsl = sys.argv[1]
    timeout = int(sys.argv[2])
    dir = sys.argv[3]
    name = sys.argv[4]
    maxseed = int(sys.argv[5])
    port = int(sys.argv[6])
    ngdl = sys.argv[7]
    simple = False
    if (sys.argv[8] == "simple"):
        simple = True

    xrsltext = open(xrsl).read()
    xrslBase = XrslBase(xrsltext, timeout)
    problemSet = ProblemSet(dir, maxseed, xrslBase)
    args = Arguments(ngdl, simple)
    curses.wrapper(communicate, "localhost", port, problemSet, args)


